#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include <math.h>

#include "mgwtypes.h"
#include "arrange.h"
#include "interface.h"
#include "image.h"
#include "utils.h"

static MgwBorderProperty bprop;

static void
border_set_background (guchar *p)
{
  p[0] = bprop.background.red;
  p[1] = bprop.background.green;
  p[2] = bprop.background.blue;
}

static void
border_set_wooden (guchar *p)
{
  p[0] = bprop.wooden.red;
  p[1] = bprop.wooden.green;
  p[2] = bprop.wooden.blue;
}

static int
border_jagged (gint position, gint pitch)
{
  return abs (5 - 10 * (position % pitch) / (gdouble) pitch);
}

static int
border_random (gint position, gint pitch, gint p0)
{
  static guint count = 0;
  static gdouble phase[10];
  static gboolean first = TRUE;
  gdouble a = 0;
  gint i;

  if (first) {
    for (i = 0; i < 10; i++)
      phase[10] = 2 * M_PI * random () / RAND_MAX;
    first = FALSE;
  }

  for (i = 0; i < 10; i++)
    a += fabs (2 * cos (count / (gdouble) (pitch * (i + 3)) + phase[i]));

  if (++count == UINT_MAX)
    count = 0;

  return a - 8;
}

static int
border_ripple_1 (gint position, gint pitch)
{
  return 7 * (1 - fabs (sin (2 * M_PI * position / pitch)));
}

static int
border_ripple_2 (gint position, gint pitch)
{
  return 7 * (1 - cos (2 * M_PI * position / pitch));
}

static void
border_buffer_function (GdkPixbuf *pbuf, int (*bf) (), gint pitch)
{
  MgwSizeInfo si;
  gint i, j, k, k_border;
  gint h, w;
  gint xpitch, ypitch, nx, ny, xmod, ymod;
  guchar *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  h = si.h;
  w = si.w;

  if (pitch < 0) {		/* random */
    xpitch = -pitch;
    ypitch = -pitch;
    xmod = BORDER_FIX_WIDTH;
    ymod = BORDER_FIX_WIDTH;
  }
  else {
    nx = (w - 2 * BORDER_FIX_WIDTH) / pitch;
    ny = (h - 2 * BORDER_FIX_WIDTH) / pitch;
    xpitch = (w - 2 * BORDER_FIX_WIDTH) / nx;
    ypitch = (h - 2 * BORDER_FIX_WIDTH) / ny;
    xmod = (w - xpitch * nx) / 2;
    ymod = (h - ypitch * ny) / 2;
  }

  for (i = xmod; i < w - xmod; i++) {
    for (j = 0; j < ymod; j++) {
      border_set_background (p0 + i * si.nc +          j  * si.rs);
      border_set_background (p0 + i * si.nc + (h - 1 - j) * si.rs);
    }

    k_border = (*bf) (i - xmod, xpitch);
    for (k = 0; k < k_border; k++) {
      border_set_background (p0 + i * si.nc +         (k + ymod) * si.rs);
      border_set_background (p0 + i * si.nc + (h - 1 - k - ymod) * si.rs);
    }
  }

  for (j = ymod; j < h - ymod; j++) {
    for (i = 0; i < xmod; i++) {
      border_set_background (p0 +          i  * si.nc + j * si.rs);
      border_set_background (p0 + (w - 1 - i) * si.nc + j * si.rs);
    }

    k_border = (*bf) (j - ymod, ypitch);
    for (k = 0; k < k_border; k++) {
      border_set_background (p0 +         (k + xmod) * si.nc + j * si.rs);
      border_set_background (p0 + (w - 1 - k - xmod) * si.nc + j * si.rs);
    }
  }

  for (j = 0; j < ymod; j++) {
    for (i = 0; i < xmod; i++) {
      border_set_background (p0 +          i  * si.nc +          j  * si.rs);
      border_set_background (p0 + (w - 1 - i) * si.nc +          j  * si.rs);
      border_set_background (p0 +          i  * si.nc + (h - 1 - j) * si.rs);
      border_set_background (p0 + (w - 1 - i) * si.nc + (h - 1 - j) * si.rs);
    }
  }
}

static void
border_wooden (GdkPixbuf *pbuf, gint bw)
{
  MgwSizeInfo si;
  gint i, j, h, w;
  guchar *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  h = si.h;
  w = si.w;

  if (bw < 0)
    bw = MAX (h, w) / 50; /* border width */

  for (i = 0; i < bw; i++) {
    for (j = 0; j < w; j++) {
      border_set_wooden (p0 +          i  * si.rs + j * si.nc);
      border_set_wooden (p0 + (h - 1 - i) * si.rs + j * si.nc);
    }
  }

  for (i = bw; i < h - bw; i++) {
    for (j = 0; j < bw; j++) {
      border_set_wooden (p0 + i * si.rs +          j  * si.nc);
      border_set_wooden (p0 + i * si.rs + (w - 1 - j) * si.nc);
    }
  }
}

static void
border_gradation (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint bw, h, w;
  gint i, j, k;
  guchar bg[3];
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  h = si.h;
  w = si.w;

  bw = MAX (h, w) / 50; /* border width */

  bg[0] = bprop.background.red;
  bg[1] = bprop.background.green;
  bg[2] = bprop.background.blue;

  for (i = 0; i < h; i++) {
    for (j = 0; j < w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      if (i < bw)
	for (k = 0; k < 3; k++)
	  p[k] += (1.0 - (gdouble) i / (bw - 1)) * (bg[k] - p[k]);
      else if (i > h - bw - 1)
	for (k = 0; k < 3; k++)
	  p[k] += (1.0 - (gdouble) (h - i - 1) / (bw - 1)) * (bg[k] - p[k]);

      if (j < bw)
	for (k = 0; k < 3; k++)
	  p[k] += (1.0 - (gdouble) j / (bw - 1)) * (bg[k] - p[k]);
      else if (j > w - bw - 1)
	for (k = 0; k < 3; k++)
	  p[k] += (1.0 - (gdouble) (w - j - 1) / (bw - 1)) * (bg[k] - p[k]);
    }
  }
}

static void
border_transparent (GdkPixbuf *pbuf, gint bw)
{
  MgwSizeInfo si;
  gint grd;
  gint h, w;
  gint i, j, k;
  gint tmp;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);
  h = si.h;
  w = si.w;

  if (bw < 0)
    bw = MAX (h, w) / 50; /* border width */
  grd = 5;                /* gradation */

  for (i = bw; i < h - bw; i++) {
    for (j = 0; j < bw; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += j * grd;
	if (255 < tmp)
	  tmp = 255;
	p[k] = tmp;
      }

      p = p0 + i * si.rs + (j + w - bw) * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += (bw - j) * grd;
	if (255 < tmp)
	  tmp = 255;
	p[k] = tmp;
      }
    }
  }
  for (i = bw; i < h - bw; i++) {
    for (j = 0; j < 3; j++) {
      p = p0 + i * si.rs + (j + bw) * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k] - (3 - j) * 10;
	if (tmp < 0)
	  tmp = 0;
	p[k] = tmp;
      }

      p = p0 + i * si.rs + (w - j - bw) * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k] - (3 - j) * 10;
	if (tmp < 0)
	  tmp = 0;
	p[k] = tmp;
      }
    }
  }

  for (i = 0; i < bw; i++) {
    for (j = bw; j < w - bw; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += i * grd;
	if (255 < tmp)
	  tmp = 255;
	p[k] = tmp;
      }

      p = p0 + (i + h - bw) * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += (bw - i) * grd;
	if (255 < tmp)
	  tmp = 255;
	p[k] = tmp;
      }
    }
  }

  for (i = 0; i < 3; i++) {
    for (j = bw; j < w - bw; j++) {
      p = p0 + (bw + i) * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k] - (3 - i) * 10;
	if (tmp < 0)
	  tmp = 0;
	p[k] = tmp;
      }

      p = p0 + (h - bw - i) * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k] - (3 - i) * 10;
	if (tmp < 0)
	  tmp = 0;
	p[k] = tmp;
      }
    }
  }
  for (i = 0; i < bw; i++) {
    for (j = 0; j < bw; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += i * grd + j * grd - bw * grd;
	p[k] = limit_byte_range (tmp);
      }

      p = p0 + i * si.rs + (j + w - bw) * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += i * grd + (bw - j) * grd - bw * grd;
	p[k] = limit_byte_range (tmp);
      }

      p = p0 + (i + h - bw) * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += (bw - i) * grd + j * grd - bw * grd;
	p[k] = limit_byte_range (tmp);
      }

      p = p0 + (i + h - bw) * si.rs + (j + w - bw) * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += (bw - i) * grd + (bw - j) * grd - bw * grd;
	p[k] = limit_byte_range (tmp);
      }
    }
  }
}

static void
border_shade (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  gint tmp;
  gint gap = 50, bw;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  bw = MAX (si.h, si.w) / 50; /* border width */

  for (i = 0; i < bw; i++) {
    for (j = i; j < si.w - i; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += gap;
	if (tmp > 255)
	  tmp = 255;
	p[k] = tmp;
      }

      p = p0 + (si.h - i - 1) * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp -= gap;
	if (tmp < 0)
	  tmp = 0;
	p[k] = tmp;
      }
    }
  }

  for (i = 0; i < bw; i++) {
    for (j = i; j < si.h - i; j++) {
      p = p0 + j * si.rs + i * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp += gap;
	if (tmp > 255)
	  tmp = 255;
	p[k] = tmp;
      }

      p = p0 + j * si.rs + (si.w - i - 1) * si.nc;
      for (k = 0; k < 3; k++) {
	tmp = p[k];
	tmp -= gap;
	if (tmp < 0)
	  tmp = 0;
	p[k] = tmp;
      }
    }
  }
}

static void
glassblock_horizontal (GdkPixbuf *pbuf, gint pitch, gint gradation)
{
  MgwSizeInfo si;
  gdouble fac;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    fac = (gdouble) (pitch - i % pitch - 1) / (pitch - 1) - 0.5;
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	p[k] = limit_byte_range (p[k] + gradation * fac);
      }
    }
  }
}

static void
glassblock_vertical (GdkPixbuf *pbuf, gint pitch, gint gradation)
{
  MgwSizeInfo si;
  gdouble fac;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (j = 0; j < si.w; j++) {
    fac = (gdouble) (pitch - j % pitch - 1) / (pitch - 1) - 0.5;
    for (i = 0; i < si.h; i++) {
      p = p0 + i * si.rs + j * si.nc;
      for (k = 0; k < 3; k++) {
	p[k] = limit_byte_range (p[k] + gradation * fac);
      }
    }
  }
}

static void
glassblock (GdkPixbuf *pbuf, gint pitch, gint gradation)
{
  glassblock_horizontal (pbuf, pitch, gradation);
  glassblock_vertical (pbuf, pitch, gradation);
}

static gint
corrugated_pattern_drift (gint i, gint p)
{
  static gint shift = 0;
  gint modulo;

  if (i % p == p - 1)
    shift = 3.0 * random () / RAND_MAX;

  modulo = (i + shift) % p;

  if (modulo == 0)
    return (gint) (50 - 100.0 * random () / RAND_MAX);
  else if (modulo == 1)
    return (gint) (15 - 50.0 * random () / RAND_MAX);
  else if (modulo == p - 2)
    return (gint) (35 - 50.0 * random () / RAND_MAX);
  else if (modulo == p - 1)
    return (gint) (100.0 * random () / RAND_MAX);
  else
    return 25 - 50.0 * random () / RAND_MAX;
}

static void
corrugated_vertical (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint drift;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      p = p0 + i * si.rs + j * si.nc;
      drift = corrugated_pattern_drift (j, 10);
      for (k = 0; k < 3; k++) {
	p[k] = limit_byte_range (p[k] + drift);
      }
    }
  }
}

static void
corrugated_horizontal (GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint drift;
  gint i, j, k;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (j = 0; j < si.w; j++) {
    for (i = 0; i < si.h; i++) {
      p = p0 + i * si.rs + j * si.nc;
      drift = corrugated_pattern_drift (i, 10);
      for (k = 0; k < 3; k++) {
	p[k] = limit_byte_range (p[k] + drift);
      }
    }
  }
}

static void
corrugated (GdkPixbuf *pbuf)
{
  corrugated_horizontal (pbuf);
  corrugated_vertical (pbuf);
}

static void
mosaic (gint pitch, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k;
  gint ref, ref_x, ref_y;
  guchar *p2, *p1, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  for (i = 0; i < si.h; i++) {
    for (j = 0; j < si.w; j++) {
      ref_y = (i / pitch) * pitch + pitch / 2;
      ref_x = (j / pitch) * pitch + pitch / 2;

      if (si.w - 1 < ref_x)
	ref_x = si.w - 1;
      if (si.h - 1 < ref_y)
	ref_y = si.h - 1;

      p1 = p0 +     i * si.rs +     j * si.nc;
      p2 = p0 + ref_y * si.rs + ref_x * si.nc;

      for (k = 0; k < 3; k++)
	p1[k] = p2[k];
    }
  }
}

static void
puzzle (gint nx, gint ny, GdkPixbuf *pbuf)
{
  MgwSizeInfo si;
  gint i, j, k, iii, jjj, dx, dy, ic, jc, idy, jdx, q;
  gint *rseq = NULL;
  guchar *tmp = NULL;
  guchar *p, *p0 = gdk_pixbuf_get_pixels (pbuf);

  get_size_info (&si, pbuf);

  if (!(tmp = stock_rgb_data (pbuf, si))) {
    g_warning ("puzzle(arrange.c): memory allocation failed(a)");
    goto FUNC_END;
  }

  dx = si.w / nx;
  dy = si.h / ny;

  if (!(rseq = (gint *) g_malloc (sizeof (gint) * nx * ny))) {
    g_warning ("puzzle(arrange.c): memory allocation failed(b)");
    goto FUNC_END;
  }

  for (i = 0; i < nx * ny; i++)
    rseq[i] = i;
  for (i = 0; i < nx * ny; i++) {
    j = nx * ny * ((gdouble) random () / RAND_MAX);
    swap (rseq + i, rseq + j);
  }

  for (ic = 0; ic < ny; ic++) {
    for (jc = 0; jc < nx; jc++) {
      for (idy = 0; idy < dy; idy++){
	for (jdx =  0; jdx < dx; jdx++) {
	  i = idy + ic * dy;
	  j = jdx + jc * dx;
	  p = p0 + i * si.rs + j * si.nc;

	  iii = idy + (rseq[ic * nx + jc] / ny) * dy;
	  jjj = jdx + (rseq[ic * ny + jc] % ny) * dx;
	  q = 3 * (iii * si.w + jjj);

	  for (k = 0; k < 3; k++) {
	    p[k] = tmp[q + k];
	  }
	}
      }
    }
  }

 FUNC_END:
  g_free (rseq);
  g_free (tmp);
}

static void
background_color_ok_button_clicked (GtkWidget * w, gpointer data)
{
  GtkColorSelectionDialog *cs;
  gdouble color[4];

  cs = GTK_COLOR_SELECTION_DIALOG (data);
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (cs->colorsel), color);

  bprop.background.red = (guchar) (color[0] * 0xFF);
  bprop.background.green = (guchar) (color[1] * 0xFF);
  bprop.background.blue = (guchar) (color[2] * 0xFF);

  gtk_grab_remove (data);
  gtk_widget_destroy (GTK_WIDGET (cs));
}

static void
background_color_cancel_button_clicked (GtkWidget * w, gpointer data)
{
  gtk_grab_remove (data);
  gtk_widget_destroy (data);
}

static void
wooden_color_ok_button_clicked (GtkWidget * w, gpointer data)
{
  GtkColorSelectionDialog *cs;
  gdouble color[4];

  cs = GTK_COLOR_SELECTION_DIALOG (data);
  gtk_color_selection_get_color (GTK_COLOR_SELECTION (cs->colorsel), color);

  bprop.wooden.red = (guchar) (color[0] * 0xFF);
  bprop.wooden.green = (guchar) (color[1] * 0xFF);
  bprop.wooden.blue = (guchar) (color[2] * 0xFF);

  gtk_grab_remove (data);
  gtk_widget_destroy (GTK_WIDGET (cs));
}

static void
wooden_color_cancel_button_clicked (GtkWidget * w, gpointer data)
{
  gtk_grab_remove (data);
  gtk_widget_destroy (data);
}

static void
border_color_select_activate (GtkWidget * w, gpointer data)
{
  GtkWidget *win;
  gchar *which = (gchar *) data;
  gdouble prev[4];

  if (!strcmp (which, "wooden")) {
    prev[0] = (gdouble) bprop.wooden.red   / 0xFF;
    prev[1] = (gdouble) bprop.wooden.green / 0xFF;
    prev[2] = (gdouble) bprop.wooden.blue  / 0xFF;
  }
  else if (!strcmp (which, "background")) {
    prev[0] = (gdouble) bprop.background.red   / 0xFF;
    prev[1] = (gdouble) bprop.background.green / 0xFF;
    prev[2] = (gdouble) bprop.background.blue  / 0xFF;
  }
  else {
    g_warning (_("border_color_select_activarte(arrange.c): internal error(which=%s)"), which);
  }
  prev[3] = 1;

  win = gtk_color_selection_dialog_new (_("Color select"));

  gtk_color_selection_set_color (GTK_COLOR_SELECTION
				 ((GTK_COLOR_SELECTION_DIALOG (win))->
				  colorsel), prev);
  gtk_color_selection_set_color (GTK_COLOR_SELECTION
				 ((GTK_COLOR_SELECTION_DIALOG (win))->
				  colorsel), prev);

  if (!strcmp (which, "wooden")) {
    gtk_signal_connect (GTK_OBJECT
			(GTK_COLOR_SELECTION_DIALOG (win)->ok_button),
			"clicked",
			GTK_SIGNAL_FUNC (wooden_color_ok_button_clicked),
			win);
    gtk_signal_connect (GTK_OBJECT
			(GTK_COLOR_SELECTION_DIALOG (win)->cancel_button),
			"clicked",
			GTK_SIGNAL_FUNC
			(wooden_color_cancel_button_clicked), win);
  }
  else if (!strcmp (which, "background")) {
    gtk_signal_connect (GTK_OBJECT
			(GTK_COLOR_SELECTION_DIALOG (win)->ok_button),
			"clicked",
			GTK_SIGNAL_FUNC
			(background_color_ok_button_clicked), win);
    gtk_signal_connect (GTK_OBJECT
			(GTK_COLOR_SELECTION_DIALOG (win)->cancel_button),
			"clicked",
			GTK_SIGNAL_FUNC
			(background_color_cancel_button_clicked), win);
  }

  gtk_widget_show (win);
  gtk_grab_add (win);
}

/* Entry point */
void
on_border_background_color_set_activate (void)
{
  border_color_select_activate (NULL, "background");
}

void
on_border_wooden_color_set_activate (void)
{
  border_color_select_activate (NULL, "wooden");
}

void
arrange_exec (GtkWidget *dummy, gchar *name)
{
  MgwImageInfo *mii = get_image_info ();
  MgwClipRect c;
  GdkPixbuf *pbuf, *org;
  gdouble zoom_factor;
  gchar buff[64];
  gboolean has_clip, now_executing = FALSE, fail = FALSE;

  if (mii->handling)
    return;
  else
    mii->handling = TRUE;

  if (!(org = get_pbuf ()))
    return;

  set_cursor (GDK_WATCH);
  menubar_hilite_control (LOCK);
  sprintf (buff, "%s: EXECUTING...", name);
  put_string_to_appbar (buff);

  if (has_clip = get_clip_rect_area (&c)) {
    pbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, c.x1, c.y1);
    gdk_pixbuf_copy_area (org, c.x0, c.y0, c.x1, c.y1, pbuf, 0, 0);
  } else {
    pbuf = org;
  }

  zoom_factor = mii->zoom;
  view_zoom (NULL, "100%");

  switch (name[0]) {
  case 'B':
    if (!strcmp (name, "Border Shade"))
      border_shade (pbuf);
    else if (!strcmp (name, "Border Transparent"))
      border_transparent (pbuf, -1);
    else if (!strcmp (name, "Border Gradation"))
      border_gradation (pbuf);
    else if (!strcmp (name, "Border Ripple-1"))
      border_buffer_function (pbuf, border_ripple_1, 50);
    else if (!strcmp (name, "Border Ripple-2"))
      border_buffer_function (pbuf, border_ripple_2, 40);
    else if (!strcmp (name, "Border Random-1"))
      border_buffer_function (pbuf, border_random, -3);
    else if (!strcmp (name, "Border Random-2"))
      border_buffer_function (pbuf, border_random, -6);
    else if (!strcmp (name, "Border Jagged-1"))
      border_buffer_function (pbuf, border_jagged, 25);
    else if (!strcmp (name, "Border Jagged-2"))
      border_buffer_function (pbuf, border_jagged, 10);
    else if (!strcmp (name, "Border Wooden-1")) {
      border_wooden (pbuf, -1);
      border_transparent (pbuf, -1);
    }
    else if (!strcmp (name, "Border Wooden-2")) {
      gint w;

      for (w = 25; w > 0; w -=5) {
	border_wooden (pbuf, w);
	border_transparent (pbuf, w);
      }
    }
    else fail = TRUE;
    break;

  case 'C':
    if (!strcmp (name, "Corrugated"))
      corrugated (pbuf);
    else if (!strcmp (name, "Corrugated_horizontal"))
      corrugated_horizontal (pbuf);
    else if (!strcmp (name, "Corrugated_vertical"))
      corrugated_vertical (pbuf);
    else fail = TRUE;
    break;

  case 'G':
    if (!strcmp (name, "Glassblock"))
      glassblock (pbuf, 10, 50);
    else if (!strcmp (name, "Glassblock_horizontal"))
      glassblock_horizontal (pbuf, 10, 50);
    else if (!strcmp (name, "Glassblock_vertical"))
      glassblock_vertical (pbuf, 10, 50);
    else fail = TRUE;
    break;

  case 'M':
    if (!strcmp (name, "Mosaic"))
      mosaic (20, pbuf);
    else fail = TRUE;
    break;

  case 'P':
    if (!strcmp (name, "Puzzle"))
      puzzle (5, 5, pbuf);
    else fail = TRUE;
    break;

  default:
    fail = TRUE;
    break;
  }

  if (!fail) {
    if (has_clip) {
      gdk_pixbuf_copy_area (pbuf, 0, 0, c.x1, c.y1, org, c.x0, c.y0);
      g_object_unref (pbuf);
    }
    sprintf (buff, "%d", (gint) (zoom_factor * 100. + 0.5));
    show_progress (0, org);
    view_zoom (NULL, buff);
    draw_clip_rect_area ();

    sprintf (buff, "%s: DONE", name);
    put_string_to_appbar (buff);
    image_changed ();
  } else {
    g_warning ("arrange_exec: FALIED(Internal error): %s", name);
    sprintf (buff, "%s: FAILED(Internal error)", name);
    put_string_to_appbar (buff);
  }

  set_cursor (GDK_TOP_LEFT_ARROW);
  menubar_hilite_control (UNLOCK);
  mii->handling = FALSE;
}

MgwBorderProperty *
get_border_property (void)
{
  return &bprop;
}
