#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "mgwtypes.h"
#include "interface.h"
#include "image.h"
#include "filters.h"
#include "utils.h"

#include "math.h"

static MgwImageInfo mii = {
  FALSE, FALSE, FALSE, FALSE
};

static void
cut_clip_rect (void)
{
  MgwClipRect c;
  GdkPixbuf *p;

  if (!get_clip_rect_area (&c))
    return;

  p = gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, c.x1, c.y1);
  gdk_pixbuf_copy_area (mii.org, c.x0, c.y0, c.x1, c.y1, p, 0, 0);
  update_pbuf (p);
  image_changed ();
}

/* Entry point */
void
view_zoom (GtkWidget *dummy, gchar *name)
{
  MgwClipRect c;
  gfloat f, p;

  f = (gdouble) atoi (name) / 100.;
  p = mii.zoom;
  mii.zoom   = f;
  mii.width  = f * gdk_pixbuf_get_width  (mii.org);
  mii.height = f * gdk_pixbuf_get_height (mii.org);

  if ((mii.pbuf != NULL) && (mii.pbuf != mii.org))
    g_object_unref (mii.pbuf);

  mii.pbuf = gdk_pixbuf_scale_simple (mii.org, mii.width, mii.height,
				      GDK_INTERP_BILINEAR);

  if (mii.has_clip) {
    mii.clip.x0 = f * (mii.clip.x0 - mii.xmargin) / p + mii.xmargin;
    mii.clip.y0 = f * (mii.clip.y0 - mii.ymargin) / p + mii.ymargin;
    mii.clip.x1 = f * mii.clip.x1 / p;
    mii.clip.y1 = f * mii.clip.y1 / p;
  }

  change_drawing_area_size (mii.width, mii.height);
}

void
shrink_wrap (void)
{
  if (mii.pbuf)
    change_drawing_area_size (mii.width, mii.height);
}

void
image_changed (void)
{
  mii.changed = TRUE;
  set_window_title (mii.filename, mii.width, mii.height, mii.changed);
}

void
image_info (GtkMenuItem *menuitem, gpointer user_data)
{
  gdouble ave = 0., gs = 0., sd = 0.;
  gint i, j, k, n;
  gint w = gdk_pixbuf_get_width (mii.pbuf);
  gint h = gdk_pixbuf_get_height (mii.pbuf);
  gint rs = gdk_pixbuf_get_rowstride (mii.pbuf);
  gint nc = gdk_pixbuf_get_n_channels (mii.pbuf);
  guchar *p, *p0 = gdk_pixbuf_get_pixels (mii.pbuf);
  gchar buff[1024];

  set_cursor (GDK_WATCH);
  redraw_image (mii.pbuf);

  n = 3 * w * h;

  for (i = 0; i < h; i++) {
    for (j = 0; j < w; j++) {
      p = p0 + i * rs + j * nc;
      gs = 0.0;
      for (k = 0; k < 3; k++) {
	gs += (gdouble) p[k];
      }
      ave += gs / n;
      gs /= 3;
      for (k = 0; k < 3; k++) {
	sd += pow (p[k] - gs, 2) / n;
      }
    }
  }

  sprintf (buff, "Image Information\n filename: %s\n (W x H): (%d x %d)\n format: %s\n description: %s\n writable: %d\n\nLevel average: %.1f (SD: %.1f)",
	   mii.filename, mii.width, mii.height,
	   gdk_pixbuf_format_get_name (mii.pfmt),
	   gdk_pixbuf_format_get_description (mii.pfmt),
	   (gint) gdk_pixbuf_format_is_writable (mii.pfmt),
	   ave, sqrt (sd));
  set_cursor (GDK_TOP_LEFT_ARROW);
  show_message_dialog (NULL, buff);
}

gboolean
get_clip_rect_area (MgwClipRect *cr)
{
  if (!mii.has_clip)
    return FALSE;

  /* X-range check */
  if ((mii.clip.x0 + mii.clip.x1 < mii.xmargin) ||
      ((mii.clip.x0 > mii.width + mii.xmargin)
       && (mii.clip.x0 + mii.clip.x1 > mii.width + mii.xmargin)))
    return FALSE;

  /* Y-range check */
  if ((mii.clip.y0 + mii.clip.y1 < mii.ymargin) ||
      ((mii.clip.y0 > mii.height + mii.ymargin)
       && (mii.clip.y0 + mii.clip.y1 > mii.height + mii.ymargin)))
    return FALSE;

  /* decide X */
  cr->x0 = MAX (0, mii.clip.x0 - mii.xmargin);
  cr->x1 = MIN (MIN (mii.clip.x1, mii.clip.x1 - (mii.xmargin - mii.clip.x0)),
		mii.width - cr->x0);
  if (cr->x1 < 0) {
    cr->x0 = MAX (0, cr->x0 + cr->x1);
    cr->x1 = MIN (-cr->x1, mii.width - cr->x0);
  }

  /* decide Y */
  cr->y0 = MAX (0, mii.clip.y0 - mii.ymargin);
  cr->y1 = MIN (MIN (mii.clip.y1, mii.clip.y1 - (mii.ymargin - mii.clip.y0)),
		mii.height - cr->y0);
  if (cr->y1 < 0) {
    cr->y0 = MAX (0, cr->y0 + cr->y1);
    cr->y1 = MIN (-cr->y1, mii.height - cr->y0);
  }

  cr->x0 /= mii.zoom;
  cr->y0 /= mii.zoom;
  cr->x1 /= mii.zoom;
  cr->y1 /= mii.zoom;

  return TRUE;
}

gboolean
key_release_event (GtkWidget *widget, GdkEventKey *event)
{
  set_cursor (GDK_TOP_LEFT_ARROW);
}

gboolean
key_press_event (GtkWidget *widget, GdkEventKey *event)
{
  if (event->state & GDK_SHIFT_MASK) {
    switch (event->keyval) {
    case GDK_Page_Down:
      mii.xscroll += 20;
      scroll_image_horizontal (mii.pbuf, mii.xscroll, mii.width, mii.height);
      break;

    case GDK_Page_Up:
      mii.xscroll -= 20;
      scroll_image_horizontal (mii.pbuf, mii.xscroll, mii.width, mii.height);
      break;
    }
  } else {
    switch (event->keyval) {
    case GDK_Page_Down:
      mii.yscroll += 20;
      scroll_image_vertical (mii.pbuf, mii.yscroll, mii.width, mii.height);
      break;

    case GDK_Page_Up:
      mii.yscroll -= 20;
      scroll_image_vertical (mii.pbuf, mii.yscroll, mii.width, mii.height);
      break;

    case GDK_BackSpace:
      revert_image ();
      break;

    case GDK_Escape:
      filter_break_execution ();
      break;

    case GDK_Control_L:
    case GDK_Control_R:
      set_cursor (GDK_FLEUR);
      break;

    default:
      break;
    }
  }

  return FALSE;
}

gboolean
on_da_configure_event (GtkWidget *widget, GdkEventConfigure *event)
{
  if (mii.has_clip) {
    mii.clip.x0 -= mii.xmargin;
    mii.clip.y0 -= mii.ymargin;
  }

  mii.xmargin = (widget->allocation.width - mii.width) / 2;
  mii.ymargin = (widget->allocation.height - mii.height) / 2;

  if (mii.has_clip) {
    mii.clip.x0 += mii.xmargin;
    mii.clip.y0 += mii.ymargin;
  }
}

gboolean
on_da_expose_event (GtkWidget *widget, GdkEventExpose *event)
{
  gint src_x, src_y, w, h;

  if (!mii.pbuf)
    return TRUE;

  src_x = MAX (0, event->area.x - mii.xmargin);
  src_y = MAX (0, event->area.y - mii.ymargin);
  w = MIN (mii.width - src_x,
	   (widget->allocation.width + mii.width) / 2 - event->area.x);
  h = MIN (mii.height - src_y,
	   (widget->allocation.height + mii.height) / 2 - event->area.y);
  
  if (w > 0 && h >0)
    gdk_draw_pixbuf (widget->window,
		     widget->style->fg_gc[GTK_WIDGET_STATE (widget)],
		     mii.pbuf,
		     src_x, src_y,
		     MAX (event->area.x, mii.xmargin),
		     MAX (event->area.y, mii.ymargin),
		     w, h,
		     GDK_RGB_DITHER_NORMAL, 0, 0);
  draw_clip_rect_area ();
  return TRUE;
}

gboolean
on_da_button_press_event (GtkWidget *widget, GdkEventButton *event)
{
  if (!mii.pbuf)
    return TRUE;

  switch (event->button) {
  case 1:
    if (event->state & GDK_CONTROL_MASK) {
      mii.drifting = TRUE;
      mii.xdrift += event->x;
      mii.ydrift += event->y;
      return TRUE;
    }

    if (mii.has_clip)
      draw_clip_rect (mii.clip.x0, mii.clip.y0, mii.clip.xp, mii.clip.yp);
    else
      mii.has_clip = TRUE;

    mii.clip.x0 = (gint) event->x;
    mii.clip.y0 = (gint) event->y;
    mii.clip.x1 = mii.clip.xp = 0;
    mii.clip.y1 = mii.clip.yp = 0;
    break;

  case 2:
    cut_clip_rect ();
    break;

  case 3:
    break;

  default:
    break;
  }
  return TRUE;
}

gboolean
on_da_motion_notify_event (GtkWidget *widget, GdkEventMotion *event)
{
  MgwClipRect cr;
  GdkModifierType state;
  gint x, y;
  gchar buff[20];

  if (!mii.pbuf)
    return TRUE;

  if (event->is_hint) {
    gint xxx, yyy;

    gdk_window_get_pointer (event->window, &xxx, &yyy, &state);
    x = xxx;
    y = yyy;
  } else {
    x = event->x;
    y = event->x;
    state = event->state;
  }

  if (mii.drifting) {
    drift_image (mii.pbuf,
		 x - mii.xdrift, y - mii.ydrift, mii.width, mii.height);
  } else {
    mii.clip.x1 = x - mii.clip.x0;
    mii.clip.y1 = y - mii.clip.y0;
    if (get_clip_rect_area (&cr)) {
      sprintf (buff, "Area: (%d, %d)-(%d x %d)", cr.x0, cr.y0, cr.x1, cr.y1);
      put_string_to_appbar (buff);
      draw_clip_rect (mii.clip.x0, mii.clip.y0, mii.clip.xp, mii.clip.yp);
      draw_clip_rect (mii.clip.x0, mii.clip.y0, mii.clip.x1, mii.clip.y1);
      mii.clip.xp = mii.clip.x1;
      mii.clip.yp = mii.clip.y1;
    }
  }

  return TRUE;
}

gboolean
on_da_button_release_event (GtkWidget *widget, GdkEventButton *event)
{
  if (!mii.pbuf)
    return TRUE;

  if (mii.drifting) {
    mii.xdrift -= event->x;
    mii.ydrift -= event->y;
    mii.drifting = FALSE;
  } else {
    if (mii.has_clip && abs (mii.clip.x1) < 20 && abs (mii.clip.y1) < 20) {
      if (mii.pbuf)
	redraw_image (mii.pbuf);
      put_string_to_appbar ("Area: CANCEL");
      mii.has_clip = FALSE;
    }
  }

  return TRUE;
}

gboolean
on_da_scroll_event (GtkWidget *widget, GdkEventScroll *event, gpointer data)
{
  if (!mii.pbuf)
    return TRUE;

  switch (event->direction) {
  case GDK_SCROLL_UP:
    mii.yscroll -= 20;
    scroll_image_vertical (mii.pbuf, mii.yscroll, mii.width, mii.height);
    break;
  case GDK_SCROLL_DOWN:
    mii.yscroll += 20;
    scroll_image_vertical (mii.pbuf, mii.yscroll, mii.width, mii.height);
    break;
  case GDK_SCROLL_LEFT:
    mii.xscroll -= 20;
    scroll_image_horizontal (mii.pbuf, mii.xscroll, mii.width, mii.height);
    break;
  case GDK_SCROLL_RIGHT:
    mii.xscroll += 20;
    scroll_image_horizontal (mii.pbuf, mii.xscroll, mii.width, mii.height);
    break;

  default:
    break;
  }

  return TRUE;
}

gboolean
on_da_leave_event (GtkWidget *widget)
{
  if (!mii.pbuf)
    return TRUE;

  if (mii.yscroll != 0 || mii.xscroll != 0) {
    mii.yscroll = 0;
    mii.xscroll = 0;
    scroll_image_horizontal (mii.pbuf, mii.xscroll, mii.width, mii.height);
  }
  if (mii.xdrift != 0 || mii.ydrift != 0) {
    mii.xdrift = 0;
    mii.ydrift = 0;
    drift_image (mii.pbuf,
		 mii.xmargin, mii.ymargin, mii.width, mii.height);
  }

  return TRUE;
}

void
revert_image (void)
{
  put_string_to_appbar (_("Revert"));
  display_image_file (mii.filename);
}

void
save_current_image_as (gchar *filename)
{
  gchar *type = gdk_pixbuf_format_get_name (mii.pfmt);
  gchar buff[128];
  gboolean rst;

  if (!mii.pbuf || !gdk_pixbuf_format_is_writable (mii.pfmt)) {
    sprintf (buff, "Image can't be saved: %s", filename);
    show_error_dialog(NULL, buff);
    return;
  }

  if (!strcmp ("OVERWRITE", filename))
    filename = mii.filename;

  if (!strcmp ("jpeg", type)) {
    rst = gdk_pixbuf_save (mii.pbuf, filename, "jpeg",
		     NULL, "quality", "85", NULL);
  } else {
    rst = gdk_pixbuf_save (mii.pbuf, filename, type, NULL, NULL);
  }

  if (rst == TRUE) {
    strcpy (mii.filename, filename);
    mii.changed = FALSE;
    set_window_title (mii.filename, mii.width, mii.height, mii.changed);
    sprintf (buff, "Saved: %s", filename);
    put_string_to_appbar (buff);
  } else {
    sprintf (buff, "Image save FAILED: %s", filename);
    show_error_dialog(NULL, buff);
  }
}

void
display_image_file (gchar *filename)
{
  GdkPixbuf *p;
  gint w0, h0;
  gchar buff[128];

  if (!(p = gdk_pixbuf_new_from_file (filename, NULL))) {
    sprintf (buff, "Can't load image: %s", filename);
    show_error_dialog (NULL, buff);
    return;
  }

  mii.pfmt = gdk_pixbuf_get_file_info (filename, &w0, &h0);
  strcpy (mii.filename, filename);
  update_pbuf (p);
  menubar_hilite_control (IMAGE_ON);
  sprintf (buff, "Loaded: %s", filename);
  put_string_to_appbar (buff);
}

void
update_pbuf (GdkPixbuf *p)
{
  if (p == mii.pbuf)
    return;

  if (mii.pbuf)
    g_object_unref (mii.pbuf);

  mii.pbuf = p;
  mii.org  = p;
  mii.zoom = 1.0;
  mii.width = gdk_pixbuf_get_width (mii.pbuf);
  mii.height = gdk_pixbuf_get_height (mii.pbuf);

  mii.xscroll  = mii.yscroll = 0;
  mii.xdrift   = mii.ydrift = 0;
  mii.changed  = FALSE;
  mii.has_clip = FALSE;

  change_drawing_area_size (mii.width, mii.height);
  redraw_image (mii.pbuf);

  set_window_title (mii.filename, mii.width, mii.height, mii.changed);
}

gchar *
get_current_filename (void)
{
  return mii.filename;
}

GdkPixbuf *
get_pbuf (void)
{
  return mii.org;
}

MgwImageInfo *
get_image_info (void)
{
  return &mii;
}
