/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#ifndef XMLRPC_CLIENT_H
#define XMLRPC_CLIENT_H

/**
 * @file xmlrpc_client.h xmlrpc_client
 */

#include <xmlrpc.h>
#include <xmlrpc_client.h>
#include "rast/db.h"
#include "rast/local_db.h"
#include "rast/macros.h"

RAST_EXTERN_C_BEGIN

/**
 * @defgroup xmlrpc_client xmlrpc_client
 * @{
 */

/** A structure that represents a database to search documents */
typedef struct {
    /** The super class instance */
    rast_db_t base;
    xmlrpc_env env;
    xmlrpc_server_info *server;
    /** The url of the database server*/
    const char *url;
    /** The property definitions */
    rast_property_t *properties;
    /** The number of properties */
    int num_properties;
} rast_xmlrpc_client_t;

rast_error_t *rast_xmlrpc_client_create_document(rast_db_t *db,
                                                 rast_document_t **doc);

rast_error_t *rast_xmlrpc_client_initialize();
void rast_xmlrpc_client_finalize();

/**
 * Open the specified database.
 * @param db The database opened
 * @param name The url of the database
 * @param flags How to open
 *          - RAST_RDRW
 *            - open for reading and writing
 *          - RAST_RDONLY
 *            - open for reading
 * @param options The options to open the database.
 * @param pool The pool to allocate the memory out of
 * @return RAST_OK if succeeded, error otherwise
 * @see rast_db_open(), rast_db_open_option_create()
 */
rast_error_t *rast_xmlrpc_client_open(rast_db_t **db,
                                      const char *name, int flags,
                                      rast_db_open_option_t *options,
                                      apr_pool_t *pool);

/**
 * Please use rast_db_close().
 * @see rast_db_close()
 */
rast_error_t *rast_xmlrpc_client_close(rast_db_t *db);

/**
 * Please use rast_db_register().
 * @see rast_db_register()
 */
rast_error_t *rast_xmlrpc_client_register(rast_db_t *base,
                                          const char *text, rast_size_t nbytes,
                                          rast_value_t *properties,
                                          rast_doc_id_t *new_doc_id);

/**
 * Please use rast_db_search().
 * @see rast_db_search()
 */
rast_error_t *rast_xmlrpc_client_search(rast_db_t *db,
                                        const char *query,
                                        rast_search_option_t *options,
                                        rast_result_t **result,
                                        apr_pool_t *pool);

/**
 * Please use rast_db_delete().
 * @see rast_db_delete()
 */
rast_error_t *rast_xmlrpc_client_delete(rast_db_t *base, rast_doc_id_t doc_id);

/**
 * Please use rast_db_update().
 * @see rast_db_update()
 */
rast_error_t *rast_xmlrpc_client_update(rast_db_t *db, rast_doc_id_t doc_id,
                                        const char *text, rast_size_t nbytes,
                                        rast_value_t *properties,
                                        rast_doc_id_t *new_doc_id);

/**
 * Please use rast_db_get_text().
 * @see rast_db_get_text()
 */
rast_error_t *rast_xmlrpc_client_get_text(rast_db_t *db, rast_doc_id_t doc_id,
                                          char **s, rast_size_t *nbytes,
                                          apr_pool_t *pool);

/**
 * Please use rast_db_encoding().
 * @see rast_db_get_text()
 */
const char *rast_xmlrpc_client_encoding(rast_db_t *base);

/**
 * Please use rast_db_properties().
 * @see rast_db_properties()
 */
const rast_property_t *rast_xmlrpc_client_properties(rast_db_t *base,
                                                     int *num_properties);

RAST_EXTERN_C_END

/** @} */

#endif

/* vim: set filetype=c sw=4 expandtab : */
