/* 
 * Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
 *
 * This file is part of Rast.
 * See the file COPYING for redistribution information.
 *
 */

#include <ctype.h>
#include <apr_buckets.h>
#include <apr_hash.h>
#include <apr_strings.h>

#include "rast/util.h"
#include "rast/filter.h"

#define SUPPORTED_FRAMEWORK_VERSION 1

static rast_error_t *
combine_lineend_hyphen_filter_invoke(rast_filter_t *filter,
                                     apr_bucket_brigade *brigade,
                                     const char *mime_type)
{
    apr_bucket_alloc_t *bucket_alloc;
    apr_bucket *bucket;
    apr_bucket_brigade *next_brigade = NULL;
    apr_pool_t *pool;
    apr_status_t status;
    rast_error_t *error = RAST_OK;

    apr_pool_create(&pool, filter->pool);
    bucket_alloc = apr_bucket_alloc_create(pool);

    for (bucket = APR_BRIGADE_FIRST(brigade);
         bucket != APR_BRIGADE_SENTINEL(brigade);
         bucket = APR_BUCKET_NEXT(bucket)) {
        apr_bucket *next_bucket;
        const char *buf, *p, *p_end, *q, *text;
        int buf_nbytes, nbytes;

        if (APR_BUCKET_IS_EOS(bucket)) {
            next_bucket = apr_bucket_eos_create(bucket_alloc);
            if (next_brigade == NULL) {
                next_brigade = apr_brigade_create(pool, bucket_alloc);
            }
            APR_BRIGADE_INSERT_TAIL(next_brigade, next_bucket);
            error = rast_text_filter_pass(filter, next_brigade, mime_type);
            apr_brigade_destroy(next_brigade);
            next_brigade = NULL;
            if (error != RAST_OK) {
                return error;
            }
            continue;
        }

        status = apr_bucket_read(bucket, &buf, &buf_nbytes, APR_BLOCK_READ);
        if (status != APR_SUCCESS) {
            return apr_status_to_rast_error(status);
        }

        q = p = buf;
        p_end = buf + buf_nbytes;
        if (buf[0] == '-') {
            p++;
        }

        while (p < p_end) {
            p = rast_memchr(p, '-', p_end - p);
            if (p == NULL || p + 2 >= p_end) {
                break;
            }
            if (p[1] == '\n' && isalpha(p[2]) && isalpha(p[-1])) {
                nbytes = p - q;
                text = apr_pmemdup(filter->pool, q, nbytes);
                next_bucket = apr_bucket_transient_create(text, nbytes,
                                                          bucket_alloc);
                if (next_brigade == NULL) {
                    next_brigade = apr_brigade_create(pool, bucket_alloc);
                }
                APR_BRIGADE_INSERT_TAIL(next_brigade, next_bucket);
                p += 2;
                q = p;
            }
            else {
                p++;
            }
        }

        p = p_end;
        nbytes = p - q;
        text = apr_pmemdup(filter->pool, q, nbytes);
        next_bucket = apr_bucket_transient_create(text, nbytes, bucket_alloc);
        if (next_brigade == NULL) {
            next_brigade = apr_brigade_create(pool, bucket_alloc);
        }
        APR_BRIGADE_INSERT_TAIL(next_brigade, next_bucket);
    }    

    if (next_brigade == NULL) {
        return RAST_OK;
    }
    return rast_text_filter_pass(filter, next_brigade, mime_type);
}

rast_error_t *
rast_combine_lineend_hyphen_filter_module_initialize(rast_filter_map_t *map)
{
    const char *name = "combine-lineend-hyphen";
    static rast_filter_module_t filter_module = {
        SUPPORTED_FRAMEWORK_VERSION,
        NULL,
        combine_lineend_hyphen_filter_invoke,
    };
    rast_error_t *error;

    error = rast_filter_map_add_text_filter(map, name, &filter_module);
    if (error != RAST_OK) {
        /* todo: error handling */
        rast_error_destroy(error);
    }

    return RAST_OK;
}
