/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_font.h"

#include <string.h>
#include <glib/gi18n.h>
#include "mozilla-prefs.h"
#include "kazehakase.h"
#include "gtk-utils.h"


#define DATA_KEY "KzPrefsFont::info"


typedef enum {
	FONT_SERIF,
	FONT_SANS_SERIF,
	FONT_MONO
} FontType;


typedef enum {
	FONT_VARIABLE_SIZE,
	FONT_FIXED_SIZE,
	FONT_MIN_SIZE
} FontSizeType;


static GtkWidget *prefs_font_create   (void);
static void       prefs_font_response (GtkWidget *widget, gint response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:          N_("/Font"),
	priority_hint: 0,
	ui_level:      KZ_UI_LEVEL_ALL,
	create:        prefs_font_create,
	response:      prefs_font_response,
};


typedef struct _KzPrefsFont
{
	GtkWidget     *main_vbox;

	GtkComboBox   *lang_option_menu;
	GtkComboBox   *propotional_menu;

	GtkComboBox  *serif_combo;
	GtkComboBox  *sans_serif_combo;
	GtkComboBox  *monospace_combo;

	GtkSpinButton *variable_size_spin;
	GtkSpinButton *fixed_size_spin;
	GtkSpinButton *min_size_spin;

	GList         *prefs_list;
	gboolean       freeze;
} KzPrefsFont;


typedef struct
{
	gchar *lang;
	gchar *serif;
	gchar *sans_serif;
	gchar *monospace;
	guint  variable_size;
	guint  fixed_size;
	guint  min_size;
} FontPref;


typedef struct
{
	gchar *title;
	gchar *style;
} PropotionalFontStyle;


static const PropotionalFontStyle font_styles[] =
{
	{ N_("Serif"),      "serif"},
	{ N_("Sans Serif"), "sans-serif"},
};
static const guint n_font_styles = G_N_ELEMENTS (font_styles);


/*
 *  This array is picked from epiphany-0.7 (lib/ephy-langs.h).
 *  Copyright (C) 2000 Marco Pesenti Gritti
 */
typedef struct
{
	gchar *title;
	gchar* code;
} FontsLanguageInfo;

static const FontsLanguageInfo fonts_language[] =
{
	{ N_("Arabic"),			"ar" },
	{ N_("Baltic"),			"x-baltic" },
	{ N_("Central European"),	"x-central-euro" },
	{ N_("Cyrillic"),		"x-cyrillic" },
	{ N_("Greek"),			"el" },
	{ N_("Hebrew"),			"he" },
	{ N_("Japanese"),		"ja" },
	{ N_("Korean"),			"ko" },
	{ N_("Simplified Chinese"),	"zh-CN" },
	{ N_("Thai"),			"th" },
	{ N_("Traditional Chinese"),	"zh-TW" },
	{ N_("Turkish"),		"tr" },
	{ N_("Unicode"),		"x-unicode" },
	{ N_("Western"),		"x-western" },
};
static const guint n_fonts_languages = G_N_ELEMENTS (fonts_language);


static FontPref *
font_pref_new (const gchar *lang)
{
	FontPref *pref = g_new0(FontPref, 1);
	if (lang)
		pref->lang  = g_strdup(lang);
	else
		pref->lang  = NULL;
	pref->serif         = NULL;
	pref->sans_serif    = NULL;
	pref->monospace     = NULL;
	pref->variable_size = 0;
	pref->fixed_size    = 0;
	pref->min_size      = 0;

	return pref;
}


static void
font_pref_free (FontPref *pref)
{
	g_return_if_fail(pref);

	g_free(pref->lang);
	g_free(pref->serif);
	g_free(pref->sans_serif);
	g_free(pref->monospace);
	g_free(pref);
}


static gint
compare_font_pref (gconstpointer a, gconstpointer b)
{
	const FontPref *pref = a;
	const gchar *lang = b;

	if (!pref || !pref->lang)
	{
		if (!lang)
			return 0;
		else
			return -1;
	}
	else if (!lang)
	{
		return 1;
	}

	return strcmp(pref->lang, lang);
}


static gboolean
combo_box_row_separator_func (GtkTreeModel *model,
			      GtkTreeIter  *iter,
			      gpointer      user_data)
{
	gchar *text;
	gboolean ret;

	gtk_tree_model_get (model, iter, 0, &text, -1);

	ret = (text && !strncmp(text, "---", 3));

	if (text)
		g_free(text);
	return ret;
}			  

static void
font_combo_set (GtkComboBox *combo,
		const gchar *lang, const gchar *type,
		const gchar *font)
{
	GList *fonts, *all_fonts = NULL, *node;
	const gchar *use_font = NULL;
	gchar *default_font = NULL;
	GtkListStore *store;
	gint select = 0, i = 0;

	/* remove old list */
	store = GTK_LIST_STORE(gtk_combo_box_get_model(combo));
	gtk_list_store_clear(store);
	
	mozilla_prefs_get_font_list(lang, type, 
				    &fonts, &all_fonts,
				    &default_font);

	if (!fonts) return;

	use_font = font && *font ? font : default_font;

	for (node = g_list_first(fonts); node; node = g_list_next(node))
	{
		if (use_font && node->data && !strcmp(use_font, node->data))
			select = i;
		gtk_combo_box_append_text(combo, node->data);	
		i++;
	}

#if GTK_CHECK_VERSION(2,6,0)
	/* for separator */
	gtk_combo_box_append_text(combo, "---");
	i++;

	for (node = g_list_first(all_fonts); node; node = g_list_next(node))
	{
		if (use_font && node->data && !strcmp(use_font, node->data))
			select = i;
		gtk_combo_box_append_text(combo, node->data);	
		i++;
	}
#endif

	gtk_combo_box_set_active(combo, select);
	
	g_free (default_font);

	g_list_foreach (fonts, (GFunc)g_free, NULL);
	g_list_free (fonts);
	
	if (all_fonts)
	{
		g_list_foreach (all_fonts, (GFunc)g_free, NULL);
		g_list_free (all_fonts);
	}
}


static void
font_spin_set (GtkSpinButton *spin,
	       const gchar *type, const gchar *lang,
	       gint size)
{
	gchar key[256];
	gint keysize = G_N_ELEMENTS(key);
	gint use_size;

	g_return_if_fail(GTK_IS_SPIN_BUTTON(spin));
	g_return_if_fail(type && *type);
	g_return_if_fail(lang && *lang);

	g_snprintf(key, keysize, "font.%s.%s", type, lang);

	if (size)
		use_size = size;
	else if (!mozilla_prefs_get_int(key, &use_size))
		use_size = 0;

	if (use_size)
		gtk_spin_button_set_value(spin, (gdouble) use_size);
}


static const gchar *
get_current_lang(KzPrefsFont *prefsui)
{
	guint i;

	g_return_val_if_fail(prefsui, NULL);

	i = gtk_combo_box_get_active(prefsui->lang_option_menu);
	if (i >= 0 && i < n_fonts_languages)
		return fonts_language[i].code;

	return NULL;
}


static void
font_prefs_reset (KzPrefsFont *prefsui)
{
	const gchar *lang;

	g_return_if_fail(prefsui);

	prefsui->freeze = TRUE;

	lang = get_current_lang(prefsui);

	if (lang && *lang)
	{
		FontPref *pref = NULL;
		GList *node;

		node = g_list_find_custom(prefsui->prefs_list, lang,
					  compare_font_pref);
		if (node)
			pref = node->data;

		if (!pref)
		{
			pref = font_pref_new(lang);
			prefsui->prefs_list
				= g_list_append(prefsui->prefs_list, pref);
		}

		font_combo_set(prefsui->serif_combo,
			       lang, "serif", pref->serif);
		font_combo_set(prefsui->sans_serif_combo,
			       lang, "sans-serif", pref->sans_serif);
		font_combo_set(prefsui->monospace_combo,
			       lang, "monospace", pref->monospace);

		font_spin_set(prefsui->variable_size_spin,
			      "size.variable", lang, pref->variable_size);
		font_spin_set(prefsui->fixed_size_spin,
			      "size.fixed", lang, pref->fixed_size);
		font_spin_set(prefsui->min_size_spin,
			      "min-size.variable", lang, pref->min_size);
	}
	else
	{
		g_warning("KzPrefsFont: "
			  "Unexpected language ID was selected!");
	}

	prefsui->freeze = FALSE;
}


static void
prefs_font_destroy (gpointer data)
{
	KzPrefsFont *prefsui = data;

	g_list_foreach(prefsui->prefs_list, (GFunc) font_pref_free, NULL);
	g_list_free(prefsui->prefs_list);
	g_free(prefsui);
}


static void
cb_font_lang_changed (GtkComboBox *option_menu, KzPrefsFont *prefsui)
{
	font_prefs_reset(prefsui);
}


static void
store_new_font (GtkComboBox *combo, KzPrefsFont *prefsui, FontType type)
{
	const gchar *lang, *new_font;
	gchar **font = NULL;
	GList *node;
	FontPref *pref = NULL;

	g_return_if_fail(GTK_IS_COMBO_BOX(combo));

	if (prefsui->freeze) return;

	lang = get_current_lang(prefsui);
	if (!lang || !*lang) return;

	node = g_list_find_custom(prefsui->prefs_list, lang,
				  compare_font_pref);
	if (!node) return;

	pref = node->data;
	if (!pref) return;

	new_font = gtk_entry_get_text(GTK_ENTRY(GTK_BIN(combo)->child));

	switch (type) {
	case FONT_SERIF:
		font = &pref->serif;
		break;
	case FONT_SANS_SERIF:
		font = &pref->sans_serif;
		break;
	case FONT_MONO:
		font = &pref->monospace;
		break;
	default:
		g_warning("KzPrefsFont: Unexpected font type!");
		break;
	}

	if (!font) return;

	if (new_font && *new_font)
	{
		g_free(*font);
		*font = g_strdup(new_font);
	}
}


static void
store_new_font_size (GtkSpinButton *spin, KzPrefsFont *prefsui,
		     FontSizeType type)
{
	const gchar *lang;
	GList *node;
	FontPref *pref = NULL;
	gint size;

	g_return_if_fail(GTK_IS_SPIN_BUTTON(spin));

	if (prefsui->freeze) return;

	lang = get_current_lang(prefsui);
	if (!lang || !*lang) return;

	node = g_list_find_custom(prefsui->prefs_list, lang,
				  compare_font_pref);
	if (!node) return;

	pref = node->data;
	if (!pref) return;

	size = (gint) gtk_spin_button_get_value(spin);

	switch (type) {
	case FONT_VARIABLE_SIZE:
		pref->variable_size = size;
		break;
	case FONT_FIXED_SIZE:
		pref->fixed_size = size;
		break;
	case FONT_MIN_SIZE:
		pref->min_size = size;
		break;
	default:
		g_warning("KzPrefsFont: Unexpected font size type!");
		break;
	}
}


static void
cb_serif_entry_changed (GtkComboBox *combo, KzPrefsFont *prefsui)
{
	store_new_font (combo, prefsui, FONT_SERIF);
}


static void
cb_sans_serif_entry_changed (GtkComboBox *combo, KzPrefsFont *prefsui)
{
	store_new_font (combo, prefsui, FONT_SANS_SERIF);
}


static void
cb_monospace_entry_changed (GtkComboBox *combo, KzPrefsFont *prefsui)
{
	store_new_font (combo, prefsui, FONT_MONO);
}


static void
cb_var_font_spin_changed (GtkSpinButton *spin, KzPrefsFont *prefsui)
{
	store_new_font_size (spin, prefsui, FONT_VARIABLE_SIZE);
}


static void
cb_fixed_font_spin_changed (GtkSpinButton *spin, KzPrefsFont *prefsui)
{
	store_new_font_size (spin, prefsui, FONT_FIXED_SIZE);
}


static void
cb_min_font_spin_changed (GtkSpinButton *spin, KzPrefsFont *prefsui)
{
	store_new_font_size (spin, prefsui, FONT_MIN_SIZE);
}


static GtkWidget *
prefs_font_create (void)
{
	KzPrefsFont *prefsui = g_new0(KzPrefsFont, 1);
	GtkWidget *main_vbox, *vbox, *hbox;
	GtkWidget *frame, *table, *label, *option_menu, *combo, *spin;
	gchar *str = NULL;

	prefsui->prefs_list = NULL;
	prefsui->freeze     = FALSE;

	main_vbox = gtk_vbox_new (FALSE, 5);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full (G_OBJECT(main_vbox),
				DATA_KEY, prefsui,
				(GDestroyNotify) prefs_font_destroy);

	label = kz_prefs_ui_utils_create_title(_("Font"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), vbox,
			   TRUE, TRUE, 0);
	gtk_widget_show(vbox);

	/*
	 *  Fonts for
	 */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
	gtk_widget_show(frame);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_frame_set_label_widget(GTK_FRAME(frame), hbox);
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("_Fonts for: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	option_menu = gtk_combo_box_new_text();
	prefsui->lang_option_menu = GTK_COMBO_BOX(option_menu);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(option_menu);

	GTKUTIL_CREATE_MENU(option_menu, fonts_language, n_fonts_languages, title);

	g_signal_connect (option_menu, "changed",
			  G_CALLBACK (cb_font_lang_changed), prefsui);

	table = gtk_table_new(3, 2, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_container_add(GTK_CONTAINER(frame), table);
	gtk_widget_show(table);

	/*
	 *  Propotional
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 0, 1,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("_Propotional: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	option_menu = gtk_combo_box_new_text();
	prefsui->propotional_menu = GTK_COMBO_BOX(option_menu);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_table_attach(GTK_TABLE(table), option_menu,
			 1, 2, 0, 1,
			 GTK_EXPAND | GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(option_menu);

	GTKUTIL_CREATE_MENU(option_menu, font_styles, n_font_styles, title);

	/* font size */
	label = gtk_label_new_with_mnemonic(_("Si_ze:"));
	gtk_table_attach(GTK_TABLE(table), label,
			 2, 3, 0, 1,  /* xpos, ypos */
			 GTK_FILL, 0,
			 4, 2);       /* padding x, y  */
	gtk_widget_show(label);

	spin = gtk_spin_button_new_with_range(1.0, 100.0, 1.0);
	prefsui->variable_size_spin = GTK_SPIN_BUTTON(spin);
	gtk_widget_set_size_request(spin, 50, -1);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), spin);
	gtk_table_attach(GTK_TABLE(table), spin,
			 3, 4, 0, 1,
			 GTK_FILL, 0,
			 2, 2);
	g_signal_connect (spin, "value-changed",
			  G_CALLBACK(cb_var_font_spin_changed), prefsui);
	gtk_widget_show(spin);

	/*
	 *  Serif
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 1, 2,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("_Serif: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	combo = gtk_combo_box_entry_new_text();
	prefsui->serif_combo = GTK_COMBO_BOX(combo);
	gtk_widget_set_size_request(combo, 90, -1);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), combo);
	g_signal_connect (combo, "changed",
			  G_CALLBACK(cb_serif_entry_changed), prefsui);
#if GTK_CHECK_VERSION(2,6,0)
	gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(combo),
					     combo_box_row_separator_func,
					     NULL, NULL);
#endif
	gtk_table_attach(GTK_TABLE(table), combo,
			 1, 2, 1, 2,
			 GTK_EXPAND | GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(combo);

	/*
	 *  Sans Serif
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 2, 3,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("Sa_ns Serif: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	combo = gtk_combo_box_entry_new_text();
	prefsui->sans_serif_combo = GTK_COMBO_BOX(combo);
	gtk_widget_set_size_request(combo, 90, -1);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), combo);
	g_signal_connect (combo, "changed",
			  G_CALLBACK(cb_sans_serif_entry_changed), prefsui);
#if GTK_CHECK_VERSION(2,6,0)
	gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(combo),
					     combo_box_row_separator_func,
					     NULL, NULL);
#endif
	gtk_table_attach(GTK_TABLE(table), combo,
			 1, 2, 2, 3,
			 GTK_EXPAND | GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(combo);

	/*
	 *  Monospace
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 3, 4,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	label = gtk_label_new_with_mnemonic(_("_Monospace: "));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	combo = gtk_combo_box_entry_new_text();
	prefsui->monospace_combo = GTK_COMBO_BOX(combo);
	gtk_widget_set_size_request(combo, 90, -1);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), combo);
	g_signal_connect (combo, "changed",
			  G_CALLBACK(cb_monospace_entry_changed), prefsui);
#if GTK_CHECK_VERSION(2,6,0)
	gtk_combo_box_set_row_separator_func(GTK_COMBO_BOX(combo),
					     combo_box_row_separator_func,
					     NULL, NULL);
#endif
	gtk_table_attach(GTK_TABLE(table), combo,
			 1, 2, 3, 4,
			 GTK_EXPAND | GTK_FILL, 0,
			 2, 2);
	gtk_widget_show(combo);

	/* font size */
	label = gtk_label_new_with_mnemonic(_("Siz_e:"));
	gtk_table_attach(GTK_TABLE(table), label,
			 2, 3, 3, 4,  /* xpos, ypos */
			 GTK_FILL, 0,
			 4, 2);       /* padding x, y  */
	gtk_widget_show(label);

	spin = gtk_spin_button_new_with_range(1.0, 100.0, 1.0);
	prefsui->fixed_size_spin = GTK_SPIN_BUTTON(spin);
	gtk_widget_set_size_request(spin, 50, -1);
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), spin);
	gtk_table_attach(GTK_TABLE(table), spin,
			 3, 4, 3, 4,
			 GTK_FILL, 0,
			 2, 2);
	g_signal_connect (spin, "value-changed",
			  G_CALLBACK(cb_fixed_font_spin_changed), prefsui);
	gtk_widget_show(spin);

	/*
	 *  Minimam font size
	 */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 4, 5, 6,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 4);       /* padding x, y  */
	gtk_widget_show(hbox);

	spin = gtk_spin_button_new_with_range(1.0, 100.0, 1.0);
	prefsui->min_size_spin = GTK_SPIN_BUTTON(spin);
	gtk_widget_set_size_request(spin, 50, -1);
	gtk_box_pack_end(GTK_BOX(hbox), spin, FALSE, FALSE, 0);
	g_signal_connect (spin, "value-changed",
			  G_CALLBACK(cb_min_font_spin_changed), prefsui);
	gtk_widget_show(spin);

	label = gtk_label_new(_("M_inimum font size: "));
	gtk_label_set_use_underline(GTK_LABEL(label), TRUE);
	gtk_box_pack_end(GTK_BOX(hbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	gtk_label_set_mnemonic_widget(GTK_LABEL(label), spin);

	mozilla_prefs_get_string("font.default", &str);
	GTKUTIL_SET_DEFAULT_HISTORY(prefsui->propotional_menu,
			    	    str,
			    	    font_styles, n_font_styles, style);
	if (str)
	{
		g_free(str);
		str =NULL;
	}
	mozilla_prefs_get_string("font.language.group", &str);
	GTKUTIL_SET_DEFAULT_HISTORY(prefsui->lang_option_menu,
			    	    str,
			    	    fonts_language, n_fonts_languages, code);
	if (str)
	{
		g_free(str);
		str =NULL;
	}
	font_prefs_reset(prefsui);

	return main_vbox;
}



static void
apply_font_prefs (KzPrefsFont *prefsui)
{
	GList *node;
	guint i;

	g_return_if_fail(prefsui);

	for (node = prefsui->prefs_list; node; node = g_list_next(node))
	{
		FontPref *pref = node->data;
		gchar key[256];
		gint keysize = G_N_ELEMENTS(key);

		if (!pref) continue;
		if (!pref->lang || !*pref->lang) continue;

		if (pref->serif)
		{
			g_snprintf(key, keysize, "font.name.serif.%s",
				   pref->lang);
			mozilla_prefs_set_string(key, pref->serif);
		}

		if (pref->sans_serif)
		{
			g_snprintf(key, keysize, "font.name.sans-serif.%s",
				   pref->lang);
			mozilla_prefs_set_string(key, pref->sans_serif);
		}

		if (pref->monospace)
		{
			g_snprintf(key, keysize, "font.name.monospace.%s",
				   pref->lang);
			mozilla_prefs_set_string(key, pref->monospace);
		}

		if (pref->variable_size)
		{
			g_snprintf(key, keysize, "font.size.variable.%s",
				   pref->lang);
			mozilla_prefs_set_int(key, pref->variable_size);
		}

		if (pref->fixed_size)
		{
			g_snprintf(key, keysize, "font.size.fixed.%s",
				   pref->lang);
			mozilla_prefs_set_int(key, pref->fixed_size);
		}

		if (pref->min_size)
		{
			g_snprintf(key, keysize, "font.min-size.variable.%s",
				   pref->lang);
			mozilla_prefs_set_int(key, pref->min_size);
			g_snprintf(key, keysize, "font.min-size.fixed.%s",
				   pref->lang);
			mozilla_prefs_set_int(key, pref->min_size);
		}
	}

	i = gtk_combo_box_get_active(prefsui->propotional_menu);
	if (i >= 0 && i < n_font_styles)
		mozilla_prefs_set_string("font.default", font_styles[i].style);

	i = gtk_combo_box_get_active(prefsui->lang_option_menu);
	if (i >= 0 && i < n_fonts_languages)
		mozilla_prefs_set_string("font.language.group",
					 fonts_language[i].code);
}


static void
prefs_font_response (GtkWidget *widget, gint response)
{
	KzPrefsFont *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
		apply_font_prefs(prefsui);
		break;
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_BEGINNER:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_font_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
