/*
 * Copyright (c) 2008-2009 Internet Initiative Japan Inc. All rights reserved.
 *
 * The terms and conditions of the accompanying program
 * shall be provided separately by Internet Initiative Japan Inc.
 * Any use, reproduction or distribution of the program are permitted
 * provided that you agree to be bound to such terms and conditions.
 *
 * $Id: dkimsignature.c 764 2009-03-22 07:12:05Z takahiko $
 */

#include "rcsid.h"
RCSID("$Id: dkimsignature.c 764 2009-03-22 07:12:05Z takahiko $");

#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <assert.h>
#include <string.h>
#include <strings.h>
#include <limits.h>
#include <time.h>
#include <stdbool.h>
#include <errno.h>

#include "dkimlogger.h"
#include "ptrop.h"
#include "pstring.h"
#include "xbuffer.h"
#include "xskip.h"
#include "xparse.h"
#include "foldstring.h"
#include "intarray.h"
#include "inetdomain.h"
#include "inetmailbox.h"

#include "dkim.h"
#include "dkimspec.h"
#include "dkimenum.h"
#include "dkimtvobj.h"
#include "dkimparse.h"
#include "dkimsignature.h"

#define DKIM_SIGNATURE_HEADER_WIDTH  78

struct DkimSignature {
    DkimTvobj_MEMBER;
    char *rawname;
    char *rawvalue;
    const char *raw_value_b_head;   // rawvalue フィールド中に保持されている sig-b-tag の値の先頭へのポインタ
    const char *raw_value_b_tail;   // rawvalue フィールド中に保持されている sig-b-tag の値の末尾へのポインタ
    time_t verify_timestamp;    // 検証を実行した時間. sig-t-tag, sig-x-tag との比較に使用する.
    dkim_pubkeyalg_t pubkeyalg; // sig-a-tag-k
    dkim_digestalg_t digestalg; // sig-a-tag-h
    XBuffer *headerhash;        // sig-b-tag
    XBuffer *bodyhash;          // sig-bh-tag
    StrArray *selecthdr;        // sig-h-tag
    dkim_canonalg_t headercanon;    // sig-c-tag
    dkim_canonalg_t bodycanon;  // sig-c-tag
    long long genarated_time;   // sig-t-tag
    long long expire_limit;     // sig-x-tag
    long long body_length_limit;    // sig-l-tag
    char *selector;             // sig-s-tag
    char *domain;               // sig-d-tag
    InetMailbox *identity;      // sig-i-tag
    IntArray *keyretr;          // sig-q-tag (dkim_keyretr_t)

    bool parsed_v;
    bool parsed_a;
    bool parsed_b;
    bool parsed_bh;
    bool parsed_c;
    bool parsed_d;
    bool parsed_h;
    bool parsed_i;
    bool parsed_l;
    bool parsed_q;
    bool parsed_s;
    bool parsed_t;
    bool parsed_x;
    bool parsed_z;
};

static dkim_stat_t DkimSignature_parse_v(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_a(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_b(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_bh(DkimTvobj *base, const DkimTagParseContext *context,
                                          const char **nextp);
static dkim_stat_t DkimSignature_parse_c(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_d(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_h(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_i(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_l(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_q(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_s(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_t(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);
static dkim_stat_t DkimSignature_parse_x(DkimTvobj *base, const DkimTagParseContext *context,
                                         const char **nextp);

static const DkimTvobjFieldMap dkim_signature_field_tbl[] = {
    {"v", DkimSignature_parse_v, true, NULL, offsetof(DkimSignature, parsed_v)},
    {"a", DkimSignature_parse_a, true, NULL, offsetof(DkimSignature, parsed_a)},
    {"b", DkimSignature_parse_b, true, NULL, offsetof(DkimSignature, parsed_b)},
    {"bh", DkimSignature_parse_bh, true, NULL, offsetof(DkimSignature, parsed_bh)},
    {"c", DkimSignature_parse_c, false, "simple/simple", offsetof(DkimSignature, parsed_c)},
    {"d", DkimSignature_parse_d, true, NULL, offsetof(DkimSignature, parsed_d)},
    {"h", DkimSignature_parse_h, true, NULL, offsetof(DkimSignature, parsed_h)},
    {"i", DkimSignature_parse_i, false, NULL, offsetof(DkimSignature, parsed_i)},
    {"l", DkimSignature_parse_l, false, NULL, offsetof(DkimSignature, parsed_l)},
    {"q", DkimSignature_parse_q, false, "dns/txt", offsetof(DkimSignature, parsed_q)},
    {"s", DkimSignature_parse_s, true, NULL, offsetof(DkimSignature, parsed_s)},
    {"t", DkimSignature_parse_t, false, NULL, offsetof(DkimSignature, parsed_t)},
    {"x", DkimSignature_parse_x, false, NULL, offsetof(DkimSignature, parsed_x)},
    {"z", NULL, false, NULL, offsetof(DkimSignature, parsed_z)},
    {NULL, NULL, false, NULL, 0},   // sentinel
};

////////////////////////////////////////////////////////////////////////
// private functions

/*
 * [RFC4871]
 * sig-v-tag   = %x76 [FWS] "=" [FWS] "1"
 */
dkim_stat_t
DkimSignature_parse_v(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    static const char *acceptable_dkim_versions[] = {
        "1", NULL,
    };

    DkimSignature *self = (DkimSignature *) base;

    for (const char **val = acceptable_dkim_versions; NULL != *val; ++val) {
        if (0 < XSkip_string(context->valuehead, context->valuetail, *val, nextp)) {
            return DSTAT_OK;
        }   // end if
    }   // end for

    *nextp = context->valuehead;
    DkimLogPermFail(self->policy, "unsupported signature version: near %.50s", context->valuehead);
    return DSTAT_PERMFAIL_SIGNATURE_INCOMPATIBLE_VERSION;
}   // end function : DkimSignature_parse_v

/*
 * [RFC4871]
 * sig-a-tag       = %x61 [FWS] "=" [FWS] sig-a-tag-alg
 * sig-a-tag-alg   = sig-a-tag-k "-" sig-a-tag-h
 * sig-a-tag-k     = "rsa" / x-sig-a-tag-k
 * sig-a-tag-h     = "sha1" / "sha256" / x-sig-a-tag-h
 * x-sig-a-tag-k   = ALPHA *(ALPHA / DIGIT)   ; for later extension
 * x-sig-a-tag-h   = ALPHA *(ALPHA / DIGIT)   ; for later extension
 */
dkim_stat_t
DkimSignature_parse_a(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *p = context->valuehead;
    const char *tailp;

    SETDEREF(nextp, context->valuehead);
    if (0 >= XSkip_alphaAlnum(p, context->valuetail, &tailp)) {
        DkimLogPermFail(self->policy, "no value for sig-a-tag-k: near %.50s", context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    self->pubkeyalg = DkimEnum_lookupPubkeyAlgByNameSlice(p, tailp);
    if (DKIM_PUBKEYALG_NULL == self->pubkeyalg) {
        DkimLogPermFail(self->policy, "unsupported public key algorithm: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_UNSUPPORTED_PUBKEYALG;
    }   // end if

    if (0 >= XSkip_char(p = tailp, context->valuetail, '-', &p)) {
        DkimLogPermFail(self->policy, "hyphen missing for sig-a-tag: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    if (0 >= XSkip_alphaAlnum(p, context->valuetail, &tailp)) {
        DkimLogPermFail(self->policy, "no value for sig-a-tag-h: near %.50s", context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    self->digestalg = DkimEnum_lookupDigestAlgByNameSlice(p, tailp);
    if (DKIM_DIGESTALG_NULL == self->digestalg) {
        DkimLogPermFail(self->policy, "unsupported digest algorithm: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_UNSUPPORTED_DIGESTALG;
    }   // end if

    SETDEREF(nextp, tailp);
    return DSTAT_OK;
}   // end function : DkimSignature_parse_a

/*
 * [RFC4871]
 * sig-b-tag       = %x62 [FWS] "=" [FWS] sig-b-tag-data
 * sig-b-tag-data  = base64string
 */
dkim_stat_t
DkimSignature_parse_b(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *p = context->valuehead;

    if (NULL != self->headerhash) {
        DkimLogImplError(self->policy, "sig-b-tag already set");
        return DSTAT_SYSERR_IMPLERROR;
    }   // end if

    SETDEREF(nextp, context->valuehead);
    XSkip_fws(p, context->valuetail, &p);
    if (context->valuetail <= p) {
        // 値が空
        DkimLogPermFail(self->policy, "sig-b-tag has empty value: near %.50s", context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    dkim_stat_t decode_stat;
    self->headerhash =
        DkimParse_decodeBase64(self->policy, p, context->valuetail, &p, &decode_stat);
    if (NULL == self->headerhash) {
        return decode_stat;
    }   // end if

    self->raw_value_b_head = context->valuehead;
    self->raw_value_b_tail = context->valuetail;
    SETDEREF(nextp, p);
    return DSTAT_OK;
}   // end function : DkimSignature_parse_b

/*
 * [RFC4871]
 * sig-bh-tag      = %x62 %x68 [FWS] "=" [FWS] sig-bh-tag-data
 * sig-bh-tag-data = base64string
 */
dkim_stat_t
DkimSignature_parse_bh(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *p = context->valuehead;

    if (self->bodyhash) {
        DkimLogImplError(self->policy, "sig-bh-tag already set");
        return DSTAT_SYSERR_IMPLERROR;
    }   // end if

    SETDEREF(nextp, context->valuehead);
    XSkip_fws(p, context->valuetail, &p);
    if (context->valuetail <= p) {
        // 値が空
        DkimLogPermFail(self->policy, "sig-bh-tag has empty value: near %.50s", context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    dkim_stat_t decode_stat;
    self->bodyhash =
        DkimParse_decodeBase64(self->policy, context->valuehead, context->valuetail, &p,
                               &decode_stat);
    if (NULL == self->bodyhash) {
        return decode_stat;
    }   // end if

    SETDEREF(nextp, p);
    return DSTAT_OK;
}   // end function : DkimSignature_parse_bh

/*
 * [RFC4871]
 * sig-c-tag       = %x63 [FWS] "=" [FWS] sig-c-tag-alg
 *                   ["/" sig-c-tag-alg]
 * sig-c-tag-alg   = "simple" / "relaxed" / x-sig-c-tag-alg
 * x-sig-c-tag-alg = hyphenated-word    ; for later extension
 */
dkim_stat_t
DkimSignature_parse_c(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *p = context->valuehead;
    const char *tailp;

    SETDEREF(nextp, context->valuehead);
    if (0 >= XSkip_hyphenatedWord(p, context->valuetail, &tailp)) {
        DkimLogPermFail(self->policy, "no value for header canonicalization algorithm: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    self->headercanon = DkimEnum_lookupCanonAlgByNameSlice(p, tailp);
    if (DKIM_CANONALG_NULL == self->headercanon) {
        DkimLogPermFail(self->policy, "unsupported header canonicalization algorithm: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_UNSUPPORTED_CANONALG;
    }   // end if

    if (0 >= XSkip_char(p = tailp, context->valuetail, '/', &p)) {
        // '/' で区切られていない場合の body canonicalization algorithm は
        // "simple" として扱うよう定められている
        self->bodycanon = DKIM_CANONALG_SIMPLE;
    } else {
        if (0 >= XSkip_hyphenatedWord(p, context->valuetail, &tailp)) {
            DkimLogPermFail(self->policy,
                            "no value for body canonicalization algorithm: near %.50s",
                            context->valuehead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if

        self->bodycanon = DkimEnum_lookupCanonAlgByNameSlice(p, tailp);
        if (DKIM_CANONALG_NULL == self->bodycanon) {
            DkimLogPermFail(self->policy, "unsupported body canonicalization algorithm: near %.50s",
                            context->valuehead);
            return DSTAT_PERMFAIL_UNSUPPORTED_CANONALG;
        }   // end if
    }   // end if

    SETDEREF(nextp, tailp);
    return DSTAT_OK;
}   // end function : DkimSignature_parse_c

/*
 * [RFC4871]
 * sig-d-tag       = %x64 [FWS] "=" [FWS] domain-name
 * domain-name     = sub-domain 1*("." sub-domain)
 *              ; from RFC 2821 Domain, but excluding address-literal
 */
dkim_stat_t
DkimSignature_parse_d(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;

    // domain-name にマッチするか確認
    if (0 >= XSkip_domainName(context->valuehead, context->valuetail, nextp)) {
        DkimLogPermFail(self->policy, "sig-d-tag doesn't match domain-name: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    self->domain = strpdup(context->valuehead, *nextp);
    if (NULL == self->domain) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimSignature_parse_d

/*
 * [RFC4871]
 * sig-h-tag       = %x68 [FWS] "=" [FWS] hdr-name
 *                   0*( *FWS ":" *FWS hdr-name )
 * hdr-name        = field-name
 */
dkim_stat_t
DkimSignature_parse_h(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *p = context->valuehead;
    const char *tailp;

    SETDEREF(nextp, context->valuehead);
    do {
        XSkip_fws(p, context->valuetail, &p);
        if (0 >= XSkip_fieldName(p, context->valuetail, &tailp)) {
            DkimLogPermFail(self->policy, "hdr-name missing: near %.50s", context->valuehead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if
        // TODO: DoS を避けるために上限を設けた方がよい?
        if (0 > StrArray_appendWithLength(self->selecthdr, p, tailp - p)) {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
        XSkip_fws(tailp, context->valuetail, &p);
    } while (0 < XSkip_char(p, context->valuetail, ':', &p));

    SETDEREF(nextp, p);
    return DSTAT_OK;
}   // end function : DkimSignature_parse_h

/*
 * [RFC4871]
 * sig-i-tag =   %x69 [FWS] "=" [FWS] [ Local-part ] "@" domain-name
 *
 * [RFC4871] 3.5.
 * i=  Identity of the user or agent (e.g., a mailing list manager) on
 *     behalf of which this message is signed (dkim-quoted-printable;
 *     OPTIONAL, default is an empty Local-part followed by an "@"
 *     followed by the domain from the "d=" tag).  The syntax is a
 *     standard email address where the Local-part MAY be omitted.  The
 *     domain part of the address MUST be the same as or a subdomain of
 *     the value of the "d=" tag.
 */
dkim_stat_t
DkimSignature_parse_i(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *errptr = NULL;

    if (NULL != self->identity) {
        DkimLogImplError(self->policy, "sig-i-tag already set");
        return DSTAT_SYSERR_IMPLERROR;
    }   // end if

    // まずは dkim-quoted-printable を解釈する.
    XBuffer *decoded_identity = XBuffer_new(0);
    if (NULL == decoded_identity) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    XParse_dkimQuotedPrintable(context->valuehead, context->valuetail, nextp, decoded_identity);
    if (0 != XBuffer_status(decoded_identity)) {
        XBuffer_free(decoded_identity);
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    // sig-i-tag の ABNF に従ってパースする.
    const char *identity_head = XBuffer_getString(decoded_identity);
    const char *identity_tail = identity_head + XBuffer_getSize(decoded_identity);
    const char *parsed_tail;
    self->identity =
        InetMailbox_buildDkimIdentity(identity_head, identity_tail, &parsed_tail, &errptr);
    XBuffer_free(decoded_identity);
    if (NULL == self->identity || parsed_tail != identity_tail) {
        *nextp = context->valuehead;
        if (NULL != self->identity) {
            InetMailbox_free(self->identity);
            self->identity = NULL;
        }   // end if
        if (NULL != errptr) {
            DkimLogPermFail(self->policy, "sig-i-tag doesn't match identity: near %.50s",
                            context->valuehead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        } else {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    }   // end if

    return DSTAT_OK;
}   // end function : DkimSignature_parse_i

/*
 * [RFC4871]
 * sig-l-tag    = %x6c [FWS] "=" [FWS] 1*76DIGIT
 */
dkim_stat_t
DkimSignature_parse_l(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    self->body_length_limit =
        DkimParse_longlong(context->valuehead, context->valuetail, DKIM_SIG_L_TAG_LEN, nextp);
    // SPEC: sig-l-tag が 2^63 を超える署名はサポートしない
    if (0 <= self->body_length_limit && context->valuetail == *nextp) {
        return DSTAT_OK;
    } else {
        DkimLogPermFail(self->policy, "sig-l-tag has invalid line length limit: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if
}   // end function : DkimSignature_parse_l

/*
 * [RFC4871]
 * sig-q-tag        = %x71 [FWS] "=" [FWS] sig-q-tag-method
 *                    *([FWS] ":" [FWS] sig-q-tag-method)
 * sig-q-tag-method = "dns/txt" / x-sig-q-tag-type ["/" x-sig-q-tag-args]
 * x-sig-q-tag-type = hyphenated-word  ; for future extension
 * x-sig-q-tag-args = qp-hdr-value
 * qp-hdr-value     = dkim-quoted-printable    ; with "|" encoded
 */
dkim_stat_t
DkimSignature_parse_q(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    const char *p = context->valuehead;
    const char *typehead, *typetail;

    SETDEREF(nextp, context->valuehead);
    do {
        XSkip_fws(p, context->valuetail, &typehead);

        if (0 >= XSkip_hyphenatedWord(typehead, context->valuetail, &typetail)) {
            DkimLogPermFail(self->policy, "no value for sig-q-tag-method: near %.50s",
                            context->valuehead);
            return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
        }   // end if
        if (0 < XSkip_char(typetail, context->valuetail, '/', &typetail)) {
            // 本当は x-sig-q-tag-args は dkim-quoted-printable になるのだが,
            // その必要性はなさそうなので hyphenated-word で代用.
            if (0 >= XSkip_hyphenatedWord(typetail, context->valuetail, &typetail)) {
                DkimLogPermFail(self->policy, "no value for x-sig-q-tag-args: near %.50s",
                                context->valuehead);
                return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
            }   // end if
        }   // end if

        dkim_keyretr_t keyretr_method = DkimEnum_lookupKeyRetrByNameSlice(typehead, typetail);
        // 未知の sig-q-tag-type は単に無視する
        if (keyretr_method != DKIM_KEYRETR_NULL) {
            // DoS 対策のため, keyretr_method 二重三重に登録されないようチェックする
            if (0 > IntArray_linearSearch(self->keyretr, keyretr_method)) {
                if (0 > IntArray_append(self->keyretr, keyretr_method)) {
                    DkimLogNoResource(self->policy);
                    return DSTAT_SYSERR_NORESOURCE;
                }   // end if
            }   // end if
        }   // end if

        SETDEREF(nextp, typetail);
        XSkip_fws(typetail, context->valuetail, &p);
    } while (0 < XSkip_char(p, context->valuetail, ':', &p));

    if (0 == IntArray_getCount(self->keyretr)) {
        DkimLogPermFail(self->policy, "no public key retrieving methods are available: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_UNSUPPORTED_KEYRETR;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimSignature_parse_q

/*
 * [RFC4871]
 * sig-s-tag    = %x73 [FWS] "=" [FWS] selector
 * selector =   sub-domain *( "." sub-domain )
 */
dkim_stat_t
DkimSignature_parse_s(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;

    if (0 >= XSkip_selector(context->valuehead, context->valuetail, nextp)) {
        DkimLogPermFail(self->policy, "sig-s-tag doesn't match selector: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if

    self->selector = strpdup(context->valuehead, *nextp);
    if (NULL == self->selector) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if

    return DSTAT_OK;
}   // end function : DkimSignature_parse_s

/*
 * [RFC4871]
 * sig-t-tag    = %x74 [FWS] "=" [FWS] 1*12DIGIT
 */
dkim_stat_t
DkimSignature_parse_t(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    self->genarated_time =
        DkimParse_longlong(context->valuehead, context->valuetail, DKIM_SIG_T_TAG_LEN, nextp);
    if (0 <= self->genarated_time && context->valuetail == *nextp) {
        return DSTAT_OK;
    } else {
        DkimLogPermFail(self->policy, "sig-t-tag has invalid timestamp: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if
}   // end function : DkimSignature_parse_t

/*
 * [RFC4871]
 * sig-x-tag    = %x78 [FWS] "=" [FWS] 1*12DIGIT
 */
dkim_stat_t
DkimSignature_parse_x(DkimTvobj *base, const DkimTagParseContext *context, const char **nextp)
{
    DkimSignature *self = (DkimSignature *) base;
    self->expire_limit =
        DkimParse_longlong(context->valuehead, context->valuetail, DKIM_SIG_X_TAG_LEN, nextp);
    if (0 <= self->expire_limit && context->valuetail == *nextp) {
        return DSTAT_OK;
    } else {
        DkimLogPermFail(self->policy, "sig-x-tag has invalid timestamp: near %.50s",
                        context->valuehead);
        return DSTAT_PERMFAIL_TAG_SYNTAX_VIOLATION;
    }   // end if
}   // end function : DkimSignature_parse_x

/*
 * [RFC4871]
 * sig-z-tag      = %x7A [FWS] "=" [FWS] sig-z-tag-copy
 *                  *( [FWS] "|" sig-z-tag-copy )
 * sig-z-tag-copy = hdr-name ":" qp-hdr-value
 * qp-hdr-value   = dkim-quoted-printable    ; with "|" encoded
 *
 * sig-z-tag は検証には使ってはならない (MUST NOT) タグなのでここでは扱わない.
 * フォーマットの検証すらおこなわない.
 */

////////////////////////////////////////////////////////////////////////
// public functions

DkimSignature *
DkimSignature_new(const DkimPolicy *policy)
{
    DkimSignature *self = (DkimSignature *) malloc(sizeof(DkimSignature));
    if (NULL == self) {
        return NULL;
    }   // end if
    memset(self, 0, sizeof(DkimSignature));

    self->selecthdr = StrArray_new(0);
    if (NULL == self->selecthdr) {
        goto cleanup;
    }   // end if
    self->keyretr = IntArray_new(0);
    if (NULL == self->keyretr) {
        goto cleanup;
    }   // end if
    self->policy = policy;
    self->ftbl = dkim_signature_field_tbl;
    self->genarated_time = -1LL;
    self->expire_limit = -1LL;
    self->body_length_limit = -1LL;

    return self;

  cleanup:
    DkimSignature_free(self);
    return NULL;
}   // end function : DkimSignature_new

/*
 * 署名ヘッダの意味的なチェック
 * [RFC4871] 6.1.1 をそのまま実行する. ただし, 署名の期限についてはチェックをおこなわない.
 * 署名の期限のチェックは DkimSignature_isExpired() によって実行する.
 */
static dkim_stat_t
DkimSignature_validate(DkimSignature *self)
{
    // From: ヘッダが sig-h-tag に含まれているかの確認
    // [RFC4871] 6.1.1.
    // If the "h=" tag does not include the From header field, the verifier
    // MUST ignore the DKIM-Signature header field and return PERMFAIL (From
    // field not signed).
    if (0 > StrArray_linearSearchIgnoreCase(self->selecthdr, FROMHEADER)) {
        DkimLogPermFail(self->policy, "sig-h-tag doesn't include " FROMHEADER " header");
        return DSTAT_PERMFAIL_FROM_FIELD_NOT_SIGNED;
    }   // end if

    if (0 > time(&(self->verify_timestamp))) {
        DkimLogImplError(self->policy, "time(2) failed: err=%s", strerror(errno));
        return DSTAT_SYSERR_IMPLERROR;
    }   // end if

    // timestamp が未来になっていないかチェック
    if (self->parsed_t && (long long) self->verify_timestamp < self->genarated_time) {
        DkimLogPermFail(self->policy,
                        "this signature had generated in the future: timestamp=%lld, now=%ld",
                        self->genarated_time, self->verify_timestamp);
        return DSTAT_PERMFAIL_TIMESTAMP_DISCREPANCY;
    }   // end if

    // sig-t-tag と sig-x-tag の整合性チェック
    if (self->parsed_t && self->parsed_x && self->expire_limit < self->genarated_time) {
        DkimLogPermFail(self->policy,
                        "signature timestamp has discrepancy: timestamp=%lld, expire=%lld",
                        self->genarated_time, self->expire_limit);
        return DSTAT_PERMFAIL_TIMESTAMP_DISCREPANCY;
    }   // end if

    // sig-i-tag と sig-d-tag の関係を検証
    if (self->parsed_i) {
        // sig-i-tag が指定されて *いる* 場合
        // -> sig-d-tag が sig-i-tag と一致するか親ドメインであることを確認する.
        // [RFC4871] 6.1.1.
        // Verifiers MUST confirm that the domain specified in the "d=" tag is
        // the same as or a parent domain of the domain part of the "i=" tag.
        // If not, the DKIM-Signature header field MUST be ignored and the
        // verifier should return PERMFAIL (domain mismatch).
        if (!InetDomain_isParent(self->domain, InetMailbox_getDomain(self->identity))) {
            DkimLogPermFail
                (self->policy,
                 "sig-d-tag and sig-i-tag domain mismatch: sig-d-tag=%s, sig-i-tag-domain=%s",
                 self->domain, InetMailbox_getDomain(self->identity));
            return DSTAT_PERMFAIL_DOMAIN_MISMATCH;
        }   // end if
    } else {
        // sig-i-tag が指定されて *いない* 場合
        // -> sig-i-tag に sig-d-tag をセットする
        // [RFC4871] 6.1.1.
        // If the DKIM-Signature header field does not contain the "i=" tag, the
        // verifier MUST behave as though the value of that tag were "@d", where
        // "d" is the value from the "d=" tag.

        // 既に identity がセットされていないことを確認
        if (NULL != self->identity) {
            DkimLogImplError(self->policy, "sig-i-tag not set");
            return DSTAT_SYSERR_IMPLERROR;
        }   // end if

        self->identity = InetMailbox_build("", self->domain);
        if (NULL == self->identity) {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    }   // end if

    return DSTAT_OK;
}   // end function : DkimSignature_validate

/**
 * 署名ヘッダの期限を確認する.
 * メッセージヘッダをパースして生成した場合のみ意味をなす.
 * 直接値を設定した DkimSignature オブジェクトに対して実行しても何もおこなわれない.
 * @return 署名の期限内ならば DSTAT_OK, 署名の期限を過ぎている場合は DSTAT_PERMFAIL_SIGNATURE_EXPIRED.
 */
dkim_stat_t
DkimSignature_isExpired(const DkimSignature *self)
{
    // [RFC4871] 6.1.1.
    // Verifiers MAY ignore the DKIM-Signature header field and return
    // PERMFAIL (signature expired) if it contains an "x=" tag and the
    // signature has expired.
    if (self->parsed_x && self->expire_limit < self->verify_timestamp) {
        DkimLogPermFail(self->policy, "signature has expired: expire=%lld, now=%ld",
                        self->expire_limit, self->verify_timestamp);
        return DSTAT_PERMFAIL_SIGNATURE_EXPIRED;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignature_isExpired

/*
 * @attention レコード末尾に余分な CRLF がついているだけでもエラーになる (ABNF にマッチしなくなるから)
 */
DkimSignature *
DkimSignature_build(const DkimPolicy *policy, const char *headerf, const char *headerv,
                    dkim_stat_t *dstat)
{
    DkimSignature *self = DkimSignature_new(policy);
    if (NULL == self) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        return NULL;
    }   // end if

    if (NULL == (self->rawname = strdup(headerf))
        || NULL == (self->rawvalue = strdup(headerv))) {
        DkimLogNoResource(policy);
        SETDEREF(dstat, DSTAT_SYSERR_NORESOURCE);
        goto cleanup;
    }   // end if

    dkim_stat_t build_stat =
        DkimTvobj_build((DkimTvobj *) self, self->rawvalue, STRTAIL(self->rawvalue), false);
    if (DSTAT_OK != build_stat) {
        SETDEREF(dstat, build_stat);
        goto cleanup;
    }   // end if

    dkim_stat_t validate_stat = DkimSignature_validate(self);
    if (DSTAT_OK != validate_stat) {
        SETDEREF(dstat, validate_stat);
        goto cleanup;
    }   // end if

    SETDEREF(dstat, DSTAT_OK);
    return self;

  cleanup:
    DkimSignature_free(self);
    return NULL;
}   // end function : DkimSignature_build

void
DkimSignature_free(DkimSignature *self)
{
    assert(self);
    free(self->rawname);
    free(self->rawvalue);
    free(self->selector);
    free(self->domain);
    if (NULL != self->identity) {
        InetMailbox_free(self->identity);
    }   // end if
    if (NULL != self->headerhash) {
        XBuffer_free(self->headerhash);
    }   // end if
    if (NULL != self->bodyhash) {
        XBuffer_free(self->bodyhash);
    }   // end if
    if (NULL != self->selecthdr) {
        StrArray_free(self->selecthdr);
    }   // end if
    if (NULL != self->keyretr) {
        IntArray_free(self->keyretr);
    }   // end if
    free(self);
}   // end function : DkimSignature_free

/*
 * DkimSignature オブジェクトに設定してある情報を元に DKIM-Signature ヘッダを生成する.
 */
dkim_stat_t
DkimSignature_buildRawHeader(DkimSignature *self, bool digestmode,
                             const char **rawheaderf, const char **rawheaderv)
{
    dkim_stat_t final_stat;
    dkim_stat_t encode_stat;

    PTRINIT(self->rawname);
    PTRINIT(self->rawvalue);

    // 初期サイズは決め打ちで問題ない
    // BUFSIZ byte までならメモリを確保し直さずに処理できる
    // BUFSIZ byte を越えてもメモリの再確保に際して少し遅くなるだけ
    FoldString *fstr = FoldString_new(BUFSIZ);
    if (NULL == fstr) {
        DkimLogNoResource(self->policy);
        final_stat = DSTAT_SYSERR_NORESOURCE;
        goto cleanup;
    }   // end if

    // メモリ再確保の際は 256 byte ずつ増加させる
    FoldString_setGrowth(fstr, 256);
    // 1 行あたり 78 byte を越えないように頑張る
    FoldString_setLineLengthLimits(fstr, DKIM_SIGNATURE_HEADER_WIDTH);
    // ダイジェスト計算用のヘッダを生成する際は CRLF,
    // メッセージへの挿入用のヘッダを生成する際は LF
    FoldString_setFoldingCR(fstr, digestmode);

    // "DKIM-Signature: " の分, 2 は ": " の分
    FoldString_consumeLineSpace(fstr, strlen(DKIM_SIGNHEADER) + 2);

    // sig-v-tag
    FoldString_appendBlock(fstr, true, "v=1;");

    // sig-a-tag
    FoldString_appendBlock(fstr, true, "a=");
    FoldString_appendBlock(fstr, true, DkimEnum_lookupPubkeyAlgByValue(self->pubkeyalg));
    FoldString_appendChar(fstr, false, '-');
    FoldString_appendBlock(fstr, false, DkimEnum_lookupDigestAlgByValue(self->digestalg));
    FoldString_appendChar(fstr, true, ';');

    // sig-c-tag
    FoldString_appendBlock(fstr, true, "c=");
    FoldString_appendBlock(fstr, true, DkimEnum_lookupCanonAlgByValue(self->headercanon));
    FoldString_appendChar(fstr, false, '/');
    FoldString_appendBlock(fstr, false, DkimEnum_lookupCanonAlgByValue(self->bodycanon));
    FoldString_appendChar(fstr, true, ';');

    // sig-d-tag
    FoldString_appendBlock(fstr, true, "d=");
    FoldString_appendBlock(fstr, true, self->domain);
    FoldString_appendChar(fstr, true, ';');

    // sig-h-tag
    size_t header_num = StrArray_getCount(self->selecthdr);
    size_t i = 0;
    // 先頭に ':' を付加しないための小細工
    FoldString_appendBlock(fstr, true, "h=");
    FoldString_appendBlock(fstr, true, StrArray_get(self->selecthdr, i));
    for (++i; i < header_num; ++i) {
        FoldString_appendChar(fstr, true, ':');
        FoldString_appendBlock(fstr, true, StrArray_get(self->selecthdr, i));
    }   // end if
    FoldString_appendChar(fstr, true, ';');

    // sig-i-tag
    if (NULL != self->identity) {
        // identity の localpart は dkim-quoted-printable でエンコードしなければならない.
        const char *identity_localpart = InetMailbox_getLocalPart(self->identity);
        XBuffer *quoted_localpart =
            DkimParse_encodeLocalpartToDkimQuotedPrintable(self->policy, identity_localpart,
                                                           strlen(identity_localpart),
                                                           &encode_stat);
        if (NULL == quoted_localpart) {
            final_stat = encode_stat;
            goto cleanup;
        }   // end if
        XBuffer_appendChar(quoted_localpart, '@');
        XBuffer_appendString(quoted_localpart, InetMailbox_getDomain(self->identity));
        if (0 != XBuffer_status(quoted_localpart)) {
            DkimLogNoResource(self->policy);
            XBuffer_free(quoted_localpart);
            final_stat = DSTAT_SYSERR_NORESOURCE;
            goto cleanup;
        }   // end if

        FoldString_appendBlock(fstr, true, "i=");
        FoldString_appendBlock(fstr, true, XBuffer_getString(quoted_localpart));
        FoldString_appendChar(fstr, true, ';');

        XBuffer_free(quoted_localpart);
    }   // end if

    // sig-q-tag
    size_t keyretr_num = IntArray_getCount(self->keyretr);
    if (0 < keyretr_num) {
        for (i = 0; i < keyretr_num; ++i) {
            if (0 == i) {
                FoldString_appendBlock(fstr, true, "q=");
            } else {
                FoldString_appendChar(fstr, true, ':');
            }   // end if
            const char *keyretr_string =
                DkimEnum_lookupKeyRetrByValue((dkim_keyretr_t) IntArray_get(self->keyretr, i));
            FoldString_appendBlock(fstr, true, keyretr_string);
        }   // end if
        FoldString_appendChar(fstr, true, ';');
    }   // end if

    // sig-s-tag
    FoldString_appendBlock(fstr, true, "s=");
    FoldString_appendBlock(fstr, true, self->selector);
    FoldString_appendChar(fstr, true, ';');

    // sig-t-tag
    FoldString_appendBlock(fstr, true, "t=");
    FoldString_appendFormatBlock(fstr, true, "%lld;", self->genarated_time);

    // sig-x-tag
    if (0 <= self->expire_limit) {
        FoldString_appendBlock(fstr, true, "x=");
        FoldString_appendFormatBlock(fstr, true, "%lld;", self->expire_limit);
    }   // end if

    // sig-bh-tag
    const void *buf = XBuffer_getBytes(self->bodyhash);
    size_t buflen = XBuffer_getSize(self->bodyhash);

    XBuffer *xbuf = DkimParse_encodeBase64(self->policy, buf, buflen, &encode_stat);
    if (NULL == xbuf) {
        final_stat = encode_stat;
        goto cleanup;
    }   // end if
    FoldString_appendBlock(fstr, true, "bh=");
    FoldString_appendNonBlock(fstr, true, XBuffer_getString(xbuf));
    FoldString_appendChar(fstr, true, ';');
    XBuffer_free(xbuf);
    xbuf = NULL;    // 重要

    // sig-b-tag
#define DKIM_EMPTY_B_TAG_VALUE "b=;"
    if (digestmode) {
        FoldString_appendBlock(fstr, true, DKIM_EMPTY_B_TAG_VALUE);
    } else {
        if (NULL == self->headerhash) {
            DkimLogImplError(self->policy, "header-hash is NULL");
            final_stat = DSTAT_SYSERR_IMPLERROR;
            goto cleanup;
        }   // end if
        buf = XBuffer_getBytes(self->headerhash);
        buflen = XBuffer_getSize(self->headerhash);
        xbuf = DkimParse_encodeBase64(self->policy, buf, buflen, &encode_stat);
        if (NULL == xbuf) {
            final_stat = encode_stat;
            goto cleanup;
        }   // end if

        FoldString_precede(fstr, strlen(DKIM_EMPTY_B_TAG_VALUE));   // digestmode が真の場合と対称に改行をおこなうために必要
        FoldString_appendBlock(fstr, false, "b=");
        FoldString_appendNonBlock(fstr, false, XBuffer_getString(xbuf));
        FoldString_appendChar(fstr, false, ';');
        XBuffer_free(xbuf);
        xbuf = NULL;    // 重要
    }   // end if

    // ここまでの追加操作で FoldString にエラーが起きていないかを確認
    if (0 != FoldString_status(fstr)) {
        DkimLogNoResource(self->policy);
        final_stat = DSTAT_SYSERR_NORESOURCE;
        goto cleanup;
    }   // end if

    // 生成したヘッダをフィールドに保存
    self->rawname = strdup(DKIM_SIGNHEADER);
    if (NULL == self->rawname) {
        DkimLogNoResource(self->policy);
        final_stat = DSTAT_SYSERR_NORESOURCE;
        goto cleanup;
    }   // end if
    self->rawvalue = strdup(FoldString_getString(fstr));
    if (NULL == self->rawvalue) {
        DkimLogNoResource(self->policy);
        final_stat = DSTAT_SYSERR_NORESOURCE;
        goto cleanup;
    }
    FoldString_free(fstr);

    SETDEREF(rawheaderf, self->rawname);
    SETDEREF(rawheaderv, self->rawvalue);

    return DSTAT_OK;

  cleanup:
    if (NULL != fstr) {
        FoldString_free(fstr);
    }   // end if
    SETDEREF(rawheaderf, NULL);
    SETDEREF(rawheaderv, NULL);
    return final_stat;
}   // end function : DkimSignature_buildRawHeader

dkim_stat_t
DkimSignature_addSelectHeaderName(DkimSignature *self, const char *headerf)
{
    if (0 > StrArray_append(self->selecthdr, headerf)) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignature_addSelectHeaderName

/**
 *  headerf で与えるヘッダが sig-h-tag に含まれるかを返す.
 * @return headerf が署名対象に含まれる場合は true, 含まれない場合は false.
 */
bool
DkimSignature_isHeaderSigned(const DkimSignature *self, const char *headerf)
{
    assert(NULL != self);
    return 0 <= StrArray_linearSearchIgnoreCase(self->selecthdr, headerf);
}   // end function : DkimSignature_isHeaderSigned

////////////////////////////////////////////////////////////////////////
// accessor

const char *
DkimSignature_getSigningDomain(const DkimSignature *self)
{
    return self->domain;
}   // end function : DkimSignature_getSigningDomain

dkim_stat_t
DkimSignature_setSigningDomain(DkimSignature *self, const char *domain)
{
    PTRINIT(self->domain);
    if (NULL != domain) {
        self->domain = strdup(domain);
        if (NULL == self->domain) {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignature_setSigningDomain

const char *
DkimSignature_getSelector(const DkimSignature *self)
{
    return self->selector;
}   // end function : DkimSignature_getSelector

dkim_stat_t
DkimSignature_setSelector(DkimSignature *self, const char *selector)
{
    PTRINIT(self->selector);
    if (NULL != selector) {
        self->selector = strdup(selector);
        if (NULL == self->selector) {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignature_setSelector

dkim_digestalg_t
DkimSignature_getDigestAlg(const DkimSignature *self)
{
    return self->digestalg;
}   // end function : DkimSignature_getDigestAlg

void
DkimSignature_setDigestAlg(DkimSignature *self, dkim_digestalg_t digestalg)
{
    self->digestalg = digestalg;
}   // end function : DkimSignature_setDigestAlg

dkim_pubkeyalg_t
DkimSignature_getPubKeyAlg(const DkimSignature *self)
{
    return self->pubkeyalg;
}   // end function : DkimSignature_getPubKeyAlg

void
DkimSignature_setPubKeyAlg(DkimSignature *self, dkim_pubkeyalg_t pubkeyalg)
{
    self->pubkeyalg = pubkeyalg;
}   // end function : DkimSignature_setPubKeyAlg

long long
DkimSignature_getGeneratedTime(const DkimSignature *self)
{
    return self->genarated_time;
}   // end function : DkimSignature_getGenerateTime

void
DkimSignature_setGeneratedTime(DkimSignature *self, long long genaratetime)
{
    self->genarated_time = genaratetime;
}   // end function : DkimSignature_setGenerateTime

long long
DkimSignature_getSignatureTTL(const DkimSignature *self)
{
    return self->expire_limit;
}   // end function : DkimSignature_setExpireTime

/**
 * @attention DkimSignature_setGeneratedTime() をあらかじめ呼んでおく必要がある.
 */
long long
DkimSignature_setSignatureTTL(DkimSignature *self, long long signature_ttl)
{
    if (0LL < self->genarated_time && 0LL < signature_ttl) {
        self->expire_limit = self->genarated_time + signature_ttl;
    } else {
        self->expire_limit = -1LL;
    }   // end if
    return self->expire_limit;
}   // end function : DkimSignature_setExpireTime

const XBuffer *
DkimSignature_getHeaderHash(const DkimSignature *self)
{
    return self->headerhash;
}   // end function : DkimSignature_getHeaderHash

dkim_stat_t
DkimSignature_setHeaderHash(DkimSignature *self, unsigned char *hashbuf, unsigned int hashlen)
{
    if (NULL == self->headerhash) {
        self->headerhash = XBuffer_new(hashlen);
        if (NULL == self->headerhash) {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    } else {
        XBuffer_reset(self->headerhash);
    }   // end if

    if (0 > XBuffer_appendBytes(self->headerhash, hashbuf, hashlen)) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignature_setHeaderHash

const XBuffer *
DkimSignature_getBodyHash(const DkimSignature *self)
{
    return self->bodyhash;
}   // end function : DkimSignature_getBodyHash

dkim_stat_t
DkimSignature_setBodyHash(DkimSignature *self, unsigned char *hashbuf, unsigned int hashlen)
{
    if (NULL == self->bodyhash) {
        self->bodyhash = XBuffer_new(hashlen);
        if (NULL == self->bodyhash) {
            DkimLogNoResource(self->policy);
            return DSTAT_SYSERR_NORESOURCE;
        }   // end if
    } else {
        XBuffer_reset(self->bodyhash);
    }   // end if

    if (0 > XBuffer_appendBytes(self->bodyhash, hashbuf, hashlen)) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if
    return DSTAT_OK;
}   // end function : DkimSignature_setBodyHash

const StrArray *
DkimSignature_getSelectHeader(const DkimSignature *self)
{
    return self->selecthdr;
}   // end function : DkimSignature_getSelectHeader

dkim_canonalg_t
DkimSignature_getHeaderCanonAlg(const DkimSignature *self)
{
    return self->headercanon;
}   // end function : DkimSignature_getHeaderCanonAlg

void
DkimSignature_setHeaderCanonAlg(DkimSignature *self, dkim_canonalg_t headercanon)
{
    self->headercanon = headercanon;
}   // end function : DkimSignature_setHeaderCanonAlg

dkim_canonalg_t
DkimSignature_getBodyCanonAlg(const DkimSignature *self)
{
    return self->bodycanon;
}   // end function : DkimSignature_getBodyCanonAlg

void
DkimSignature_setBodyCanonAlg(DkimSignature *self, dkim_canonalg_t bodycanon)
{
    self->bodycanon = bodycanon;
}   // end function : DkimSignature_setBodyCanonAlg

long long
DkimSignature_getBodyLengthLimit(const DkimSignature *self)
{
    return self->body_length_limit;
}   // end function : DkimSignature_getBodyLengthLimit

void
DkimSignature_setBodyLengthLimit(DkimSignature *self, long long body_length_limit)
{
    self->body_length_limit = body_length_limit;
}   // end function : DkimSignature_setBodyLengthLimit

const char *
DkimSignature_getRawHeaderName(const DkimSignature *self)
{
    return self->rawname;
}   // end function : DkimSignature_getRawHeaderName

const char *
DkimSignature_getRawHeaderValue(const DkimSignature *self)
{
    return self->rawvalue;
}   // end function : DkimSignature_getRawHeaderValue

void
DkimSignature_getReferenceToBodyHashOfRawHeaderValue(const DkimSignature *self, const char **head,
                                                     const char **tail)
{
    *head = self->raw_value_b_head;
    *tail = self->raw_value_b_tail;
}   // end function : DkimSignature_getReferenceToBodyHashOfRawHeaderValue

const InetMailbox *
DkimSignature_getIdentity(const DkimSignature *self)
{
    return self->identity;
}   // end function : DkimSignature_getIdentity

dkim_stat_t
DkimSignature_setIdentity(DkimSignature *self, const InetMailbox *mailbox)
{
    InetMailbox *new_mailbox = InetMailbox_duplicate(mailbox);
    if (NULL == new_mailbox) {
        DkimLogNoResource(self->policy);
        return DSTAT_SYSERR_NORESOURCE;
    }   // end if
    if (NULL != self->identity) {
        InetMailbox_free(self->identity);
    }   // end if
    self->identity = new_mailbox;
    return DSTAT_OK;
}   // end function : DkimSignature_setIdentity

const IntArray *
DkimSignature_getKeyRetrMethod(const DkimSignature *self)
{
    return self->keyretr;
}   // end function : DkimSignature_getKeyRetrMethod
