/* This file is part of the KDE project

   Copyright (C) 2005 Dario Massarin <nekkar@libero.it>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
*/

#ifndef JOB_H
#define JOB_H

/**
 *  @brief Job class
 *
 *  We want to abstract this common interface in order to simplify the 
 *  Scheduler code. A Job can be either a Transfer or a search through the net.
 *  It is basically something you execute in background and that the scheduler 
 *  can decide to start, stop or cancel. In this way we don't expose the complex 
 *  API of a Transfer (or a Search), to the scheduler.
 *  By definition a job must always belong to a JobQueue (see jobqueue.h).
 **/

#include "kget_export.h"

class QDomNode;

class Scheduler;
class JobQueue;

class KGET_EXPORT Job
{
    public:
        /**
         * The status property describes the current job status
         */
        enum Status {
            Running, /// The job is being executed
            Delayed, /// The job is delayed. This means that the scheduler should
                     /// not start it until it exits from the delayed state
            Stopped, /// The job is stopped
            Aborted, /// The job is stopped, but this also indicates that it
                     /// stopped because an error occurred
            Finished /// The job exited from its Running state successfully
        };

        /**
         * The policy property describes how the scheduler should manage this job.
         */
        enum Policy {
            Start, /// The scheduler should start this job even if its queue
                   /// isn't in a Running status
            Stop,  /// The scheduler shouldn't never start this job, even if
                   /// if its queue is in a Running status
            None   /// The scheduler should start this job depending on its
                   /// queue status
        };

        Job(JobQueue * parent, Scheduler * scheduler);
        virtual ~Job();

        //Job commands
        virtual void start()=0;
        virtual void stop()=0;

        virtual void setDelay(int seconds)=0;
        virtual void delayTimerEvent()=0;

        virtual void postDeleteEvent() {}

        JobQueue * jobQueue() {return m_jobQueue;}

        //Job properties
        void setStatus(Status jobStatus);
        void setPolicy(Policy jobPolicy);

        Status status() const {return m_status;}
        Policy policy() const {return m_policy;}

        virtual int elapsedTime() const =0;
        virtual int remainingTime() const =0;

        //Job capabilities
        virtual bool isResumable() const =0;

    protected:
        Scheduler * scheduler() const {return m_scheduler;}

        void read(QDomNode * n);
        void write(QDomNode * n);

        /**
         * This one posts a job event to the scheduler
         */
        void postJobEvent(Status); //do we need a JobEvent instead of JobStatus?

        JobQueue *  m_jobQueue;
        Scheduler * m_scheduler;

    private:
        Status m_status;
        Policy m_policy;
};

#endif
