/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QEdje.
 *
 * QEdje is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QEdje is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QEdje.  If not, see <http://www.gnu.org/licenses/>.
 *
 * This file incorporates work covered by the license described in the
 * file COPYING.libedje.
 */

#include "qedje.h"
#include "qedjeloader.h"

#if !defined(QT_NO_DEBUG_STREAM)

/*!
    Returns a string representation of a QColor.
*/
static QString strColor(const QColor s)
{
    return QString("(%1, %2, %3, %4)").arg(s.red())
        .arg(s.green()).arg(s.blue()).arg(s.alpha());
}

/*!
    Print all internal data of a QEdjeColorClass.
*/
QDebug operator<<(QDebug dbg, const QEdjeColorClass &e)
{
    dbg.nospace() << "\n.. [colorClass: name = "
                  << e.name << "]";
    dbg.nospace() << "\n..... color: " << strColor(e.color);
    dbg.nospace() << "\n..... color2: " << strColor(e.color2);
    dbg.nospace() << "\n..... color3: " << strColor(e.color3);

    return dbg.space();
}

/*!
    Print all internal data of a QEdjePartDescription.
*/
QDebug operator<<(QDebug dbg, const QEdjePartDescription &desc)
{
    dbg.nospace() << "\n........ [description: name = " << desc.state.name
                  << ", value = " << desc.state.value << "]";
    dbg.nospace() << "\n........... align: " << desc.align;

    dbg.nospace() << "\n........... visible: " << desc.visible;
    dbg.nospace() << "\n........... imageId: " << desc.imageId;

    dbg.nospace() << "\n........... fixedWidth: " << desc.fixedWidth;
    dbg.nospace() << "\n........... fixedHeight: " << desc.fixedHeight;

    dbg.nospace() << "\n........... min: " << desc.min;
    dbg.nospace() << "\n........... max: " << desc.max;
    dbg.nospace() << "\n........... step: " << desc.step;

    dbg.nospace() << "\n........... color: " << strColor(desc.color);
    dbg.nospace() << "\n........... color2: " << strColor(desc.color2);
    dbg.nospace() << "\n........... color3: " << strColor(desc.color3);

    dbg.nospace() << "\n........... aspect min: " << desc.aspect.min;
    dbg.nospace() << "\n........... aspect max: " << desc.aspect.max;
    dbg.nospace() << "\n........... aspect pref: " << desc.aspect.prefer;

    dbg.nospace() << "\n........... [rel1]";
    dbg.nospace() << "\n.............. idx: " << desc.rel1.idX;
    dbg.nospace() << "\n.............. idy: " << desc.rel1.idY;
    dbg.nospace() << "\n.............. offset: " << desc.rel1.offset;
    dbg.nospace() << "\n.............. relative: " << desc.rel1.relative;

    dbg.nospace() << "\n........... [rel2]";
    dbg.nospace() << "\n.............. idx: " << desc.rel2.idX;
    dbg.nospace() << "\n.............. idy: " << desc.rel2.idY;
    dbg.nospace() << "\n.............. offset: " << desc.rel2.offset;
    dbg.nospace() << "\n.............. relative: " << desc.rel2.relative;

    dbg.nospace() << "\n........... [text]";
    dbg.nospace() << "\n.............. text: " << desc.text.text;
    dbg.nospace() << "\n.............. textClass: " << desc.text.textClass;
    dbg.nospace() << "\n.............. style: " << desc.text.style;
    dbg.nospace() << "\n.............. font: " << desc.text.font;
    dbg.nospace() << "\n.............. align: " << desc.text.align;
    dbg.nospace() << "\n.............. elipsis: " << desc.text.elipsis;
    dbg.nospace() << "\n.............. size: " << desc.text.size;
    dbg.nospace() << "\n.............. idSource: " << desc.text.idSource;
    dbg.nospace() << "\n.............. idTextSource: " << desc.text.idTextSource;
    dbg.nospace() << "\n.............. fitX: " << desc.text.fitX;
    dbg.nospace() << "\n.............. fitY: " << desc.text.fitY;
    dbg.nospace() << "\n.............. minX: " << desc.text.minX;
    dbg.nospace() << "\n.............. minY: " << desc.text.minY;
    dbg.nospace() << "\n.............. maxX: " << desc.text.maxX;
    dbg.nospace() << "\n.............. maxY: " << desc.text.maxY;

    // TWEEN LIST TODO

    return dbg.space();
}

/*!
    Print all internal data of a QEdjePartCollection.
*/
QDebug operator<<(QDebug dbg, QEdjePartCollection *e)
{
    dbg.nospace() << "\n.. [group: id = " << e->id
                  << ", name = " << e->part << "]";
    dbg.nospace() << "\n..... min: " << e->propMin;
    dbg.nospace() << "\n..... max: " << e->propMax;

    foreach (QEdjeProgram *p, e->programs) {
        dbg.nospace() << "\n..... [program: id = " << p->id
                      << ", name = " << p->name << "]";
        dbg.nospace() << "\n........ signal: " << p->signal;
        dbg.nospace() << "\n........ source: " << p->source;
        dbg.nospace() << "\n........ action: " << p->action;
        dbg.nospace() << "\n........ state: " << p->state;
        dbg.nospace() << "\n........ state2: " << p->state2;
        dbg.nospace() << "\n........ value: " << p->value;
        dbg.nospace() << "\n........ value2: " << p->value2;
        dbg.nospace() << "\n........ inFrom: " << p->inFrom;
        dbg.nospace() << "\n........ inRange: " << p->inRange;
        dbg.nospace() << "\n........ tweenMode: " << p->tweenMode;
        dbg.nospace() << "\n........ tweenTime: " << p->tweenTime;
        dbg.nospace() << "\n........ targets: " << p->targets;
        dbg.nospace() << "\n........ after: " << p->after;
    }

    foreach (QEdjePart *p, e->parts) {
        dbg.nospace() << "\n..... [part: id = " << p->id
                      << ", name = " << p->name
                      << ", type = " << p->type << "]";
        dbg.nospace() << "\n........ effect: " <<  p->effect;
        dbg.nospace() << "\n........ clip_to: " << p->clipToId;
        dbg.nospace() << "\n........ source: " << p->source;
        dbg.nospace() << "\n........ mouse_events: " << p->mouseEvents;
        dbg.nospace() << "\n........ repeat_events: " << p->repeatEvents;
        dbg.nospace() << "\n........ precise_is_inside: " << p->preciseIsInside;
        dbg.nospace() << "\n........ use_alternate_font_metrics: " << p->useAlternateFontMetrics;

        dbg.nospace() << "\n........ [dragable]";
        dbg.nospace() << "\n........... x: " << p->dragable.x;
        dbg.nospace() << "\n........... y: " << p->dragable.y;
        dbg.nospace() << "\n........... step: " << p->dragable.step;
        dbg.nospace() << "\n........... count: " << p->dragable.count;
        dbg.nospace() << "\n........... eventsId: " << p->dragable.eventsId;
        dbg.nospace() << "\n........... confineId: " << p->dragable.confineId;

        dbg.nospace() << p->defaultDescription;

        foreach (QEdjePartDescription *e, p->otherDescriptions)
            dbg.nospace() << *e;
    }

    foreach (QEdjeData *p, e->data) {
        dbg.nospace() << "\n..... [data]";
        dbg.nospace() << "\n........ key: " << p->key;
        dbg.nospace() << "\n........ value: " << p->value;
    }

    return dbg.space();
}

/*!
    Print all internal data of a QEdjeStyle.
*/
QDebug operator<<(QDebug dbg, const QEdjeStyle &e)
{
    dbg.nospace() << "\n.. [style: name = "
                  << e.name << "]";
    foreach(QEdjeStyleTag *tag, e.tags) {
        dbg.nospace() << "\n..... [tag]";
        dbg.nospace() << "\n........ key: " << tag->key;
        dbg.nospace() << "\n........ value: " << tag->value;
        dbg.nospace() << "\n........ font: " << tag->font;
        dbg.nospace() << "\n........ fontSize: " << tag->fontSize;
        dbg.nospace() << "\n........ textClass: " << tag->textClass;
    }
}

/*!
    Print all internal data of a QEdjeFile.
*/
QDebug operator<<(QDebug dbg, const QEdjeFile &s)
{
    dbg.nospace() << "[QEdjeFile]";
    dbg.nospace() << "\n.. version: " << s.version;
    dbg.nospace() << "\n.. compiler: " << s.compiler;
    dbg.nospace() << "\n.. feature_ver: " << s.featureVer;

    foreach (QEdjeStyle *e, s.styles) {
        dbg.nospace() << *e;
    }

    foreach (QEdjeColorClass *e, s.colorClasses) {
        dbg.nospace() << *e;
    }

    dbg.nospace() << "\n.. [font_dir_entry]";
    foreach (QEdjeFontDirectoryEntry *e, s.fontDir) {
        dbg.nospace() << "\n..... (entry = " << e->entry
                      << ", path = " << e->path << ")";
    }

    dbg.nospace() << "\n.. [image_dir_entry]";
    foreach (QEdjeImageDirectoryEntry *e, s.imageDir) {
        dbg.nospace() << "\n..... (id = " << e->id
                      << ", entry: " << e->entry
                      << ", sourceType: " << e->sourceType
                      << ", sourceParam: " << e->sourceParam
                      << ")";
    }

    dbg.nospace() << "\n.. [collection_dir_entry]";
    foreach (QEdjePartCollectionDirectoryEntry *e, s.collectionDir) {
        dbg.nospace() << "\n..... (id = " << e->id
                      << ", entry = " << e->entry << ")";
    }

    dbg.nospace() << "\n.. [data]";
    foreach (QEdjeData *e, s.data) {
        dbg.nospace() << "\n..... (key = " << e->key
                      << ", value = " << e->value << ")";
    }

    foreach (QEdjePartCollection *e, s.collectionHash) {
        dbg.nospace() << e;
    }

    return dbg.space();
}

#endif
