/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A Graphic view with more functionalities for skrooge.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skggraphicsview.h"
#include "skgservices.h"
#include "skgtraces.h"
#include "skgmainpanel.h"

#include <kmenu.h>
#include <kfiledialog.h>

#include <QDesktopServices>
#include <QPrinter>
#include <QPrintDialog>
#include <QClipboard>
#include <QApplication>
#include <QDomDocument>
#include <QSvgGenerator>
#include <qmath.h>

SKGGraphicsView::SKGGraphicsView(QWidget *parent)
                : QWidget(parent)
{
        ui.setupUi(this);

        ui.kGraphicsView->setBackgroundBrush(Qt::white);

        //Set icons
        ui.kZoomIn->setIcon(KIcon("zoom-in"));
        ui.kZoomOut->setIcon(KIcon("zoom-out"));
        ui.kZoomOriginal->setIcon(KIcon("zoom-original"));
        ui.kPrint->setIcon(KIcon("printer"));
        ui.kCopy->setIcon(KIcon("edit-copy"));

        //Build contextual menu
        ui.kGraphicsView->setContextMenuPolicy(Qt::CustomContextMenu);
        connect(ui.kGraphicsView, SIGNAL(customContextMenuRequested(const QPoint & ) ),this,SLOT(showMenu( const QPoint& )));

        mainMenu = new KMenu(ui.kGraphicsView);
        mainMenu->addTitle(i18n("Graphic"));

        QAction* actZoomOriginal = mainMenu->addAction(ui.kZoomOriginal->icon(), ui.kZoomOriginal->toolTip());
        connect(actZoomOriginal, SIGNAL(triggered(bool)), this, SLOT(onZoomOriginal()));

        QAction* actZoomIn = mainMenu->addAction(ui.kZoomIn->icon(), ui.kZoomIn->toolTip());
        connect(actZoomIn, SIGNAL(triggered(bool)), this, SLOT(onZoomIn()));

        QAction* actZoomOut = mainMenu->addAction(ui.kZoomOut->icon(), ui.kZoomOut->toolTip());
        connect(actZoomOut, SIGNAL(triggered(bool)), this, SLOT(onZoomOut()));

        mainMenu->addSeparator();

        QAction* actCopy = mainMenu->addAction(ui.kCopy->icon(), ui.kCopy->toolTip());
        connect(actCopy, SIGNAL(triggered(bool)), this, SLOT(onCopy()));

        QAction* actPrint = mainMenu->addAction(ui.kPrint->icon(), ui.kPrint->toolTip());
        connect(actPrint, SIGNAL(triggered(bool)), this, SLOT(onPrint()));

        QMenu* exp=mainMenu->addMenu(i18n("Export"));

        QAction* actPDF = exp->addAction(KIcon("application-pdf"), i18n("Export PDF..."));
        connect(actPDF, SIGNAL(triggered(bool)), this, SLOT(onSavePDF()));

        QAction* actSvg = exp->addAction(KIcon("image-svg+xml"), i18n("Export SVG..."));
        connect(actSvg, SIGNAL(triggered(bool)), this, SLOT(onSaveSvg()));

        QAction* actImage = exp->addAction(KIcon("image-png"), i18n("Export image..."));
        connect(actImage, SIGNAL(triggered(bool)), this, SLOT(onSaveImage()));

        mainMenu->addSeparator();

        actShowToolBar = mainMenu->addAction(i18n("Show tool bar"));
        if (actShowToolBar) {
                actShowToolBar->setCheckable(true);
                actShowToolBar->setChecked(true);
                connect(actShowToolBar, SIGNAL(triggered(bool)), this, SLOT(onSwitchToolBarVisibility()));
        }

        actAntialiasing = mainMenu->addAction(i18n("Antialiasing"));
        if (actAntialiasing) {
                actAntialiasing->setCheckable(true);
                actAntialiasing->setChecked(true);
                onSwitchAntialiasing();
                connect(actAntialiasing, SIGNAL(triggered(bool)), this, SLOT(onSwitchAntialiasing()));
        }
}

SKGGraphicsView::~SKGGraphicsView()
{
        actAntialiasing=NULL;
        actShowToolBar=NULL;
        mainMenu=NULL;
}

void SKGGraphicsView::setScene ( QGraphicsScene * iScene )
{
        ui.kGraphicsView->setScene(iScene);
}

QGraphicsView* SKGGraphicsView::graphicsView()
{
        return ui.kGraphicsView;
}

QString SKGGraphicsView::getState() const
{
        SKGTRACEIN(10, "SKGGraphicsView::getState");
        QDomDocument doc("SKGML");
        QDomElement root = doc.createElement("parameters");
        doc.appendChild(root);

        root.setAttribute("isToolBarVisible", isToolBarVisible() ? "Y" : "N");
        if (actAntialiasing) root.setAttribute("antialiasing", actAntialiasing->isChecked() ? "Y" : "N");

        return doc.toString();
}

void SKGGraphicsView::setState(const QString& iState )
{
        SKGTRACEIN(10, "SKGGraphicsView::setState");
        QDomDocument doc("SKGML");
        if (doc.setContent(iState)) {
                QDomElement root = doc.documentElement();

                QString isToolBarVisible=root.attribute ("isToolBarVisible");
                if (!isToolBarVisible.isEmpty()) setToolBarVisible(isToolBarVisible=="Y");

                QString antialiasing=root.attribute ("antialiasing");
                if (!antialiasing.isEmpty() && actAntialiasing) actAntialiasing->setChecked(antialiasing=="Y");
        }
}

KMenu* SKGGraphicsView::getContextualMenu() const
{
        return mainMenu;
}

void SKGGraphicsView::showMenu(const QPoint& pos)
{
        if (mainMenu) mainMenu->popup(this->mapToGlobal(pos));
}

void SKGGraphicsView::setToolBarVisible(bool iVisibility)
{
        ui.toolBar->setVisible(iVisibility);
        if (actShowToolBar) actShowToolBar->setChecked(iVisibility);
}

bool SKGGraphicsView::isToolBarVisible() const
{
        return ui.toolBar->isVisible();
}

void SKGGraphicsView::onSwitchToolBarVisibility()
{
        setToolBarVisible(!isToolBarVisible());
}

void SKGGraphicsView::onSwitchAntialiasing()
{
        if (actAntialiasing) ui.kGraphicsView->setRenderHint(QPainter::Antialiasing, actAntialiasing->isChecked());
}

void SKGGraphicsView::onZoom()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onZoom");

        int sliderValue=ui.kZoomSlider->value();
        qreal scale = qPow(qreal(1.5), ( sliderValue) / qreal(50));
        QMatrix matrix;
        matrix.scale(scale*originalMatrix.m11(), scale*originalMatrix.m22());

        ui.kGraphicsView->setMatrix(matrix);
        if (sliderValue==0 && ui.kGraphicsView->scene()) {
                ui.kGraphicsView->fitInView(ui.kGraphicsView->scene()->sceneRect(), Qt::KeepAspectRatio);
                originalMatrix=ui.kGraphicsView->matrix();
        }
}

void SKGGraphicsView::onZoomIn()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onZoomIn");
        ui.kZoomSlider->setValue(ui.kZoomSlider->value() + 50);
}

void SKGGraphicsView::onZoomOut()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onZoomOut");
        ui.kZoomSlider->setValue(ui.kZoomSlider->value() - 50);
}

void SKGGraphicsView::onZoomOriginal()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onZoomOriginal");
        ui.kZoomSlider->setValue(0);
        onZoom();
}

void SKGGraphicsView::onPrint()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onPrint");
        QPrinter printer;
        QPrintDialog dialog(&printer, this);
        if (dialog.exec() == QDialog::Accepted) {
                QPainter painter(&printer);
                ui.kGraphicsView->render(&painter);
        }
}

void SKGGraphicsView::onSavePDF()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onSavePDF");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "application/pdf", this);
        if (fileName.isEmpty()) return;

        {
                QPrinter printer;
                printer.setOutputFileName(fileName);
                QPainter painter(&printer);
                ui.kGraphicsView->render(&painter);
        }
        QDesktopServices::openUrl(QUrl(fileName));
}

void SKGGraphicsView::onSaveSvg()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onSaveSvg");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "image/svg+xml", this);
        if (fileName.isEmpty()) return;

        {
                QSvgGenerator generator;
                generator.setFileName(fileName);
                QPainter painter(&generator);
                ui.kGraphicsView->render(&painter);
        }
        QDesktopServices::openUrl(QUrl(fileName));
}

void SKGGraphicsView::onSaveImage()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onCopy");
        QString fileName=SKGMainPanel::getSaveFileName(KUrl("kfiledialog:///IMPEXP"), "image/png image/jpeg image/gif image/tiff" , this);
        if (fileName.isEmpty()) return;

        {
                QImage image(ui.kGraphicsView->size(), QImage::Format_ARGB32);
                QPainter painter(&image);
                ui.kGraphicsView->render(&painter);
                image.save(fileName);
        }
        QDesktopServices::openUrl(QUrl(fileName));
}

void SKGGraphicsView::onCopy()
{
        _SKGTRACEIN(10, "SKGGraphicsView::onCopy");
        QClipboard *clipboard = QApplication::clipboard();
        if (clipboard) {

                QImage image(ui.kGraphicsView->size(), QImage::Format_ARGB32);
                QPainter painter(&image);
                ui.kGraphicsView->render(&painter);
                clipboard->setImage(image);
        }
}
#include "skggraphicsview.moc"

