/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGOBJECTMODELBASE_H
#define SKGOBJECTMODELBASE_H
/** @file
 * This file is part of Skrooge and defines classes skgbjectmodelbase.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectbase.h"

#include <kglobal.h>
#include <QAbstractItemModel>
#include <QStringList>

class SKGDocument;

/**
 * The Table model managing SKGObjectBase
 */
class KDE_EXPORT SKGObjectModelBase : public QAbstractItemModel
{
        Q_OBJECT
        Q_ENUMS(AttributeType);

public:
        /**
         * This structure represents a template to display data
         */
        typedef struct {
                /** Identifier of the schema */
                QString id;
                /** The nls name */
                QString name;
                /** The icon */
                QString icon;
                /** The schema. @see setSupportedAttributes */
                QString schema;
        } SKGModelTemplate;

        /**
         * This structure defines status for operations
         */
        enum AttributeType {INTEGER, 	/**< Integer */
                            DOUBLE, 	/**< Double */
                            CURRENCY, 	/**< Currency */
                            STRING, 	/**< String */
                            DATE, 	/**< Date */
                            OTHER 	/**< Other */
                           };
        /**
         * Default constructor
         * @param iDocument the document where to search
         * @param iTable the table where to search
         * @param iWhereClause the where clause
         * @param parent parent QT object
         * @param iWhereClauseChild the where clause to find children. Must be something like:
         * r_node_id=#ID#
         * #ID# will be replaced by id of the parent
         * @param iResetOnCreation to reset data during creation
         */
        SKGObjectModelBase(SKGDocument* iDocument,
                           const QString& iTable,
                           const QString& iWhereClause,
                           QObject *parent=NULL,
                           const QString& iWhereClauseChild="",
                           bool iResetOnCreation=true);

        /**
         * Destructor
         */
        virtual ~SKGObjectModelBase();

        /**
         * Refresh the model.
         */
        virtual void refresh();

        /**
         * Returns the number of row for the given parent.
         * @param parent the parent
         * @return the number of row
         */
        virtual int rowCount(const QModelIndex &parent = QModelIndex() ) const;

        /**
         * Returns the index of the item in the model specified by the given row, column and parent index.
         * @param row row
         * @param column column
         * @param parent parent
         * @return the index
         */
        virtual QModelIndex index ( int row, int column, const QModelIndex & parent = QModelIndex() ) const;

        /**
         * Returns the parent of the model item with the given index, or QModelIndex() if it has no parent.
         * @param index index
         * @return the parent
         */
        virtual QModelIndex parent ( const QModelIndex & index ) const;

        /**
         * Returns the number of columns for the given parent.
         * @param parent the parent
         * @return the number of column
         */
        virtual int columnCount( const QModelIndex & parent = QModelIndex()) const;

        /**
         * Returns the data stored under the given role for the item referred to by the index.
         * @param index the index
         * @param role the role
         * @return the returned value
         */
        virtual QVariant data ( const QModelIndex & index, int role = Qt::DisplayRole ) const;

        /**
         * Returns the SKGObjectBase for the item referred to by the index.
         * @param index the index
         * @return the returned SKGObjectBase
         */
        virtual SKGObjectBase getObject(const QModelIndex &index) const;

        /**
         * Returns the SKGObjectBase for the item referred to by the index.
         * @param index the index
         * @return the returned SKGObjectBase. Do not delete it !
         */
        virtual SKGObjectBase* getObjectPointer(const QModelIndex &index) const;

        /**
         * Returns the data for the given role and section in the header with the specified orientation.
         * @param section the section
         * @param orientation the orientation
         * @param role the role
         * @return the header data
         */
        virtual QVariant headerData ( int section, Qt::Orientation orientation, int role = Qt::DisplayRole ) const;

        /**
         * Returns the item flags for the given index.
         * @param index index of the object
         * @return flags of the given index
         */
        virtual Qt::ItemFlags flags(const QModelIndex &index) const;


        /**
         * Return the index of the attribute, -1 if not found
         * @param iAttributeName the attribute name
         * @return index of this attibute
         */
        virtual int getIndexAttribute (const QString& iAttributeName ) const;

        /**
         * Return the attribute for an index
         * @param iIndex the index
         * @return attribute
         */
        virtual QString getAttribute (int iIndex ) const;

        /**
         * Return the type of attribute for an index
         * @param iIndex the index
         * @return type of attribute
         */
        virtual SKGObjectModelBase::AttributeType getAttributeType (int iIndex ) const;

        /**
         * To know if the model can be fetched
         * @param parent parent
         * @return true of false
         */
        virtual bool canFetchMore ( const QModelIndex & parent ) const;

        /**
         * Set the table. Don't forget to do a reset after that.
         * @param iTable the table name
         */
        virtual void setTable ( const QString & iTable );

        /**
         * Get table name
         * @return table name
         */
        virtual QString getTable() const;

        /**
         * Get real table name
         * @return real table name
         */
        virtual QString getRealTable() const;

        /**
         * Get where clause
         * @return where clause
         */
        virtual QString getWhereClause() const;

        /**
         * Get document
         * @return document
         */
        virtual SKGDocument* getDocument() const;

        /**
         * Set the list of supported attributes. Don't forget to do a reset after that.
         * @param iListAttribute the list of attributes. If the list is empty then the default list is set.
         * The format of this string is the following one: attribute name[|visibility Y or N[|size]];attribute name[|visibility Y or N[|size]];...
         */
        virtual void setSupportedAttributes ( const QStringList & iListAttribute );

        /**
         * Set the filter. Don't forget to do a reset after that.
         * @param iWhereClause the where clause
         */
        virtual void setFilter ( const QString & iWhereClause );

        /**
         * Returns the actions supported by the data in this model.
         * @return Qt::DropActions
         */
        virtual Qt::DropActions supportedDragActions() const;

        /**
         * Returns the actions supported by the data in this model.
         * @return Qt::DropActions
         */
        virtual Qt::DropActions supportedDropActions() const;

        /**
         * Get list of supported schemas
         * @return list of schemas @see SKGModelTemplate
         */
        virtual  QList<SKGObjectModelBase::SKGModelTemplate> getSchemas() const;

        /**
         * Returns a list of MIME types that can be used to describe a list of model indexes.
         * @return list of mime types
         */
        virtual QStringList mimeTypes() const;

        /**
         * Returns an object that contains serialized items of data corresponding to the list of indexes specified.
         *The formats used to describe the encoded data is obtained from the mimeTypes() function.
         * If the list of indexes is empty, or there are no supported MIME types, 0 is returned rather than a serialized empty list.
         * @param indexes the index object
         * @return the mime data
         */
        virtual QMimeData* mimeData(const QModelIndexList &indexes) const;

        /**
         * Handles the data supplied by a drag and drop operation that ended with the given action.
         * Returns true if the data and action can be handled by the model; otherwise returns false.
         * Although the specified row, column and parent indicate the location of an item in the model where the operation ended,
         *it is the responsibility of the view to provide a suitable location for where the data should be inserted.
         * @param data mime data
         * @param action action
         * @param row row
         * @param column column
         * @param parent parent
         * @return true if the dropping was successful otherwise false.
         */
        virtual bool dropMimeData(const QMimeData *data,
                                  Qt::DropAction action,
                                  int row, int column,
                                  const QModelIndex &parent);

protected slots:
        /**
         * This method is called by refresh to build the cache (to improve performance)
         */
        virtual void buidCache();

        /**
         * data are modified
         * @param iTableName table name
         * @param iIdTransaction the id of the transaction for direct modifications of the table (update of modify objects is enough)
         *or 0 in case of modifications by impact (full table must be refreshed)
         */
        virtual void dataModified(const QString& iTableName, int iIdTransaction);

protected:

        /**
         *list of attributes
         */
        QStringList listAttibutes;

        /**
         *list of attributes
         */
        QList<AttributeType> listAttibuteTypes;

        /**
         *list of supported attributes
         */
        QStringList listSupportedDefault;

        /**
         *list of supported schemas
         */
        QList<SKGObjectModelBase::SKGModelTemplate> listSchema;

        /**
         *To disable/enable reset
         */
        bool isResetRealyNeeded;

private:
        Q_DISABLE_COPY(SKGObjectModelBase);
        void clear();
        int getID(const SKGObjectBase* obj, int row) const;

        SKGDocument* document;
        QString table;
        QString realTable;
        QString whereClause;
        QString whereClauseChild;

        QHash<int, SKGObjectBase::SKGListSKGObjectBase*> parentChildRelations;
        QHash<int, int> childParentRelations;
        QHash<int, SKGObjectBase*> objectsHashTable;
        QHash<int, int> objectsHashTableRows;

        QStringList listSupported;
        QList<bool> listVisibility;
        QList<int> listSize;
        bool doctransactionTable;
        bool nodeTable;
};

#endif
