/* bwlimit.c by Roman Drahtmller <draht@suse.de>
 * 
 * turns out to be useful every once in a while if you have small bandwidth
 * and need to transfer files without losing interactivity. TOS routing is
 * actually the way to go, but it doesn't help too much in most cases.
 *
 * This program is published under the terms of the GNU general public license.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <sys/time.h>
#define _GNU_SOURCE
#include <getopt.h>

#define DEFAULT_BUFSIZE 8*1024

#define PROGNAME "bwlimit"

void usage(void);

void usage() {
    fprintf(stderr, "usage: %s -b <limit>\n", PROGNAME);
    fprintf(stderr, "where limit is kB/s\n");
}

int bw_do_read(int fd, char *buf, int len)
{
    int i = 0;
    while (1) {
	i = read(fd, buf, len);
	if (i < 0) {
	    switch (errno) {
	    case EINTR:
	    case EAGAIN:
		break;
	    case 0:		/* shouldn't happen! */
		return (0);
		break;
	    default:
		perror("read");
		exit(errno);
		break;
	    }
	}

	if (i == 0) {
	    switch (errno) {
	    case EINTR:
	    case EAGAIN:
		break;
	    case 0:		/* EOF */
		return (0);
		break;
	    default:
		perror("read");
		exit(errno);
		break;
	    }
	}

	if (i > 0) {
	    return (i);
	}
    }
}

int bw_do_write(int fd, char *buf, int len)
{
    int i = 0;
    while (1) {
	i = write(fd, buf, len);
	if (i == len)		/* handles len==0 as well */
	    return (i);
	/* now get rid of the rest */
	if (i < 0) {
	    switch (errno) {
	    case EINTR:
	    case EAGAIN:
		break;
	    case 0:		/* shouldn't happen! */
		break;
	    default:
		perror("write");
		exit(errno);
		break;
	    }
	}

	if (i == 0) {
	    switch (errno) {
	    case 0:		/* could be a filesystem driver error */
		return (0);
		break;
	    case EINTR:
	    case EAGAIN:
		break;
	    default:
		perror("write");
		exit(errno);
	    }
	}

	len = len - i;
	buf = buf + i;
    }
}



int main(int argc, char **argv) {

    int c = 0;
    int bandwidth = 0;
    int len = 0;
/* usleep expects unsigned int. make sure it's > 0! */
    int usecs = 0;
    int usecs_should = 0;
    int usecs_inter = 0;
    int usecs_diff = 0;
    char *buf;
    struct timeval timeofday_last = { 0, 0 };
    struct timeval timeofday_now;

    if (argc != 3) {
	usage();
	exit(-1);
    }
    while ((c = getopt(argc, argv, "b:")) != -1) {
	switch (c) {
	case 'b':
	    bandwidth = atoi(optarg);
	    break;
	default:
	    usage();
	    break;
	}
    }

    if (bandwidth < 1) {
	usage();
	exit(-1);
    }

#ifdef B_DEBUG
    fprintf(stderr, "bw: %i\n", bandwidth);
#endif

    buf = malloc(DEFAULT_BUFSIZE);
    if (buf == NULL) {
	perror("malloc");
	exit(-1);
    }


    while (1) {
	len = bw_do_read(0, buf, DEFAULT_BUFSIZE);

	if (len == 0 && errno == 0) { /* EOF */
	    bw_do_write(1, buf, len);
	    exit(0);
	}

	if(!usecs) { /* first run, initialize */
	    usecs = len * 1000 / bandwidth;
	    usecs_should = usecs;
#ifdef B_DEBUG
fprintf(stderr, "usecs init: %i\n", usecs);
#endif
	    gettimeofday(&timeofday_now, NULL);
	    timeofday_last.tv_usec = timeofday_now.tv_usec;
	    timeofday_last.tv_sec  = timeofday_now.tv_sec;
	} else {
/* we measure the elapsed time because we don't know how long the
   read()-write() cycles have lasted. Then we calculate the reduced
   sleep time as in increase/decrease of the last one. Be adaptive,
   meaning only change sleep time gradually so that we have damping
   instead of strange oscillations.
 */   
	    gettimeofday(&timeofday_now, NULL);
	    usecs_diff = ( ( timeofday_now.tv_sec - timeofday_last.tv_sec)
		* 1000000 ) + timeofday_now.tv_usec - timeofday_last.tv_usec;
	/* remember */
	    timeofday_last.tv_usec = timeofday_now.tv_usec;
	    timeofday_last.tv_sec  = timeofday_now.tv_sec;

	    usecs_inter = usecs_should - usecs_diff;
#ifdef B_DEBUG
fprintf(stderr, "usecs_diff: %i   usecs_inter: %i   ", usecs_diff, usecs_inter);
#endif
	    usecs_inter = usecs_inter / 8;
	    usecs += usecs_inter;
	}

	if(usecs <= 0) { 
			/* reset. last read or write took so long that
			   this got out of bounds. No sleep. */
#ifdef B_DEBUG
fprintf(stderr, "\nreset\n");
#endif
		usecs = usecs_should / 2; 
	} else {
#ifdef B_DEBUG
fprintf(stderr, "    usecs: %i    \r", usecs);
#endif
	usleep(usecs);
	}
	
	len = bw_do_write(1, buf, len);

    }
}
