//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: SolverExpression.cpp,v 1.11 2006/03/12 20:17:23 delpinux Exp $

#include <SolverExpression.hpp>

#include <Solver.hpp>
#include <PDEProblem.hpp>
#include <PDESystem.hpp>
#include <SolverDriver.hpp>
#include <BoundaryConditionSet.hpp>

#include <PDESystemExpression.hpp>
#include <VariationalProblemExpression.hpp>

#include <UserFunction.hpp>

#include <PDESolution.hpp>

#include <VariationalProblem.hpp>

#include <Scene.hpp>

#include <Structured3DMesh.hpp>
#include <Information.hpp>

#include <DegreeOfFreedomSetBuilder.hpp>

#include <ParameterCenter.hpp>

#include <FEMFunction.hpp>
#include <FEMFunctionBuilder.hpp>

SolverExpression::SolverExpression(const SolverExpression& e)
  : Expression(e),
    __solver(e.__solver),
    __solverType(e.__solverType)
{
  ;
}

SolverExpression::SolverExpression(const SolverExpression::SolverType& t)
  : Expression(Expression::solver),
    __solverType(t)
{
  ;
}

SolverExpression::~SolverExpression()
{
  ;
}


std::ostream& SolverExpressionSolveFDM::put(std::ostream& os) const
{
  os << __FILE__ << ':' << __LINE__ << ": NOT IMPLEMENTED\n";
  return os;
}

void SolverExpressionSolveFDM::
__setScene(ReferenceCounting<DomainExpression> __domainExp) const
{
  switch((*__domainExp).domainType()) {
  case DomainExpression::set: {
    DomainExpressionSet& D = dynamic_cast<DomainExpressionSet&>(*__domainExp);
    SceneExpression& SE = dynamic_cast<SceneExpression&>(*(D.scene()));
    Information::instance().setScene(SE.scene());
    break;
  }
  case DomainExpression::variable: {
    const DomainExpressionVariable& D
      = dynamic_cast<const DomainExpressionVariable&>(*__domain);
    __setScene(D.domainExpression());
    break;
  }
  case DomainExpression::undefined: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "undefined domain",
		       ErrorHandler::normal);
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected domain type",
		       ErrorHandler::unexpected);
  }
  }
}

void SolverExpressionSolveFDM::execute()
{
  //! A mesh is used there. We inform others about that ...
  Information::instance().setMesh(__mesh);

  //! The unknowns
  Information::instance().setUnknownList(__unknownList);

  //! Reset solver options.
  ParameterCenter::instance().reset();

  (*__mesh).execute();
  (*__unknownList).execute();

  (*__domain).execute();

  __setScene(__domain);

  (*__solverOptions).execute();
  (*__problemExpression).execute();

  switch ((*__problemExpression).problemType()) {
  case ProblemExpression::pdeSystem: {
    (*dynamic_cast<PDESystemExpression&>(*__problemExpression).pdeSystem()).setDomain((*__domain).domain());
    break;
  }
  case ProblemExpression::variationalProblem: {
    (*dynamic_cast<VariationalProblemExpression&>(*__problemExpression).variationalProblem()).setDomain((*__domain).domain());
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected problem type",
		       ErrorHandler::unexpected);
  }
  }

  ffout(2) << "Solving Problem:\n" << '\n';
  ffout(2) << "\tUnknowns: " << (*__unknownList) << '\n';
  ffout(2) << "\tProblem:  " << (*__problemExpression) << '\n';

  // Not very nice coding ...
  DiscretizationType discretization((*(*(*__unknownList).begin())).discretizationType());

  DegreeOfFreedomSetBuilder dofBuilder((*__unknownList).size(),
				       discretization,
				       *(*__mesh).mesh(),
				       *(*__domain).domain());

  const DegreeOfFreedomSet& degreeOfFreedomSet
    = dofBuilder.degreeOfFreedomSet();

  PDESolution u(degreeOfFreedomSet);

  const DegreeOfFreedomPositionsSet& dofPositionsSet
    = dofBuilder.positionsSet();

  typedef
    std::vector<ReferenceCounting<FEMFunctionBase > >
    UnknownList;
  UnknownList UL;
  Mesh& M = (*(*__mesh).mesh());

  FEMFunctionBuilder femFunctionBuilder;
  for (UnknownListExpression::iterator i= (*__unknownList).begin();
       i != (*__unknownList).end(); ++i) {
    (*(*i)).execute();
    UserFunctionLanguage givenU ((*(*i)).value());
    UL.push_back(femFunctionBuilder.build(discretization, M, givenU));      
  }

  u.setUserFunction(UL);

  ConstReferenceCounting<Problem> P;
  switch ((*__problemExpression).problemType()) {
  case ProblemExpression::pdeSystem: {
    P = (PDESystem*)(dynamic_cast<PDESystemExpression&>(*__problemExpression).pdeSystem());
    break;
  }
  case ProblemExpression::variationalProblem: {
    P = (VariationalProblem*)(dynamic_cast<VariationalProblemExpression&>(*__problemExpression).variationalProblem());
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected problem type",
		       ErrorHandler::unexpected);
  }
  }

  SolverDriver sd(P, u, discretization,(*__mesh).mesh(), degreeOfFreedomSet,
		  SolverDriver::fictitiousDomainMethod);

  sd.run();

  u.getUserFunction(UL);

  UnknownListExpressionSet& UList
    = dynamic_cast<UnknownListExpressionSet&>(*__unknownList);
  size_t i=0;
  for(UnknownListExpressionSet::listType::iterator u = UList.giveList().begin();
      u != UList.giveList().end(); ++u) {
    FunctionExpressionVariable& computedU = dynamic_cast<FunctionExpressionVariable&>((*(*u)));

    (*computedU.variable()) = new FunctionExpressionFEM(__mesh, UL[i]);

    ++i;
  }

  //! The unknown are no more there.
  Information::instance().unsetUnknownList();

  //! The mesh is nomore used in this region.
  Information::instance().unsetMesh();
}

SolverExpressionSolveFDM::
SolverExpressionSolveFDM(ReferenceCounting<UnknownListExpression> unknownList,
			 ReferenceCounting<DomainExpression> domain,
			 ReferenceCounting<MeshExpression> mesh,
			 ReferenceCounting<SolverOptionsExpression> solverOptions,
			 ReferenceCounting<ProblemExpression> problemExpression)
  : SolverExpression(SolverExpression::solve),
    __unknownList(unknownList),
    __domain(domain),
    __mesh(mesh),
    __solverOptions(solverOptions),
    __problemExpression(problemExpression)
{
  ;
}

SolverExpressionSolveFDM::
SolverExpressionSolveFDM(const SolverExpressionSolveFDM& s)
  : SolverExpression(s),
    __unknownList(s.__unknownList),
    __domain(s.__domain),
    __mesh(s.__mesh),
    __solverOptions(s.__solverOptions),
    __problemExpression(s.__problemExpression)
{
  ;
}

SolverExpressionSolveFDM::~SolverExpressionSolveFDM()
{
  ;
}


std::ostream& SolverExpressionSolveFEM::put(std::ostream& os) const
{
  os << __FILE__ << ':' << __LINE__ << ": NOT IMPLEMENTED\n";
  return os;
}


void SolverExpressionSolveFEM::execute()
{
  //! A mesh is used there. We inform others about that ...
  Information::instance().setMesh(__mesh);

  //! The unknowns
  Information::instance().setUnknownList(__unknownList);

  //! Reset solver options.
  ParameterCenter::instance().reset();

  (*__mesh).execute();
  (*__unknownList).execute();

  (*__solverOptions).execute();
  (*__problemExpression).execute();

  if ((*__problemExpression).hasPOVBoundary()) {
    throw ErrorHandler(__FILE__,__LINE__,
		       "cannot use POVRay references in standard FEM discretization.\n"
		       "You probably forgot to specify the domain in the 'solve' bloc.\n"
		       "Refere to the Fictitious Domain part of the documentation",
		       ErrorHandler::normal);
  }

  ffout(2) << "Solving Problem:\n" << '\n';
  ffout(2) << "\tUnknowns: " << (*__unknownList) << '\n';
  ffout(2) << "\tProblem:  " << (*__problemExpression) << '\n';

  // Not very nice coding ...
  DiscretizationType discretization((*(*(*__unknownList).begin())).discretizationType());

  DegreeOfFreedomSetBuilder dofBuilder((*__unknownList).size(),
				       discretization,
				       (*(*__mesh).mesh()));

  const DegreeOfFreedomSet& degreeOfFreedomSet
    = dofBuilder.degreeOfFreedomSet();

  const DegreeOfFreedomPositionsSet& dofPositionsSet
    = dofBuilder.positionsSet();

  PDESolution u(degreeOfFreedomSet);

  typedef
    std::vector<ReferenceCounting<FEMFunctionBase > >
    UnknownList;
  UnknownList UL;
  Mesh& M = dynamic_cast<Mesh&>(*(*__mesh).mesh());

  FEMFunctionBuilder femFunctionBuilder;
  for (UnknownListExpression::iterator i= (*__unknownList).begin();
       i != (*__unknownList).end(); ++i) {
    FunctionExpression& f = (*(*i));
    f.execute();

    UserFunctionLanguage givenU (f.value());
    UL.push_back(femFunctionBuilder.build(discretization, M, givenU));      
  }

  u.setUserFunction(UL);

  ConstReferenceCounting<Problem> P;
  switch ((*__problemExpression).problemType()) {
  case ProblemExpression::pdeSystem: {
    P = (PDESystem*)(dynamic_cast<PDESystemExpression&>(*__problemExpression).pdeSystem());
    break;
  }
  case ProblemExpression::variationalProblem: {
    P = (VariationalProblem*)(dynamic_cast<VariationalProblemExpression&>(*__problemExpression).variationalProblem());
    break;
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "unexpected problem type",
		       ErrorHandler::unexpected);
  }
  }

  SolverDriver sd(P, u, discretization,
		  (*__mesh).mesh(),
		  degreeOfFreedomSet,
		  SolverDriver::standardFEM);

  sd.run();

  u.getUserFunction(UL);

  UnknownListExpressionSet& UList
    = dynamic_cast<UnknownListExpressionSet&>(*__unknownList);
  size_t i=0;
  for(UnknownListExpressionSet::listType::iterator u = UList.giveList().begin();
      u != UList.giveList().end(); ++u) {
    FunctionExpressionVariable& computedU = dynamic_cast<FunctionExpressionVariable&>((*(*u)));

    (*computedU.variable()) = new FunctionExpressionFEM(__mesh, UL[i]);

    ++i;
  }

  //! The unknown are no more there.
  Information::instance().unsetUnknownList();

  //! The mesh is nomore used in this region.
  Information::instance().unsetMesh();
}

SolverExpressionSolveFEM::
SolverExpressionSolveFEM(ReferenceCounting<UnknownListExpression> unknownList,
			 ReferenceCounting<MeshExpression> mesh,
			 ReferenceCounting<SolverOptionsExpression> solverOptions,
			 ReferenceCounting<ProblemExpression> problemExpression)
  : SolverExpression(SolverExpression::solve),
    __unknownList(unknownList),
    __mesh(mesh),
    __solverOptions(solverOptions),
    __problemExpression(problemExpression)
{
  ;
}

SolverExpressionSolveFEM::
SolverExpressionSolveFEM(const SolverExpressionSolveFEM& s)
  : SolverExpression(s),
    __unknownList(s.__unknownList),
    __mesh(s.__mesh),
    __solverOptions(s.__solverOptions),
    __problemExpression(s.__problemExpression)
{
  ;
}

SolverExpressionSolveFEM::~SolverExpressionSolveFEM()
{
  ;
}

