//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FEMFunctionBuilder.cpp,v 1.2 2005/10/29 14:42:00 delpinux Exp $

#include <FEMFunctionBuilder.hpp>
#include <FiniteElementTraits.hpp>

#include <Structured3DMesh.hpp>
#include <MeshOfTetrahedra.hpp>
#include <MeshOfHexahedra.hpp>

template <typename MeshType>
ReferenceCounting<FEMFunctionBase>
FEMFunctionBuilder::__build(const DiscretizationType& d,
			    const MeshType& mesh) const
{
  typedef typename MeshType::CellType CellType;

  switch(d.type()) {
  case DiscretizationType::LagrangianFEM1: {
    typedef
      FiniteElementTraits<CellType,DiscretizationType::LagrangianFEM1>
      FETraits;

    return new FEMFunction<MeshType, FETraits>(mesh,
					       d.type());

  }
  case DiscretizationType::LagrangianFEM2: {
    typedef
      FiniteElementTraits<CellType,DiscretizationType::LagrangianFEM2>
      FETraits;

    return new FEMFunction<MeshType, FETraits>(mesh,
					       d.type());

  }
  }
  // This point should not be reached
  return 0;
}


ReferenceCounting<FEMFunctionBase>
FEMFunctionBuilder::build(const DiscretizationType& d,
			  const Mesh& mesh) const
{
  switch (mesh.type()) {
  case Mesh::cartesianHexahedraMesh: {
    const Structured3DMesh& m = static_cast<const Structured3DMesh&>(mesh);
    return this->__build(d, m);
  }
  case Mesh::hexahedraMesh: {
    const MeshOfHexahedra& m = static_cast<const MeshOfHexahedra&>(mesh);
    return this->__build(d, m);
  }
  case Mesh::tetrahedraMesh: {
    const MeshOfTetrahedra& m = static_cast<const MeshOfTetrahedra&>(mesh);
    return this->__build(d, m);
  }
  default: {
    throw ErrorHandler(__FILE__,__LINE__,
		       "not supported mesh type",
		       ErrorHandler::unexpected);
  }
  }
}

ReferenceCounting<FEMFunctionBase>
FEMFunctionBuilder::build(const DiscretizationType& d,
			  const Mesh& mesh,
			  const UserFunction& u) const
{
  ReferenceCounting<FEMFunctionBase> f
    = this->build(d,mesh);
  (*f) = u;
  return f;
}


ReferenceCounting<FEMFunctionBase>
FEMFunctionBuilder::build(const DiscretizationType& d,
			  const Mesh& mesh,
			  const Vector<real_t>& values) const
{
  ReferenceCounting<FEMFunctionBase> f
    = this->build(d,mesh);
  (*f) = values;
  return f;  
}
