//////////////////////////////////////////////////////////////////////////////
// dl.C

#define WITH_DEPGRAPH
#define WITH_GLOBAL_VARIABLES
#define BUILTIN_IMPL

#include "dl.h"
#include "odbc.h"
#include "diag.h"
#include "sql.h"
#include "plan.h"
#include "inheritance.h"
#include "stats.h"
#include "generate.h"
#include "grounding.h"
#include <algorithm>
#include <ctype.h>
#include <stdio.h>                        // The interface to FLEX uses FILE*s
#include <limits.h>

//////////////////////////////////////////////////////////////////////////////
// Global variables

const char LicenseText[]=
#include "LICENSE.h"
;

const char*  WhoAmI;

STATS_TIMER timer;

size_t       GINTERPRET::SIZE = 0;
size_t       GATOMSET::SIZE = 0;

// Initialise static members of AGGREGATEFUNCTION.
AGGREGATEFUNCTION::AGGREGATESETHASH AGGREGATEFUNCTION::h2iAggregateSetTable;
vector<AGGREGATEFUNCTION::AGGREGATESETHASH::const_iterator>
                                    AGGREGATEFUNCTION::i2hAggregateSetTable;

// Initialise static members of AGGREGATEATOM.
AGGREGATEATOM::AGGREGATEFUNCTIONHASH AGGREGATEATOM::
                                                   h2iAggregateFunctionTable;
vector<AGGREGATEATOM::AGGREGATEFUNCTIONHASH::const_iterator>
                                    AGGREGATEATOM::i2hAggregateFunctionTable;

unsigned AGGREGATEATOM::aggregateFunctionsCount;  
// Initialise static members of GATOM...
GATOM::GATOMHASH                         GATOM::h2iTable;
vector<GATOM::GATOMHASH::const_iterator> GATOM::i2hTable;
bool                                     GATOM::frozen(false);

// ...and of TERM and ATOM.
STRING_CONSTANTS        TERM::Strings(0);
INTEGER_CONSTANTS       TERM::Integers(0);
PREDICATE_NAMES         ATOM::Predicates(0);

RULES        IDB;
CONSTRAINTS  Constraints;
WEAKCONSTRAINTS  WConstraints;
CONJUNCTION *Query = 0;
static bool  QueryAnswered = false;
static GINTERPRET *QueryI;
static void  printQueryI(ostream&, const INTERPRET&, const GINTERPRET*);

static bool  IsIDBGround = true;
static bool  IsConstraintsGround = true;
static bool  IsWConstraintsGround = true;
static bool  IsQueryGround = true;

GRULES       GroundIDB;
GCONSTRAINTS GroundConstraints;
GWEAKCONSTRAINTS GroundWConstraints;
GWEAKCONSTRAINTS  ViolatedWConstraints;

INTERPRET    I;
VATOMSET     EDB;

OIDs         ObjectIDs;
bool         ObjectsExist = false;
bool         StrictRulesExist = false;

unsigned     CurrentOID = 0,
             RuleID = 0;
unsigned     AuxCounter     = 0;

#ifndef NDEBUG
unsigned    TraceLevel      = 0,
            DTraceLevel     = 0,
            FTraceLevel     = 0,
            PTraceLevel     = 0,
            GTraceLevel     = 0,
            HTraceLevel     = 0,
            CTraceLevel     = 0;
#endif

int         MaxInteger                 = -1;
int         MaxIntegerSeen             = -1;

unsigned    OptionStatsLevel           = 0;
bool OptionProjectionRewrite           = true;
bool OptionBasicRewrite                = true;
static bool OptionSubsumptionChecking  = false;
// For the moment, OptionRewriteCountEDB is assigned true by default.
// In the future it should be set by a command-line option.
bool        OptionRewriteCountEDB      = true;
bool        OptionRewriteDeleteRules   = true;
static bool OptionMagicSets            = false;
static bool OptionGroundingProp        = true;
unsigned    OptionGroundingReorder     = 2;
bool        OptionGroundingSemiNaive   = true;
bool        OptionUseHeuristics        = true;
unsigned    OptionModelCheckMode       = 0;
#if CHECK_OPT >= 3
bool        OptionPartialChecksForwards = true;
#else
bool        OptionPartialChecksForwards = false;
#endif
unsigned    OptionVerbosity            = 0;
static bool OptionWait                 = false;
bool        OptionSilent               = false;
bool        OptionPrintFacts           = true;
bool        OptionPrintSATlike         = false;

bool        OptionTraceComputationWC   = false;
bool        OptionCostBound            = false;

bool        OptionUseOldPTs            = true;

// 5 seems to be magic for unknown reasons.
unsigned    OptionPTCacheSize          = 5;

bool        OptionHeuristicsConsiderComplementaryPT = true;
unsigned    OptionHeuristicsCombineComplementaryPT  = 0;
bool        OptionHeuristicsCanChooseComplementaryPT   = true;

bool        OptionComputeGUS           = true;

STRINGSET   PredFilter;
STRINGSET   PredPFilter;
bool        ProjectionsExist = false;

unsigned    MaxWeakConstraintLevel=0;

const char * heuristicSequenceDefault = "0,2,4,15,7,8,9,10,11";
const char * heuristicSequenceCheckerDefault = "6";
CriteriaSequence HeuristicSequence;
CriteriaSequence HeuristicSequenceChecker;
CriteriaSequence HeuristicCombinationSequence;

#ifdef HEURISTICSTATS
CriteriaSequence statsHCriterionApplied;
#endif

//////////////////////////////////////////////////////////////////////////////
// Frontend-specific stuff.

FRONTEND    OptionFrontend  = FRONTEND_DATALOG;
FDIAG       FDiag           = FRONTEND_DIAG_DEFAULT;
FPLAN       FPlan           = FRONTEND_PLANNING_DEFAULT;

unsigned    parser_line;
static const char* parser_file;
unsigned    parser_errors   = 0;
bool        ParserStateInternal = true;

void        (*CallbackAfterGrounding)() = 0;
void        (*CallbackAfterParsing)() = 0;

//////////////////////////////////////////////////////////////////////////////
// Helper functions as expected by LEX/FLEX 

extern "C" int yywrap()    // End-of-file handler for LEX
    {
    return 1;
    }

extern "C" FILE* yyin;     // Where LEX reads its input from

//////////////////////////////////////////////////////////////////////////////
// Helper functions as expected by YACC/BISON 

extern int diagyyparse();
extern int planyyparse();
extern int sqlyyparse();

int yyerror(const char* s) // Error handler for YACC
    {
    if( !ParserStateInternal )
        {
        if( OptionVerbosity )
            cerr << WhoAmI << " (";
        
        if( strlen(parser_file) > 0 )
            cerr << parser_file << ": ";
        cerr << "line " << parser_line;
        
        if( OptionVerbosity )
            cerr << ')';

        cerr << ": ";
        }
    else
        cerr << "Internal parser invocation: ";
    
    cerr << s << "." << endl;

    return 0;
    }

#include "parser2.c"

//////////////////////////////////////////////////////////////////////////////
// Parse dlv core input from a stringstream instead of a file.
// Resets the stringstream and returns
// whether parser errors have occurred.
bool parseStringStream(ostringstream &s)
    {
    unsigned current_errors = parser_errors;

    yy_buffer_state *ptr = yy_scan_string(s.str().c_str());
    yyparse();
    yy_delete_buffer(ptr);
    s.str("");

    return current_errors == parser_errors;
    }	

//////////////////////////////////////////////////////////////////////////////
// Print a fatal error message and terminate abnormaly.
void InternalError(const char* msg)
    {
    cerr << endl
         << "An internal error occurred ("  << msg << ")."
         << endl
         << "Please contact <pfeifer+faber@dbai.tuwien.ac.at>!"
         << endl;
    exit(99);
    }

//////////////////////////////////////////////////////////////////////////////
// Exit safely, i.e. supply the right output.
void SafeExit(const int status)
    {
    if( status == 0 )
	{
        // Exit normally
	switch( OptionFrontend )
	    {
	    case FRONTEND_BRAVE:
                if( ! QueryAnswered) 
                    {
                    if( IsQueryGround )
                        cout << *Query << " is bravely false." << endl;
                    else
                        {
                        if( QueryI )
                            printQueryI(cout,I,QueryI);
                        else
                            cout << "No stable model found." << endl;
                        }
                    }
		break;
	    case FRONTEND_CAUTIOUS:
                if( ! QueryAnswered )
                    {
                    if( IsQueryGround )
                        cout << *Query << " is cautiously true." << endl;
                    else
                        {
                        if( QueryI )
                            printQueryI(cout,I,QueryI);
                         else
                            cout << "No stable model found." << endl;
                        }
                    }
                break;
	    default:
		break;
	    }
	}

    if( OptionWait )
        {
        char s[1024];

        cout << endl;
        fgets(s,sizeof s,stdin); 
        }

    exit(status);
    }

//////////////////////////////////////////////////////////////////////////////
static void printLogo(const char *prefix="")
    {
    cout << prefix 
         << "DLV [build " 
#ifdef NDEBUG
         << "BEN"
#else
         << "DEV"
#endif
#ifdef WITH_ODBC 
         << "+ODBC"
#endif
#ifdef TIMERS
         << "+TIMERS"
#endif
         << "/" << __DATE__ 
#ifdef __GNUC__
         << "   gcc " << __VERSION__ 
#endif
         << "]" << endl
         << endl;

    if( OptionVerbosity )
        {
        const char *p=LicenseText;
        char prevchar=' ';

        for( unsigned blanklines=0; blanklines < 3  &&  *p != '\0'; )
            {
            if( prevchar == '\n'  &&  *p == '\n' )
                blanklines++; 
            else
                cout << *p;
            prevchar=*p++;
            }
        }
        cout << endl;
    }

//////////////////////////////////////////////////////////////////////////////
static void printUsage(ostream &out, const bool full)
//
    {
         // 123456789-123456789-123456789-123456789-123456789-123456789-123456789-123456789-  
    out  << "usage: " << WhoAmI 
         << " {FRONTEND} {OPTIONS}"
#ifndef NDEBUG
            // Supported, though undocumented
            " [-stats++] [--depgraph]"
            " [--print-{edb,idb,gidb,constraints,gconstraints,wconstraints,"
             "gwconstraints,interpret}]"
            " [--negpts] [--pospts] [--ptcache=<size>]"
            " [--complpt] [--nocomplpt] [--combinecomplpt=<n>]"
            " [--canchoosecomplpt] [--cannotchoosecomplpt]"
            " [--GUS] [--noGUS]"
            " [--heuristic=([0-9]+,)*[0-9]+]"
            " [--combineheuristic=([0-9]+,)*[0-9]+]"
            // Really unsupported
            " [-trace=(0|1|2|3)]"
            " [-dtrace=(0|1)]"
            " [-ftrace=(0|1)]"
            " [-ptrace=(0|1)]"
            " [-gtrace=(0|1|2)] "
            " [-htrace=(0|1|2|3)] "
            " [-ctrace=(0|1|2)] "
#endif
            " [filename [filename [...]]]" << endl
         << endl;

    if( ! full )
        { 
        cout << "Specify -help for more detailed usage information." << endl
             << endl;
        return;
        }
        
    out  << "Frontends:" << endl
         << endl
         << "-FD           Abductive diagnosis" << endl
         << "-FDmin        Abductive diagnosis, subset minimal" << endl
         << "-FDsingle     Abductive diagnosis, single error" << endl
         << "-FDmincost    Abductive diagnosis with Penalization" << endl 
         << "-FR           Reiter's diagnosis" << endl
         << "-FRmin        Reiter's diagnosis, subset minimal" << endl
         << "-FRsingle     Reiter's diagnosis, single error" << endl
         << "-FP           Planning with \"K\" Action Language" << endl
         << "-FPopt        ...find optimistic plans in batch mode" << endl
         << "-FPsec        ...find secure plans in batch mode" << endl
         << "-FPc          Planning with \"C\" Action Language" << endl
         << "-FPcheck=<n>  Secure Check method <n>. Currently "
                           "implemented: 1,2" << endl
         << "-FPsoundcheck=<n>    A sound secure check method.    "<< endl
         << "-FPcompletecheck=<n> A complete secure check method. "<< endl
         << "-planlength=<N> Maximum plan length for planning frontend" << endl
         << "-planminactions=<N> Minimum number of actions per time" << endl
         << "-planmaxactions=<N> Maximum number of actions per time" << endl
         << "-plancache=<N>  Size of the plan cache (number of plans,"
                            " defaults to "<< PLANCACHE <<")" << endl
         << "-FS           SQL3" << endl
         << endl
         << "General Options:" << endl
         << endl
         << "--            Also read input from stdin." << endl
         << "-brave        Perform brave reasoning." << endl
         << "-cautious     Perform cautious reasoning." << endl   
         << "-costbound=<N,..,N> Compute the models with cost <= <N,..,N>"
                          " (N=_ indicates that no bound is required)." << endl
         << "-det          Only compute the deterministic consequences."
         << endl
         << "-instantiate  Only ground and print the instantiation." << endl
         << "-n=<n>        Compute at most <n> stable models "
                          "(-n=0 and -n=all give all)." << endl
         << "-N=<N>        Limit integers to [0,<N>]." << endl
         << "-wait         Before terminating, wait until Return is pressed." 
         << endl
         << endl
         << "Output Options:" << endl
         << endl
         << "-filter=X[,Y[,...] Output only instances of the specified predicate(s)." 
         << endl
         << " -pfilter=X[,Y[,...]" << endl
         << "              Output only positive instances of the specified predicate(s)."
         << endl
         << "-facts        Include facts as part of the output." << endl
         << " -nofacts     Don't include facts as part of the output." << endl
         << "-license      Print the license of this program." << endl
         << "-silent       Suppress the startup banner and blank lines." <<endl
         << "-stats        Print statistics and timings regarding the "
                          "computation." << endl
         << "-v            Be a bit more verbose than usual." << endl
         << "-wctrace      Print all (possibly not optimal) models " << endl 
         << "              during computation of weak constraints." << endl 
         << endl
         << "Optimizations:" << endl
         << endl

         << "-OMS[-]       Magic Sets Rewriting." << endl
         << "-OS[-]        Input subsumption checking." << endl
         << "-O0           Disable all optimizations." << endl
         << "-OR[-]        Input rewriting." << endl 
         << "-ORb[-]       Basic input rewriting." << endl 
         << "-ORp[-]       Projection rewriting." << endl 
         << "-ORdr[-]      Rewriting deletes rules whose body is" << endl
         << "              always false." << endl
         << "-OGp[-]       Use special grounder for propositional input." 
         << endl
         << "-OGo(0|1|2)   Grounding performs (no|simple|advanced)" << endl
         << "              dynamic body reordering." << endl
         << "-OGs          Grounding employs semi-naive evaluation." << endl
         << "-OH[-]        Heuristics." << endl
         << "-OM-          Special Model Checker mode. Not for general use!" 
         << endl
#ifndef NDEBUG
         << "-OMm=<n>      Model-checking mode <n>." << endl
#endif
         << "-OPf[-]       Partial model checking ``forwards''." << endl
         << endl
         << "Default options: -OR -ORdr -OGp -OGo2 -OGs -OH"
#if CHECK_OPT >= 3 
            " -OPf"
#endif
         << endl
         << endl;
    }


//////////////////////////////////////////////////////////////////////////////
/// generates a constraint for a negated atom.
/// The constraint is of the form ":- f(...), -f(...)." and is created only if
/// both forms (positive and negative) are not undefined.
/// @param neg a reference to the negative form of the atom.
static bool createConstraint(const PREDICATE_NAMES::index_t neg_index)
    {
    // Check whether this predicate does not occur with 
    // different negative and positive arities.
    assert( ATOM::Predicates.getItem(neg_index).isNegative() );
    pair<PREDICATE_NAMES::index_t,bool> thePos =
        ATOM::Predicates.find(ATOM::Predicates.getItem(neg_index).getPositiveName());
    if( thePos.second &&
        ATOM::Predicates.getArity( neg_index ) !=
        ATOM::Predicates.getArity( thePos.first ) )
        {
        cerr << ATOM::Predicates.getItem(thePos.first).getName()
             << " and " 
             << ATOM::Predicates.getItem(neg_index).getName()
             << " seen with different arities (" 
             << ATOM::Predicates.getArity( thePos.first )
             << " and " 
             << ATOM::Predicates.getArity( neg_index ) 
             << ", respectively)." << endl;
        return false;
        }
        
    TERMS *params=0;

    if( ATOM::Predicates.getArity(neg_index) )
        {
        // Even if the entire input is ground, now it is not any longer!
        IsConstraintsGround=false;
 
        params=new TERMS;

        for (unsigned v=0; v < ATOM::Predicates.getArity(neg_index); v++)
            params->push_back(TERM(v));
        }

    ATOM neg(neg_index,params);

    if( TraceLevel >= 1 )
        cdebug << "createConstraint(" << neg;

    if ( neg.isUndef() )
        {
        if( TraceLevel >= 1 )
            cdebug << "). Literal undefined. Constraint not added."
                   << endl; 
        return true;
        }

    ATOM pos(neg);
    bool pos_exists = neg.getPositive(params,pos);

    if( ! pos_exists  ||  pos.isUndef() )
        {
        if( TraceLevel >= 1 )
            cdebug << "). Positive form undefined. Constraint not added."
                   << endl; 

        return true;
        }

    ATOM neg1(neg,params);

    if( params )
        delete params;

    CONJUNCTION con;
    con.add(pos);
    con.add(neg1);
    Constraints.push_back(CONSTRAINT(con,true));

    if( TraceLevel >= 1 )
        cdebug << "). Constraint added." << endl;
    
    return true;
    }

/** sets up the table of built-in atom names.
 */
static void initBuiltins()
    {
    for (unsigned i=0; i<NBuiltins; i++)
        {
#ifndef NDEBUG
        pair<PREDICATE_NAMES::index_t,bool> addresult =
#endif

	ATOM::Predicates.add(builtin_name[i],builtin_arity[i],
                             PREDICATE_NAMES::typeBuiltin);

#ifndef NDEBUG
        // The builtin must not have existed before in Predicate_Names.
        assert(addresult.second);
        // The returned index must be the same as the builtin number.
        assert(addresult.first == i);
#endif
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Magic() checks whether the magic set transformation can be applied, and if
// so, it actually transforms the program.
//
// @return false if some error occurred

static bool Magic()
    {
    if( Query )
        {
        if( TraceLevel >= 1 )
            cdebug << "Magic Sets rewriting of query " << *Query << " ?"
                   << endl;

        // Is the query appropriate for the magic sets rewriting?
        for( CONJUNCTION::const_iterator i=Query->begin();
             i != Query->end();
             i++ )
            {
            if( ! (*i).isGround() )
                {
                // FIXME: this must be corrected when non-ground
                // queries will be introduced
                cerr << "Non-ground queries are not supported yet "
                    "for the Magic Sets Technique." << endl;
                return false;
                }
            if( (*i).isEDB() )
                {
                cerr << "EDB predicates are not allowed in queries ("
                     << *i << ")." << endl;
                return false;
                }
            }

        MagicRewriting();
        }
    else
        {
        // No query exists.
        cerr << "No query supplied. Cannot apply Magic Sets Optimization."
             << endl;
        return false;
        }

    return true;
    }

////////////////////////////////////////////////////////////////////////

/** creates appropriate constraints for implementing queries efficiently
 * Note: Error messages are not terminated by newline, since it is expected
 *       that some other message (indicating how the program will proceed)
 *       is printed immediately afterwards.
 * @return false if some error occurred
 */
static bool HandleQuery()
    {
    if( ! Query )
        {
        // No query has been specified, but one is required when brave
        // or cautious reasoning are requested.
        if( OptionFrontend == FRONTEND_BRAVE
            || OptionFrontend == FRONTEND_CAUTIOUS )
            {
            cerr << "No query supplied. Cannot continue." << endl;
            return false;
            }
        else
            return true;
        }

    if( TraceLevel >= 1 )
        cdebug << "Handling Query " << *Query << "?" << endl;

    unsigned maxVar=0;                // maximum variable index in the query

    // Make one pass through the query, checking whether it is ground
    // or not and looking for the maximum variable index.
    for( CONJUNCTION::const_iterator i=Query->begin();
         i != Query->end();
         i++ )
        {
        if( ! i->isGround() )
            {
            IsQueryGround=false;

            for( TERMS::const_iterator j=i->getParams()->begin();
                 j != i->getParams()->end();
                 j++ )
                if( j->getType() == TERM::Variable )
                    maxVar=max(maxVar,j->getVar());
            }
        }

    if( OptionFrontend != FRONTEND_CAUTIOUS )
        {
        // We either do brave reasoning, or just have a query without
        // reasoning (possibly coming from a frontend).  For now, we
        // forbid the latter for non-ground queries.
        if( OptionFrontend != FRONTEND_BRAVE  &&  ! IsQueryGround )
            {
            cerr << "Non-ground queries are only supported with brave and "
                    "cautious reasoning." << endl;
            return false;
            }

        // Each ground literal in the query is transformed into a constraint,
        // which we add to the program (marked as internal).
        // Example: a, not b? becomes :- not a.
        //                            :- b.
        for( CONJUNCTION::const_iterator i=Query->begin();
             i != Query->end();
             i++ )
            if( (*i).isGround() )
                {
                CONSTRAINT c(true);
                c.add( LITERAL( ! (*i).isNegative(), (ATOM)*i) );

                Constraints.push_back(c);
                
                if( TraceLevel >= 1 )
                    cdebug << "                " << c << endl;
                }
        }
    else if( IsQueryGround )
        {
        // We do cautious reasoning and have a ground query.

        // The query is treated as an internal constraint. 
        // Example: a, not b ? 
        //  becomes :- a, not b.
        // 
        // If a model is computed, this means that this model did not
        // violate the constraint, i.e. at least one of the query's
        // literals is false in this model.  This model can be seen as
        // a witness that the query is not true in all possible models.
        // 
        // If no model is computed, the query is cautiously true.

        // Create a constraint containing exactly the query
        // conjunction and mark it as internal.
        CONSTRAINT c(*Query,true);
        Constraints.push_back(c);

        if( TraceLevel >= 1 )
            cdebug << "                " << c << endl;
        }
    else
        // We do cautious reasoning and have a non-ground query.
        ;

    // If the query is non-ground, we need to create a new "query rule" with
    // the query in the body, and initialize the interpretation that will
    // hold the union or intersection, respectively, of all models found.
    if( ! IsQueryGround )
        {
        TERMS t;
        for(unsigned i=0; i <= maxVar; i++)
            t.push_back( TERM(i) );

        DISJUNCTION head;
        head.add( ATOM(PREDNAME_QUERY,&t) );
        IDB.push_back( RULE(&head,Query) );

        if( TraceLevel >= 1 )
            cdebug << "Adding rule for non-ground query: "
                   << IDB.back() << endl;

        assert( IDB.back().isSafe() );
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// Output and callback functions.
//////////////////////////////////////////////////////////////////////////////

class InterpretAtomAsRulePrinter : public unary_function<ATOM&,void>
    {
    ostream &out;
public:
    InterpretAtomAsRulePrinter(ostream &out2)
        : out(out2)
        {
        }

    void operator() (const ATOM &a) const
        {
        out << a << '.' << endl;
        }
    };

//////////////////////////////////////////////////////////////////////////////
static void printModel(
//
    ostream &out,
    const INTERPRET *A,
    const GINTERPRET *I)
    {
    out << '{';

    bool cont=false;

    if( OptionPrintFacts )
        cont = EDB.printCensored(out,false);

    // Print the part which is in all models, using the "continuing" feature
    // of print().

    if( A )
        cont = A->printCensored(out,cont) || cont;

    // Print the part which has been computed by the Model Generator, again
    // using the "continuing" feature of print().

    if( I )
        I->printCensored(out,cont);

    out << "}";

    // Export tuples to an external database (only for the first model)
    static bool doExport = true;
    if ( doExport )
        {
        odbcExport(A,I);
        doExport = false;
        }
    }

//////////////////////////////////////////////////////////////////////////////
static void printQueryI(
//
    ostream &out,
    const INTERPRET &A,
    const GINTERPRET *I )
    {
    // Some (or even all) instances of the query predicate may already
    // have been solved by the grounding, so we first need to consider
    // the interpretation derived by the grounding.
    ATOM pattern(PREDNAME_QUERY,0,PREDICATE_NAMES::typeQuery);
    ATOMSET::FIND_RESULT f;

    for(A.findInPositivePart(pattern,f); ! f.atEnd(); f++)
        {
        const TERMS *params=f->getParams();

        print_list(out,*params,", ");
        out << endl;
        }

    assert( I );
    for(size_t i=0; i < GINTERPRET::size(); i++)
        if( ! strcmp(GATOM(i).getTheRealOne().getPredName(), PREDNAME_QUERY) )
            if( I->isTrue( GATOM(i) ) )
                {
                const TERMS *params=GATOM(i).getTheRealOne().getParams();

                print_list(out,*params,", ");
                out << endl;
                }
    }

//////////////////////////////////////////////////////////////////////////////
bool DatalogCallback(
          MODEL_GENERATOR *mg,
    const INTERPRET  *A,
    const GINTERPRET *I )
    {
    assert( mg );

    if( ! OptionSilent  &&  mg->getNumberOfModelsPrinted() )
        cout << endl;

    if( mg->groundedWC || mg->parsedWC )
        {
        if(mg->isOptimalCostKnown && ! mg->optCostBound)
            cout << "Best model: ";
        else if(mg->optTraceWC)
            cout << "Current model [maybe not optimal]: ";
        }

    printModel(cout,A,I);
    cout << endl;

    if( mg->isOptimalCostKnown /*|| !OptionTraceComputationWC*/ )
        mg->incrNumberOfModelsPrinted();

    if( mg->groundedWC || mg->parsedWC )
        {
        cout << "Cost ([Weight:Level]): ";
        I->printCost(cout);
        cout << endl;
        }

    return true;
    }

bool BraveCallback(
          MODEL_GENERATOR *mg,
    const INTERPRET  *A,
    const GINTERPRET *I )
    {
    if( IsQueryGround )
        {
        cout << *Query
             << " is bravely true, evidenced by ";
        printModel(cout,A,I);
        cout << endl;
        if( ! OptionSilent )
            cout << endl;

        QueryAnswered = true;    

        if( mg )
            mg->incrNumberOfModelsPrinted();

        return false;
        }
    else
        {
        if( mg )
            {
            assert( I );

            if( TraceLevel >= 1 )
                {
                printModel(cdebug,A,I);
                cdebug << endl;
                }

            if( mg->getNumberOfModelsComputed() == 1 )
                {
                // We are here for the first time and need to initialize.
                assert( ! QueryI );
                QueryI=new GINTERPRET(0,FALSE_V);
                }

            QueryI->unionPositive(*I);
            }
        else
            {
            // The problem was completely solved by the grounding.
            assert( ! QueryI );
            QueryI=new GINTERPRET(0,FALSE_V);
            }

        return true;
        }
    }

bool CautiousCallback(
          MODEL_GENERATOR *mg,
    const INTERPRET  *A,
    const GINTERPRET *I )
    {
    if( IsQueryGround )
        {
        cout << *Query
             << " is cautiously false, evidenced by ";
        printModel(cout,A,I);
        cout << endl;

        QueryAnswered = true;

        if( mg )
            mg->incrNumberOfModelsPrinted();

        return false;
        }
    else
        {
        if( mg )
            {
            assert( I );

            if( TraceLevel >= 1 )
                {
                printModel(cdebug,A,I);
                cdebug << endl;
                }

            if( mg->getNumberOfModelsComputed() == 1 )
                {
                // We are here for the first time and need to initialize.
                assert( ! QueryI );
                QueryI=new GINTERPRET(0,TRUE_FROM_UNDEFINED_V);
                }

            // Check whether any instance of the query predicate (the head
            // of the new "query rule") is contained in this model as well
            // as QueryI, and remove all instances which are not in this
            // intersection from QueryI.
            bool empty=true;
            for(size_t i=0; i < GINTERPRET::size(); i++)
                if( ! strcmp(GATOM(i).getTheRealOne().getPredName(),
                             PREDNAME_QUERY) )
                    if( I->isTrue(GATOM(i)) )
                        {
                        if( QueryI->isTrue(GATOM(i)) )
                            empty=false;
                        }
                    else
                        QueryI->forceFalse(GATOM(i));

            if( empty )
                {
                // We do not need to proceed with further models; QueryI
                // is now "empty" and we will print it that way.
                if( TraceLevel >= 1 )
                    cdebug << "The (relevant) intersection of all models up"
                              "to now is empty; stopping computation."
                           << endl;

                return false;
                }
            }
        else
            {
            // The problem was completely solved by the grounding.
            assert( ! QueryI );
            QueryI=new GINTERPRET(0,FALSE_V);
            }

        return true;
        }
    }

//////////////////////////////////////////////////////////////////////////////
// Statistics
//////////////////////////////////////////////////////////////////////////////

static void PrintStatisticsAfterGrounding()
    {
    cstats << endl << endl
           << "Options: "
           << "  ORb = "      << OptionBasicRewrite
           << "  ORp = "      << OptionProjectionRewrite
           << "  ORdr = "     << OptionRewriteDeleteRules
           << "  OMS = "      << OptionMagicSets
           << "  OS = "       << OptionSubsumptionChecking
           << "  OGp = "      << OptionGroundingProp
           << "  OGo = "      << OptionGroundingReorder
           << "  OGs = "      << OptionGroundingSemiNaive
           << "  OH = "       << OptionUseHeuristics
           << "  OPf = "      << OptionPartialChecksForwards
           << endl << endl;

    size_t NConstants=TERM::Strings.size()+TERM::Integers.size();

    if( OptionStatsLevel == 1 )
        {
        cstats << endl
               << "Residual Instantiation"       << endl
               << " Rules                    : " << GroundIDB.size() << endl
               << " Constraints              : " << GroundConstraints.size()
                                                 << endl
               << " Weak Constraints         : " << GroundWConstraints.size()
                                                 << endl;
        }
    else if( OptionStatsLevel > 1 )
        {
        cstats << endl 
               << "Predicates                : " << ATOM::Predicates.size() << endl
               << "|Herbrand Base|           : " << tallyGroundAtoms(
                                                      PREDICATE_NAMES::typeEDB,
                                                                    NConstants)
                                                  + tallyGroundAtoms(
                                                      PREDICATE_NAMES::typeIDB,
                                                                    NConstants)
                                                 << endl
               << "|Residual Herbrand Univ.| : " << NConstants << endl
               << "|Residual Herbrand Base|  : " << GINTERPRET::size()
                                                 << endl
               << "Facts (EDB)               : " << EDB.size() << endl
               << "Rules (IDB)               : " << IDB.size() << endl
               << " are ground?              : " << (IsIDBGround ? "yes" : "no")
                                                 << endl
               << " full grounding           : " << tallyGroundRules(IDB,
                                                                     NConstants)
                                                 << endl
               << " actual grounding         : " << GroundIDB.size() << endl
               << "Constraints               : " << Constraints.size() << endl
               << " are ground?              : " << (IsConstraintsGround ? "yes" : "no")
                                                 << endl
               << " full grounding           : " << tallyGroundConstraints(
                                                         Constraints,NConstants)
                                                 << endl
               << " actual grounding         : " << GroundConstraints.size()
                                                 << endl
               << "Weak Constraints          : " << WConstraints.size()
                                                 << endl
               << " (actual) grounding       : " << GroundWConstraints.size()
                                                 << endl;
        }

    cstats     << "Structural Size           : " << tallyStructuralSize(
                                                        GroundIDB,
                                                        GroundConstraints,
                                                        GroundWConstraints )
                                                    + EDB.size()
                                                 << endl;
    }

static void PrintFinalStatistics(const MODEL_GENERATOR *mg)
    {
    bool dummy_mg = ( mg == 0 );

    if( ! dummy_mg )
        mg->printStatistics(cstats); 
    else
        {
        GRULES r;
        GCONSTRAINTS c;
        GWEAKCONSTRAINTS w;
        CriteriaSequence s;

        MODEL_GENERATOR::FLAGS flags = static_cast<MODEL_GENERATOR::FLAGS>
                                       ( MODEL_GENERATOR::DEFAULT
                                         | MODEL_GENERATOR::PERFORM_CHECK );

        mg=new MODEL_GENERATOR( r, c, w,
                                0, 0, 0,
                                0,
                                0,
                                flags,
                                0,
                                0,
                                false,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                0,
                                s,
                                s );
        }

    cstats << "Time for first answer set : ";
    if( dummy_mg )
        cstats << timer.preMG << endl;
    else if( mg->getNumberOfModelsComputed() )
        cstats << timer.preMG+mg->timerFirst << endl;
    else
        cstats << "n/a" << endl;
    cstats << " (including instantiation)" << endl;

    cstats << "Time for all answer sets  : ";
    if( dummy_mg )
        cstats << timer.preMG << endl;
    else if( mg->getNumberOfModelsComputed() )
        cstats << timer.preMG+mg->timerTotal << endl;
    else
        cstats << "n/a" << endl;
    cstats << " (including instantiation)" << endl;

    if( OptionStatsLevel > 1 )
        cstats << "Parsing                   : " << timer.Parsing
                                                 << endl 
               << "Rewriting-Basic           : " << timer.RewritingBasic
                                                 << endl
               << "Rewriting-Projection      : " << timer.RewritingProjection
                                                 << endl
               << "SubsumptionChecking       : " << timer.SubsumptionChecking
                                                 << endl;
    cstats << "Instantiation time        : " << timer.Grounding << endl;

    if( OptionStatsLevel > 1 )
        cstats << " Depgraph                 : " << timer.GroundingDepgraph
                                                 << endl
               << " Postprocessing           : " << timer.GroundingPostprocess
                                                 << endl
               << " Constraints              : " << timer.GroundingConstraints
                                                 << endl;
    
    // interesting stats for secure planning:
    // FIXME: do this in a general callback-stats function?
    if( OptionStatsLevel > 1  && mg &&
        OptionFrontend == FRONTEND_PLANNING_NEW &&
        ( FPlan & FRONTEND_PLANNING_SECURE )
        )
        cstats << "Plan sec.checks/cache hits: " << planCheckCounter
                                                 << "/" 
                                                 << planCacheHit 
                                                 << endl;
    
    mg->printTimers(cstats);

    cstats << endl
           << "(Timers are informational only and must not be used for "
              "benchmarks.  See"
           << endl
           << "<http://www.dbai.tuwien.ac.at/proj/dlv/bench/> for "
              "further information.)"
           << endl;

    if( dummy_mg )
        delete mg;
    }

//////////////////////////////////////////////////////////////////////////////
// Command-line processing
//////////////////////////////////////////////////////////////////////////////

static void ParseCostBoundSpecification(
    const char *argstring,
    unsigned startindex,
    vector<unsigned> &optionCostBound )
    {
    assert( startindex < strlen(argstring) );
    for( const char *j = argstring+startindex; *j != '\0'; )
        {      
        unsigned x;
        if( *j == '_' )
            {
            x=UINT_MAX;
            j++;
            }
        else if( sscanf(j,"%u",&x) < 0 )
            {
            cerr << "Warning: costbound specification is bad: '"
                 << j << '\'' << endl;
            break;
            }
        optionCostBound.push_back(x);
        for( ; isdigit(*j); j++ )
            if( x==UINT_MAX )
                break;
        if( *j == ',' )
            {
            j++;
            if( *j == '\0' )
                cerr << "Warning: costbound specification "
                        "terminated by comma." << endl;
            }
        else if( *j != '\0' )
            {
            cerr << "Warning: Heuristic specification "
                    "contains a bad delimiter: '" 
                 << *j << '\'' << endl;
            break;
            }
        }
    if( optionCostBound.size() == 0 )
        cerr << "Warning: Empty costbound specification. "
                "Using default costbound." << endl;
    }

static void ParseHeuristicSpecification(
    const char* argstring,
    unsigned startindex,
    CriteriaSequence &HeuristicObject )
    {
    assert( startindex < strlen(argstring) );

    for( const char *j = argstring+startindex; *j != '\0'; )
        {
        unsigned x;
        if( sscanf(j,"%u",&x) < 1)
            {
            cerr << "Warning: Heuristic specification is bad: '"
                 << j << '\'' << endl;
            break;
            }
        HeuristicObject.push_back(x);
        for( ; isdigit(*j); j++ );
        if( *j == ',' )
            {
            j++;
            if( *j == '\0' )
                cerr << "Warning: Heuristic specification "
                        "terminated by comma." << endl;
            }
        else if( *j != '\0' )
            {
            cerr << "Warning: Heuristic specification "
                    "contains a bad delimiter: '" 
                 << *j << '\'' << endl;
            break;
            }
        }

    if( HeuristicObject.size() == 0 )
        cerr << "Warning: Empty heuristic specification. "
                "Using default heuristic." << endl;
    }

static void ParseFilteredPredicates(
    const char *s,
    STRINGSET &filter )
    {
    while( *s != '\0' )
        {
        // Locate the first comma in input.  If there is none, we consider
        // the rest of the input and are done.
        const char *p=strchr(s,',');
        if( p == NULL ) 
            {
            filter.insert(s);
            break;
            }
        else
            {
            // Extract the substring between the current position and the
            // comma we located. Note that we must not free the buffer, as
            // STRINGSETs only maintain pointers, not full strings.
            char *buf=new char[p-s+1];
            strncpy(buf,s,p-s);
            buf[p-s]='\0';
            filter.insert(buf);
            
            s=p+1;
            }
        }
    }

//////////////////////////////////////////////////////////////////////////////
static bool readNumberOption(
//
// If the prefix of the command-line argument arg matches name (which usually
// is of the form "-XYZ="), try to convert the suffix of arg to an integer,
// verify that it is in the intervall [lower,upper] and finally store it
// into *variable.
//
// In case of problems, do not modifiy *variable but issue an error message.
//
// Returns bool, indicating whether the prefix of arg matched name.
//
    const char *name,
    const char *arg,
    int *variable,
    int lower = 0,
    int upper = MAX_MAXINT)
    {
    if( strncmp(arg, name, strlen(name)) )
        return false; 

    char *err;
 
    int n=strtol(&arg[strlen(name)], &err, 10);
    if (*err || n < lower || n > upper )
        {
        cerr << "Argument to " << name 
             << "<n> must be an integer between "
             << lower << " and " << upper << " (inclusive) - ignored."
             << endl;
        }
    else
        *variable = n;

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
static bool readNumberOption(
// For unsigned variables: Assert positive lower and upper bounds and call
// the original function with a cast.
//
    const char *name,
    const char *arg,
    unsigned *variable,
    int lower = 0,
    int upper = MAX_MAXINT)
    {
    assert( lower >= 0  &&  upper <= MAX_MAXINT );

    return readNumberOption(name, arg, (int *) variable, lower, upper);
    }

//////////////////////////////////////////////////////////////////////////////
// isSuffix(string,suffix) returns true if string ends with suffix.
//
static bool isSuffix( const char *string, const char *suffix )
    {
    if( ! string || ! suffix || strlen(string) < strlen(suffix) )
	return false;
    return ! strcmp(string+strlen(string)-strlen(suffix),suffix);
    }

//////////////////////////////////////////////////////////////////////////////
int main(int argc, char* argv[])
//
    {
    bool     optionPipe=false;
    bool     optionOmitModelCheck=false;
    unsigned optionModels=0;
    vector<unsigned> optionCostBound;
    bool     optionPrintDepgraphs=false;
    enum
        {
        PRINT_EDB           = 0x0001,
        PRINT_IDB           = 0x0002,
        PRINT_GIDB          = 0x0004,
        PRINT_CONSTRAINTS   = 0x0008,
        PRINT_GCONSTRAINTS  = 0x0010,
        PRINT_WCONSTRAINTS  = 0x0020,
        PRINT_GWCONSTRAINTS = 0x0040,
        PRINT_INTERPRET     = 0x0080
        };
    unsigned optionDebugPrint=0;
    bool     optionDetCons=false;
    bool     optionInstantiate=false;


#ifdef BISON_DEBUG
    yydebug=1;                   // Set to 1, and YACC will output debug info
#endif
    WhoAmI=argv[0];

    // Analyse command-line switches (any parameter starting with a dash is
    // considered a switch) and count the number of filenames provided.
    unsigned filenames=0;

    for(int i=1; i < argc; i++)
        if( argv[i][0] == '-' )
            {
            if( ! strcmp(argv[i],"-n=all") )
                // Special form for -n=0, upon request by Nicola Leone.
                optionModels=0;
            else if( ! strncmp(argv[i],"-n=", 3) )
                optionModels=strtoul(&argv[i][3],NULL,10);

	    else if ( readNumberOption("-N=", argv[i], &MaxInteger) )
		; // cf. function readNumberOption()

            else if ( !strncmp(argv[i],"-costbound=",11) )
                {
                ParseCostBoundSpecification(argv[i],11,optionCostBound );
                OptionCostBound=true;
                }

            else if( ! strcmp(argv[i],"-det") )
                optionDetCons=true;

            else if( ! strcmp(argv[i],"-instantiate") )
                optionInstantiate=true;

	    else if ( ! strncmp(argv[i],"-filter=",8) )
                ParseFilteredPredicates(argv[i]+8,PredFilter);
            else if ( ! strncmp(argv[i],"-pfilter=",9) )
                ParseFilteredPredicates(argv[i]+9,PredPFilter);

            else if( ! strcmp(argv[i],"-facts") )
                OptionPrintFacts=true;
            else if( ! strcmp(argv[i],"-nofacts") )
                OptionPrintFacts=false;
            else if( ! strcmp(argv[i],"-satlike") )
                OptionPrintSATlike=true;
            else if( ! strcmp(argv[i],"-wctrace") )
                OptionTraceComputationWC=true;
            else if( ! strcmp(argv[i],"-silent") )
                OptionSilent=true;
            else if( ! strcmp(argv[i],"-v") )
                OptionVerbosity=1;

            else if( ! strcmp(argv[i],"--depgraph") )
                optionPrintDepgraphs=true;

            else if( ! strcmp(argv[i],"--negpts") )
                OptionUseOldPTs=true;
            else if( ! strcmp(argv[i],"--pospts") )
                OptionUseOldPTs=false;
            else if( ! strncmp(argv[i],"--ptcache=",10) )
                OptionPTCacheSize = strtoul(&argv[i][10],NULL,10);
            else if( ! strcmp(argv[i],"--complpt") )
                OptionHeuristicsConsiderComplementaryPT = true;
            else if( ! strcmp(argv[i],"--nocomplpt") )
                OptionHeuristicsConsiderComplementaryPT = false;
            else if( ! strncmp(argv[i],"--combinecomplpt=",17) )
                OptionHeuristicsCombineComplementaryPT 
                    = strtoul(&argv[i][17],NULL,10);
            else if( ! strcmp(argv[i],"--canchoosecomplpt") )
                OptionHeuristicsCanChooseComplementaryPT = true;
            else if( ! strcmp(argv[i],"--cannotchoosecomplpt") )
                OptionHeuristicsCanChooseComplementaryPT = false;

            else if( ! strcmp(argv[i],"--GUS") )
                OptionComputeGUS=true;
            else if( ! strcmp(argv[i],"--noGUS") )
                OptionComputeGUS=false;

            else if( ! strncmp(argv[i],"--heuristic=",12) )
                ParseHeuristicSpecification(argv[i],12,HeuristicSequence);
            else if( ! strncmp(argv[i],"--cheuristic=",13) )
                ParseHeuristicSpecification(argv[i],13,
                                            HeuristicSequenceChecker);
            else if( ! strncmp(argv[i],"--combineheuristic=",19) )
                ParseHeuristicSpecification(argv[i],19,
                                            HeuristicCombinationSequence);

            else if( ! strcmp(argv[i],"-stats") )
                OptionStatsLevel=1;
            else if( ! strcmp(argv[i],"-stats++") )
                OptionStatsLevel=2;

            else if( ! strcmp(argv[i],"--print-edb") )
                optionDebugPrint |= PRINT_EDB;
            else if( ! strcmp(argv[i],"--print-idb") )
                optionDebugPrint |= PRINT_IDB;
            else if( ! strcmp(argv[i],"--print-gidb") )
                optionDebugPrint |= PRINT_GIDB;
            else if( ! strcmp(argv[i],"--print-constraints") )
                optionDebugPrint |= PRINT_CONSTRAINTS;
            else if( ! strcmp(argv[i],"--print-gconstraints") )
                optionDebugPrint |= PRINT_GCONSTRAINTS;
            else if( ! strcmp(argv[i],"--print-wconstraints") )
                optionDebugPrint |= PRINT_WCONSTRAINTS;
            else if( ! strcmp(argv[i],"--print-gwconstraints") )
                optionDebugPrint |= PRINT_GWCONSTRAINTS;
            else if( ! strcmp(argv[i],"--print-interpret") )
                optionDebugPrint |= PRINT_INTERPRET;
#ifndef NDEBUG                             
	    else if ( readNumberOption("-trace=", argv[i], &TraceLevel,0,3) )
		; // cf. function readNumberOption()

	    else if ( readNumberOption("-dtrace=", argv[i], &DTraceLevel,0,1) )
		; // cf. function readNumberOption()

	    else if ( readNumberOption("-ftrace=", argv[i], &FTraceLevel,0,1) )
		; // cf. function readNumberOption()

	    else if ( readNumberOption("-ptrace=", argv[i], &PTraceLevel,0,1) )
		; // cf. function readNumberOption()

	    else if ( readNumberOption("-gtrace=", argv[i], &GTraceLevel,0,2) )
		; // cf. function readNumberOption()

	    else if ( readNumberOption("-ctrace=", argv[i], &CTraceLevel,0,2) )
		; // cf. function readNumberOption()

	    else if ( readNumberOption("-htrace=", argv[i], &HTraceLevel,0,3) )
		; // cf. function readNumberOption()
#endif

            else if( ! strcmp(argv[i],"-O0") )
                {
                OptionMagicSets = false;
                OptionSubsumptionChecking = false;
                OptionBasicRewrite = false;
                OptionProjectionRewrite = false;
                OptionRewriteDeleteRules = true;
                OptionGroundingProp = false;
                OptionGroundingReorder = 0;
                OptionGroundingSemiNaive = false;
                OptionUseHeuristics = false;
                OptionPartialChecksForwards=false;
                }

            else if( ! strcmp(argv[i],"-OMS") )
                OptionMagicSets=true;
            else if( ! strcmp(argv[i],"-OMS-") )
                OptionMagicSets=false;

            else if( ! strcmp(argv[i],"-OS") )
                OptionSubsumptionChecking = true;
            else if( ! strcmp(argv[i],"-OS-") )
                OptionSubsumptionChecking = false;
            
            else if( ! strcmp(argv[i],"-OR") )
                {
                OptionBasicRewrite=true;
                OptionProjectionRewrite=true;
                }
            else if( ! strcmp(argv[i],"-OR-") )
                {
                OptionBasicRewrite=false;
                OptionProjectionRewrite=false;
                }
            else if( ! strcmp(argv[i],"-ORb") )
                OptionBasicRewrite=true;
            else if( ! strcmp(argv[i],"-ORb-") )
                OptionBasicRewrite=false;
            else if( ! strcmp(argv[i],"-ORp") )
                OptionProjectionRewrite=true;
            else if( ! strcmp(argv[i],"-ORp-") )
                OptionProjectionRewrite=false;
 
            else if( ! strcmp(argv[i],"-ORdr") )
                OptionRewriteDeleteRules=true;
            else if( ! strcmp(argv[i],"-ORdr-") )
                OptionRewriteDeleteRules=false;
 
            else if( ! strcmp(argv[i],"-OGp") )
                OptionGroundingProp=true;
            else if( ! strcmp(argv[i],"-OGp-") )
                OptionGroundingProp=false;

            else if( ! strcmp(argv[i],"-OGo0") )
                OptionGroundingReorder=0;
            else if( ! strcmp(argv[i],"-OGo1") )
                OptionGroundingReorder=1;
            else if( ! strcmp(argv[i],"-OGo2") )
                OptionGroundingReorder=2;

            else if( ! strcmp(argv[i],"-OGs") )
                OptionGroundingSemiNaive=true;
            else if( ! strcmp(argv[i],"-OGs-") )
                OptionGroundingSemiNaive=false;

            else if( ! strcmp(argv[i],"-OH") )
                OptionUseHeuristics=true;
            else if( ! strcmp(argv[i],"-OH-") )
                OptionUseHeuristics=false;

            else if( ! strcmp(argv[i],"-OM-") )
                optionOmitModelCheck=true;
            else if( ! strncmp(argv[i],"-OMm=", 5) )
                OptionModelCheckMode=strtoul(&argv[i][5],NULL,10);

            else if( ! strcmp(argv[i],"-OPf-") )
                OptionPartialChecksForwards=false;
            else if( ! strcmp(argv[i],"-OPf") )
                OptionPartialChecksForwards=true;

            else if( ! strcmp(argv[i],"-brave")
                     || ! strcmp(argv[i],"-FB") )
                OptionFrontend=FRONTEND_BRAVE;
            else if( ! strcmp(argv[i],"-cautious")
                     || ! strcmp(argv[i],"-FC") )
                OptionFrontend=FRONTEND_CAUTIOUS;

            else if( ! strcmp(argv[i],"-FD") )
                {
                OptionFrontend=FRONTEND_DIAG_ABDUCTIVE;
                FDiag = FRONTEND_DIAG_DEFAULT;
                }
            else if( ! strcmp(argv[i],"-FDmincost") )
                {       
                OptionFrontend=FRONTEND_DIAG_ABDUCTIVE;
                FDiag = FRONTEND_DIAG_WEIGHTED;
                }
            else if( ! strcmp(argv[i],"-FDmin") )
                {
                OptionFrontend=FRONTEND_DIAG_ABDUCTIVE;
                FDiag = FRONTEND_DIAG_MIN;
                }
            else if( ! strcmp(argv[i],"-FDsingle") )
                {
                OptionFrontend=FRONTEND_DIAG_ABDUCTIVE;
                FDiag = FRONTEND_DIAG_SINGLE;
                }
            else if( ! strcmp(argv[i],"-FR") )
                {
                OptionFrontend=FRONTEND_DIAG_REITER;
                FDiag = FRONTEND_DIAG_DEFAULT;
                }
            else if( ! strcmp(argv[i],"-FRmin") )
                {
                OptionFrontend=FRONTEND_DIAG_REITER;
                FDiag = FRONTEND_DIAG_MIN;
                }

            else if( ! strcmp(argv[i],"-FRsingle") )
                {
                OptionFrontend=FRONTEND_DIAG_REITER;
                FDiag = FRONTEND_DIAG_SINGLE;
                }

            else if( ! strcmp(argv[i],"-FPc") )
                {
                OptionFrontend=FRONTEND_PLANNING_C;
                FPlan=FRONTEND_PLANNING_DEFAULT;
                }
            else if( ! strcmp(argv[i],"-FP") )
                {
                OptionFrontend=FRONTEND_PLANNING_NEW;
                FPlan=FRONTEND_PLANNING_ASK;
                }
            else if( ! strcmp(argv[i],"-FPopt") )
                {
                OptionFrontend=FRONTEND_PLANNING_NEW;
                FPlan=FRONTEND_PLANNING_OPTIMISTIC;
                }
            else if( ! strcmp(argv[i],"-FPsec") )
                {
                OptionFrontend=FRONTEND_PLANNING_NEW;
                FPlan=FRONTEND_PLANNING_SECURE;
                }
            else if( ! strncmp(argv[i],"-FPcheck=", 9) )
                {
                char *err;
                unsigned n=strtoul(&argv[i][9], &err, 10);
                if (*err || n < 1 || n > 2)
                    {
                    cerr << "-FPcheck Ignored - Implemented checks are " 
                            "1 and 2. Check defaults to FPcheck=1."
                         << endl;
                    }
                else
                    // Check1: set variable FPcheck = 0
                    // Check2: set variable FPcheck = 1
                    FPcheck=n-1;

                OptionFrontend=FRONTEND_PLANNING_NEW;
                FPlan=FRONTEND_PLANNING_SECURE;
                }
            else if( ! strncmp(argv[i],"-FPsoundcheck=", 14) )
                {
                char *err;
                unsigned n=strtoul(&argv[i][14], &err, 10);
                if (*err || n < 1 || n > 2)
                    {
                    cerr << "-FPsoundcheck Ignored - Implemented checks are " 
                            "1 and 2."
                         << endl;
                    }
                else
                    FPsoundcheck=n-1;

                OptionFrontend=FRONTEND_PLANNING_NEW;
                FPlan=FRONTEND_PLANNING_SECURE;
                }
            else if( ! strncmp(argv[i],"-FPcompletecheck=", 17) )
                {
                char *err;
                unsigned n=strtoul(&argv[i][17], &err, 10);
                if (*err || n < 1 || n > 2)
                    {
                    cerr << "-FPcompletecheck Ignored - Implemented checks "
                            "are 1 and 2."
                         << endl;
                    }
                else
                    FPcompletecheck=n-1;

                OptionFrontend=FRONTEND_PLANNING_NEW;
                FPlan=FRONTEND_PLANNING_SECURE;
                }

            else if( readNumberOption("-planlength=", argv[i], 
                                      &maxPlanLength) )
                ; // cf. function readNumberOption() 

            else if( readNumberOption("-planminactions=", argv[i], 
                                      &minActions) )
                ; // cf. function readNumberOption()

            else if( readNumberOption("-planmaxactions=", argv[i], 
                                      &maxActions) )
                ; // cf. function readNumberOption()

            else if( readNumberOption("-plancache=", argv[i], 
                                      &planCacheSize) )
                ; // cf. function readNumberOption()

            else if( ! strcmp(argv[i],"-FS") )
                OptionFrontend=FRONTEND_SQL;
            
            else if( ! strcmp(argv[i],"-wait") )
                OptionWait=true;

            else if( ! strcmp(argv[i],"--") )
                optionPipe=true;

            else if( ! strcmp(argv[i],"-license") )
                {
                printLogo();
                cout << LicenseText << endl;
                return 0;
                }

            else if( ! strcmp(argv[i],"-h")  ||  ! strcmp(argv[i],"-help") )
                {
                printLogo();
                printUsage(cout,true);
                return 0;
                }
            else 
                {
                printLogo();
                printUsage(cerr,false);
                return -1;
                }
            }                 
        else
            filenames++;

    if( optionInstantiate )
        printLogo("% Instantiation generated by\n% ");
    else if( ! OptionSilent )
        printLogo();

    // We need some kind of input: either a file or stdin is mandatory
    // (unless -v has been specified, in which case we exit silently).
    if( filenames == 0  &&  ! optionPipe )
        {
        if( OptionVerbosity )
            return 0;
        else
            {
            printUsage(cerr,false);
            return -1;
            }
        }

    if( optionDetCons  &&  OptionFrontend != FRONTEND_DATALOG )
        {
        cerr << "Option -det is not applicable for front-ends." << endl;
        SafeExit(1);
        }
    else if( OptionTraceComputationWC && OptionCostBound )
        {
        cerr << "Options -costbound and -wctrace are not applicable "
             << "together." << endl;
        SafeExit(1);
        }

    // Read input database(s)

    timer.preMG.start();
    timer.Parsing.start();

    initBuiltins();
    ObjectIDs.push_back(OID("#defaultobject",0,&EDB,&IDB));

    // While parsing user input, we don't accept INTERNAL_IDs.
    ParserStateInternal = false;

    for(int j=1; j < argc; j++)
        if( argv[j][0] != '-' )
            {
            parser_file=argv[j];
            parser_line=0;

            FILE* fp=fopen(parser_file,"r");
            if( fp )
                {
                if( OptionVerbosity )
                    cout << "Reading \"" << parser_file << "\"" << endl;

                parser_line=1;
                
                // We should make the following a bit more elegant, 
                // especially remove some code duplication!

                if( isSuffix(parser_file,".hyp") )
                    {
                    diagParseLiterals=false;

                    diagyyin=fp;
                    diagyyparse();
                    fclose(diagyyin);
                    }
                else if( isSuffix(parser_file,".obs") )
                    {
                    diagParseLiterals=true;

                    diagyyin=fp;
                    diagyyparse();
                    fclose(diagyyin);
                    }
                else if( isSuffix(parser_file,".sql") )
                    {
                    if( OptionFrontend != FRONTEND_SQL )
                        { 
                        cerr << "SQL frontend not selected." << endl;
                        exit(1); 
                        }

                    sqlyyin=fp;
                    sqlyyparse();
                    fclose(sqlyyin);
                    }
		else if( isSuffix(parser_file,".plan") )
                    {
                    if( OptionFrontend != FRONTEND_PLANNING_C
                     && OptionFrontend != FRONTEND_PLANNING_NEW )
                        {
                        cerr << "No planning frontend selected." << endl;
                        exit(1); 
                        }

                    planyyin=fp;
                    planyyparse();
                    fclose(planyyin);
                    } 
                else
                    {
                    yyin=fp;
                    yyparse();
                    fclose(yyin);
                    }
                }
            else
                yyerror("can't open input");
            }

    if( optionPipe )
        {
        parser_file="";
        parser_line=1;

        yyin=stdin;

        yyparse();
        }

    ParserStateInternal = true;

    if( parser_errors )
        {
        cerr << "Aborting due to parser errors."
             << endl;

        SafeExit(1);
        }

    ObjectsExist = ( ObjectIDs.size() > 1 );

    if( StrictRulesExist && ! ObjectsExist )
        {
        cerr << "Error: Strict rules used without objects." << endl;
        SafeExit(1);
        }


    //////////////////////////////////////////////////////////////////////////
    // Some Debugging Output
    if( TraceLevel >= 1 )
        {
        cdebug << "  Filtered predicates: ";
        print_list(cdebug,PredFilter,"; ");
        cdebug << endl
               << "only in positive form: ";
        print_list(cdebug,PredPFilter,"; ");
        cdebug << endl;
        }
    if( TraceLevel >= 3 )
        {
        cdebug << "sizeof GATOM        = " << sizeof(GATOM)        << endl
               << "sizeof GDISJUNCTION = " << sizeof(GDISJUNCTION) << endl
               << "sizeof GCONJUNCTION = " << sizeof(GCONJUNCTION) << endl
               << "sizeof GRULE        = " << sizeof(GRULE)        << endl;
        }
        
    //////////////////////////////////////////////////////////////////////////
    // Handle frontends. (SQL does most of its work during parsing.)

    // Non-Datalog frontends (usually) produce non-ground programs.

    if( ( OptionFrontend != FRONTEND_DATALOG
          && OptionFrontend != FRONTEND_BRAVE
          && OptionFrontend != FRONTEND_CAUTIOUS )
        || ObjectsExist )
        {
        IsIDBGround=false;
        IsConstraintsGround=false;
        IsWConstraintsGround=false;
        }

    bool (*callback)(MODEL_GENERATOR*,const INTERPRET*,const GINTERPRET*) = 0;
    bool (*callbackCheck)(MODEL_GENERATOR*,const INTERPRET*,const GINTERPRET*) = 0;

    switch( OptionFrontend )
        {
        case FRONTEND_DATALOG:                
            callback=DatalogCallback;
            break;
        case FRONTEND_BRAVE:
            callback=BraveCallback;
            break;
        case FRONTEND_CAUTIOUS:
            callback=CautiousCallback;
            break;
        case FRONTEND_DIAG_ABDUCTIVE:
        case FRONTEND_DIAG_REITER:
            CreateDiagnosis(IDB,diagH,diagO);
            callback=DiagCallback;
            break;
        case FRONTEND_SQL:
            break;
        case FRONTEND_PLANNING_C:
        case FRONTEND_PLANNING_NEW:
            CallbackAfterGrounding=PlanCallbackAfterGrounding;
            CallbackAfterParsing=PlanCallbackAfterParsing;
            callbackCheck = PlanCallbackCheck;
            callback=PlanCallback;
            break;
        default:
            assert( 0 );
        }

    // FIXME: Is this the correct position? There is some
    //        stuff done before the frontends are handled...  
    if( CallbackAfterParsing )
        CallbackAfterParsing();

    // Create constraints "representing" true negation.

    bool failure=false;
    for( NEG_ATOMS::const_iterator i=NegativeAtoms.begin();
         i != NegativeAtoms.end();
         i++)
        {
        if( ! createConstraint(*i) )
            failure=true;
        }
 
    if( failure ) 
        SafeExit(1);

    //////////////////////////////////////////////////////////////////////////
    // Process Queries, possibly exiting at this point.

    if( ! HandleQuery() )
	SafeExit(1);

    if( ObjectsExist )
        InheritanceRewriting(ObjectIDs);
        
    timer.Parsing.stop();

    //////////////////////////////////////////////////////////////////////////
    // Import EDB from databases.   
#ifdef WITH_ODBC
    if (ODBC_Imports.size())
        odbcImport();
#endif

    //////////////////////////////////////////////////////////////////////////
    // Rewriting

    assert( I.isPositivePartEmpty() );

    timer.RewritingBasic.start();
    bool hasBuiltins;
    bool hasAggregates;
    BasicRewriting(IDB,Constraints,hasBuiltins,hasAggregates);
    timer.RewritingBasic.stop();

    if( OptionProjectionRewrite )
        {
        timer.RewritingProjection.start();

        if( ! IsIDBGround )
            ProjectionRewriting(IDB);

        if( ! IsConstraintsGround )
            ProjectionRewriting(Constraints,IDB);

        timer.RewritingProjection.stop();
        }

    //////////////////////////////////////////////////////////////////////////
    // Subsumption checking of IDB rules

    if( OptionSubsumptionChecking )
        {
        timer.SubsumptionChecking.start();

        if( !SubsumptionCheckingRewriting() )
	    SafeExit(1);

        timer.SubsumptionChecking.stop();
        }

    //////////////////////////////////////////////////////////////////////////
    //Magic Sets
    
    if( OptionMagicSets )
        if( ! Magic() )
            SafeExit(1);

    //////////////////////////////////////////////////////////////////////////
    // Debug output (non-ground structures)

    if( TraceLevel >= 2 )
        {
        cdebug << "MaxInteger     = " << MaxInteger << endl
               << "MaxIntegerSeen = " << MaxIntegerSeen << endl;
        }

    if( optionDebugPrint & PRINT_EDB )
        cout << EDB << endl;
    if( optionDebugPrint & PRINT_IDB )
        {
        print_list(cout,IDB,"\n");
        cout << endl;
        }
    if( optionDebugPrint & PRINT_CONSTRAINTS )
        {
        print_list(cout,Constraints,"\n");
        cout << endl;
        }
    if( optionDebugPrint & PRINT_WCONSTRAINTS )
        {
        print_list(cout,WConstraints,"\n");
        cout << endl;
        }

    //////////////////////////////////////////////////////////////////////////
    // Grounding

    bool groundingFailed = false;
    timer.Grounding.start();

    // Use the Propositional Grounder only if the input is completely
    // ground and does not contain builtins.

    if( OptionGroundingProp
        && IsIDBGround && IsConstraintsGround && IsWConstraintsGround
        && IsQueryGround
        && ! hasBuiltins )
        {
        if( TraceLevel >= 1  ||  GTraceLevel >= 1 )
            cdebug << "Invoking propositional \"grounder\"." << endl;
     
        groundingFailed = ! GroundPropositionalInput( 
                                IDB,Constraints,WConstraints,EDB,
                                GroundIDB,GroundConstraints,
                                GroundWConstraints,ViolatedWConstraints, 
                                optionInstantiate);
        }
    else
        {
        timer.GroundingDepgraph.start();

        DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM> NonGroundDepGraph(
            IDB, CONSTRAINTS(),
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::SIMONASORDERING |
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::NONGROUND );

        timer.GroundingDepgraph.stop();

        GROUNDING grounding(NonGroundDepGraph,optionInstantiate);
        groundingFailed = ! grounding.Ground();
        }

    timer.Grounding.stop();

    //////////////////////////////////////////////////////////////////////////
    // Debug output (ground structures)

    if( optionDebugPrint & PRINT_GIDB )
        {
        print_list(cout,GroundIDB,"\n");
        cout << endl;
        }
    if( optionDebugPrint & PRINT_GCONSTRAINTS )
        {
        print_list(cout,GroundConstraints,"\n");
        cout << endl;
        }
    if( optionDebugPrint & PRINT_GWCONSTRAINTS )
        {
        print_list(cout,GroundWConstraints,"\n");
        cout << endl;
        }
    if( optionDebugPrint & PRINT_INTERPRET )
        cout << I << endl;

    //////////////////////////////////////////////////////////////////////////
    // Weak Constraints and their COSTs.

    //Initialise costBound.
    COST* costBound;
    costBound = new COST(MaxWeakConstraintLevel,UINT_MAX);
    for( unsigned k=1; k <= optionCostBound.size(); k++)
        {
        if (k >  MaxWeakConstraintLevel)
            break;
        (*costBound).setCost(k,(*(optionCostBound.begin()+k-1)*SCALEFACT));
        }
    // Initialise groundingCost.
    COST *groundingCost;
    groundingCost = new COST(MaxWeakConstraintLevel,0);

    // Check whether there are weak constraints violated during the grounding
    // and in this case increase groundingCost.
    if ( ! ViolatedWConstraints.empty() )
        {
        for ( GWEAKCONSTRAINTS::const_iterator i = ViolatedWConstraints.begin();
                                               i != ViolatedWConstraints.end(); 
                                               i++ )
            {
            // Traverse all violated weak constraints and update groundingCost
            // accordingly.
            (*groundingCost).updateCost((*i).getWeights());
            if( *groundingCost > *costBound )
                SafeExit(1);
            }
        }

    //////////////////////////////////////////////////////////////////////////
    // Finally, we're done with the Grounding.

    GATOM::freeze();
    GINTERPRET::setSize( GATOM::h2iTable.size() );
    GATOMSET::setSize( GATOM::h2iTable.size() );

    if( OptionStatsLevel )
        PrintStatisticsAfterGrounding();

    if( optionInstantiate )
        { 
        if( OptionPrintFacts )
            {
            // print EDB as facts
            cout << "% EDB facts:" << endl;
            EDB.foreach( InterpretAtomAsRulePrinter(cout) );
            }

        if( ! I.isPositivePartEmpty() )
            cout << "% Facts derived to be true in every answer set:" << endl;

        I.foreachPositive( InterpretAtomAsRulePrinter(cout) );

        if( GroundIDB.size()
            || GroundConstraints.size()
            || GroundWConstraints.size() )
            cout << "% Residual ground instantiation of the program:" << endl;

        print_list(cout,GroundIDB,"\n");
        cout << endl;

        print_list(cout,GroundConstraints,"\n");
        cout << endl;

        print_list(cout,GroundWConstraints,"\n");
        cout << endl;

        if( OptionPrintSATlike )
            GINTERPRET::printSATlike(cout);
        }
    
     if( optionInstantiate || groundingFailed )
	{
        if( OptionStatsLevel )
            PrintFinalStatistics(0);

        SafeExit(0);
        }

    if( CallbackAfterGrounding )
        CallbackAfterGrounding();

    //////////////////////////////////////////////////////////////////////////
    // Debugging Output
    
    if( TraceLevel >= 3 )
        {
        cdebug << endl 
               << "The following string constants have been introduced: ";
        print_list(cdebug,TERM::Strings.begin(),TERM::Strings.end(),", ");
        cdebug << endl
               << "The following numeric constants have been introduced: ";
        print_list(cdebug,TERM::Integers.begin(),TERM::Integers.end(),", ");
        cdebug << endl;

        cdebug << "The following predicates have been introduced: ";
        print_list(cdebug,ATOM::Predicates.begin(),ATOM::Predicates.end(),", ");
        cdebug << endl 
               << endl;
        }

    if( optionPrintDepgraphs )
        {
	testDEPGRAPH(IDB,Constraints,GroundIDB,GroundConstraints);
        exit(EXIT_SUCCESS);
        }

    bool nongroundHCF;

    // If IDB is "small" compared to GroundIDB, compute its DEPGRAPH and check
    // for head-cycle freeness. This will be used by the Model Generator to
    // avoid construction of a (possibly much larger) ground DEPGRAPH. 
    //
    // FIXME: We should re-evaluate this heuristics!
    if( GroundIDB.size() > 3*IDB.size() )
        {
        DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM> dg(
            IDB, CONSTRAINTS(),
            DEPGRAPH<ATOM,CONSTRAINTS,PROGRAM>::NONGROUND );
        
        nongroundHCF=dg.isHCF();

        if( TraceLevel >= 1 )
            cdebug << "Non-ground dep.graph has been built. "
                      "HCF = " << nongroundHCF
                   << endl;
        }
    else
        {
        nongroundHCF=false;

        if( TraceLevel >= 1 )
            cdebug << "Non-ground dep.graph has not been built. "
                      "Assuming HCF = 0"
                   << endl;
        }

    //////////////////////////////////////////////////////////////////////////
    // Main computation

    if( GroundIDB.begin() == GroundIDB.end() )
        {
        for( GCONSTRAINTS::const_iterator i=GroundConstraints.begin();
             i != GroundConstraints.end();
             i++ )
            {
            // The GroundIDB is empty, so the constraint only contains
            // ``undefined'' literals.
            // If all of these are negative, it is always violated.
            
            if( (*i).pos_begin() == (*i).pos_end() )
                {
                if( TraceLevel >= 1 )
                    {
                    cdebug << "Constraint  " << *i 
                           << "  is always violated [dl.C]." << endl;
                    }
                SafeExit(0);
                }
            }
        
        if( OptionVerbosity )
            cout << endl;
        
        timer.preMG.stop();

        switch( OptionFrontend )
            {
            case FRONTEND_DATALOG:
                // Check whether some weak constraints were parsed.
                if( !WConstraints.empty() )
                    cout << "Best model: ";

                printModel(cout,&I,0);
                cout << endl;

                if(!WConstraints.empty())
                    {
                    cout << "Cost ([Weight:Level]): ";
                    // Check whether there are violated weak constraints
                    // during grounding and print out Best Cost.
                    if(!ViolatedWConstraints.empty())
                        {
                        assert(groundingCost);
                        cout << *groundingCost << endl;
                        }
                    else // Non violated weak constraints.
                        cout << "<0>" << endl;
                    }
                break;
            case FRONTEND_SQL:
                SQLCallback(&I);
                break;
            default:
                callback(0,&I,0);
            }

        if( OptionStatsLevel )
            PrintFinalStatistics(0);
        }
    else
        {
        assert( OptionFrontend != FRONTEND_SQL );

        if( OptionVerbosity )
            cout << endl;

        // Set the default heuristic (unless some heuristic has been
        // set explicitly).
        if( HeuristicSequence.size() == 0 )
            ParseHeuristicSpecification(heuristicSequenceDefault,0,
                                        HeuristicSequence);

        if( OptionHeuristicsConsiderComplementaryPT )
            {
            if ( HeuristicCombinationSequence.size() != 0 )
                {
                // If a heuristic combination sequence has been
                // explicitly specified, check that its length matches
                // the length of the heuristic sequence.
                if( HeuristicCombinationSequence.size()
                    != HeuristicSequence.size() )
                    {
                    cerr << "Length of --heuristic does not match length "
                        "of --combineheuristic." << endl;
                    SafeExit(1);
                    }
                }
            else
                {
                // If a heuristic combination sequence has not been
                // explicitly specified, construct one out of the uniform
                // combination criterion (which has a default value).
                for( CriteriaSequence::const_iterator i
                         = HeuristicSequence.begin();
                     i != HeuristicSequence.end();
                     i++ )
                    {
                    HeuristicCombinationSequence.push_back(
                        OptionHeuristicsCombineComplementaryPT);
                    }
                }
            }

        // Set the default heuristic for model checks (unless some
        // heuristic has been set explicitly).

        if( HeuristicSequenceChecker.size() == 0 )
            ParseHeuristicSpecification(heuristicSequenceCheckerDefault,0,
                                        HeuristicSequenceChecker);

#ifdef HEURISTICSTATS
	statsHCriterionApplied.insert(statsHCriterionApplied.end(),
                                      CRITERIA_MAX+2,0);
#endif

        timer.preMG.stop();

        MODEL_GENERATOR::FLAGS flags=MODEL_GENERATOR::DEFAULT;

        if( OptionComputeGUS )
            flags = static_cast<MODEL_GENERATOR::FLAGS>
                    (flags | MODEL_GENERATOR::PERFORM_GUS);

        if( nongroundHCF )
            flags = static_cast<MODEL_GENERATOR::FLAGS> 
                    (flags | MODEL_GENERATOR::KNOWN_HCF);
        
        if( ! optionOmitModelCheck )
            flags = static_cast<MODEL_GENERATOR::FLAGS>
                    (flags | MODEL_GENERATOR::PERFORM_CHECK);

        bool parsedWC = ! WConstraints.empty();
        bool groundedWC = (! GroundWConstraints.empty() )
                            || (! ViolatedWConstraints.empty() );

        MODEL_GENERATOR MG( GroundIDB,GroundConstraints,
                            GroundWConstraints,
                            Query,&EDB,&I,
                            optionModels,
                            costBound,
                            flags,
                            callback,
                            callbackCheck,
                            hasAggregates,
                            groundingCost,
                            groundedWC,
                            parsedWC,
                            OptionTraceComputationWC,
                            OptionCostBound,
                            MaxWeakConstraintLevel,
                            AGGREGATEATOM::getAggregateFunctionsCount(),
                            HeuristicSequence,
                            HeuristicCombinationSequence);

        if( optionDetCons )
            {
            MG.detcons();
            }
        else
            {
            MG.run();
            // Check whether there are ground weak constraints and in
            // this case print the model with the best cost.
            // If optimal Cost is already known, then the second pass 
            // must not be done as the models have been already printed.
            if ( !MG.isOptimalCostKnown )
                {
                if ( ! MG.bestModel )
                    {
                    if ( TraceLevel >=1 )
                        cdebug << "No stable model found." << endl;
                    }
                else
                    {
                    MG.isOptimalCostKnown = true;

                    if( optionModels == 1 )
                        MG.runCallback(&I,MG.bestModel);
                    else
                        {
                        assert( MG.wfCost && groundingCost );
                        if ( TraceLevel >=1 )
                            cdebug << endl
                                   << endl
                                   << "Beginning of the second MG run, "
                                      "where we now know the optimal cost."
                                   << endl
                                   << endl;

                        *(MG.wfCost) = *groundingCost;
                        MG.run();
                        }
                    }
                }
            }

        if( OptionStatsLevel )
            PrintFinalStatistics(&MG);

#ifdef HEURISTICSTATS
        string statHCriteriaLabels[CRITERIA_MAX+1] = {
            "MBTDifference",
            "MBTDifferenceUnlessExactlyOneHasNoMBTEliminated",
            "MBT2Difference",
            "MBT2DifferenceUnlessExactlyOneHasNoMBTEliminated",
            "MBT3Difference",
            "MBT3DifferenceUnlessExactlyOneHasNoMBTEliminated",
            "EliminatedClauses",
            "HeuristicCost",
            "EliminatedWCCost",
            "MBTEliminated",
            "MBT2Eliminated",
            "MBT3Eliminated",
            "HardWired",
            "EliminatedMBTsExist",
            "WeightedHardWired",
            "EliminatedPVClauses",
            "PVClausesDifference",
            "EliminatedPVClauses2",
            "EliminatedPVClauses3",
            "PVClauses2Difference",
            "PVClauses3Difference",
            "EliminatedUndefined",
            "PVCLinearCombination1",
            "PVCLinearCombination2",
            "LinearCombination3",
            "LinearCombination4",
            "LinearCombination5",
            "LinearCombination6",
            "LinearCombination7",
            "LinearCombinationMBT1",
            "LinearCombinationMBT2",
            "LinearCombinationMBT3",
            "LinearCombinationPVC1",
            "LinearCombinationMBT4",
            "LinearCombinationMBT5",
            "LinearCombinationPVCI1",
            "LinearCombinationPVCI2"
        };

        if( OptionStatsLevel )
            {
            cstats << "Heuristic Criteria Statistics:" << endl;
            for( CriteriaSequence::const_iterator i =
                     HeuristicSequence.begin();
                 i != HeuristicSequence.end();
                 i++ )
                cstats << "  " << statHCriteriaLabels[*i] << " decided "
                       << statsHCriterionApplied[*i] << " times." << endl;
            cstats << "  No criterion applied " 
                   << statsHCriterionApplied[CRITERIA_MAX+1]
                   << " times." << endl;
            }
#endif
        }

    //////////////////////////////////////////////////////////////////////////

    // We don't need these anymore.
    assert(costBound);
    delete costBound;
    assert(groundingCost);
    delete groundingCost;

    SafeExit(0);
    }

// Local Variables:
// c-file-style: "dl"
// End:
