//////////////////////////////////////////////////////////////////////////////
// dl.h

#ifndef DL_H
#define DL_H

#define HAVE_GCC_VERSION(MAJOR, MINOR) \
  (__GNUC__ > (MAJOR) || (__GNUC__ == (MAJOR) && __GNUC_MINOR__ >= (MINOR)))

#include <iostream>

//////////////////////////////////////////////////////////////////////////////
// Global Settings

// Max. number of distinct variables in a rule.
#define n_par 60

// Max. integer allowed in input and/or as #maxint.
#define MAX_MAXINT 999999999

// Max. length of numeric constants. This should match MAX_MAXINT.
#define NUM_LENGTH 9

// Names of aggregate predicates.
#define PRED_COUNT_NAME "#count"
#define PRED_SUM_NAME   "#sum"
#define PRED_MIN_NAME   "#min"
#define PRED_MAX_NAME   "#max"
// The name of the predicate used for non-ground queries.
#define PREDNAME_QUERY  "#query"

// Initial part of auxiliary predicates names.
#define AUXNAME "aux#"
	
// The character that separates multiple subdirectories in a fully
// qualified filename.
#ifdef __WIN32__
    #define DIR_SEPARATOR '\\'
#else
    #define DIR_SEPARATOR '/'
#endif

//////////////////////////////////////////////////////////////////////////////
// Debugging

#define cdebug cerr
#define cstats cerr

#ifdef NDEBUG
const unsigned    TraceLevel  = 0,
                  DTraceLevel = 0,
                  FTraceLevel = 0,
                  PTraceLevel = 0,
                  GTraceLevel = 0,
                  HTraceLevel = 0,
                  CTraceLevel = 0;
#else
extern unsigned   TraceLevel,        // General + Model Generator traces 
                  DTraceLevel,       // DEPGRAPH traces
                  FTraceLevel,       // Frontend traces
                  PTraceLevel,       // Parser traces (disabled for now)
                  GTraceLevel,       // Grounding traces
                  HTraceLevel,       // Heuristic traces
                  CTraceLevel;       // Model Checker traces
#endif

// If BISON_DEBUG is set, set YYDEBUG to != 0 so that bison will generate
// code for the variables yydebug, diagyydebug, sqlyydebug (and possibly
// further ones).

#ifdef BISON_DEBUG
#define YYDEBUG 1
#endif


//////////////////////////////////////////////////////////////////////////////
// Global Options (usually set from the command-line)

enum FRONTEND {   FRONTEND_DATALOG=0,
                  FRONTEND_BRAVE,
                  FRONTEND_CAUTIOUS,
                  FRONTEND_DIAG_ABDUCTIVE,
                  FRONTEND_DIAG_REITER, 
                  FRONTEND_SQL,
                  FRONTEND_PLANNING_C,
                  FRONTEND_PLANNING_NEW
              };


extern int        MaxInteger;
extern int        MaxIntegerSeen;

extern unsigned   OptionStatsLevel;
extern bool       OptionProjectionRewrite;
extern bool       OptionBasicRewrite;
extern bool       OptionRewriteCountEDB;
extern bool       OptionRewriteDeleteRules;
extern unsigned   OptionGroundingReorder;
extern bool       OptionGroundingSemiNaive;
extern bool       OptionUseOldPTs;
extern unsigned   OptionPTCacheSize;
extern bool       OptionHeuristicsConsiderComplementaryPT;
extern unsigned   OptionHeuristicsCombineComplementaryPT;
extern bool       OptionHeuristicsCanChooseComplementaryPT;
extern bool       OptionUseHeuristics;
extern unsigned   OptionModelCheckMode;
extern bool       OptionPartialChecksForwards;
extern unsigned   OptionVerbosity;
extern bool       OptionSilent;
extern bool       OptionPrintFacts;
extern bool       OptionPrintSATlike;
extern bool       OptionTraceComputation;
extern FRONTEND   OptionFrontend;
extern enum FDIAG {
                  FRONTEND_DIAG_DEFAULT,
                  FRONTEND_DIAG_MIN,
                  FRONTEND_DIAG_SINGLE,
                  FRONTEND_DIAG_WEIGHTED
                  } FDiag;

extern enum FPLAN {
                  FRONTEND_PLANNING_DEFAULT   =0,
                  // These flags can be or-ed bitwise, except the following
                  // two, where at most one may be set. If none of these is
                  // set, ..._SECURE is assumed.
                  FRONTEND_PLANNING_SECURE       =1,
                  FRONTEND_PLANNING_OPTIMISTIC   =2,
                  FRONTEND_PLANNING_ASK          =4
                  } FPlan;

extern unsigned   parser_line;
extern unsigned   parser_errors;
extern bool       ParserStateInternal;

//////////////////////////////////////////////////////////////////////////////
// Misc

namespace std { }

using namespace std;

#include <string>
#include <set>
#include <functional>
#include <sstream>

struct string_less : public binary_function<char *, char *, bool> {
    bool operator()(const char *x, const char *y) const
	{ return strcmp(x,y)<0; } 
};

typedef set<const char *,string_less > STRINGSET;

extern STRINGSET  PredFilter;
extern STRINGSET  PredPFilter;

extern bool       ObjectsExist;
extern bool       StrictRulesExist;
extern bool       ProjectionsExist;

extern unsigned int  MaxWeakConstraintLevel;

void InternalError(const char*);
void SafeExit(int status);

#include "dltypes.h"
#include "interpret.h"
#include "ginterpret.h"
#ifdef WITH_DEPGRAPH
    #include "depgraph.h"
#endif

//////////////////////////////////////////////////////////////////////////////
// Global Variables

extern const char*  WhoAmI;
extern CriteriaSequence HeuristicSequence;
extern CriteriaSequence HeuristicSequenceChecker;
extern CriteriaSequence HeuristicCombinationSequence;

#ifdef HEURISTICSTATS
extern CriteriaSequence statsHCriterionApplied;
#define CRITERIA_MAX 36
#endif

#ifdef WITH_GLOBAL_VARIABLES

extern NEG_ATOMS    NegativeAtoms;

extern RULES        IDB;
extern CONSTRAINTS  Constraints;
extern WEAKCONSTRAINTS  WConstraints;

extern CONJUNCTION *Query;

extern GRULES       GroundIDB;
extern GCONSTRAINTS GroundConstraints;
extern GWEAKCONSTRAINTS  GroundWConstraints;
extern GWEAKCONSTRAINTS  ViolatedWConstraints;

extern INTERPRET   I;
extern VATOMSET    EDB;

extern OIDs         ObjectIDs;
extern unsigned     RuleID;

// Auxiliary counter used for the special predicates, representing 
// aggregate atoms in ground rules and constraints. 
extern unsigned     AuxCounter; 
     
// FIXME: The following could be put into a included-from-parsers-only.h
// file, if something like that is created.

//////////////////////////////////////////////////////////////////////////////
// LOCAL_VARIABLES is a container that stores rule-local variables.
// These are mapped to integers, so that only equally named variable mapt to
// equal values.
class LOCAL_VARIABLES
    {
    typedef map<string,unsigned,less<string> > VARIABLES;

    unsigned count;
    VARIABLES vars;

public:
    // creates an empty LOCAL_VARIABLES container.
    LOCAL_VARIABLES()
        :count(0),vars()
        {
        }

    LOCAL_VARIABLES(const LOCAL_VARIABLES &v)
        : count(v.count), vars(v.vars)
        {
        }

    void operator=(const LOCAL_VARIABLES&)
        {
        assert( 0 );
        }    

    // @return the number of variables
    unsigned getNumber() const
        {
        return count;
        }

    // @return the number with which the next variable is identified
    unsigned next_variable_number() const
        {
        return count+1;
        }

    // empties the LOCAL_VARIABLES.
    // This should be invoked before processing a new rule.
    void clear() { count=0; vars.erase(vars.begin(),vars.end()); };

    // registers the variable with the given name and returns it index.
    // Does not register another variable if the name is already registered,
    // but returns the index of the already registered one in that case.
    // @param v the variable name
    // @return the integer this name is mapped to
    unsigned record(char const *v)
	{
	VARIABLES::iterator i;

	i=vars.find(v);
	if (i==vars.end())
	    i=vars.insert(pair<const string,unsigned>(string(v),count++))
		.first;
	return (*i).second;
	}

    // gives a unique integer id to an anonymous variable.
    // @return the integer id.
    unsigned anonymous()
	{
	return count++;
	}
    };

extern LOCAL_VARIABLES LocalVariables;

#endif


//////////////////////////////////////////////////////////////////////////////
// Global Functions

TERM*  newTERM(const char*);
TERMS* newTERMS(const TERM*,const TERM* new2=0, const TERM* new3=0);
ATOM*   newNegativeATOM(const char*, const TERMS*);
ATOM*   newBuiltin(BUILTIN, const TERMS*);

void BasicRewriting(RULES&,CONSTRAINTS&,bool&,bool&);
void ProjectionRewriting(RULES&);
void ProjectionRewriting(CONSTRAINTS&,RULES&);

bool SubsumptionCheckingRewriting();

// flex functions we use to process strings instead of files as input.
extern "C" struct yy_buffer_state * yy_scan_string(const char*); 
extern "C" void   yy_delete_buffer(struct yy_buffer_state *);
bool parseStringStream(ostringstream &);

void MagicRewriting();

bool GroundPropositionalInput(
         const RULES&, const CONSTRAINTS&, const WEAKCONSTRAINTS&,
         const VATOMSET&,
         GRULES&, GCONSTRAINTS&,GWEAKCONSTRAINTS&,GWEAKCONSTRAINTS&, 
         bool);

void ModelCheckerStats();

//////////////////////////////////////////////////////////////////////////////

/** efficiently removes an element from a back-insertion-sequence in which
 *  order is not relevant. The elements of the sequence must support
 *  assignment.
 *  This is particularly useful for vectors, where erase is O(n), while
 *  unordered_erase is O(1).
 *  We simply overwrite the element to be erased by the last element and
 *  remove the last element. Note that this technique also works if the
 *  element to be erased is the last one (provided that the assignment
 *  operator of the elements handles self-assignment correctly).
 *  @param sequence the back insertion sequence from which an element
 *                  should be erased
 *  @param erase_it iterator to the element to be erased
 */
template< class T >
void unordered_erase( T &sequence, typename T::iterator erase_it )
    {
    *erase_it = sequence.back();
    sequence.pop_back();
    }

//////////////////////////////////////////////////////////////////////////////
// Bug Fixes for Various Environments

// BUG-FIX: bison[1.24] fails to prototype its interfaces yylex() and yyerror().
extern "C" int yylex();
extern "C" int yyerror(const char*);

#endif
