//////////////////////////////////////////////////////////////////////////////
// generate-pt.h

#include <list>

#ifndef GENERATE_PT_H
#define GENERATE_PT_H

#define level 0

//////////////////////////////////////////////////////////////////////////////
// This class implements the "positional" or "incremental" approach to
// computing possibly true literals.
//
// Upon initialisation, all rules are searched for possibly true (PT)
// literals (negative ones before positive ones). This position is
// stored (atEnd() holds if no PT literals were found).
//
// The ++ operator looks for the next PT literal in the sequence of
// rules. The application of this operator implies that another PT
// literal has been found before. Because the interpretation must be
// allowed to change after a PT literal has been found, it is
// necessary that the search starts again from the beginning
// ("wrapping") if the end of the rules is encountered. Otherwise some
// literals, which have become PT literals in a rule before the last PT
// literal due to the change of the interpretation, might be missed.
//
class MODEL_GENERATOR::PT
//
    {
    MODEL_GENERATOR &MG;
    GRULES::const_iterator beginRule;
    GRULES::const_iterator currRule;
    GRULES::const_iterator endRule;
    GDISJUNCTION::const_iterator      currHeadAtom;
    GCONJUNCTION::const_iterator      currBodyLit;

    const GINTERPRET&                 I;

    bool                              allowsPositive;
    bool                              noMorePTs;

    inline bool findPT(const GRULE&);
    inline void findNextPTRule(bool);

public:
    inline bool atEnd() const  { return noMorePTs; }

    inline bool isReallyPossible() const;

    inline void operator++();
    inline void operator++(int);

    const GRULE& getRule() const  { return *currRule; }
        
    const GLITERAL getLiteral() const
        {
        if( allowsPositive )
            return GLITERAL(false,*currHeadAtom);
        else
            return *currBodyLit;
        }

    bool isNegative() const  { return !allowsPositive; }
    bool isPositive() const  { return  allowsPositive; }

    void init(const GRULES& prog)
        {
        beginRule = prog.begin();
        endRule = prog.end();
        currRule = beginRule;

        // Find the first PT literal (if it exists). No need to wrap
        // the search, since we start at the very beginning.
        findNextPTRule(false);

        if( ! atEnd() )
            MG.statChoicePoints++;

        if( TraceLevel >= 1 )
            cdebug << indent(level) << *this << endl;
        }

    PT(MODEL_GENERATOR &mg,const GINTERPRET &I2)
        : MG(mg), I(I2), allowsPositive(false), noMorePTs(false)
        {
        // init() will do the real initialisation.
        }

    PT()
        // This is a cute hack to avoid warnings. [(C) Gerald]
        : MG(MG), I(I)
        {
        assert( 0 );
        }

    PT(const PT&)
        // This is a cute hack to avoid warnings. [(C) Gerald]
        : MG(MG), I(I)
        {
        assert( 0 );
        }

    void operator= (const PT&)
        {
        assert( 0 );
        }

    };

inline ostream& operator<< (ostream& out, const MODEL_GENERATOR::PT& pt)
    {
    if( pt.atEnd() ) 
        out << "PT = {}";
    else
        {
        out << "PT = " << pt.getRule() 
            << " [" << pt.getLiteral() << "]";
        }

    return out;
    }

//////////////////////////////////////////////////////////////////////////////
// Look for the first (if any) PT literal in the given rule and, if
// one PT literal has been found, set the variables accordingly.
//
// @param grule, the rule to be examined
//
// @return true, if a PT has been found
//         false else
//
inline bool MODEL_GENERATOR::PT::findPT( const GRULE& grule )
    {
    // If the rule is satisfied (either the head is true or the
    // body is false), no condition for a PT literal can be met.
    if( grule.satisfied )
        return false;


    if( OptionUseOldPTs )
        {
        // At this point we can be sure that the head is not true and
        // that the body is not false.
        if( grule.undefPosBody == 0 && grule.undefNegBody == 0 )
            {
            // The body is true, so this rule allows positive PT literals.

            // Its head must not be false; this is an inconsistency and
            // the deterministic operators should have noticed that.
            assert( ! I.isFalse(grule.getHead()) );

            // It should be disjunctive, otherwise the deterministic
            // operators should have derived the head.
            assert( grule.isDisjunctive() );

            // Set everything to indicate the positive PT literals.
            allowsPositive = true;

            // Find the first undefined atom.
            for( GDISJUNCTION::const_iterator i = grule.getHead().begin();
                 i != grule.getHead().end();
                 i++ )
                if( I.isUndefined(*i) )
                    {
                    currHeadAtom = i;
                    break;
                    }

            assert( currHeadAtom != grule.getHead().end() );

            return true;
            }

        // grule.undefPosBody != 0 || grule.undefNegBody != 0 holds at
        // this point. Only negative PT literals may occur.

        else if( grule.undefPosBody == 0 )
            {
            const GCONJUNCTION* body=grule.getBody();

            assert( body );
            assert( grule.undefNegBody != 0 );

            // At this point the rule satisfies the criteria for allowing
            // negative PT literals.

            allowsPositive = false;

            // Find the first undefined literal.
            for( GCONJUNCTION::const_iterator i = body->neg_begin();
                 i != body->neg_end();
                 i++ )
                if( I.isUndefined(*i) )
                    {
                    currBodyLit = i;
                    break;
                    }

            assert( currBodyLit != body->neg_end() );

            return true;
            }
        }
    else
        {
        if( grule.undefPosBody == 0 )
            {
            currHeadAtom = grule.getHead().end();

            // Find the first undefined atom.
            for( GDISJUNCTION::const_iterator i = grule.getHead().begin();
                 i != grule.getHead().end();
                 i++ )
                if( I.isUndefined(*i) )
                    {
                    currHeadAtom = i;
                    break;
                    }

            allowsPositive = ( currHeadAtom != grule.getHead().end() );

            return allowsPositive;
            }
        }

    // This rule does not support any PT literals.
    return false;
    }

//////////////////////////////////////////////////////////////////////////////
// Look for the first (if any) rule which supports PT literals. If one
// has been found, set all variables accordingly. The search is
// started at currRule (inclusively).
//
// @param wrap, if this parameter is true and the end of the rules is
//              reached while searching, the search is continued at the
//              beginning of the rules ("wrapped") until the end is
//              reached again or until a rule which supports PT literals
//              has been found.
//
inline void MODEL_GENERATOR::PT::findNextPTRule(bool wrap)
    {
    // Begin the search at currRule. As long as there are further
    // rules and the current rule does not support PT literals, try
    // the next rule.
    while( currRule != endRule && ! findPT(*currRule) )
        {
        currRule++;
        }  

    // A PT has been found. The position of the PT in the head
    // resp. body has been set by findPT().
    if( currRule != endRule )
        return;

    // If the search must be wrapped (once) at the end, and the final
    // rule has been reached, reset currRule and increment it as long
    // as the end is not reached and no PT literals are supported by
    // it.
    if( wrap )
        {
        currRule = beginRule;
        while( currRule != endRule && ! findPT(*currRule) )
            {
            currRule++;
            }

        // A PT literal has been found. The position of the PT literal
        // in the head resp. body has already been set by findPT().
        if( currRule != endRule )
            return;
        }

    // Set the flag which indicates that no possibly true atoms are
    // left.
    noMorePTs = true;
    }

inline bool MODEL_GENERATOR::PT::isReallyPossible() const
    {
    if( allowsPositive )
        return ( I.isUndefined(*currHeadAtom) 
                 || I.isMustBeTrue(*currHeadAtom) );
    else
        return I.isUndefined(*currBodyLit);
    }

inline void MODEL_GENERATOR::PT::operator++()
    {
    // should not be called if there are no PTs left
    assert( ! noMorePTs );

    // Look whether some other PT literals occur in the current rule.
    if( allowsPositive )
        {
        do
            {
            currHeadAtom++;

            assert( currHeadAtom == (*currRule).getHead().end() 
                    || ! I.isMustBeTrue(*currHeadAtom) );
            }
        while( currHeadAtom != (*currRule).getHead().end()
               && ! I.isUndefined(*currHeadAtom) );

        if( currHeadAtom != (*currRule).getHead().end() )
            {
            if( TraceLevel >= 1 )
                cdebug << indent(level) << *this << endl;
            return;
            }
        }
    else
        {
        do
            {
            currBodyLit++;
            }
        while( currBodyLit != ( * (*currRule).getBody() ).neg_end()
               && ! I.isUndefined(*currBodyLit) );

        if( currBodyLit != ( * (*currRule).getBody() ).neg_end() )
            {
            if( TraceLevel >= 1 )
                cdebug << indent(level) << *this << endl;
            return;
            }
        }

    // If this point is reached, neither positive nor negative PT
    // literals are left in the current rule.

    currRule++;

    // Find the next rule which supports PT literals.

    // If the end is reached, the search must be wrapped, since some
    // additional PTs may have become true "in the meantime" (by a
    // more determined interpretation).

    findNextPTRule(true);

    if( TraceLevel >= 1 )
        cdebug << indent(level) << *this << endl;
    }

inline void MODEL_GENERATOR::PT::operator++(int)
    {
    ++(*this);
    }

#undef level

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////


class MODEL_GENERATOR::HeuristicPT
//
    {
    MODEL_GENERATOR    &MG;
    GINTERPRET&         I;
    bool                noMorePTs;
    list<HeuristicItem> PTCache;

    const unsigned      maxCacheSize;
    unsigned            currentCacheSize;

    inline void getFirstRealPT();

public:

    HeuristicPT(MODEL_GENERATOR &mg, GINTERPRET& I2, unsigned cacheSize)
        : MG(mg), I(I2), noMorePTs(false), maxCacheSize(cacheSize), currentCacheSize(0)
        {
        }

    HeuristicPT(MODEL_GENERATOR &mg, const HeuristicPT& hpt)
        : MG(mg), I(hpt.I), noMorePTs(hpt.noMorePTs), maxCacheSize(hpt.maxCacheSize),
          currentCacheSize(hpt.currentCacheSize)
        {
        }

    // operator= is provided just for completeness. It should (at
    // least currently) not be used, hence assert(0).
    HeuristicPT& operator= (const HeuristicPT& hpt)
        {
        assert(0);
        // The following is a hack to suppress a warning stating that
        // hpt is not used.
        noMorePTs = hpt.noMorePTs;
#ifdef NEVER
        if( this != &hpt )
            {
            MG = hpt.MG;
            assert(&I == &(hpt.I));
            noMorePTs = hpt.noMorePTs;
            PTCache = hpt.PTCache;

            assert( maxCacheSize == hpt.maxCacheSize );
            assert( hpt.currentCacheSize <= maxCacheSize );
            currentCacheSize = hpt.currentCacheSize;
            }
#endif
        return *this;
        }

    ~HeuristicPT()
        {
        }

    bool init();
    bool next();

    inline bool atEnd() const  { return noMorePTs; }

    inline const GLITERAL& getLiteral(bool& reallyPT) const
        {
        assert( ! atEnd() );
        assert( ! PTCache.empty() );

        const HeuristicItem& bestPT = PTCache.front();

        if( HTraceLevel >= 1 )
            cdebug << indent(MG.level) << "PT: " << PTCache.front() << endl;

        if( OptionHeuristicsCanChooseComplementaryPT )
            {
            assert(bestPT.hccompl);

            GreaterEqualHeuristic geh(MG.heuristicSequence, MG.heuristicCombinationSequence);

            if( geh(bestPT.hc,*(bestPT.hccompl)) )
                reallyPT = true;
            else
                // use the complementary literal
                reallyPT = false;
            }
        else
            reallyPT = true;


        return bestPT.literal;
        }

private:
    void insertIntoCache(
        const GATOM &a,
        const bool neg,
        HeuristicCounters *hcptr,
        HeuristicCounters *hccomplptr )
        {
        assert(hcptr);

        if( TraceLevel >= 3 )
            {
            cdebug << indent(MG.level) << GLITERAL(neg,a) << endl
                   << " with " << *hcptr << endl;
            if(hccomplptr)
                cdebug << indent(MG.level) << "and, " 
                       << *hccomplptr << endl;
            }

        HeuristicItem heur_item(GLITERAL(neg,a),*hcptr,hccomplptr);

        if( HTraceLevel >= 3 )
            {
            cdebug << "Considering " << heur_item << " >= ";

            if( PTCache.empty() )
                cdebug << "nothing" << endl;
            else
                cdebug << PTCache.back() << endl;
            }

        GreaterEqualHeuristic geh(MG.heuristicSequence,
                                  MG.heuristicCombinationSequence);

        if( ! PTCache.empty() && geh(heur_item,PTCache.back()) )
            {
            list<HeuristicItem>::iterator i = find_if(PTCache.begin(),
                                                      PTCache.end(),
                                                      bind1st(GreaterEqualHeuristic(MG.heuristicSequence,MG.heuristicCombinationSequence),heur_item));
            assert( i != PTCache.end() );
            // insert into list
            PTCache.insert(i, heur_item);

            if( HTraceLevel >= 3 )
                cdebug << "Inserted " << heur_item << " before " 
                       << *i << endl;

            if( currentCacheSize == maxCacheSize )
                PTCache.pop_back();
            else
                currentCacheSize++;
            }
        else
            {
            if( currentCacheSize < maxCacheSize )
                {
                PTCache.push_back(heur_item);
                currentCacheSize++;
                }
            else
                if( HTraceLevel >= 3 )
                    cdebug << "Not inserted." << endl;
            }
        }


    bool HandlePT(
        const GATOM &a,
        const bool neg,
        GINTERPRET &I,
        unsigned &numberOfConsPTs,
        unsigned &numberOfInconsPTs )
        {
        HeuristicCounters *hccomplptr;

        if( OptionHeuristicsConsiderComplementaryPT )
            hccomplptr = &(MG.hccomplbuffer);
        else
            hccomplptr = 0;

        MG.setAlreadyConsideredAsPT(a,neg);

        if( ! MG.CalculateHeuristicValue(a,neg,I,&(MG.hcbuffer),hccomplptr) )
            {
            if( TraceLevel >= 1 )
                cdebug << indent(MG.level)
                       << "Calculation of heuristic values detected "
                          "inconsistency."
                       << endl;
            numberOfInconsPTs++;
            if( ! MG.LookaheadPropagate(I) )
                return false;
            }
        else
            {
            numberOfConsPTs++;
            insertIntoCache(a,neg,&(MG.hcbuffer),hccomplptr);

            // If a has only one potentially supporting rule, and the
            // potentially supporting rule has only two undefined
            // atoms which are in the head, the heuristic counters for
            // the second head atom are equal to a's (negative and
            // positive reversed), if the rule is also the last
            // potentially supporting rule for it.
            if( OptionHeuristicsConsiderComplementaryPT
                && ! neg && MG.getPotentialSupport(a) == 1 )
                {
                // Find a's last potentially supporting rule.
                GRULEPTRS::const_iterator rule;

                for( rule = MG.headRules_begin(a);
                     rule != MG.headRules_end(a) 
                         && ! (**rule).potentiallySupports(a);
                     rule++ )
                    {}

                // Check counter-correctness.
                assert( rule != MG.headRules_end(a) );

#ifndef NDEBUG
                // No other rule with a in the head should potentially
                // support a.
                GRULEPTRS::const_iterator rule1 = rule;
                for( rule1++;
                     rule1 != MG.headRules_end(a);
                     rule1++ )
                    assert( ! (**rule1).potentiallySupports(a) );
#endif

                // Now check whether there are only two undefined
                // atoms left, both of which must be in the head.
                if( (**rule).undefHead == 2
                    && (**rule).undefPosBody + (**rule).undefNegBody == 0 )
                    {
                    GDISJUNCTION::const_iterator i;
                    for( i = ((**rule).getHead()).begin();
                         i != ((**rule).getHead()).end();
                         i++ )
                        if( I.isUndefined(*i) && *i != a )
                            break;

                    // Check counter-correctness.
                    assert( i != ((**rule).getHead()).end() );
                    const GATOM &b = *i;

#ifndef NDEBUG
                    // No atom in the head apart from a and b should
                    // be undefined.
                    for( GDISJUNCTION::const_iterator i = 
                             ((**rule).getHead()).begin();
                         i != ((**rule).getHead()).end();
                         i++ )
                        assert( ! I.isUndefined(*i) || *i == a || *i == b );
#endif

                    if( ! MG.isAlreadyConsideredAsPT(b,false) )
                        {
                        MG.setAlreadyConsideredAsPT(b,false);

                        // HC[not a] == HC[b] holds

                        if( MG.getPotentialSupport(b) == 1 )
                            {
                            // HC[not b] == HC[a] holds
                            // counters are already fully calculated
                            }
                        else
                            {
                            // Determine HC[not b].
                            if( ! MG.CalculateHeuristicValue(b,false,I,0,
                                                              &(MG.hcbuffer)) )
                                {
                                if( TraceLevel >= 1 )
                                    cdebug << indent(MG.level)
                                       << "Calculation of heuristic "
                                        "values detected inconsistency."
                                           << endl;
                                numberOfInconsPTs++;
                                if( ! MG.LookaheadPropagate(I) )
                                    return false;
                                }
                            else
                                {
                                numberOfConsPTs++;
                                }
                            }
                        insertIntoCache(b,false,hccomplptr,&(MG.hcbuffer));
                        }
                    }
                }                
            }

        return true;
        }

    bool createPTs();
    };

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
