//////////////////////////////////////////////////////////////////////////////
// Intelligent Grounding

#define WITH_DEPGRAPH
#define WITH_GLOBAL_VARIABLES

#include <cmath>
#include "dl.h"
#include "stats.h"
#include "grounding.h"
#include "grounding-body.h"

const char *RangeName[] =
{
    "rI",
    "rDeltaNF",
    "rI_DeltaNF"
};

// Only used in assertions.
#ifndef NDEBUG

/** Check if a vector contains only null-elements.
 * A vector containing only elements equal to the null-element is considered
 * empty.
 * @param vec the vector to examine.
 * @param dim the size of the vector.
 * @param elNul the null-element.
 * @return whether the given vector is empty.
 */
template< class T1, class T2 >
bool empty(const T1& vec,unsigned dim,T2 elNul)
    {
    for( unsigned i=0; i < dim; i++ )
	if( vec[i] != elNul )
	    {
	    if( GTraceLevel >=1 )
		cdebug << "VEC not empty for i=" << i << endl;
	    return false;
	    }
    return true;
    }

#endif

/* Given two pointers p1 and p2 pointing to two different objects
 * A and B respectively, invert them in order that p1 points to B
 * and p2 points to A.
 */ 
template< class T >
void switchPointers(T* &p1, T* &p2)
    {
    T& TEMP = *p1;
    p1 = p2;
    p2= &TEMP;
    }

void GROUNDING::init()
    {
    if( GTraceLevel >= 1 )
        cdebug << "Ground() begins." << endl;

#ifdef NEVER
    assert( Const_Names.begin() != Const_Names.end() );
#endif

    for( unsigned i = 0; i < n_par; i++ )
        vTOc[i] = TERM();

    deltaNF = new INTERPRET();
    newNF = new INTERPRET();
    solved = new bool[ATOM::Predicates.size()];
    done = new bool[ATOM::Predicates.size()];
    size = new unsigned[ATOM::Predicates.size()];
    deltaSize = new unsigned[ATOM::Predicates.size()];
    newSize = new unsigned[ATOM::Predicates.size()];
    dictionary = new DBDICTIONARY();
    for( unsigned i = 0; i < ATOM::Predicates.size(); i++ )
        {
        solved[i] = true;
        done[i] = false;
        size[i] = 0;
        deltaSize[i] = 0;
        newSize[i] = 0;
        }

    EDB.foreach( initializeSize(size) );
    }

/**Give an error message if an aggregate has been used in a wrong way.
 * Moreover, in this case or if a constraint is always violated, 
 * clean memory and stop instantiation, because it has already failed. 
 * Uses GROUNDING::status.
 * @param T the current rule/constraint/weakConstraint to be instantiated.
 * @return true whether the instantiation has failed.
 */
template<class T>
bool GROUNDING::failedInstantiation( const T &r ) const
    {
    if( status == AggregateInvalidUse )
        cerr << "Error occurs in " << endl << r << endl;
    if( status != OK )
        {
        // Stop grounding because the instantiation has failed.
        cleanMemory();
        return true;
        }

    return false;
    }

/** Try to match a literal with a ground atom.
 * First, the current ground atom is tried, upon failure the next, and so
 * forth. When a match is found, arguments of the literal that are unbound
 * variables are bound with vTOc to their counterparts in the ground atom.
 * On failure, no bindings are made.
 * Uses globals: vTOc
 * @param lit the literal to match.
 * @param StackCell the cell holding argument status and pointers to the
 *	current ground atom
 * @param vTOc the current variable mapping.
 * @return whether a match was found.
 */
bool GROUNDING::Match( const LITERAL &lit, cell &StackCell, TERM vTOc[n_par] )
    {
    if (lit.isBuiltin())
	return MatchBuiltin(lit,StackCell);
    if (StackCell.atEnd())
	{
	if( GTraceLevel >= 2 )
	    cdebug << " OOPS!";
	return false;
	}
    BOOLS &status=StackCell.get_Status();
    bool found=false;
    while (!found)
	{
	int k=0;
	found=true;
	for( BOOLS::const_iterator i=status.begin(); 
	     ! (i == status.end()); 
	     i++, k++ )
	    {
	    const TERM& k_thParLit=(*lit.getParams())[k];
	    const TERM& k_thCstAtEDB=(*(*(*StackCell.getSubstAtom()))
				       .getParams())[k];

            if ((*i) && (vTOc[k_thParLit.getVar()]).isNull()) 
                {
		vTOc[k_thParLit.getVar()]=k_thCstAtEDB;
                }
	    else
		{
		const TERM* temp;
		if (k_thParLit.isVar())
		    temp=&vTOc[k_thParLit.getVar()];
		else
		    temp=&k_thParLit;
		if ( (*temp).diffConstants(k_thCstAtEDB) )
		    {
		    for(int j=k-1; j>=0; j--)
			if (status[j])
			    vTOc[(*lit.getParams())[j].getVar()]=TERM();
                    StackCell.goToNextSubstAtom(lit,I,deltaNF);
		    if (StackCell.atEnd())
			return false;
		    found=false;
		    break;
		    }
		}
	    }
	}
    StackCell.goToNextSubstAtom(lit,I,deltaNF);
    return true;
    }

/** Try to match a built-in predicate.
 * If all parameters are bound, only checks for satisfaction. 
 * Otherwise, fills in the next possible candidates that satisfy the 
 * predicate. When no more candidates are left, no bindings are made.
 * Uses globals: vTOc
 * @param lit the built-in to match.
 * @param StackCell the cell holding argument status and counters.
 * @return whether the built-in was satisfied.
 */
bool GROUNDING::MatchBuiltin( const LITERAL &lit, cell &StackCell )
    {
    int id=lit.getBuiltinID();
    if (id>=NSimpleBuiltins)
	{
	if( MaxInteger == -1 )
            InternalError("#maxint undefined");
	int par[3];
	int unbound1=-1, unbound2=-1, unbound3=-1, bind1=-1, bind2=-1;
	bool failure=false;   // Have we failed to ground this literal?

        // For duplicate variable checking we keep a vector of parameter
        // offsets of the duplicate variable occurrences. Currently, only one
        // variable may occur twice or three times, so there is no need to
        // store the variable IDs. All this only applies to unbound variables.
        // This is int and not unsigned because we compare against unbound1
        // which is also int, as it might be -1 (indicating no variable).
        vector<int> duplicateUnbound;

	unsigned k=0;
	for(TERMS::const_iterator i=(lit.getParams())->begin();
	    i!=(lit.getParams())->end(); i++, k++)
	    {
            const TERM *p;
            if( i->isVar() )
                p=&vTOc[i->getVar()];
            else
                p=&*i;
            if( p->isNull() )
		{
		if (unbound1==-1)
		    unbound1=k;
		else if (unbound2==-1)
		    unbound2=k;
		else
		    unbound3=k;

                // Check whether the variable occurs also elsewhere in this
                // literal. Do this only if no duplicates were detected yet.
                if( duplicateUnbound.empty() )
                    {
                    unsigned k1 = k+1;
                    bool isDuplicate = false;
                    for(TERMS::const_iterator j=i+1;
                        j!=(lit.getParams())->end();
                        j++,k1++)
                        {
                        if( j->isVar() )
                            if( j->getVar() == i->getVar() )
                                {
                                if( ! isDuplicate )
                                    {
                                    isDuplicate = true;
                                    duplicateUnbound.push_back(k);
                                    }
                                duplicateUnbound.push_back(k1);
                                }
                        }
                    }
                }
            else if ( p->getType() == TERM::Integer )
                par[k]=p->getInt();
            else
                {
                // In case we encounter a non-integer here, take note (We
                // cannot ground this), but be silent about it.
                failure=true;
                }
	    }
        if( failure )
            return false;

	if (unbound1==-1)
	    {
            // No unbound arguments.    
	    if (StackCell.counter1()>0) // there's only one chance
		return false;
	    StackCell.advance1();
	    switch (id)
		{
		case BuiltinInt:
		    return true; // we already asserted that it's an int
		case BuiltinSucc:
		    return par[0]+1==par[1];
		case BuiltinAddition:
		    return par[0]+par[1]==par[2];
		case BuiltinMultiplication:
		    return par[0]*par[1]==par[2];
		default:
		    assert(0);
		}
	    }
	/* else */
	if (builtin_arity[id]==1)
	    {
	    bind1=0;
	    unbound1=-1;
	    }
	else if (unbound2==-1)
	    {
	    if (StackCell.counter1()>0) // there's only one chance
		return false;
	    StackCell.advance1();
	    }
	else
	    {
	    if (unbound3==-1) // two unbound variables 
		{
                // check whether the two unbound variables are equal
		if( ! duplicateUnbound.empty() )
                    {
                    switch (id)
                        {
                        case BuiltinInt:
                            // has only one parameter
                            assert(0);
                        case BuiltinSucc:
                            // no number is its own successor
                            assert( unbound1 == 0 && unbound2 == 1 );
                            return false;
                        case BuiltinAddition:
                            break;
                        case BuiltinMultiplication:
                            break;
                        default:
                            assert(0);
                        }
                    }
		bind1=unbound1;
		unbound1=unbound2;
		}
	    else		// three unbound variables
		{		// bind both operands
		assert(unbound1==0);
		bind1=0;
		bind2=1;
		unbound1=2;
		unbound3=-1;
		}
	    unbound2=-1;
	    }
	int result=0;
	do
	    {
	    if (bind1!=-1)
		{
		par[bind1]=StackCell.counter1();
		if (bind2!=-1)
		    {
		    par[bind2]=StackCell.counter2();
		    if (par[bind2]>MaxInteger)
			return false;
		    }
		if (par[bind1]>MaxInteger)
		    if (bind2==-1)
			return false;
		    else
			{
			StackCell.reset1();
			par[bind1]=0;
			StackCell.advance2();
			}
		StackCell.advance1();
                }

	    switch (id)
		{
		case BuiltinInt:
		    break;
		case BuiltinSucc:
                    // if the two operands are equal we should have handled it
                    // above
                    assert( duplicateUnbound.empty() );
		    if (unbound1==0)
			result=par[1]-1;
		    else
			result=par[0]+1;
		    break;
		case BuiltinAddition:
		    if (unbound1==2)
			result=par[0]+par[1];
		    else
			result=par[2]-par[1-unbound1];
		    break;
		case BuiltinMultiplication:
		    if (unbound1==2)
			result=par[0]*par[1];
		    else
			{
			assert(bind2==-1); // would have bound the result
                        if (par[1-unbound1] != 0 && 
                           (par[2]%par[1-unbound1] == 0))
                            result = par[2]/par[1-unbound1];
                        else
			    if (bind1!=-1)
				continue;
			    else
				return false;
			}
		    break;
		default:
		    assert(0);
		}

	    if (result<0 || result>MaxInteger)
		{
		if (bind1!=-1)
		    if (bind2!=-1)
			continue;
		return false;
		}
	    else
                {
                if ( !duplicateUnbound.empty() )
                    {
                    vector<int>::const_iterator i
                        = duplicateUnbound.begin();
                    assert( i != duplicateUnbound.end() );
                    // store the first value of the duplicates
                    int duplicateValue = par[*i];
                    // use a flag indicating an unadmissible binding (i.e. one
                    // which assigns two differing values to the same variable
                    bool admissibleValue = true;
                    for( i++ ; i != duplicateUnbound.end(); i++ )
                        {
                        // if one of the duplicates has a differing value,
                        // do not produce this binding
                        if ( unbound1 == *i )
                            {
                            if ( duplicateValue != result )
                                admissibleValue = false;
                            }
                        if ( unbound1 != *i )
                            if ( duplicateValue != par[*i] )
                                admissibleValue = false;
                        }
                    if ( ! admissibleValue )
                        continue;
                    }
		break;
                }
	    } while (1);
	TERM *tmp;
	if (bind1!=-1)
	    {
	    tmp=&vTOc[(*lit.getParams())[bind1].getVar()];
            (*tmp).~TERM();
	    new (tmp) TERM(par[bind1],0);
	    if (bind2!=-1)
		{
		tmp=&vTOc[(*lit.getParams())[bind2].getVar()];
                (*tmp).~TERM();
		new (tmp) TERM(par[bind2],0);
		}
	    }
	if (unbound1!=-1)
	    {
	    tmp=&vTOc[(*lit.getParams())[unbound1].getVar()];
            (*tmp).~TERM();
	    new (tmp) TERM(result,0);
	    }
	return true;
	}
    else
	{
	if (StackCell.counter1()>0) // there's only one chance
	    return false;
	StackCell.advance1();
	// we don't mess around with dynamic arrays, maximum builtin
	// arity should be a small constant
	const TERM *params[MaxBuiltinArity];
	unsigned k=0;
	for(TERMS::const_iterator i=(lit.getParams())->begin();
	    i!=(lit.getParams())->end(); i++, k++)
	    if ((*i).isVar())
		{
		params[k]=&vTOc[(*i).getVar()];
		// all variables MUST be bound
		assert(!(*params[k]).isNull());
		}
	    else
		params[k]=&*i;

	switch (id)
	    {
	    case BuiltinEquality:
		return *params[0]==*params[1];
	    case BuiltinInequality:
		return *params[0]!=*params[1];
	    case BuiltinLess:
		return params[0]->lexicographic_compare(*params[1]) < 0;
	    case BuiltinGreater:
		return params[0]->lexicographic_compare(*params[1]) > 0;
	    case BuiltinLessEqual:
		return params[0]->lexicographic_compare(*params[1]) <= 0;
	    case BuiltinGreaterEqual:
		return params[0]->lexicographic_compare(*params[1]) >= 0;
	    default:
		assert(0);	// illegal builtin
                return false;
	    }
	}
    }

/** Try to match an aggregate literal.
 * This function is a template because for recursive aggregates it
 * is called for matching a ground literal.  
 * We don't need to consider deltaNF while instantiating an aggregate 
 * literal because those ground atoms we use for that already are in 
 * the positive and undefined parts of I, both for recursive and non 
 * recursive aggregates.
 * Use globals: I. 
 * @param lit the aggregate literal to match.
 * @param vTOc the current variable mapping.
 * @return true. 
 */
template<class T>
bool GROUNDING::MatchAggregate( 
    const T &lit, 
    TERM vTOc[n_par] )
    {
    AGGREGATEATOM &aggr = lit.getAggregate();
    ATOM auxAt(aggr.getAuxAtom());

    // Check whether there are no ground instances for the auxiliary atom. 
    // In this case nothing has to be done for matching the aggregate. 
    if( size[auxAt.getIndex()] == 0 )
        return true;

    if( ! aggr.isAssignment() )
        {
        // If the lower guard is a variable, the matched value
        // should be an integer.
        if( ( aggr.getLowerGuardAsVar() ).isVar() )
            {
            if( ! ( vTOc[aggr.getLowerGuardAsVar().getVar()] ).isInt() )
                {
                cerr << "Lower guard must be an integer in " << lit
                     << endl;
                exit(1);
                }
            }

        // If the upper guard is a variable, the matched value
        // should be an integer.
        if ( aggr.getUpperGuardAsVar().isVar() )
            {
            if( ! ( vTOc[aggr.getUpperGuardAsVar().getVar()] ).isInt() )
                {
                cerr << "Upper guard must be an integer in " << lit
                     << endl;
                exit(1);
                }
            }
        }

    cell StackCell(auxAt, vTOc, I, 0, size, 0, rI); 

    if( GTraceLevel >= 2 )
        cdebug << "     next pLiteral to be matched: "      
               << auxAt << " with interpretations range="
               << RangeName[rI] << endl;
 
    while( Match(auxAt, StackCell, vTOc) )
        {
        const ATOM &a = ATOM(auxAt, vTOc);

        if( GTraceLevel >= 1 )
                cdebug << " -> successful matching on " 
                       << a << endl;

        // Get the value of the free variables.
        TERMS terms;
        for( TERMS::const_iterator i = aggr.getFreeVars().begin();
             i != aggr.getFreeVars().end();
             i++ )
            terms.push_back( vTOc[(*i).getVar()] );

        if( ( auxAt.isEDB() && EDB.contains(a) ) || I.isTrue(a) )
            {
            // Update bounds: there is another true ground
            // instance of the auxiliary atom.
            aggr.updateBounds( *terms.begin(), true );
            }
        else if( I.isUndefined(a) )
            {
            // Add the undefined ground instance of the auxiliary
            // atom to the aggregateSet.
            AGGREGATESETVALUE setValue = AGGREGATESETVALUE(terms,GATOM(a));
            aggr.addToAggregateSet(setValue);
            }

        // Update bounds: there is another ground instance 
        // of the auxiliary atom.
        aggr.updateBounds( *terms.begin() );
        Restore(auxAt,StackCell.get_Status(),vTOc);
        } 

    aggr.finishBounds();

    return true;
    }

/** Create a ground instance of a rule, with given variable mappings.
 *
 * The new ground instance rule is obtained by constructing a new rule R
 * and substituting all variables of R with their correspondent constants
 * in vTOc.
 *
 * - If the head of the grounded rule is found to be true or the body is 
 *   found to be false it does not make sense to add the ground instance 
 *   to GroundIDB and we finish the procedure without changing GroundIDB.
 * - Otherwise, 
 *     o if the body of the grounded rule is not empty or the rule is 
 *       disjunctive, the head atoms are undefined (they are neither true 
 *       nor false at this point) and we add them to the undefined part 
 *       of newNF, if they have not been generated yet;
 *     o if the body of the grounded rule is empty and the rule is not 
 *       disjunctive we obtained a so called IDB fact, that is, the head 
 *       atom must be in each stable model and we add it to the positive 
 *       part of newNF. If the head atom H was undefined, then it was 
 *       either in the undefined part of either (i) I, or (ii) deltaNF, 
 *       or (iii) newNF, and we add it to the positive part of newNF. 
 *       (Note that when we add an atom to the positive part of an instance 
 *       of INTERPRET, that atom is automatically removed from the undefined 
 *       part of that instance if the atom is present there.) If we are in 
 *       the case (i), H will be automatically removed from the undefined 
 *       part of I when I is updated with the atoms of newNF and H is found 
 *       in the positive part of newNF (see the procedure UpdateI). If we 
 *       are in the case (ii), H will first be stored in the undefined part 
 *       of I when I is updated with the atoms from deltaNF, and then in the 
 *       positive part of I when I is updated with the atoms from newNF.
 *
 * Uses globals: vTOc, solved, I
 */
void GROUNDING::AddToGroundIDB(
    const RULE &R, 
    bool &NFvEDBwasChanged,
    INTERPRET *deltaNF, 
    INTERPRET *newNF,
    unsigned *newSize,
    const unsigned component )
    {
    if( trueHead(R.getHead(),*deltaNF,*newNF,vTOc) )
        {
        if( GTraceLevel >= 2 )
            cdebug << R
                   << " -- head is already true. Rule not added."
                   << endl;
        return;
        }
   
    // Creation of the body of the new ground rule.
    GCONJUNCTION* body;

    // Keep track of the recursive aggregates in the current rule.
    // Note that the code concerning recursive aggregates is not 
    // currently used. 
    vector<LITERAL> recursiveAggregates;

    if( R.getBody() )
        {
        assert( R.getBody()->begin() != R.getBody()->end() );
        body = new GCONJUNCTION();
        fillGConjunction(*body,*R.getBody(),vTOc,deltaNF,
                         component,&recursiveAggregates);
        }
    else
        body=0;
    
#ifndef NEVER
    if( body && body->begin() == body->end() 
        && ! recursiveAggregates.size() )
        {
        delete body;
        body=0;
        }
#endif

    if( body || R.isDisjunctive() )
        {
        // The head of the new ground rule is created only if the ground 
        // body is not empty or the *non-ground* rule R is disjunctive.
        GDISJUNCTION head;
        for( DISJUNCTION::const_iterator i = R.getHead().begin(); 
             i != R.getHead().end(); 
             i++ ) 
            {
            if( (*i).isPropositional() )
                head.addUnique(GATOM(*i));
            else
                {
                ATOM head_element(*i,vTOc);
                head.addUnique(GATOM(head_element));
                }
            }

        // The ground rule is created and added to GroundIDB only if
        // it is not a fact, i.e. the ground body is not empty or the 
        // ground head is disjunctive. 
        // This further check is needed because it is possible 
        // that a non-ground rule with disjunctive head and body not empty 
        // has a ground instance with empty ground body and non-disjunctive 
        // ground head. As an example consider the rule
        // p(X) v p(1) :- q(X). and the fact q(1). Clearly, the ground 
        // instance of this rule is p(1). 
        if( body || head.size() > 1 )
            {
            GRULE groundR(&head,body);

            // Note that the code concerning recursive aggregates is 
            // not currently used.

            // Check whether recursive aggregates have been found in the
            // current rule.
            if( recursiveAggregates.size() )
                {
                if( GTraceLevel >= 2 )
                    cdebug << groundR 
                           << " -- postponed due to recursive aggregates." 
                           << endl;

                // Store a new RecursiveAggrRule containing the information
                // about the current rule, the recursive aggregates found in 
                // it and the current vTOc.
                recursiveAggrRules.push_back( 
                    RecursiveAggrRule( groundR, recursiveAggregates, vTOc ) );
                }
            else
                {
                if( GTraceLevel >= 1 )
                    cdebug << groundR << " -- ADDED." << endl;

                GroundIDB.push_back(groundR);
                }

            for( GDISJUNCTION::const_iterator 
                 H = head.begin();
                 H != head.end();
                 H++ )
                {
                // We are sure that !I.isTrue(*H) && !deltaNF->isTrue(*H)
                // && !newNF->isTrue((*H).getTheRealOne())
                // holds because of some previous test.
                assert( !I.isTrue((*H).getTheRealOne())
                        && !deltaNF->isTrue((*H).getTheRealOne()) 
                        && !newNF->isTrue((*H).getTheRealOne()) );

                if( (! I.isUndefined((*H).getTheRealOne())) && 
                    (! deltaNF->isUndefined((*H).getTheRealOne())) )
                    {
                    if( !newNF->isUndefined((*H).getTheRealOne()) )  
                        {
                        NFvEDBwasChanged = true;
                        newNF->addUndefined((*H).getTheRealOne());
                        solved[((*H).getTheRealOne()).getIndex()] = false;
                        newSize[((*H).getTheRealOne()).getIndex()]++;
                        if( GTraceLevel >= 2 )
                            cdebug << "IDB: counter_of( "
                                   << (*H).getTheRealOne().getPredName() 
                                   << " ) = " 
                                   << newSize[((*H).getTheRealOne()).
                                              getIndex()]
                                   << endl;
                        }
                    else
                        if( GTraceLevel >=2 )
                            cdebug << (*H).getTheRealOne() 
                                   << " is already in newNF" << endl;
                    }
                }
            if( body )
                delete body;
            return;
            }
        }
    
    // The following code is executed only if the ground instance of R 
    // is not constructed (i.e, it is a fact).
    // In this case, it is necessary to create the ground head atom
    // and add it to the interpretation.
    DISJUNCTION::const_iterator H = R.getHead().begin();
    ATOM a(*H, vTOc);
        
    if( GTraceLevel >= 1 )
        cdebug << a << " -- ADDED as definite knowledge.";

    // Note that the code concerning recursive aggregates is not 
    // currently used.

    // Check whether recursive aggregates have been found in the current 
    // rule. In this case create a ground rule if its body (without 
    // recursive aggregates) is empty.
    if( recursiveAggregates.size() )
        {
        GDISJUNCTION head;

        if( (*H).isPropositional() )
            head.add(GATOM(*H));
        else
           {
           ATOM head_element(*H,vTOc);
           head.add(GATOM(head_element));
           }

        body = new GCONJUNCTION();

        GRULE groundR( &head, body );

        // Store a new RecursiveAggrRule containing the information
        // about the current rule, the recursive aggregates found in 
        // it and the current vTOc.
        recursiveAggrRules.push_back( 
            RecursiveAggrRule( groundR, recursiveAggregates, vTOc ) );

        delete body;

        if( ! I.isUndefined(a) && 
            ! deltaNF->isUndefined(a) )
            {
            if( ! newNF->isUndefined(a) )  
                {
                NFvEDBwasChanged = true;
                newNF->addUndefined(a);
                solved[a.getIndex()] = false;
                newSize[a.getIndex()]++;
                if( GTraceLevel >= 2 )
                    cdebug << "IDB: counter_of( "
                           << a.getPredName() 
                           << " ) = " 
                           << newSize[a.getIndex()]
                           << endl;
                }
            else
                if( GTraceLevel >= 2 )
                    cdebug << a 
                           << " is already in newNF" << endl;
            }

        return;
        }

    if( body )
        delete body;

    // We are sure that !I.isTrue(*H) && !deltaNF->isTrue(*H)
    // && !newNF->isTrue((*H).getTheRealOne()) holds
    // because of some previous test.
    assert( !I.isTrue(a)
            && !deltaNF->isTrue(a)
            && !newNF->isTrue(a) );

    if( GTraceLevel >= 1 )
        cdebug << endl;
    if( !newNF->isUndefined(a) ) 
        { 
        NFvEDBwasChanged = true;
        newNF->addPositive(a,false);
        newSize[a.getIndex()]++;
        if( GTraceLevel >= 2 )
            cdebug << "EDB: counter_of( "
                   << a.getPredName() 
                   << " ) = " 
                   << newSize[a.getIndex()]
                   << endl;
        }
    else
        if( newNF->isUndefined(a) )
            {
            if( GTraceLevel >=2 )
                cdebug << a 
                       << " was already in newNF::UndefinedPart" 
                       << " but was moved in newNF::PositivePart"
                       << endl;
            newNF->addPositive(a,false);
            }
    }

/** Create all ground instances of one rule.
 *
 * If the body of R is empty or contains only negative literals, we have 
 * nothing to instantiate and we simply call AddToGroundIDB which add 
 * R to GroundIDB.
 *
 * An object OrdB of type OrderedBody is defined.
 *
 * pLiteral is the current literal of the rule body to be grounded.
 * The technique we use for grounding is a backtracking technique 
 * implemented by means of a stack and a do while cycle. On the stack 
 * we save the status of the current literal before we go to the next in 
 * order that when we return to the same literal by backtracking we can 
 * attempt the matching with a new ground atom that we have not used before.
 *
 * The technique is the following: we initialise the stack S with the 
 * first literal of OrdB to be grounded. At the generic iteration of the
 * do while cycle we get the literal on the top of the stack S to
 * be grounded and we try to match it with some ground atom by calling the
 * function Match(...).
 * - If the matching was successful we get the next literal from OrdB to 
 *   be grounded and if we did not reach the end of OrdB we save the
 *   literal  on the stack S with its status in order that at next
 *   iteration be the next literal to be grounded by calling Match(...).
 *   If the end of OrdB was reached that means we got a new ground
 *   instance of the rule and we call the procedure AddToGroundIDB(...)
 *   which adds the new ground  instance to GroundIDB. Then we backtrack
 *   to the last literal we grounded  and call again Match(...) trying
 *   to match it with another ground atom.
 * - If the matching failed, we remove first pLiteral with its status from
 *   the stack, and then
 *     o if pLiteral was the first literal of OrdB we check if the rule
 *       is recursive. If the rule is an exit rule that means we generated 
 *       all possible ground instances of the rule and we finish the 
 *       procedure (we assign outBody to pLiteral which is the exit 
 *       condition of the cycle). If the rule is a recursive rule we first 
 *       attempt to move the token in OrdB to another recursive literal. 
 *       If the attempt fails we finish the procedure as in the case of 
 *       exit rule, otherwise the body of OrdB is reordered and we assign 
 *       to pLiteral the new first literal in OrdB and we save it on the 
 *       stack S.
 *     o if pLiteral was not the first literal of OrdB we backtrack to
 *       the previous grounded literal of OrdB and at the next iteration 
 *       we call again Match(...) trying to match it with another ground 
 *       atom and backtracking again in case no matching is possible, etc.
 *
 * Uses globals: vTOc, I, size, done
 */
void GROUNDING::InstantiateRule(
    const RULE &R,
    bool isExitRule,
    bool &NFvEDBwasChanged, 
    INTERPRET *deltaNF,
    INTERPRET *newNF, 
    unsigned *deltaSize,
    unsigned *newSize,
    unsigned iteration,
    const unsigned component )
    {
    if( R.getBody() )
        {
        if( isAlwaysFalse(R) )
        // FIXME: (possible improvement) these rules could be marked and
        // not considered again
            return;
        }
    else
        // Each rule having empty body must be disjunctive; if it's not
        // disjunctive it's a (IDB) fact and its head has been already stored
        // in EDB table during either the parsing or a previous step of
        // grounding
        {
        AddToGroundIDB(R,NFvEDBwasChanged,deltaNF,newNF,newSize,component);
        return;
        }

    bool *headVars = new bool [n_par];
    fill_n(headVars, n_par, false);
    computeHeadVars(R, headVars);

    OrderedBody OrdB(*R.getBody(),size,deltaSize,done,isExitRule,iteration);
    if( isExitRule )
        {
        if( OrdB.noGround() )
            {
            if( GTraceLevel >= 2 )
                cdebug << "Grounded aborted immediately for this EXIT rule!"
                       << endl;
            return;
            }
        }
    else
        while( OrdB.noGround() )
            {
            if( OrdB.atEnd() )
                {
                if( GTraceLevel >= 2 )
                    cdebug<<"All iterations failed for RECURSIVE rule!"
                          <<endl;
                return;
                }
            else
                {
                if( GTraceLevel >= 2 )
                    cdebug << "The current iteration don't produce anything!"
                           << " Next iteration..." << endl;
                OrdB.Reorder();
                }
            }
    InstantiateBody(OrdB, rule, isExitRule, R, NFvEDBwasChanged, 
                    deltaNF, newNF, deltaSize, newSize, 
                    &GROUNDING::AddToGroundIDB, component, headVars);
    delete [] headVars;
    }

/** Create a ground instance of a constraint, with given variable mappings.
 * This is a just simpler version of AddToGroundIDB (c.f.).
 * Uses globals: vTOc, GroundConstraints
 * @param C the constraint to ground.
 */
void GROUNDING::AddToGroundCONSTRAINTS(
    const CONSTRAINT &C, 
    bool&, 
    INTERPRET*, 
    INTERPRET*, 
    unsigned*,
    const unsigned )
    {
    // Creation of a ground constraint.
    GCONSTRAINT GC;
    fillGConjunction(GC,C,vTOc,0,UINT_MAX,0);
   
    if( GC.begin() == GC.end() )
        {
        if( TraceLevel >= 1 || GTraceLevel >= 1 )
            {
            if( C.isInternal() )
                cdebug << "Internal ";
            
            cdebug << "Constraint  " << C
                   << "  is always violated [grounding.C]." << endl;
            }
        
        status = ConstraintAlwaysViolated;
        if ( printInstantiation ) 
            {
            cout << "% The following constraint will always be violated" 
                 << endl << "% " << C << endl
                 << "% so this program does not have any answer set." << endl
                 << ":- not a_Constant_Never_Occurring_In_Regular_Programs." 
                 << endl;
            GroundIDB.erase(GroundIDB.begin(), GroundIDB.end());
            GroundConstraints.erase(GroundConstraints.begin(), GroundConstraints.end());
            }	
        return;
        }
    else
        {
        if( GTraceLevel >= 1 )
            cdebug << GC << " -- ADDED." << endl;
        GroundConstraints.push_back(GC);
        }
    }

/** Create all ground instances of one constraint.
 * This is very much a simpler version of InstantiateRule: constraints
 * have no head, they cannot be recursive, etc.
 * Uses globals: vTOc, I
 * @param C the constraint to ground.
 */
void GROUNDING::InstantiateConstraint(
    const CONSTRAINT &C, 
    unsigned iteration )
    {
    OrderedBody OrdB(C,size,0,done,true,iteration);
  
    if( OrdB.noGround() )
	{
	if( GTraceLevel >= 2 )
	    cdebug << "Grounded aborted immediately for " << C << endl;
	return;
	}

    bool dummy=false;
    InstantiateBody(OrdB,strongConstraint, dummy, C, dummy, 0, 0, 0, 0,
                    &GROUNDING::AddToGroundCONSTRAINTS, UINT_MAX, 0);
    }

/** Create all ground instances of one constraint.
 * This is very much a simpler version of InstantiateRule: constraints
 * have no head, they cannot be recursive, etc.
 * Uses globals: vTOc, I
 * @param C the constraint to ground.
 */
void GROUNDING::AddToGroundWEAKCONSTRAINTS(
    const WEAKCONSTRAINT &C, 
    bool&, 
    INTERPRET*, 
    INTERPRET*, 
    unsigned*,
    const unsigned )
    {
    if( GTraceLevel >= 1 )
        {
        cdebug << "Non-ground weak constraint has weight " << C << ": " 
               << C.getWeights().first << endl;
        }

    // Creation of the ground weak constraint
    GWEAKCONSTRAINT GC;
    fillGConjunction(GC,C,vTOc,0,UINT_MAX,0);
    // Instantiate Weight. Check whether Weight is a variable.
    TERM w1;
    if( C.getWeights().first.isVar() )
        // Check whether the variable is a positive integer.
        if( ! vTOc[C.getWeights().first.getVar()].isInt() 
            || vTOc[C.getWeights().first.getVar()].getInt() < 0 )
            {
            cerr << "Weight must be a non-negative integer in " << C
                 << endl;
            exit(1);
            }
        else
            // Instantiate Weight.
            w1 = TERM( vTOc[C.getWeights().first.getVar()].getInt() 
                       * SCALEFACT, 0);
    else
        w1 = C.getWeights().first;

    // Instantiate Level. Check whether Level is a variable.
    TERM w2;
    if( C.getWeights().second.isVar() )
        // Check whether the variable is a positive integer.
        if( ! vTOc[C.getWeights().second.getVar()].isInt()
            || ! vTOc[C.getWeights().second.getVar()].getInt() > 0 )
            {
            cerr << "Level must be a positive integer in " << C
                 << endl;
            exit(1);
            }
        else
            // Instantiate Level.
            w2 = TERM( vTOc[C.getWeights().second.getVar()].getInt(),0);
    else
        w2 = C.getWeights().second;

    GC.setWeights(w1,w2);

    assert( w1.isInt() );
    if( w1.getInt() == 0 )
        {
        if( GTraceLevel >= 1 )
            {
            cdebug << "Not generating " << GC << ": Weight is 0."
                   << endl;
            }
        }
    else
        {
        if( GTraceLevel >= 1 )
            {
            cdebug << "Groundweakconstraint weight of " << GC << ": " 
                   << GC.getWeight() << endl;
            }
        if( GC.begin() == GC.end() )
            {
            // Add weak to the violated.
            ViolatedWConstraints.push_back(GC);
            if( TraceLevel >= 1 || GTraceLevel >= 1 )
                {
                cdebug << "Weak Constraint  " << GC
                       << "  is always violated [grounding.C]." << endl;
                }
            }
        else
            {
            if( GTraceLevel >= 1 )
                cdebug << GC << " -- ADDED." << endl;
            GroundWConstraints.push_back(GC);
            }
        }

    // Increase, if necessary, the max level available
    if( GC.getLevel() > MaxWeakConstraintLevel )
        MaxWeakConstraintLevel = GC.getLevel();
    }

/** Create all ground instances of one weak constraint.
 * This is essentially the same as InstantiateConstraint.
 */
void GROUNDING::InstantiateWeakConstraint(
    const WEAKCONSTRAINT &C, 
    unsigned iteration )
    {
    OrderedBody OrdB(C,size,0,done,true,iteration);
  
    if( OrdB.noGround() )
	{
	if( GTraceLevel >= 2 )
	    cdebug << "Grounded aborted immediately for " << C << endl;
	return;
	}

    bool dummy=false;
    InstantiateBody(OrdB,weakConstraint , dummy, C, dummy, 0, 0, 0, 0,
                    &GROUNDING::AddToGroundWEAKCONSTRAINTS, UINT_MAX, 0);
    }

// BEGIN functions needed by the backjumping algorithm.

// Stores the variables of the head of R in the array headVars.
void GROUNDING::computeHeadVars(const RULE &R, bool *headVars)
    {
    for( DISJUNCTION::const_iterator i = R.getHead().begin();  
         i !=  R.getHead().end();
         i++ )
        if ( i->getParams() ) 
            for( TERMS::const_iterator j = i->getParams()->begin();
                 j != i->getParams()->end();
                 j++ )
                if( j->isVar() )
                    headVars[j->getVar()] = true;
    }

// Auxiliary function used during the backjumping phase.
// Move the iterator pLiteral back to the literal that has to 
// be matched again, cleaning the stack and restoring variables in
// vTOc. 
// @param OrdB the body of the rule to examine.
// @param pLiteral the current literal.
// @param lit the index of the literal where to jump back.
// @param S the stack of cells.
// @param vTOc the current variable mapping.
void  GROUNDING::backToLit(
    OrderedBody &OrdB,
    CONJUNCTION::const_iterator &pLiteral,
    const int &lit,
    vector<cell> &S,
    TERM vTOc[n_par])
    {
    while( OrdB.getIndexLiteral(pLiteral) != lit
           && pLiteral != OrdB.begin() )
        {
        // Negative regular literals (non-aggregates) are never
        // pushed in the stack. So for them avoid the pop of an 
        // element from the stack.    
        if( ! pLiteral->isNegative() || pLiteral->isAggregate() )
            { 
            Restore( *pLiteral, S.back().get_Status(), vTOc );
            S.pop_back();
            }
        pLiteral = OrdB.getPrevious();    
        }
    // After the while, pLiteral points to the literal in the body
    // which has to be matched again. The cell corresponding to this
    // literal has not to be removed from the stack but vTOc need 
    // to be restored.
    Restore(*pLiteral, S.back().get_Status(), vTOc);
    // if lit is -1, i.e. the instantiation of the current rule can 
    // be stopped, all the cells have to be removed from the 
    // stack. In the previous while, we removed all the cells 
    // except the first one. So, we make here one further pop to remove it. 
    if( lit == -1 )
        S.pop_back();
    }

/** Update the CSB after a failed match according to the type of failure. 
 * @param PreviousMatch_OK stores the status of the previous
 * match( failed or not).
 * @param CB the closest binder of the variables of the current literal. 
 * @param CSB the Closest Successful Binder.
 * @param lit the index of the current literal.
 * @param bJ the BACKJUMPING object. 
 */
void GROUNDING::updateCSB(
    const bool& previousMatch_OK, 
    const int& CB,
    int& CSB,
    const int& lit,
    const BACKJUMPING& bJ )
    {
    if( previousMatch_OK )
        {
        // CSB has to be updated if it follows CB in OrdB.        
        if( CSB > CB )
            CSB = CB;
        }
    else
        // If it coincides with the current literal, CSB is pushed back 
        // to the closest binder binding an output var.
        if( lit == CSB )
            CSB = bJ.getClosestBinderOV(lit);
    }

/** Backjump to look for another solution.
 * Jump on the CSB literal; if no CSB is found, set done to true. This
 * will stop the instantiation of the current rule.
 * @param OrdB the body of the rule to examine.
 * @param pLiteral the current literal.
 * @param bJ the BACKJUMPING object.
 * @param CSB the index of the closest successful binder. 
 * @param done is set to true to indicate if the instantiation
 *  has to be ended.
 * @param S the stack of cells.
 * */

void GROUNDING::backFromSolutionFound(
    OrderedBody &OrdB,
    CONSTRAINT::const_iterator &pLiteral,
    BACKJUMPING &bJ,
    int &CSB,
    bool &done,
    vector<cell> &S)
    {   
    // Move pLiteral on the last literal in the body.
    pLiteral = OrdB.getPrevious();
          
    CSB = bJ.getGreatestBinderOV();
    // If there exists a CSB, go back to it and look for another
    // solution. Else, exit restoring all the cells and emptying
    // the stack.     
    if( CSB != -1 )
        {   
        backToLit(OrdB, pLiteral, CSB, S, vTOc);
        if( GTraceLevel >= 2 )
            cdebug << " Back to " << *pLiteral
                   << " and look for another solution " <<endl;
        }
    else
        {
        if( GTraceLevel >= 2 )
            cdebug << " No further relevant solutions exist. "
                   << " Exit "<< endl;
        backToLit(OrdB, pLiteral, -1, S, vTOc);
        done = true;
        }
    }
// END functions needed by backjumping algorithm. 

/** Instantiate the body of a rule or a constraint. Use the backjumping 
 * algorithm described below. 
 * A crucial notion in this algorithm is the Closest Successful Binder 
 * (CSB), which, intuitively, represents the literal with greatest index 
 * that is a binder of some variable whose current assigned value belongs 
 * to the last computed solution. The CSB acts as a barrier for some kind 
 * of jumps, i.e. it can not be jumped, otherwise we can miss some relevant 
 * solution.
 * The basic schema of the algorithm is the following:
 * At each iteration, try to find a match for a literal. If it succeeds and 
 * the current literal is not the last literal then, proceed with the next 
 * literal in the body, else we have to backtrack and thus, we have to 
 * decide where to jump and possibly update the current CSB. 
 * There are three different cases to be handled:
 * 1) Solution Found (backtrack to find another solution): jump back to the 
 * closest regular literal that binds an output variable and set the CSB
 * to it.
 * 2) Failure at the first attempt to find a match for a literal: jump back
 * to the closest literal that binds any of its variables. If this literal 
 * preceeds the CSB in the ordering, push back the CSB to it.
 * 3) Failure while looking for another match for a literal: jump back
 * to the greatest literal between the CSB and the closest binder of a 
 * variable of its dependencies set.
 * @param OrdB the body of the rule or the constraint already ordered.
 * @param type says if OrdB is the body of a rule or of a constraint or of 
 * a weak constraint.
 * @param C the constraint or the rule to ground.
 * @params isExitRule, NFVEDBwasChanged, deltaNF, newNf, deltaSize, newSize
 * are necessary only if C is a rule.
 * @param component the current component in the dependency graph. 
 * It is set to UINT_MAX if C is a constraint or a weakConstraint. 
 */
template<class TYPE>
void GROUNDING::InstantiateBody(
    OrderedBody & OrdB,
    const TYPECLAUSE &type,
    bool & isExitRule,
    const TYPE& C, 
    bool & NFVEDBwasChanged, 
    INTERPRET * deltaNF,
    INTERPRET * newNF, 
    unsigned* deltaSize,
    unsigned* newSize,
    void(GROUNDING::*AddFunction)(
        const TYPE& , 
        bool&, 
        INTERPRET*, 
        INTERPRET*, 
        unsigned*,
        unsigned),
    const unsigned component,
    bool *headVars)
    {
    CONSTRAINT::const_iterator outBody=OrdB.end();
    CONSTRAINT::const_iterator pLiteral=OrdB.begin();

    //Create the structures needed by the backjumping algorithm.
    BACKJUMPING bJ(OrdB, headVars);
    bJ.computeStructures(OrdB, solved, type, false);
    int CSB = -1;  

    /** Save choice-points in the grounding of one RULE.
     * When there are more than one possible matches for one predicate, the
     * current state is pushed on this stack. Later, the grounding backtracks
     * to this point againt, to consider the other matches.
     */
    vector<cell> S;
    bool previousMatch_OK = true;

    S.push_back(cell(*pLiteral, vTOc, I, deltaNF, size, deltaSize, OrdB.getRange()));
    for( bool done=false; ! done; )
	{
	cell &StackCell=S.back();
        bool successfulMatch;
        if( (*pLiteral).isAggregate() )
            {
            // Note that the code concerning recursive aggregates is 
            // not currently used.

            if( ! isRecursiveAggregate( (*pLiteral).getAggregate()
                                         .getAuxAtom(), component ) )
                {
                successfulMatch = MatchAggregate(*pLiteral, vTOc);
                // Check whether the violation of the aggregate causes the 
                // current literal to become false w.r.t the interpretation. 
                // In this case go back to the first non aggregate literal 
                // in the rule.   
                if( checkViolatedAggregate(*pLiteral, vTOc) )      
                    {
                    if( GTraceLevel >= 2 )
                        cdebug << " Constraint satisfied,"
                                  " grounding aborted immediately!" 
                               << endl;

                    previousMatch_OK = false;
                    
                    if( bJ.getClosestBinderVars(OrdB,pLiteral) == -1 )
                        done = true;

                    backToLit(OrdB, pLiteral,
                              bJ.getClosestBinderVars(OrdB,pLiteral),
                              S, vTOc);
                                        
                    if( GTraceLevel >= 2 )
                        { 
                        cdebug << " Aggregate is violated " << endl;
                        if ( ! done )
                            cdebug << " back to " << *pLiteral << endl;
                        else cdebug << " Exit " << endl;
                        }
                    continue;
                    }
                if( status == AggregateInvalidUse )
                    return;
                }
            else 
                successfulMatch = true;
            }
        // If the first literal in the body is negative check whether
        // it's not already false. In this case the instantiation
        // process can continue, otherwise it has to be stopped.
        else if ( (*pLiteral).isNegative() )
            if ( ! negativeLiteralIsFalse(*pLiteral, deltaNF) )
                successfulMatch = true;
            else 
                successfulMatch = false;
        else
            successfulMatch = Match(*pLiteral,StackCell,vTOc);            
	if( successfulMatch )
	    {
            previousMatch_OK = true;
	    if( GTraceLevel >= 1 )
                { 
 	        cdebug << " -> successful matching of "
                       << *pLiteral;
                if( !(*pLiteral).isNegative() && !(*pLiteral).isAggregate() ) 
                    cdebug << " with -> "
                           << ATOM(*pLiteral,vTOc);
                cdebug << endl;
                }
            pLiteral=OrdB.getNext();

            if( pLiteral != outBody 
                && ( ! (*pLiteral).isNegative() 
                     || (*pLiteral).isAggregate() ) )
                S.push_back(cell(*pLiteral, vTOc, I, deltaNF, size, 
                                 deltaSize, OrdB.getRange()));
            else if( pLiteral == outBody )
                {
                (this->*AddFunction)( C, NFVEDBwasChanged,deltaNF, newNF, 
                                      newSize, component );
                if( GTraceLevel >= 2 )
                    cdebug << " Solution found ... " << endl;

                if( status == ConstraintAlwaysViolated )
                    return;

                // Back From Solution Found: 
                previousMatch_OK = false;
                backFromSolutionFound(OrdB, pLiteral, bJ, CSB, done, S);
                }
            }
        else // FAILED MATCH
            {
            if( GTraceLevel >= 1 )
                { 
 	        cdebug << " -> failed matching of "
                       << *pLiteral;
                if( !(*pLiteral).isNegative() && !(*pLiteral).isAggregate() ) 
                    cdebug << " with -> "
                           << ATOM(*pLiteral,vTOc);
                cdebug << endl;
                }
       
            int CB = bJ.getClosestBinderVars(OrdB, pLiteral);
            int CBDS = bJ.getClosestBinderDS(OrdB, pLiteral);

            // Update the CSB after the failed match;
            updateCSB(previousMatch_OK, CB, CSB, 
                      OrdB.getIndexLiteral(pLiteral), bJ );
 
            // Decide on which literal jump back depending on the type of 
            // failure (failureOnFirstMatch or failureOnNextMatch), and
            // possibly exit.

            // The following if checks if it is possible to safely
            // terminate the instantiation of the current rule.
            // This is possible in two cases: when a first match
            // on a literal fails and it has not a closest binder
            // or when a next match on a literal fails, it has
            // not a closest binder of its dep set and CSB is -1

            // if EXIT
            if( ( previousMatch_OK && CB == -1 ) 
                || ( !previousMatch_OK && CBDS == -1 && CSB == -1 ) )
                {
                if( GTraceLevel >= 2 )
                    cdebug << " EXIT: no further solutions " << endl;
                backToLit(OrdB, pLiteral, -1, S, vTOc);
                }
            if( pLiteral == OrdB.begin() )
                {
                assert(empty(vTOc,n_par,TERM()));
                if ( type != rule || isExitRule)
                    done = true;
                else
                    { 
                    if( OrdB.atEnd() )
                        done = true;
                    else
                        {
                        OrdB.Reorder();
                        while( OrdB.noGround() )
                            {
                            if( OrdB.atEnd() )
                                {
                                if( GTraceLevel >= 2 )
                                    cdebug << " No further iteration "
                                           << " delivered to a new instance "
                                           << endl;
                                return;
                                }
                            else
                                {
                                if( GTraceLevel >= 2 )
                                    cdebug << " The current iteration didn't"
                                           << " produce anything! "
                                           << " Next iteration..."
                                           << endl;
                                OrdB.Reorder(); 
                                }
                            }
                        // After that the body of the recursive rule
                        // has been reordered, structures have to be
                        // resetted and computed again.
                        bJ.resetStructures(headVars);
                        CSB = -1;
                        bJ.computeStructures(OrdB, solved, type, true);
                        pLiteral = OrdB.begin();
                        S.push_back(cell(*pLiteral, vTOc, I, deltaNF, 
                                         size, deltaSize, OrdB.getRange()));
                        }
                    } 
           
                } // end if EXIT
            else
                {
                // Backjump after a failure while looking for a first match 
                // for a literal. 
                // Jump back to CB, the closest literal to pLiteral binding 
                // some of its variables.
                if( previousMatch_OK )
                    {
                    backToLit(OrdB, pLiteral, CB, S, vTOc);
                    if( GTraceLevel >= 2 )
                        cdebug << " Failure On First Match: "; 
                    previousMatch_OK = false;
                    }
                else
                    {
                    // Backjump after a failure while looking for another 
                    // match for a literal.
                    // Jump back to the closest literal between the CSB and 
                    // CBDS, the closest literal to pLiteral binding some of 
                    // the variables of its dependencies set.
                    backToLit(OrdB, pLiteral, max(CBDS,CSB), S, vTOc);
                    if( GTraceLevel >=2 )
                        cdebug << " Failure On Next Match: ";
                    }
                
                if( GTraceLevel >=2 )
                    cdebug << " back to " << *pLiteral << endl;
                } 
            }
        }
    }

/** Return the aggregate atom to a previous state.
 * Uses globals: vTOc
 * @param a the aggregate atom to restore.
 */
void GROUNDING::RestoreAggregate( AGGREGATEATOM &a )
    {
    // Restore guards only if it's an assignment.
    // It's sufficient to restore only a guard because in 
    // this case the two guards are the same variable.
    if( a.isAssignment() && a.getLowerGuardAsVar().isVar() )
        {
        assert( a.getLowerGuardAsVar() == a.getUpperGuardAsVar() );
        vTOc[a.getLowerGuardAsVar().getVar()]=TERM();
        }
    a.reset();
    }

/** Return the variable bindings to a previous state.
 * Since the only state transformation is unbound->bound_to_value we only
 * unbind those variables that were unbound and are bound now.
 * Uses globals: vTOc
 * @param lit the literal for which variables are restored.
 * @param state the vector which states what variables should be unbound.
 * @param vTOc the current variable mappings.
 */
void GROUNDING::Restore( 
    const LITERAL &lit, 
    const BOOLS &state, 
          TERM vTOc[n_par] )
    {
    if( GTraceLevel >= 2 )
        cdebug << " Restore literal " << lit << endl;
    if(lit.isAggregate())
        RestoreAggregate(lit.getAggregate());
    unsigned k=0;
    for(BOOLS::const_iterator i=state.begin(); i != state.end(); i++, k++)
	if (*i)
	    {
            unsigned nVar=0;
            if(lit.isRegularAtom())
	        nVar=(*lit.getParams())[k].getVar();

	    assert(nVar < n_par);
            if( GTraceLevel >= 2 )
                cdebug << "     --> reset _" << nVar << endl;
	    vTOc[nVar]=TERM();
	    }
    }

/** Add new ground atoms to I.
 * New ground atoms are first stored in an auxiliary database (deltaNF) and
 * then put into I at once with this function.
 * @param deltaNF the collection of ground atoms to add.
 * @param deltaSize the number of ground atoms in the collection.
 */
void GROUNDING::UpdateI( const INTERPRET &deltaNF, unsigned *deltaSize )
    {
    deltaNF.foreachPositive( UpdateIpositive(deltaSize,size) );
    deltaNF.foreachUndefined( UpdateIundefined(deltaSize,size) );
    }

/** Return true if the given rule has already been considered
 *  in a previous component.
 *  @param R the RULE to examine.
 *  @param idxComp the index of the component of R.
 */
bool GROUNDING::isAlreadyInstantiated(
    const RULE &R,
    unsigned &idxComp) const
    {
    // Check whether one of the head atoms of the current rule
    // belongs to a previous component.
    for( DISJUNCTION::const_iterator H = R.getHead().begin();
         H != R.getHead().end();
         H++ )
        if( NonGroundDepGraph.getAtomComponent(*H,false) < idxComp )
            {
            if( GTraceLevel >= 2 )
                cdebug<<" Rule "<<R<<": SKIPPED "<<endl
                      <<" already instantiated in a previous component. "
                      <<endl;
            return true;
            }
    return false;
    }

/** Check whether one of the positive ATOMs of the given RULE's body is 
 * undefined. The RULE can never fire in this case.
 * @param R the RULE to examine.
 * @return whether R's body can be satisfied.
 */
bool GROUNDING::isAlwaysFalse( const RULE &R ) const
    {
    for( CONJUNCTION::const_iterator i  = R.getBody()->pos_begin(); 
				     i != R.getBody()->pos_end();
				     i++ )
	if( (*i).isUndef() )
	    return true;
                
    return false;
    }

/** Remove unnecessary ground rules.
 * This function goes once more over all ground rules and eliminates rules
 * whose head is already true, or whose body can never be satisfied.
 * Uses globals: I, GroundIDB
 */
void GROUNDING::SimplifyGroundIDB()
    {
    I.clearUndefinedPart();
    for(GRULES::iterator R = GroundIDB.begin(); R != GroundIDB.end(); )
	{
	assert( (*R).getBody() || (*R).isDisjunctive() );
	bool theHeadIsTrue = false;
	bool theBodyIsFalse = false;
	if( GTraceLevel >= 1 )
	    cdebug << " processing " << *R << endl;

	// each head atom is an IDB atom
	for( GDISJUNCTION::const_iterator H = R->getHead().begin();
					  H != R->getHead().end();
					  H++ )
	    if( I.isTrue((*H).getTheRealOne()) )
		{
		if( GTraceLevel >= 2 )
		    cdebug << "   Atom " << *H 
			   << " already true in the head of rule " << *R
			   << " (was removed from GroundIDB)" << endl;
		theHeadIsTrue = true;
		break;
		}

	if( theHeadIsTrue )
            unordered_erase(GroundIDB,R);
	else
	    {
	    if( (*R).getBody() )
		{
		for( GCONJUNCTION::const_iterator
			 L = (*R).getBody()->neg_begin();
			 L != (*R).getBody()->neg_end();
			 L++ )
		    if( I.isTrue((*L).getTheRealOne()) )
			{
			if( GTraceLevel >= 2 )
			    cdebug << "   Literal " << *L 
				   << " already false in the body of rule "
				   << *R << " (was removed from GroundIDB)"
				   << endl;
			theBodyIsFalse = true;
			break;
			}
		}

	    if( theBodyIsFalse )
                unordered_erase(GroundIDB,R);
	    else
		{
		for( GDISJUNCTION::const_iterator 
                     H = R->getHead().begin();
                     H != R->getHead().end();
                     H++ )
		    {
		    if( !I.isUndefined((*H).getTheRealOne()) )
			I.addUndefined((*H).getTheRealOne());
		    else
			if( GTraceLevel >= 1 )
			    cdebug << "  " << *H
				   << " is defined by more than one rule!"
				   << endl;
		    }
		if( (*R).getBody() )
		    {
                    for( GCONJUNCTION::iterator 
                         L = (*R).getBodyForModification()->pos_begin();
			 L != (*R).getBody()->pos_end(); )
                        {
			if( I.isTrue((*L).getTheRealOne()) )
                            {
                            if( GTraceLevel >= 2 )
                                cdebug << *L
                                       << " is always true. " 
                                       << "Removing it from "
                                       << *R << endl;

                            L=(*R).getBodyForModification()->remove(L);
                            // Iterator now points to the element following 
                            // the removed one.
                            }
                        else
                            // Increment this iterator only if we have not
                            // removed the current element.
                            L++;
                        }
		    }
		}
	    }

	if( !theHeadIsTrue && !theBodyIsFalse )
	    R++;
	}
    }

 void GROUNDING::cleanMemory() const
     {
     delete deltaNF;
     delete newNF;
     delete[] solved;
     delete[] done;
     delete[] size;
     delete[] deltaSize;
     delete[] newSize;
     delete dictionary;
     }

/** Generate ground instances of all rules, constraints, and weak constraints.
 *
 * Variables:
 *
 * deltaNF is the set of the new non false ground atom instances 
 * derived at the previous step; 
 *
 * newNF is the set of the new non false ground atom instances derived 
 * at the current step; 
 *
 * Algorithm:
 *
 * Let P be a non-ground program and C a node of NonGroundDepGraph,
 * the non-ground collapsed dependency graph.
 * We denote by recursive_rules_P(C) the set of the rules r from P s.t. 
 * predicates from C occur both in H(r) and in B+(r), and by exit_rules_P(C)
 * the remaining set of rules r in P with a predicate from C in H(r).
 *
 * We remove a node C_i from NonGroundDepGraph which has no incoming 
 * arc (i.e., a source), and generates all instances r' of rules r in
 * GroundSubP defining predicates in C_i. 
 * This is done by calls to InstantiateRule which inserts in newNF
 * each new ground atom of the heads of the ground rules. 
 *
 * For efficiency reasons, first non-recursive rules are instantiated once 
 * and for all. Then, the recursive rules are repeatedly instantiated until 
 * no new ground atom is added to newNF. At each recursive rules 
 * instantiating iteration we apply a differentiate technique which uses 
 * efficiently the ground atoms generated at the previous iteration. 
 * At this scope we need two sets:
 * deltaNF in which we find the ground atoms generated at the previous
 * iteration and newNF in which we store the new ground atoms we generate
 * in the current iteration that we need in the next iteration. 
 * In order to avoid at the end of each iteration to empty deltaNF and to 
 * store in it the atoms of newNF we will simply invert their pointers 
 * changing in this way the roles of the two sets. 
 * At this scope we used the function switchPointers. The same function is 
 * used also to exchange deltaSize and newSize.
 * At the end of each iteration the procedure UpdateI store in the
 * interpretation the new ground atoms.
 *
 * After all exit and recursive rules of the current component have been
 * grounded we set done[i]=true for each predicate i of that component.
 * That means we cannot further derive new instances for the predicate i.
 *
 * After all components of the non-ground collapsed dependency graph
 * were considered we call the procedure SimplifyGroundIDB which removes
 * from GroundIDB all rules whose head is already known to be true or
 * whose body is already known to be false. From the body of the remaining
 * rules, it removes the literals which are already known to be true.
 *
 * Finally, the procedure InstantiateConstraint is called for each
 * constraint rule and the execution halts if at least one constraint is
 * violated. If a weak constraint is violated we just store it in a vector
 * called ViolatedWConstraints without aborting the computation.
 */
bool GROUNDING::Ground()
    {
    for( unsigned i=0; i < NonGroundDepGraph.getComponents().size(); i++ )
	{
	const PROGRAM &GroundSubP=NonGroundDepGraph.getComponentRules(i);
#ifdef NEVER
        assert( GroundSubP.size() != 0 );
#endif
        deltaNF->clear();
        newNF->clear();
        if( GTraceLevel >= 1 )
	    cdebug << endl <<"comp " << (i+1) << " : GroundSubP = " 
		   << GroundSubP << endl;
	bool NFvEDBwasChanged = true;

        for( PROGRAM::const_iterator R = GroundSubP.exit_begin();
             R != GroundSubP.exit_end();
             R++ )
            {
            if( hasUnstratifiedAggregates( (*R),i ) )
                return false;

            if( ! isAlreadyInstantiated(*R,i) )
                {
                if( GTraceLevel >= 1 )
                    cdebug << " processing EXIT rule: " << *R << endl;

                unsigned iteration = 1;
                InstantiateRule(*R, true, NFvEDBwasChanged,
                                deltaNF, newNF, deltaSize,
                                newSize, iteration, i);
                if( failedInstantiation(*R) )
                    return false;
                }
            }
	
#ifdef NEVER
	if( ! newNF->isPositivePartEmpty() )
	    {
	    for(GRULES::iterator R = GroundIDB.begin(); R != GroundIDB.end(); )
		{
		bool GroundIDBwasChanged=false;
		if( GTraceLevel >= 1 )
		    cdebug << " processing GROUND EXIT rule: " << *R << endl;
		for( GDISJUNCTION::const_iterator H = R->getHead().begin();
						  H != R->getHead().end();
						  H++ )
		    {
		    // each head atom is an IDB atom
		    if( newNF->isTrue((*H).getTheRealOne()) )
			{
			if( GTraceLevel >= 2 )
			    cdebug << "   Atom " << *H 
				   << " already true in the head of rule " 
				   << *R << " (was removed from GroundIDB)" 
				   << endl;

                        unordered_erase(GroundIDB,R);

			GroundIDBwasChanged=true;
			break;
                        }
		    }
		if( !GroundIDBwasChanged )
		    R++;
		}
	    }
#endif

	NFvEDBwasChanged = true;
        unsigned nIt=0;

	while( NFvEDBwasChanged )
	    {
	    nIt++;

	    if( GTraceLevel >= 1 )
	      cdebug << endl << "ITERATION " << nIt;

	    NFvEDBwasChanged = false;
            // Invert deltaNF and newNF switching their pointers.
            switchPointers(deltaNF, newNF);
            switchPointers(deltaSize, newSize);
            if( GTraceLevel >= 1 )
                cdebug << "   deltaNF: " << *deltaNF << endl;
            
            newNF->clear();
            assert(empty(newSize, ATOM::Predicates.size(), (unsigned)0));
            
            for( PROGRAM::const_iterator R = GroundSubP.rec_begin();
                 R != GroundSubP.rec_end();
                 R++ )
                {
                if( hasUnstratifiedAggregates( (*R),i ) )
                    return false;
                
                if( ! isAlreadyInstantiated(*R,i) )
                    {
                    if( GTraceLevel >= 1 )
                        cdebug << " processing rule: " << *R << endl;
                    InstantiateRule(*R, false, NFvEDBwasChanged,
                                    deltaNF, newNF, deltaSize,
                                    newSize, nIt-1, i);
                    
                    if( failedInstantiation(*R) )
                        return false;
                    }
                }
            UpdateI(*deltaNF,deltaSize);
            if( GTraceLevel >= 1 )
                cdebug << "newNF: " << *newNF << endl;
            }     
        unsigned ComponentSize=NonGroundDepGraph.getComponents()[i].size();
        for(unsigned j=0;j<ComponentSize;j++)
            //set done[i]=true for each atom "i" in the component.
            done[NonGroundDepGraph.getAtoms()[NonGroundDepGraph.getComponents()[i][j]].getIndex()]=true;
	
        if( GTraceLevel >= 1 )
            cdebug << endl << "Instantiating aggregates: "<< endl << endl;

        // Note that the code concerning recursive aggregates is 
        // not currently used.
 
        // Store the recursive aggregates already matched.
        // We use this to avoid matching the same aggregate more than once
        // by processing only those aggregates not already in this set.
        set<GLITERAL> matchedAggregates;
    
        for( vector< RecursiveAggrRule >::iterator
             j = recursiveAggrRules.begin();
             j != recursiveAggrRules.end();
             j++ )
            {
            GRULE &r = (*j).rule;
            bool toBeAdded = true;

            for( vector<LITERAL>::iterator
                 l = (*j).recursiveAggregates.begin();
                 l != (*j).recursiveAggregates.end();
                 l++ )
                {
                GLITERAL lit( (*l).isNegative(), GATOM(ATOM(*l, (*j).v2c)) );

                if( matchedAggregates.find(lit ) == matchedAggregates.end() )
                    {
                    (void)matchedAggregates.insert(lit);
                    MatchAggregate( lit, (*j).v2c );
                    lit.getAggregate().moveToHashes();
                    }

                if( GTraceLevel >= 2 )
                        cdebug << endl << "-- Aggregate instantiated: "
                               << endl << lit << endl;

                // Check whether the aggregate is violated. In this case the
                // rule in which it appears has not to be added to GroundIDB.
                const AGGREGATEATOM &a = lit.getAggregate();
                if( ( a.isSatisfied() && lit.isNegative() )
                   || ( a.isViolated() && ! lit.isNegative() ) )
                    toBeAdded = false;
                // Otherwise, check whether the aggregate is not satisfied:
                // only in this case it has to be added to the ground rule.
                else if( ! ( ( a.isSatisfied() && ! lit.isNegative() )
                        || ( a.isViolated() && lit.isNegative() ) ) )
                    r.getBodyForModification()->addUnique(lit);
                }
   
            if( toBeAdded )
                {   
                if( GTraceLevel  >= 2 )
                    cdebug << endl << "-- Rule added to GroundIDB: "
                           << endl << r << endl;

                GroundIDB.push_back(r);
                }
            }

        recursiveAggrRules.clear();
        }
    
    timer.GroundingPostprocess.start();

    if( !I.isPositivePartEmpty() )
        {
        if( GTraceLevel >= 1 )
            cdebug << endl << "Invoking SimplifyGroundIDB():" << endl;
	SimplifyGroundIDB();
        }

    timer.GroundingPostprocess.stop();


    if( GTraceLevel >= 1 )
	cdebug << endl << endl << "Constraints grounding:" << endl << endl;

    timer.GroundingConstraints.start();

    for( CONSTRAINTS::const_iterator C =  Constraints.begin();
				     C !=Constraints.end();
				     C++ )
	{
 	if( GTraceLevel >= 1 )
 	    cdebug << " Processing " << *C << endl;
        if( ignoreConstraint(*C) )
            {
            if( GTraceLevel >= 1 )
                cdebug << " Constraint superfluos. Not grounded." << endl;
            }
        else
            {
            unsigned iteration=1;
            InstantiateConstraint(*C,iteration);

            if( failedInstantiation(*C) )
                {
                timer.GroundingConstraints.stop();
                return false;
                }
            }
	}

    timer.GroundingConstraints.stop();
    for(WEAKCONSTRAINTS::const_iterator C =  WConstraints.begin();
                                        C != WConstraints.end();
                                        C++ )
        {
        if( GTraceLevel >= 1 )
            cdebug << " Processing " << *C << endl;
        unsigned iteration=1;

        InstantiateWeakConstraint(*C,iteration);

        if( failedInstantiation(*C) )
            return false;
        }

    cleanMemory();    

    if( GTraceLevel >= 1 )
	cdebug << "Ground() ends." << endl;

    return true;
    }

bool GROUNDING::findSingleDefinition(
//
// Given a program and a predicate, try to find the single rule that defines
// that predicate. Fail if no such rule exists or there is more than one rule
// defining that predicate.
//
// @param rule, points to the rule upon success, is set to 0 else.
// @return true upon success, false else.
//
    const PROGRAM &program,
    const ATOM &atom,
    const RULE **rule ) const
    {
    unsigned definitions=0;

    // Look at each rule...
    for( PROGRAM::const_iterator i=program.begin();
         i != program.end();
         i++ )
        {
        const DISJUNCTION &head=i->getHead();

        // ...and at each head atom there.
        for( DISJUNCTION::const_iterator j=head.begin();
             j != head.end();
             j++ )
            {
            // If we have a match and it's not the first one, fail; else
            // note the current rule and proceed with the next one.
            if( (*j).getPredItem() == atom.getPredItem() )
                {
                if( ++definitions > 1 )
                    {
                    *rule=0;
                    return false;
                    }

                *rule=&*i;
                break;
                }
            }
        }

    return definitions == 1;
    }

bool GROUNDING::ignoreConstraint(
//
// If a constraint has the following properties, it can never be violated
// and thus we can ignore it during grounding:
//  o The constraint consists of at least two IDB literals and
//    no IDB predicate appears more than once in the constraint.
//  o These IDB literals are defined by means of just one rule (not one
//    per literal, but one for all literals).
//  o All atoms in the head of that rule have the same parameter list.
//  o All atoms in the constraint have the same parameter list.
//
    const CONSTRAINT &c ) const
    {
    if( GTraceLevel >= 1 )
        cdebug << "  Do we really need this?";

    // Find the first IDB literal in the positive part.
    CONSTRAINT::const_iterator j=c.pos_begin();
    while( j != c.pos_end()  &&  ! (*j).isIDB() )
        j++;

    // If there is none, then the constraint consists of EDB or built-in
    // literals only and the optimization does not apply.
    if( j == c.pos_end() )
        {
        if( GTraceLevel >= 1 )
            cdebug << "  No IDB literal found." << endl;

        return false;
        }

    // Remember the first IDB literal found as well as its component.
    const LITERAL &firstIDB=*j;
    const unsigned comp= NonGroundDepGraph.getAtomComponent(static_cast<const ATOM>(*j),false);

    // If an IDB literal does not have a valid component, it does not occur
    // in (the head of) a rule any longer and thus never becomes true and
    // the constraint cannot be violated at all.
    if( comp == INVALID_COMPONENT_INDEX )
        {
        if( GTraceLevel >= 1 )
            cdebug << "  IDB literal with invalid component." << endl;

        return true;
        }

    const RULE *rule;

    if( ! findSingleDefinition(NonGroundDepGraph.getComponentRules(comp),*j,&rule) )
        {
        if( GTraceLevel >= 1 )
            cdebug << "  IDB literal which is not `single definition'." << endl;
        return false;
        }

    unsigned nIDB=1;

    for(j++; j != c.pos_end(); j++)
        {
        if( (*j).isIDB() )
            {
            nIDB++;

            unsigned comp2=NonGroundDepGraph.getAtomComponent(*j,false);

            if( comp2 == INVALID_COMPONENT_INDEX )
                {
                if( GTraceLevel >= 1 )
                    cdebug << "  IDB literal with invalid component." << endl;

                return true;
                }

            // Verify that the parameter lists match.
            if( compare(firstIDB.getParams(),(*j).getParams()) )
                {
                if( GTraceLevel >= 1 )
                    cdebug << "  Param mismatch in constraint." << endl;

                return false;
                }


            // Also check that the current predicate does not appear more
            // than once. We do this by comparing with all previous ones.
            for(CONSTRAINT::const_iterator k=c.pos_begin(); k != j; k++)
                if( (*k).getPredItem() == (*j).getPredItem() )
                    {
                    if( GTraceLevel >= 1 )
                        cdebug << "  IDB literals with same predicate."
                               << endl;

                    return false;
                    }

            const RULE *rule2;
            if( ! findSingleDefinition(NonGroundDepGraph.getComponentRules(comp),*j,&rule2) )
                {
                if( GTraceLevel >= 1 )
                    cdebug << "  IDB literal which is not `single definition'."
                           << endl;

                return false;
                }
            else if( rule2 != rule )
                {
                if( GTraceLevel >= 1 )
                    cdebug << "  IDB literals defined by different rules."
                           << endl;

                return false;
                }
            }
        }

    if( nIDB == 1 )
        {
        if( GTraceLevel >= 1 )
            cdebug << "  Just one IDB literal." << endl;

        return false;
        }

    // Finally check that all parameter lists of the single definition
    // rule match. We do this by comparing all atoms with the first one.
    const DISJUNCTION &head=(*rule).getHead();
    for( DISJUNCTION::const_iterator i=head.begin()+1;
         i != head.end();
         i++ )
        {
        if( compare((*head.begin()).getParams(),(*i).getParams()) )
            {
            if( GTraceLevel >= 1 )
                cdebug << "  Param mismatch in defining rule." << endl;

            return false;
            }
        }

    if( GTraceLevel >= 1 )
        cdebug << "  At the end with no previous condition matching." << endl;

    return true;
    }

/** Check whether aggregate literal is false.
 * @return true if the aggregate literal is false, i.e.
 * if a positive literal represents a violated aggregate
 * or if a negative literal represents a satisfied
 * aggregate, false else. 
 * If the aggregate is used as an assignment bind the guards 
 * with the value of the aggregate if the auxiliary atom is solved,
 * exit otherwise.
 */
bool GROUNDING::checkViolatedAggregate(
    const LITERAL &lit,
    TERM v2c[] ) 
    {
    if( GTraceLevel >= 1 )
        cdebug << "Check whether aggregate is violated."
               << endl; 

    AGGREGATEATOM &a = lit.getAggregate();

    if( a.getLowerGuardAsVar().isVar()
       && ! v2c[a.getLowerGuardAsVar().getVar()].isNull() )
        {
        // This aggregate is not an assignment, as the
        // variable has been bound before.
        a.setAssignment(false);
        }

    if( a.isAssignment() )
        {
        if( solved[a.getAuxAtom().getIndex()] )
            {
            assert( a.getLowerBound() == a.getUpperBound() );

            // Assign count value to the guards.
            v2c[a.getLowerGuardAsVar().getVar()]=TERM(a.getLowerBound(),0);
            } 
        else 
            {
            cerr << "Aggregate function cannot be applied on "
                    "disjunctive or unstratified predicates, "
                    "if it is used in an assignment with an "
                    "unbounded variable." << endl;       					 
            status = AggregateInvalidUse; 
            }
        }

    // Check whether the aggregate is violated.
    else if( a.isViolated(v2c) )
        {
        if( ! lit.isNegative() )
            {
            if( GTraceLevel >= 1 )
                cdebug<< "Positive aggregate is violated."
                      << endl;
            return true;
            }
        }
    // Check whether the aggregate is satified.
    else if( a.isSatisfied(v2c) )
        {        
        if( lit.isNegative() )
            {
            if( GTraceLevel >= 1 )
                cdebug<< "Negative aggregate is satisfied."
                      << endl;
            return true;
            }
        }
    
    return false;
    }

/** Check whether literal L is already false.
 * Uses globals: I, vTOc
 * @param L the LITERAL to examine.
 * @param deltaNF the database to consider in addition to I.
 * @return whether literal L is false in I or deltaNF.
 */
bool GROUNDING::negativeLiteralIsFalse(
    const LITERAL &L,
    const INTERPRET *deltaNF ) const
    {
    const ATOM& tmp=ATOM(L,vTOc);
    
    if( (L).isEDB() )
        {
        if( EDB.contains(tmp) )
            {
            if( GTraceLevel >= 2 )
                cdebug << "   Negative literal " << tmp 
                       << " always false in the body of rule ";
            return true;
            }
        }
    else
        {
        if( I.isTrue(tmp) || (deltaNF && deltaNF->isTrue(tmp)) )
            {
            if( GTraceLevel >= 2 )
                cdebug << "   Negative literal " << tmp 
                       << " always false in the body of rule ";
            return true;
            }
        }
    
    return false;
    }

/** Check whether one of the ATOMs in the given DISJUNCTION is already true.
 * @param Head the ATOMs to examine.
 * @params deltaNF and newNF the databases to consider in addition to I.
 * @return whether an element of Head is true in I or deltaNF or newNF.
 */
bool GROUNDING::trueHead(
    const DISJUNCTION &Head, 
    const INTERPRET &deltaNF, 
    const INTERPRET &newNF,
    const TERM vTOc[] ) const 
    {
    for( DISJUNCTION::const_iterator H = Head.begin();
                                     H != Head.end();
                                     H++ )
        {
        // each head atom is an IDB atom
        const ATOM& tmp=ATOM(*H,vTOc);
        if( I.isTrue(tmp) || deltaNF.isTrue(tmp) || newNF.isTrue(tmp) )
            {
            if( GTraceLevel >= 2 )
                cdebug << "   Atom " << tmp 
                       << " already true in the head of rule ";
            return true;
            }
        }
    return false;
    }

/** Return true if the the predicates appearing in the aggregate atom are
 * recursive with the head of the rule. Return false, otherwise.
 */
bool GROUNDING::hasUnstratifiedAggregates(
    const RULE &R,
    unsigned i ) const
    {
    if ( R.hasBody() )
        {
        for( CONJUNCTION::const_iterator B = R.getBody()->begin();
             B != R.getBody()->end();
             B++ )
            {
            if( (*B).isAggregate() )
                {
                if( NonGroundDepGraph.getAtomComponent( (*B).
                    getAggregate().getAuxAtom(), false ) >= i )
                    {
                    cerr << "Error on rule " << R
                         << " : the predicats appearing in the aggregate"
                            " atom cannot be recursive with the head of"
                            " the rule." << endl;
                                                         
                    return true;
                    }
                }
            }
        }
                                                                                
    return false;
    }


/** Return true if the the predicates appearing in the aggregate atom are
 * recursive with the head of the rule. Return false, otherwise.
 * @param g the ATOM appearing in the aggregate.
 * @param i the current component in the dependency graph. It is set to 
 * UINT_MAX for the aggregates appearing in constraints or weakConstraints.  
 *
 * Note that the code concerning recursive aggregates is not currently used.
 */
bool GROUNDING::isRecursiveAggregate( 
    const ATOM &g, 
    unsigned i ) const
    {
    return( i != UINT_MAX 
            && NonGroundDepGraph.getAtomComponent(g,false) >= i ); 
    }

/** Build a new cell for a given state.
 * See the declaration of cell for a more thorough discussion of its contents.
 * @param lit the literal in processing.
 * @param varTOconst the current variable mappings.
 * @param I the ground atoms computed in before the last iteration.
 * @param deltaNF the ground atoms computed in the last iteration.
 * @param deltaSize the number of ground atoms computed in the last iteration.
 * @param range the interpretations range of the literal.
 * @return the newly constructed cell.
 */
cell::cell(
    const LITERAL &lit, 
    const TERM varTOconst[],
    INTERPRET &I, 
    INTERPRET *deltaNF, 
    unsigned* size, 
    unsigned* deltaSize, 
    INTERPRETATIONS_RANGE range )
    : count1(0), count2(0), interpretationsRange(range)
    {
    type=lit.getType();
    
    switch( type )
        {
        case PREDICATE_NAMES::typeEDB:
            // Match with ground atoms in EDB. 
            pInstances = new TABLE();
            EDB.find(lit, *pInstances);
            break;
        case PREDICATE_NAMES::typeIDB:
            pInstances = new TABLE();
            if ( interpretationsRange == rI ) // Match with atoms in I
                {
                // Look for instances of lit in the positive part of I
                // and set the variable lookingIn in order to remember
                // in which interpretation we are looking.   
                I.findInPositivePart(lit, *pInstances);
                lookingIn = PositiveI;
                // If the positive part is empty, move the pointer to the
                // undefined part of I and update lookingIn.
                if ( pInstances->atEnd() )
                    {
                    I.findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedI;
                    }
                }
            else if ( interpretationsRange == rDeltaNF )
                // Match with ground atoms in DeltaNF.
                {
                // Look for instances of lit in the positive part of 
                // deltaNF and set the variable lookingIn accordingly. 
                deltaNF->findInPositivePart(lit, *pInstances);
                lookingIn = PositiveDeltaNF;
                // If the positive part is empty, move the pointer to the
                // undefined part of deltaNF and update lookingIn.
                if ( pInstances->atEnd() )
                    {
                    deltaNF->findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedDeltaNF;
                    }
                }
            else // interpretationsRange == rI_DeltaNF
                // Match with ground atoms in I and in DeltaNF.
                {
                // For literals having rI_DeltaNF both I and deltaNF
                // have to be used. Start by setting to pointer to the
                // positive part of I; if it is empty, move the pointer
                // to the undefined part of I; if also this is empty,
                // move again the pointer to the positive part of deltaNF
                // and finally to the undefined part of deltaNF. Set
                // the variable lookingIn accordingly.
                I.findInPositivePart(lit, *pInstances);
                lookingIn = PositiveI;
                if ( pInstances->atEnd() )
                    {
                    I.findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedI;
                    if ( pInstances->atEnd() )
                        {
                        deltaNF->findInPositivePart(lit, *pInstances);
                        lookingIn = PositiveDeltaNF;
                        if (pInstances->atEnd())
                            {
                            deltaNF->findInUndefinedPart(lit, *pInstances);
                            lookingIn = UndefinedDeltaNF;
                            }
                        }
                    }
                }   
            break;   		    
        default:
            assert( type==PREDICATE_NAMES::typeBuiltin 
                    || type==PREDICATE_NAMES::typeAggregate );
            pInstances=0;
        }
    if( GTraceLevel >= 2 )
        {
        if( pInstances )
            switch( interpretationsRange )
                {
                case rI:
                    cdebug << "      cardinality=" 
                           << size[lit.getIndex()] << endl;
                    break;
                case rDeltaNF:
                    cdebug << "      cardinality=" 
                           << deltaSize[lit.getIndex()] << endl;
                    break;
                case rI_DeltaNF:
                    cdebug << "      cardinality in I=" 
                           << size[lit.getIndex()] <<" --- "
                           << " cardinality in deltaNF=" 
                           << deltaSize[lit.getIndex()] << endl;
                    break;
                default: 
                    assert(0);
                }
        }
    initializeStatus(lit, varTOconst);
    }

/** Copy an existing cell.
 * @return the new copy.
 */
cell::cell( const cell& c )
    : type(c.type),
      count1(c.count1),
      count2(c.count2),
      interpretationsRange(c.interpretationsRange),
      lookingIn(c.lookingIn)
    {
    if( c.pInstances )
        pInstances = new TABLE(*c.pInstances); 
    else 
        pInstances=0;
    
    copy(c.Status.begin(),c.Status.end(),inserter(Status,Status.end()));
    }

/** Destroy a cell.
 */
cell::~cell()
    {
    if( pInstances )
       delete pInstances;
    }

void cell::operator=( const cell& )
    {
    assert( 0 );
    }

/** Initialize the Status.
 * Status contains true for each variable in the literal that is unbound.
 * This is used in restoring the current state later in the computation.
 * @param lit the literal for which parameter status is collected.
 * @param varTOconst the variable mapping.
 */
void cell::initializeStatus(
    const LITERAL &lit,
    const TERM varTOconst[] )
    {
    assert( Status.empty() );

    if( lit.isRegularAtom() && lit.getParams() )
	{
	for( TERMS::const_iterator i=(lit.getParams())->begin();
             i != (lit.getParams())->end();
             i++ )
	    {
	    if ( ((*i).isVar()) && (varTOconst[(*i).getVar()]).isNull() )
		Status.push_back(true);
	    else
		Status.push_back(false);
	    }
	}
    }

/** Check whether all ground atoms have been considered.
 * Should not be called for aggregates and builtins.
 * @return whether there are no further ground atoms to match.
 */
bool cell::atEnd()
    {
    assert( type != PREDICATE_NAMES::typeBuiltin 
            && type != PREDICATE_NAMES::typeAggregate );
        
    // This function is always called immediately after function
    // goToNextSubstAtom or after the constructor of cell.
    // These other two functions move the pointer pInstances to the 
    // next ground atom in the current part of interpretation, and if no next 
    // atom exists there, the pointer is moved to the next interpretation 
    // (or part of interpretation) to be considered. 
    // For that reason, in order to check whether we examined all the possible
    // ground atoms, we can simply check that the pointer has 
    // reached the end of the current set of ground atoms. 
    assert( !pInstances->atEnd() 
            || (( interpretationsRange == rI && lookingIn == UndefinedI)
                || (lookingIn == UndefinedDeltaNF) 
                || type == PREDICATE_NAMES::typeEDB) );
    return ( pInstances->atEnd() );
    }

/** Provide access to the current ground atom.
 * @return the current ground atom - to be matched with the literal.
 */
cell::TABLE* cell::getSubstAtom()
    {
    return pInstances;
    }

/** Advance to the next ground atom.
 * The pointer to the interpretation is moved to the next atom. 
 * If no next atom exists there, the pointer is moved to the next 
 * interpretation (or part of interpretation) to be considered according 
 * to the interpretations range and to the variable lookingIn.
 * After the pointer has reached the end of the last interpretation 
 * atEnd() will return true.
 */
void cell::goToNextSubstAtom(
    const LITERAL &lit, 
    INTERPRET &I, 
    INTERPRET *deltaNF)
    {
    // Should not be called for builtins or aggregates!
    assert( type != PREDICATE_NAMES::typeBuiltin 
            && type != PREDICATE_NAMES::typeAggregate );
    if ( type == PREDICATE_NAMES::typeEDB )
        {
        // For EDB predicates the pointer can move only on
        // the EDB atomset.
        assert( pInstances );
        (*pInstances)++;
        }
    else
        {
        assert( type == PREDICATE_NAMES::typeIDB );
        // For IDB predicates the interpretation to consider are I 
        // and/or deltaNF according to the interpretations range.
        // Move the pointer to the next atom.
        (*pInstances)++;
        switch ( interpretationsRange )
            {
            // range rI: atoms can be matched with ground atoms in the 
            // positive and undefined parts of I. 
            case rI:
                // If the pointer has reached the end of the positive 
                // part of I move the pointer to the undefined part
                // and update lookingIn.  
                if ( pInstances->atEnd() && (lookingIn == PositiveI) )
                    {
                    I.findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedI;
                    }
                break;
            // range rDeltaNF: atoms can be matched with ground atoms in the 
            // positive and undefined parts of deltaNF. 
            case rDeltaNF:
                // If the pointer has reached the end of the positive 
                // part of deltaNF move the pointer to the undefined part
                // and update lookingIn.  
                if ( pInstances->atEnd() && (lookingIn == PositiveDeltaNF) )
                    {
                    deltaNF->findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedDeltaNF; 
                    }
                break;
            // range rI_DeltaNF: atoms can be matched with ground atoms in 
            // the positive and undefined parts of I and deltaNF.   
            case rI_DeltaNF:
                // If the pointer has reached the end of the positive 
                // part of I move the pointer to the undefined part of I
                // and update lookingIn.  
                if (pInstances->atEnd() && (lookingIn == PositiveI) )
                    {
                    I.findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedI;
                    }
                // If the pointer has reached the end of the undefined 
                // part of I move the pointer to the positive part of deltaNF
                // and update lookingIn.  
                if( pInstances->atEnd() && (lookingIn == UndefinedI) )
                    {
                    deltaNF->findInPositivePart(lit, *pInstances);
                    lookingIn = PositiveDeltaNF;
                    }    
                // If the pointer has reached the end of the positive  
                // part of deltaNF move the pointer to the undefined part of 
                // deltaNF and update lookingIn.  
                if ( pInstances->atEnd() && (lookingIn == PositiveDeltaNF) )
                    {
                    deltaNF->findInUndefinedPart(lit, *pInstances);
                    lookingIn = UndefinedDeltaNF; 
                    }
                break;
            default:
                assert(0); // range is neither rI, rDeltaNF, nor rI_DeltaNF
            }
        }
    }

BACKJUMPING::BACKJUMPING(
    const OrderedBody &OrdB, 
    bool *headVars): 
    bodySize(OrdB.getTotalSize()),binder(n_par),closestBinderVars(bodySize),
    closestBinderDS(bodySize),closestBinderOV(bodySize),
    depSet(bodySize), outputVars(n_par), globalVarsInAggr(bodySize)
    {
    if ( headVars )
        copy(headVars,headVars+n_par,outputVars.begin());
    else fill(outputVars.begin(),outputVars.end(),false);

    fill(binder.begin(),binder.end(),-1);   
    fill(closestBinderVars.begin(),closestBinderVars.end(),-1);
    fill(closestBinderDS.begin(),closestBinderDS.end(),-1);
    fill(closestBinderOV.begin(),closestBinderOV.end(),-1);

    for( vector< vector<bool> >::iterator it = depSet.begin();
         it < depSet.end(); 
         it++ )
        {
        it->reserve(n_par);
        it->insert(it->end(),n_par, false);
        }

    for( vector< vector<bool> >::iterator it = globalVarsInAggr.begin();
         it < globalVarsInAggr.end(); 
         it++ )
        {
        it->reserve(n_par);
        it->insert(it->end(), n_par, false);
        }

    lastRegularAtom = -1;
    greatestBinderOV = -1;
    }

/** Given a literal, compute the binders of its variables.
 * Fill binder and alreadyBound vectors.
 * @param OrdB the body of the rule to examine. 
 * @param i the index of the current literal. 
 * @param alreadyBound stores variables appearing in the literals  
 * preceding the current one in OrdB.
 */
void BACKJUMPING::computeBinders(
    const OrderedBody &OrdB,
    const int &i,
    vector<bool> &alreadyBound)
    {
    assert( OrdB.getLiteral(i)->getParams() ); 
    for( TERMS::const_iterator j = OrdB.getLiteral(i)->getParams()->begin();
         j != OrdB.getLiteral(i)->getParams()->end(); 
         j++ )
        if( j->isVar() && ! alreadyBound[j->getVar()] )
            {
            alreadyBound[j->getVar()] = true;
            binder[j->getVar()] = i;
            
            if( GTraceLevel >= 2 )
                cdebug << " The binder of " << j->getVar() << " is "
                       << *(OrdB.getLiteral(binder[j->getVar()])) << endl;
            }
    }

// Auxiliary function used to compute the closest binder of a literal.
void BACKJUMPING::updateClosestBinder(
    vector<int>& CB, 
    const int &i, 
    const unsigned &indexVar)
    {
    if( binder[indexVar] > CB[i] 
        && binder[indexVar] != i ) 
        CB[i] = binder[indexVar];
    }

/** Given a literal, compute its closest binder, i.e. the closest literal
 * binding one of its variable. 
 * @param OrdB the body of the rule to examine.
 * @param i the index of the current literal.
 */
void BACKJUMPING::computeClosestBinderVars(
    const OrderedBody &OrdB,
    const int &i)
    {
    LITERAL lit = *OrdB.getLiteral(i);
    if( lit.isRegularAtom() && lit.getParams() )
        {
        for( TERMS::const_iterator j = lit.getParams()->begin();
             j != lit.getParams()->end(); 
             j++ )
            if( j->isVar() )
                updateClosestBinder(closestBinderVars, i, j->getVar());
        }
    else 
        for( int j = 0; j < n_par; j++)
            if( globalVarsInAggr[i][j] )
                updateClosestBinder(closestBinderVars, i, j);
    }

/** Auxiliary function used to fill some BACKJUMPING structures.
 * Given a literal and a vector of bools with size n_par, for each variable
 * i of the literal, set vec[i] to true. For aggregates consider only
 * global variables.
 */
void  BACKJUMPING::addVarsTo(
    vector<bool>& vec, 
    const int &currLit,
    const OrderedBody &OrdB)
    {
    const LITERAL &lit = *OrdB.getLiteral(currLit);
    if( lit.isRegularAtom() && lit.getParams() )
        {
        for( TERMS::const_iterator j = lit.getParams()->begin();
             j !=  lit.getParams()->end(); 
             j++ )
            if( j->isVar() ) 
                vec[j->getVar()] = true;
        }
    else if( ! lit.isRegularAtom() )
        for( unsigned i = 0; i < n_par; i++ )
            if( globalVarsInAggr[currLit][i] )
                vec[i] = true;
    }

// Determine the global variables of aggregate literals. 
// Note that this function supposes that aggregates are
// placed after regular literals in the body.
void BACKJUMPING::computeGlobalVarsInAggregates(const OrderedBody &OrdB)
    {
    vector<bool> alreadyConsidered(n_par);
    fill(alreadyConsidered.begin(), alreadyConsidered.end(), false);
    for( int i = 0; i < bodySize; i++ )
        {
        const LITERAL &lit = *OrdB.getLiteral(i);
        // Traverse the body and set alreadyConsidered[v] to
        // true if variable v appears in a regular literal of the body. 
        if( lit.isRegularAtom() && lit.getParams() )
            {
            for( TERMS::const_iterator it = lit.getParams()->begin();
                 it != lit.getParams()->end(); 
                 it++)
                if( it->isVar() )
                    alreadyConsidered[it->getVar()] = true;
            }
        // An aggregate variable v is global if it appears also in
        // some regular literal, that is if alreadyConsidered[v] is true.
        else if( lit.isAggregate() )
            {
            for( TERMS::const_iterator t = 
                     lit.getAggregate().getAuxAtom().getParams()->begin();
                 t != lit.getAggregate().getAuxAtom().getParams()->end(); 
                 t++ )
                if( t->isVar() )
                    if( alreadyConsidered[t->getVar()] )
                        globalVarsInAggr[i][t->getVar()] = true;
            const TERM &lg = lit.getAggregate().getLowerGuardAsVar();
            if ( lg.isVar() && alreadyConsidered[lg.getVar()] )
                globalVarsInAggr[i][lg.getVar()] = true;
            const TERM &ug = lit.getAggregate().getUpperGuardAsVar();
            if ( ug.isVar() && alreadyConsidered[ug.getVar()] )
                globalVarsInAggr[i][ug.getVar()] = true;
            }
        }
    }


// The next four auxiliary functions are used to determine if two literals 
// have some variable in common. 
// For aggregates consider only global variables. We access to aggregates by 
// means of their indices because we don't need to consider the whole aggregate 
// but rather its global variables that are stored in the structure 
// globalVarsInAggr.

// Intersection between a regular literal and an aggregate.
bool BACKJUMPING::intersectionR_A( 
    const LITERAL &lit1, 
    const unsigned &i2)
    {
    for( TERMS::const_iterator it = lit1.getParams()->begin(); 
         it != lit1.getParams()->end(); 
         it++ )
        if( it->isVar() && globalVarsInAggr[i2][it->getVar()] )
            return true;
    return false;
    }

// Intersection between two regular literals.
bool BACKJUMPING::intersectionR_R( 
    const LITERAL &lit1, 
    const LITERAL &lit2)
    {
    for( TERMS::const_iterator j = lit1.getParams()->begin();
         j < lit1.getParams()->end(); 
         j++ )
        for( TERMS::const_iterator k = lit2.getParams()->begin();
             k < lit2.getParams()->end();
             k++ )
            if( j->isVar() && *j == *k )
                return true;
    return false;
    }

// Intersection between two aggregates.
bool BACKJUMPING::intersectionA_A(
    const unsigned &i1, 
    const unsigned &i2)
    {
    for( unsigned k = 0; k < n_par; k++ )
        if( globalVarsInAggr[i1][k] 
                && globalVarsInAggr[i2][k] )
                return true;
    return false;
    }

// Distinguish between the three possible types of intersection
// among literals and call the appropriate function.
bool BACKJUMPING::intersection(
    const OrderedBody &OrdB, 
    const unsigned &i1, 
    const unsigned &i2)
    {
    const LITERAL &lit1 = *OrdB.getLiteral(i1);
    const LITERAL &lit2 = *OrdB.getLiteral(i2);
    if( lit1.isRegularAtom() && lit2.isRegularAtom() )
        {
        if ( lit1.getParams() && lit2.getParams() )
            return intersectionR_R(lit1, lit2);
        }
    else if( ! lit1.isRegularAtom() && ! lit2.isRegularAtom() )
        return intersectionA_A(i1,i2);
    else if( lit1.isRegularAtom() && ! lit2.isRegularAtom() )
        return intersectionR_A(lit1, i2);
    else
        return intersectionR_A(lit2, i1);
    return false;
    }

// Compute the dependencies set of a given literal with index currLit. 
// Add to depSet its variables, examine the literals having 
// index > currLit and push in a vector those having non-empty intersection 
// with the literal with index currLit. For each literal pushed in the vector 
// repeat the same operations done for currLit starting from the
// addition of its variables to depSet.
void BACKJUMPING::computeDepSet(
    const OrderedBody &OrdB, 
    const int &currLit)
    {
    assert((*OrdB.getLiteral(currLit)).isRegularAtom());

    vector<unsigned> depLiterals;
    depLiterals.reserve(bodySize);
    vector<bool> alreadyConsidered(bodySize);
    fill(alreadyConsidered.begin(), alreadyConsidered.end(), false);
    
    // Add the variables of currLit to its dependencies set.
    addVarsTo(depSet[currLit], currLit, OrdB);
    depLiterals.push_back(currLit);

    for( vector<unsigned>::const_iterator it = depLiterals.begin();
         it != depLiterals.end();
         it ++ )
        for( int h = *it + 1; h < bodySize; h++ )
            if( intersection(OrdB, *it, h) )
                if( ! alreadyConsidered[h] )
                    {
                    addVarsTo(depSet[currLit], h, OrdB);
                    depLiterals.push_back(h);
                    alreadyConsidered[h] = true;
                    }
    
    if( GTraceLevel >= 2)
        {
        cdebug << " The dependencies set of " << *OrdB.getLiteral(currLit)
               << " is ";
        for( unsigned i = 0; i < n_par; i++)
            if( depSet[currLit][i] )
                cdebug << i << ", "; 
        cdebug << endl;
        }
    }

/** Determine whether the variables of a given literal currLit are  
 * output vars and add them to the vector outputVars using function addVarsTo.
 * The variables of currLit are output vars if
 * either currLit is unsolved or if its dependencies set contains already
 * some output variable.
 * For aggregates add the global variables of the auxAtom only if it 
 * is unsolved.
 * @param OrdB the body of the rule to examine.
 * @param currLit the index of the current literal.
 * @param solved stores whether predicates are completely defined.
 */
void BACKJUMPING::computeOutputVars( 
    const OrderedBody &OrdB, 
    const int &currLit, 
    const bool *solved )
    {
    const LITERAL &lit = *OrdB.getLiteral(currLit);
    if( lit.isRegularAtom() && lit.getParams() )
        {
        if( ! solved[lit.getIndex()] )
            addVarsTo(outputVars, currLit, OrdB);
        else if( ! lit.isNegative() )
            {
            bool found = false;
            for( unsigned j = 0; j < n_par && !found; j++ )
                if( depSet[currLit][j] && outputVars[j] )
                    {
                    addVarsTo(outputVars, currLit, OrdB);
                    found = true;
                    }     
            }
        }
    else if( lit.isAggregate() ) 
        if( ! solved[lit.getAggregate().getAuxAtom().getIndex()] )
            addVarsTo(outputVars, currLit, OrdB);
    }      		 

// Compute all the structures needed by the BJ algorithm.
// This function is called exactly once for constraints and exit rules,
// but is called more than once in case of recursive rules. This is because, 
// when recursive rules are reordered, structures have to be recomputed,
// because they depend on the order of literals in the body. 
// The only structure that remains always the same is globalVarsInAggr,
// and so, in order to avoid to recompute it again, we use the bool 
// recRuleReordered.  
void  BACKJUMPING::computeStructures(
    const  OrderedBody &OrdB,
    const bool *solved,
    const TYPECLAUSE &type,
    const bool &recRuleReordered )
    {
    vector<bool> alreadyBound(n_par);
    fill(alreadyBound.begin(), alreadyBound.end(), false);
    
    // Determine the global variables of the aggregates
    // and store them in an appropriate structure.
    if( !recRuleReordered ) 
        computeGlobalVarsInAggregates(OrdB);
    
    for( int i = 0; i < bodySize; i++ )
        {
        const LITERAL &lit = *(OrdB.getLiteral(i));
        
	// Compute the dependencies set of all the regular positive 
        // literals. For negative literals and aggregates, the depSet is 
        // useless, because we look at the depSet when there is a failure 
        // while looking for another match and, in this case, there is no 
        // possibility that the literal to match again is negative or 
        // aggregate.
        // Furthermore, compute the binders of all the variables appearing 
        // in OrdB and update lastRegularAtom.
        if( lit.isRegularAtom() && lit.getParams() )
            {
            if( ! lit.isNegative() )
                {
                lastRegularAtom = i;
                computeDepSet(OrdB, i);
                }
            computeBinders(OrdB, i, alreadyBound);
            }
        
        // Compute the closest binder of the variables of each literal.
        computeClosestBinderVars(OrdB, i);

        // Compute the output variables. 
        if( type != weakConstraint )
            computeOutputVars(OrdB, i, solved);
        // If we are instantiating a weak constraint, 
        // all the variables must be ouptut variables 
        // (this is because, we can not avoid the generation of 
        // instances of the weak, otherwise we risk to have wrong
        // cost of the models).
        else if( lit.isRegularAtom() )
            addVarsTo(outputVars, i, OrdB);

        // For each regular positive literal compute the closest binder
        // of its depSet and of outputVars. Disregard the first literal
        // (because it has no preceding literals), 
        // the negative literals, and aggregates.
        if( i != 0 && ! lit.isNegative() && ! lit.isAggregate() )
            for( unsigned j = 0; j < n_par; j++ )
                {
		if( depSet[i][j] )
                    // Compute the closest binder of the 
                    // dependencies set of i-th literal.
                    updateClosestBinder(closestBinderDS, i, j);
		if( outputVars[j] )
                    // Compute the closest binder of the 
                    // output vars.
                    updateClosestBinder(closestBinderOV, i, j);
                }
        if( GTraceLevel >= 2 )
            {
            cdebug << " The closest binder of The DepSet of "
                   << lit << " is ";
            if( closestBinderDS[i] != -1 )
                cdebug << *OrdB.getLiteral(closestBinderDS[i]) << endl;
            else
                cdebug << " NULL " << endl;
            cdebug <<" The closest binder of The Output Vars of "
                   << lit << " is ";
            if( closestBinderOV[i] != -1 )
                cdebug << *OrdB.getLiteral(closestBinderOV[i]) << endl;
            else
                cdebug << " NULL " << endl;
            cdebug << " The closest binder of " << *OrdB.getLiteral(i)
                   << " is ";
            if( closestBinderVars[i] != -1 )
                cdebug << *OrdB.getLiteral(closestBinderVars[i]) << endl;
            else
                cdebug << " NULL " << endl;
            cdebug << endl;
            }
        }
    if( GTraceLevel >= 2 )
        {
        cdebug << " output vars: ";
        for( unsigned k = 0; k < n_par; k++ )
            if( outputVars[k] )
                cdebug << "  _" << k ;
            cdebug << endl << endl;
        }
    
    greatestBinderOV = computeGreatestBinderOV(OrdB);
    if( GTraceLevel >= 2 )
        {
        if( greatestBinderOV > -1 )
            cdebug << " The last literal in the body binding "
                   << "an output variable is "
                   << *OrdB.getLiteral(greatestBinderOV) << endl;
        else
            cdebug << " There are no output variables " << endl;
        }
   }

// Return the binder of a given variable.
inline int BACKJUMPING::getBinder(const int &i) const
    {
    return binder[i];
    }

// Return the Closest Binder of a given literal.
inline int BACKJUMPING::getClosestBinderVars(
    const OrderedBody &OrdB,
    CONSTRAINT::const_iterator pLiteral)  const
    { 
    return closestBinderVars[OrdB.getIndexLiteral(pLiteral)];
    }						    

// Return the Closest Binder of the dependencies set of a given literal.
inline int BACKJUMPING::getClosestBinderDS(
    const OrderedBody &OrdB, 
    CONSTRAINT::const_iterator pLiteral) const
    {
    return closestBinderDS[OrdB.getIndexLiteral(pLiteral)];
    }

// Return the Closest Binder of output vars of a 
// given literal.
inline int BACKJUMPING::getClosestBinderOV(const unsigned &i) const
    {
    return closestBinderOV[i];
    }

int BACKJUMPING::computeGreatestBinderOV(const OrderedBody &OrdB) const
    {
    // CSB is always a regular atom and it is computed and
    // returned by function getClosestBinderOV. 
    // However this function does not consider the case in 
    // which the CSB is the last regular literal in the body. 
    // This possibility is handled by the following if expressions.
    if( lastRegularAtom != -1 )
        {
        const LITERAL &lit = *OrdB.getLiteral(lastRegularAtom);
        for( TERMS::const_iterator it =lit.getParams()->begin();
             it != lit.getParams()->end();
             it++)
            if( it->isVar() 
                && outputVars[it->getVar()] 
                && binder[it->getVar()] == lastRegularAtom )
                return lastRegularAtom;
    
        return getClosestBinderOV(lastRegularAtom);
        }
    return -1;
    }

inline int BACKJUMPING::getGreatestBinderOV() const
    {
    return greatestBinderOV;
    }

// Return if a variable is an output var.
inline bool BACKJUMPING::isOutputVar(const int  &i) const
    {
    return outputVars[i];
    }

// Reset the bj structures.  
void BACKJUMPING::resetStructures(bool *headVars) 
    { 
    assert( headVars );
    copy(headVars, headVars+n_par, outputVars.begin());

    fill(binder.begin(), binder.end(), -1);

    lastRegularAtom = -1;
    fill(closestBinderVars.begin(), closestBinderVars.end(), -1);
    fill(closestBinderDS.begin(), closestBinderDS.end(), -1); 
    fill(closestBinderOV.begin(), closestBinderOV.end(), -1);
    
    for( int i= 0; i != bodySize; i++)
        fill(depSet[i].begin(), depSet[i].end(), false);

    greatestBinderOV = -1;
    }

// Local Variables:
// c-file-style: "dl"
// fill-column: 78
// End:
