//////////////////////////////////////////////////////////////////////////////
// grule.h

#ifndef GRULE_H
#define GRULE_H

template<>
class TRULE<GATOM,GLITERAL>
//////////////////////////////////////////////////////////////////////////////
/// Template specialisation
//////////////////////////////////////////////////////////////////////////////
    {
public:
    // SATISFIED_BY_HEAD_AND_BODY == SATISFIED_BY_BODY | SATISFIED_BY_HEAD
    // should be guaranteed.
    enum SATISFACTION { UNSATISFIED=0, SATISFIED_BY_BODY=1,
                        SATISFIED_BY_HEAD=2, SATISFIED_BY_HEAD_AND_BODY=3 };

private:
    TDISJUNCTION<GATOM>* head;
    TCONJUNCTION<GLITERAL>* body;

    // A counter used by the computeGUS algorithm [see (generate.C)].
    // NOTE: it is not handled by constructors, but only by public
    // methods posBodyUndefInCSet(), posBodyUndefInCIncr(),
    // posBodyUndefInCDecr(), posBodyUndefInC(), because its value has
    // sense only during a single execution of computeGUS().
    unsigned posBodyUndefInComponent;

    // This boolean is used actually only by the computeGUS algorithm,
    // to remember if we have to take care about it during the current
    // computation of the GUS on the current component.
    bool usefulForGUS;

    void init(const TDISJUNCTION<GATOM>* head2, const TCONJUNCTION<GLITERAL>* body2)
        {
        if( head2 )
            head=new TDISJUNCTION<GATOM>(head2->begin(), head2->end());
        else
            head=0;

        if( body2 )
            body=new TCONJUNCTION<GLITERAL>(body2->begin(), body2->end());
        else
            body=0;
        }

    void erase()
        {
        if( head )
            delete head;
        if( body )
            delete body;
        }
public:

    // The methods that handle the field posBodyUndefInC.
    unsigned posBodyUndefInC() const
        {
        return posBodyUndefInComponent;
        } // posBodyUndefInC()

    void posBodyUndefInCSet(unsigned value) 
        {
        posBodyUndefInComponent = value;        
        } // posBodyUndefInCSet()

    void posBodyUndefInCIncr() 
        {
        posBodyUndefInComponent++;
        } // posBodyUndefInCIncr()
    
    void posBodyUndefInCDecr() 
        {
        posBodyUndefInComponent--;
        } // posBodyUndefInCDecr()

    // The methods that handle the field usefulForGUS
    void setUsefulForGUS(bool useful=true) 
        {
        usefulForGUS = useful;
        } // setUsefulForGUS()

    bool isUsefulForGUS() const
        {
        return usefulForGUS;
        } // isUsefulForGUS()


    unsigned short satisfied;

    // The undef... counters keep track of how many literals in the
    // corresponding part of the rule are undefined or whose atoms are
    // must-be-true in the current interpretation.
    unsigned short undefHead;
    unsigned short undefPosBody;
    unsigned short undefNegBody;

    // The mbt... counters keep track of how many atoms of the
    // literals in the corresponding part of the rule are must-be-true
    // in the current interpretation. mbtX <= undefX should always
    // hold.
    unsigned short mbtHead;
    unsigned short mbtPosBody;

    // The value of potentiallySupportedAtom is 0 if the rule does not
    // potentially support an atom in the head. This is the case if
    // the body is false, or no atom is true in the head or if more
    // than one atoms (which are not equal) are true in the head.
    // Otherwise, if the body is true and exactly one atom in the head
    // is true, then potentiallySupportedAtom gives the position of the first
    // occurrence of this atom, where the first atom is 1. If b is the
    // begin() iterator of the head, b + potentiallySupportedAtom - 1 gives an
    // iterator to the first occurrence of the potentially supported atom.
    unsigned short potentiallySupportedAtom;

    TRULE()
        {
        assert( 0 );
        }

    TRULE(const TRULE& rule2)
        : usefulForGUS(false),
          satisfied(rule2.satisfied), 
          undefHead(rule2.undefHead), 
          undefPosBody(rule2.undefPosBody), 
          undefNegBody(rule2.undefNegBody),
          mbtHead(rule2.mbtHead), 
          mbtPosBody(rule2.mbtPosBody),
          potentiallySupportedAtom(rule2.potentiallySupportedAtom)
        {
        init(rule2.head,rule2.body);
        }

    TRULE(const TDISJUNCTION<GATOM>* head2, 
          const TCONJUNCTION<GLITERAL>* body2 = 0) 
        : usefulForGUS(false), 
          satisfied(UNSATISFIED),
          undefHead(0), undefPosBody(0), undefNegBody(0),
          mbtHead(0), mbtPosBody(0),
          potentiallySupportedAtom(0)
        {
        init(head2,body2);
        }

    ~TRULE()
        {
        erase();
        }

    TRULE& operator= (const TRULE& rule2)
        {
        if( this != &rule2 )
            {
            erase();
            init(rule2.head,rule2.body);
            satisfied = rule2.satisfied;
            undefHead = rule2.undefHead;
            undefPosBody = rule2.undefPosBody;
            undefNegBody = rule2.undefNegBody;
            mbtHead = rule2.mbtHead;
            mbtPosBody = rule2.mbtPosBody;
            potentiallySupportedAtom = rule2.potentiallySupportedAtom;
            usefulForGUS = rule2.usefulForGUS;
            }
        
        return *this;
        }


    void removeBody()
        {
        if( body )
            {
            delete body;
            body=0;
            }
        }


    bool hasHead() const
        {
        return ( head != 0 );
        }

    bool hasBody() const
        {
        return ( body != 0 );
        }

    const TDISJUNCTION<GATOM> &getHead() const
        {
        assert( head );
        return *head;
        }

    TDISJUNCTION<GATOM> &getHeadForModification() const
        {
        assert( head );
        return *head;
        }

    const TCONJUNCTION<GLITERAL>* getBody() const
        {
        return body;
        }

    TCONJUNCTION<GLITERAL>* getBodyForModification() const
        {
        return body;
        }

    bool isDisjunctive() const
        {
        return ( head->end() - head->begin() > 1 );
        }

    bool potentiallySupports( const GATOM& a ) const
        {
        // If no particular atom is potentially supported, then either
        // all [non-false] head atoms are potentially supported if the
        // rule is not satisfied, or no atom is potentially supported
        // if the rule is satisfied.
        if( potentiallySupportedAtom == 0 )
            return ! satisfied;

        // Otherwise we check whether the given atom is the
        // potentially supported one.
        return ( *(getHead().begin() + potentiallySupportedAtom - 1) == a );
        }

    bool partiallyViolated() const
        {
        // An unsatisfied rule is partially violated if its head is
        // false (and its body undefined), or if its body is true or
        // mbt (and its head undefined).
        assert( ! satisfied );
        return ( partiallyViolatedByHead() || partiallyViolatedByBody() );
        }

    bool partiallyViolatedByHead() const
        {
        // An unsatisfied rule is partially violated by its head if its head
        // is false (and its body undefined).
        assert( ! satisfied );
        return ( undefHead == 0 );
        }

    bool partiallyViolatedByBody() const
        {
        // An unsatisfied rule is partially violated by its body if
        // if its body is true or mbt (and its head undefined).
        assert( ! satisfied );
        assert( undefPosBody >= mbtPosBody );
        return ( undefPosBody - mbtPosBody + undefNegBody == 0 );
        }

    unsigned partialViolationLevel() const
        {
        // An unsatisfied rule is partially violated if its head is
        // false (and its body undefined), or if its body is true or
        // mbt (and its head undefined).
        assert( partiallyViolated() );
        return ( undefHead + undefPosBody - mbtPosBody + undefNegBody );
        }

    bool isBinaryClause() const
        {
        return ( ! satisfied 
                 && (undefHead + undefPosBody - mbtPosBody + undefNegBody 
                     == 2) );
        }
    };

template<>
class TCONSTRAINT<GLITERAL> : public TCONJUNCTION<GLITERAL>
//////////////////////////////////////////////////////////////////////////////
/// Template specialisation
///
/// (No internal flag is required in the ground case.)
//////////////////////////////////////////////////////////////////////////////
    {
public:
    unsigned short satisfied;
    unsigned short undefPosBody;
    unsigned short undefNegBody;

    TCONSTRAINT<GLITERAL>()
        : TCONJUNCTION<GLITERAL>(), satisfied(false)
	{
	}

    TCONSTRAINT<GLITERAL>(const TCONJUNCTION<GLITERAL>& c)
        : TCONJUNCTION<GLITERAL>(c), satisfied(false)
	{
	}

    TCONSTRAINT<GLITERAL>(const_iterator begin2, const_iterator end2)
        : TCONJUNCTION<GLITERAL>(begin2,end2), satisfied(false)
	{
	}

    bool isBinaryClause() const
        {
        return ( ! satisfied && (undefPosBody + undefNegBody == 2) );
        }
    };

#endif

// Local Variables:
// mode: c++
// c-file-style: "dl"
// End:
