//////////////////////////////////////////////////////////////////////////////
// odbc.C
//
// Optional ODBC interface, which allows importing input facts and exporting
// (parts of) models by virtue of the function odbcImport() and the class
// ODBC_Export.

#define WITH_GLOBAL_VARIABLES
#undef  IDENT_LENGTH                      // We must not use this here.
#define ODBC_MAX_STRING_LENGTH   1001     // maximum ID length, not counting trailing '\0'

#include "dl.h"
#include "odbc.h"

vector<ODBC_EXPORT> ODBC_Exports;
vector<ODBC_IMPORT> ODBC_Imports;

#ifdef WITH_ODBC

#include <algorithm>
#include <iostream>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <cmath>
#include <cstring>
#include <sql.h>
#include <sqlext.h>
#include <sqltypes.h>

//////////////////////////////////////////////////////////////////////////////

static bool row_open = false;

static TERMS *current_idents = new TERMS;

static SQLHDBC  V_OD_hdbc;    // connection handle
static SQLHENV  V_OD_Env;     // ODBC environment handle
static SQLHSTMT V_OD_hstmt;

enum ValueConv { U_INT = 0, UT_INT, UR_INT, CONST, Q_CONST, DEFAULT };

//////////////////////////////////////////////////////////////////////////////
bool odbcConnect(
// Allocate needed handles and connect to the database.
//
// Return true upon success, false else.
//
    char *dbName,
    char *userName,
    char *passw )
    {
    //  Allocate Environment handle and register version.
    int res=SQLAllocHandle(SQL_HANDLE_ENV,SQL_NULL_HANDLE,&V_OD_Env);
    if ( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (SQL_HANDLE_ENV, " << res << ")" << endl;
        return false;
        }

    res = SQLSetEnvAttr(V_OD_Env, SQL_ATTR_ODBC_VERSION,
                             (void*)SQL_OV_ODBC3, 0);
    if ( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (SetEnvAttr, " << res << ")"<< endl;
        SQLFreeHandle(SQL_HANDLE_ENV, V_OD_Env);
        return false;
        }

    // Allocate connection handle, set timeout.
    res = SQLAllocHandle(SQL_HANDLE_DBC, V_OD_Env, &V_OD_hdbc);
    if ( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (SQL_HANDLE_DBC, " << res << ")" << endl;
        SQLFreeHandle(SQL_HANDLE_ENV, V_OD_Env);
        return false;
        }
    SQLSetConnectAttr(V_OD_hdbc, SQL_LOGIN_TIMEOUT, (SQLPOINTER *)5, 0);

    // Connect to the database.
    res = SQLConnect(V_OD_hdbc, (SQLCHAR*)dbName , SQL_NTS,
                                (SQLCHAR*)userName, SQL_NTS,
                                (SQLCHAR*)passw, SQL_NTS);
    if ( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (SQLConnect, " << res << ")" << endl;
        SQLFreeHandle(SQL_HANDLE_ENV, V_OD_Env);
        return false;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
void odbcDisconnect()
// Free allocated handles disconnect the specified connection.
//
    {
    SQLDisconnect(V_OD_hdbc);
    SQLFreeHandle(SQL_HANDLE_DBC,V_OD_hdbc);
    SQLFreeHandle(SQL_HANDLE_ENV, V_OD_Env);
    }

//////////////////////////////////////////////////////////////////////////////
static bool getNumberOfColumns(
// Set numcol to number of columns in the table.
//
    SQLHSTMT *V_OD_hstmt,
    SQLSMALLINT &numcol )
    {
    numcol=0;

    int res = SQLNumResultCols(*V_OD_hstmt, &numcol);
    if ( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (getNumberOfColumns, " << res << ")" << endl;
        odbcDisconnect();
        return false;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
static void describeColumn(
// Insert the type of every column in the selected table into vectype.
//
    SQLSMALLINT *vectype,
    SQLSMALLINT numcol,
    SQLHSTMT *V_OD_hstmt )
    {
    SQLSMALLINT namelength,dataType,decimaldigits,nullable;
    SQLUINTEGER columnsize;

    for (SQLUSMALLINT j=0; j < numcol; j++)
        {
        char columname[ODBC_MAX_STRING_LENGTH];

        SQLDescribeCol(*V_OD_hstmt, j+1,
                       (SQLCHAR*)columname, ODBC_MAX_STRING_LENGTH,
                       &namelength,
                       &dataType, &columnsize,
                       &decimaldigits, &nullable);
        vectype[j]=dataType;
        }
    }

//////////////////////////////////////////////////////////////////////////////
static char *rtrim(
// Trim whitespace from the end of a string.
//
    char *st )
    {
    int l;
    l = strlen(st)-1;
    while(st[l] == ' ')
        st[l--] = '\0';
    return st;
    }

//////////////////////////////////////////////////////////////////////////////
static char *ltrim(
// Return the substring of a string that does not start with whitespace.
//
    char *st )
    {
    while( *st == ' ' )
        st++;
    return st;
    }
    
//////////////////////////////////////////////////////////////////////////////
static char* UCase(char *s, int n)
// Transform the first n characters of the string s to upper case.
    {
    char *us=new char[strlen(s)+1];

    for( int i=0; i < n; i++)
        us[i]=toupper(s[i]);
    us[n] = '\0';
    return us;
    }

//////////////////////////////////////////////////////////////////////////////
static bool prepareConversion(
// Create a vector with conversion operators for every column of the table
// and return the size of this vector in r.
//
    const char *s,
    ValueConv v[],
    int &r )
    {
    r=0;

    char *sp= new char [ strlen(s) +1];
    int t = 0;
    for (unsigned int i = 0;  i <= strlen(s); i++)
        {
        if( i == strlen(s)  || s[i] == ',' )
            {
            if (strncmp (UCase(ltrim(sp),6),"UT_INT",6) == 0)
                v[r] = UT_INT;
            else if (strncmp (UCase(ltrim(sp),5),"CONST",5) == 0)
                v[r] = CONST;
            else if (strncmp (UCase(ltrim(sp),7),"Q_CONST",7) == 0)
                v[r] = Q_CONST;
            else if (strncmp (UCase(ltrim(sp),6),"UR_INT",6) == 0)
                v[r] = UR_INT;
            else if (strncmp (UCase(ltrim(sp),5),"U_INT",5) == 0)
                v[r] = U_INT;
            else
                {
                cerr << "ODBC error in type statement." << endl;
                odbcDisconnect();
                delete[] sp;
                return false;
                }
            r++;
            t = 0;
            }
        else
            {
            sp[t] = s[i];
            t++;
            }
        }
    delete[] sp;
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
static bool insertString(
// Execute an SQL statement.
//
    char *string,
    SQLHSTMT *V_OD_hstmt )
    {
    int res = SQLAllocHandle(SQL_HANDLE_STMT, V_OD_hdbc, V_OD_hstmt);
    if( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (AllocHandle, " << res << ")" << endl;
        odbcDisconnect();
        return false;
        }

    res = SQLExecDirect(*V_OD_hstmt,(SQLCHAR*)string, SQL_NTS);
    if( res != SQL_SUCCESS  &&  res != SQL_SUCCESS_WITH_INFO )
        {
        cerr << "ODBC error (Insert, " << res << ")" << endl;
        odbcDisconnect();
        return false;
        }

    return true;
    }

//////////////////////////////////////////////////////////////////////////////
// This function creates a record to be inserted into a given table on the
// currently connected database.
//
static char *insert_tuple (
    char *ins,
    char **pa,
    unsigned arity,
    SQLSMALLINT *vectype )
    {
    char *s = new char[ODBC_MAX_STRING_LENGTH+1];
    strcpy( s, ins);
    for (unsigned k = 0; k < arity; k++)
        {
        if ( vectype[k] == SQL_C_LONG
             || vectype[k] == SQL_C_NUMERIC
             || vectype[k] == SQL_C_SLONG
             || vectype[k] == SQL_C_ULONG
             || vectype[k] == SQL_C_DOUBLE
             || vectype[k] == SQL_C_FLOAT
             || vectype[k] == SQL_C_SHORT
             || vectype[k] == SQL_C_SSHORT
             || vectype[k] == SQL_C_USHORT 
             || vectype[k] == SQL_BIGINT
             || vectype[k] == SQL_DECIMAL
             || vectype[k] == SQL_FLOAT
             || vectype[k] == SQL_TINYINT )
            strcat(s, pa[k]);
        
        if ( vectype[k] == SQL_C_CHAR
             || vectype[k] == SQL_C_BINARY
             || vectype[k] == SQL_C_TYPE_DATE
             || vectype[k] == SQL_C_TYPE_TIMESTAMP 
             || vectype[k] == SQL_LONGVARCHAR
             || vectype[k] == SQL_VARCHAR
             || vectype[k] == SQL_C_BINARY )
           {
           strcat( s, "'");
           strcat( s,  pa[k]);
           strcat( s, "'");
           }

        if( k == arity-1 )
            strcat(s, ")");
        else
            strcat( s, ", ");
        }

    return s;
    }

//////////////////////////////////////////////////////////////////////////////
// This function deletes all or some records from a given table in the
// currently connected database.
//
// @param ntable, the name of the output table.
// @param query,  the WHERE condition of the SQL statement for the table tuple 
//                deletion; a query param is on the form:
//                     REPLACE        WHERE_CONDITION
//                    (keyword)       (SQL_STATEMENT)
//                keyword must be removed in order to properly use this param.
//
static bool delete_tuple(
    char *ntable,
    char *query )
    {
    char *lquery = ltrim(query);

    // Give up if the query does not start with "REPLACE" or if the 
    // condition following "REPLACE" is not empty, but is not separated from
    // "REPLACE" by whitespace.
    if ( strncmp (lquery, "REPLACE", 7) || 
         ( strlen(lquery+7) == strlen(ltrim(lquery+7)) && 
         strlen(ltrim(lquery+7)) > 0))
        {
        cerr << "ODBC: error in the writing of REPLACE." << endl;
        odbcDisconnect();
        return true;
        }

    char *st = new char[ODBC_MAX_STRING_LENGTH+1];
    strcpy(st, "DELETE FROM ");
    strcat(st, ntable);
    
    strcat(st, lquery + 7);

    SQLHSTMT V_OD_hstmt ;
    if (!insertString(st, &V_OD_hstmt))
        {
        return false;
        }
    // to free a handle for a SQL-statement
    SQLFreeHandle (SQL_HANDLE_STMT,V_OD_hstmt);
    delete[] st;
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
static bool odbcExport(
// Insert an atom with predicate name 'pname' in the table table_name.
//
    const char* pname,
    const GINTERPRET *modello,
    const INTERPRET *A,
    const VATOMSET &EDB,
    const char* table_name )
    {
    unsigned cont = 0;
    char* sel = new char[ODBC_MAX_STRING_LENGTH+1];
    SQLHSTMT V_OD_hstmt ;
    strcpy(sel,"SELECT * FROM ");
    strcat(sel, table_name);
    if (!insertString(sel,&V_OD_hstmt))
        return false;

    SQLSMALLINT numberCol=0;
    if ( ! getNumberOfColumns(&V_OD_hstmt,numberCol) )
       return false;

    SQLSMALLINT *vectype =  new SQLSMALLINT [numberCol];
    describeColumn(vectype,numberCol,&V_OD_hstmt);

    // to free a handle for a SQL-statement
    SQLFreeHandle (SQL_HANDLE_STMT,V_OD_hstmt);
    delete[] sel;
    vector<ATOM>* atoms = new vector<ATOM>();
    TERMS * idents = new vector<TERM> ();
    for (SQLSMALLINT i = 0; i < numberCol; i++)
        {
        TERM id (0,0);
        idents->push_back(id);
        }
    if (modello != 0)
        modello->exportMatching(pname, *atoms, cont);
    bool find = false;

    if (OptionPrintFacts)
        EDB.exportMatching (pname, *atoms, cont);
    if (A != 0 )
        {
        find = A->isPositivePartEmpty();
        if (!find)
            {
            ATOM *ato1 = new ATOM(pname,idents,PREDICATE_NAMES::typeIDB);
            ATOMSET::FIND_RESULT *pres = new ATOMSET::FIND_RESULT();
            A->findInPositivePart(*ato1,*pres);
            ATOMSET::FIND_RESULT &res = *pres;
            for (; !res.atEnd(); res++)
                {
                if (strcmp(pname, (*res).getPredName())==0)
                    {
                    atoms->push_back(*res);
                    cont++;
                    }
                }
            delete[] pres;
            }
        }
    if (cont == 0)
        {
        odbcDisconnect();
        return false;
        }
    for( vector<ATOM>:: const_iterator j = atoms->begin();
         j != atoms->end();
         j++ )
        {
        char* ins =  new char[ODBC_MAX_STRING_LENGTH+1];
        sprintf(ins,"INSERT INTO %s VALUES ( ",table_name );

        unsigned t = 0;
        char **pa = new char* [numberCol+1];
        for( vector<TERM>:: const_iterator k = (*j).getParams()->begin();
             k != (*j).getParams()->end();
             k++ )
            {
            switch( k->getType() )
                {
                case TERM::String:
                    {
                    char *ba = new char [ODBC_MAX_STRING_LENGTH+1];
                    pa[t] = new char[ODBC_MAX_STRING_LENGTH+1];
                    strcpy (ba, (*k).getNameAsString());
                    char *qa = new char[strlen(ba)+1];
                    unsigned int j = 0;
                    for(unsigned int i = 0; i < strlen(ba); i++, j++)
                        {
                        if (ba[i] == '\'')
                            {
                            qa[j] = ba[i];
                            j++;
                            qa[j] = '\'';
                            }
                        else
                            qa[j] = ba[i];
                        } // FOR
                    qa[j] = '\0';
                    if (qa[0] == '"')
                        {
                        qa[strlen(qa)-1] = 0;
                        strcpy(pa[t], qa+1);
                        }
                    else
                        strcpy(pa[t],qa);
                    t++;
                    break;
                    }
                case  TERM::Integer:
                    {
                    int *p = new int(k->getInt());
                    pa[t] = new char[50];
                    sprintf(pa[t],"%d",*p);
                    t++;
                    break;
                    }
                case TERM::Variable:
                    assert(0);
                    break;
                case TERM::NullConst:
                    assert(0);
                    break;
                } //  switch
            } // FOR
        char *tup = insert_tuple(ins, pa, numberCol, vectype);
        if (! insertString(tup, &V_OD_hstmt) )
           {
           return false;
           }
        // to free a handle for a SQL-statement
        SQLFreeHandle (SQL_HANDLE_STMT,V_OD_hstmt);
        delete[] tup;
        for (SQLSMALLINT i = 0; i < numberCol; i++)
            delete[] pa[i];
        delete[] pa;
        delete[] ins;
        }

    odbcDisconnect();
    delete[] vectype;
    return true;
    }

//////////////////////////////////////////////////////////////////////////////
static void rdbms_rowcheck(
// Start a new atom if necessary.
//
    const char *pName )
    {
    if (row_open)
        {
        if(TraceLevel >= 2)
            cdebug << ", ";
        }
    else  // new atom starts
        {
        if(TraceLevel >= 2)
            cdebug << "(" << pName << endl;
        row_open = true;

        if (current_idents != 0)
            delete current_idents;
        current_idents = new TERMS;
        }
    }

//////////////////////////////////////////////////////////////////////////////
static void rdbms_sendInt(
// Set the n_th term to an integer.
    unsigned long i,
    const char *pName )
    {
    rdbms_rowcheck(pName);

    if(TraceLevel >= 2)
       cdebug << i ;

    TERM ident(i,0);
    current_idents->push_back(ident);
    }

//////////////////////////////////////////////////////////////////////////////
static void rdbms_sendConst(
// Set n_th term to a string without quotes.
//
    char *str,
    const char *pName )
    {
    rdbms_rowcheck(pName);

    if(TraceLevel >= 2)
        cdebug << str;

    TERM ident(rtrim(str));
    current_idents->push_back(ident);
    }

//////////////////////////////////////////////////////////////////////////////
static void rdbms_sendString(
// Set the n_th term to a string with quotes.
//
    char *str,
    unsigned len,
    const char *pName )
    {
    rdbms_rowcheck(pName);
    str[len] = '\0';
    rtrim(str);
    len = strlen(str);
    unsigned len2 = (unsigned)len + 2;
    if(len2 > ODBC_MAX_STRING_LENGTH)
        {
        cerr << "ODBC error: character string read from DB is too long ("
             << str << ")." << endl;
        exit(1);
        }

    char str2[ODBC_MAX_STRING_LENGTH + 1];
    sprintf(str2,"\"%s\"",str);

    if( TraceLevel >= 2 )
        cdebug << str2;

    TERM ident(rtrim(str2));
    current_idents->push_back(ident);
    }

//////////////////////////////////////////////////////////////////////////////
static void rdbms_sendRowEnd(
// Build the fact.
//
    const char *pName )
    {
    if(row_open)
        {
        if(TraceLevel >= 2)
            cdebug << ").";
        row_open = false;
        ATOM atom( pName, current_idents, PREDICATE_NAMES::typeUndef );
        if( atom.refineType(PREDICATE_NAMES::typeEDB) )
            EDB.add(atom);
        else
            {
            atom.refineType(PREDICATE_NAMES::typeIDB);
            DISJUNCTION disj;
            disj.add(atom);
            IDB.push_back(RULE(&disj, 0));
            }
        }
    }


//////////////////////////////////////////////////////////////////////////////
void ODBC_IMPORT::connect()
    {
    if ( ! odbcConnect(DBdns, DBuser, DBpass) )
        {
        cerr << "Error during ODBC connect." << endl;
        SafeExit(1);
        }
    }

//////////////////////////////////////////////////////////////////////////////
bool ODBC_IMPORT::import()
// Execute the query requested by the user and build an new predicate pName
// whose atoms are the results of the query.
//
    {
#define pName predicate
#define type DBtype

    int         NCol_type;
    SQLSMALLINT numcol;

    if( strncmp(UCase(ltrim(query),6),"SELECT",6) != 0 )
        {
        cerr << "ODBC: error in the writing of SELECT." << endl;
        odbcDisconnect();
        return false;
        }

    if ( !insertString(query,&V_OD_hstmt)
         || ! getNumberOfColumns(&V_OD_hstmt,numcol) )
        return false; 

    ////////////////////////////////////////

    ValueConv *typeConv = new ValueConv[numcol];
    if ( strcmp(type, "") )
        {
        if( ! prepareConversion(type,typeConv,NCol_type) )
            {
            odbcDisconnect();
            return false;
            }

        if (NCol_type != numcol)
            {
            odbcDisconnect();
            return false;
            }
        }
    else
        for (int c=0; c < numcol; c++)
            typeConv[c] = DEFAULT;

    ////////////////////////////////////////

    SQLSMALLINT *vectype = new SQLSMALLINT[numcol];
    describeColumn(vectype,numcol,&V_OD_hstmt);

    void  **values = new void *[numcol];
    SQLINTEGER *V_OD_err = new SQLINTEGER[numcol];
    int V_OD_erg = 0;
    for (SQLUSMALLINT k = 0; k < numcol; k++)
        {
        switch (vectype[k])
            {
            case SQL_C_TYPE_DATE:
            case SQL_LONGVARCHAR:
            case SQL_VARCHAR:
            case SQL_C_CHAR:
            case SQL_C_TYPE_TIMESTAMP:
            case SQL_C_BINARY:
                values[k] = new char [ODBC_MAX_STRING_LENGTH+1];
                V_OD_erg = SQLBindCol( V_OD_hstmt,k+1,
                                       SQL_C_CHAR,
                                       values[k],
                                       ODBC_MAX_STRING_LENGTH,
                                       &V_OD_err[k] );
                if( typeConv[k] == DEFAULT )
                    typeConv[k]=Q_CONST;
                break;
            case SQL_C_ULONG:
            case SQL_C_SHORT:
            case SQL_BIGINT:
            case SQL_C_LONG:
            case SQL_C_SLONG:
            case SQL_TINYINT:
            case SQL_C_SSHORT:
            case SQL_C_USHORT:
                switch (typeConv[k] )
                    {
                    case UR_INT:
                    case UT_INT:
                    case U_INT:
                    case DEFAULT:
                        values[k] = new (unsigned long)();
                        V_OD_erg = SQLBindCol( V_OD_hstmt,
                                               k+1,
                                               vectype[k],
                                               values[k],
                                               sizeof(vectype[k]),
                                               &V_OD_err[k] );
                        if( typeConv[k] == DEFAULT )
                            typeConv[k]=U_INT;
                        break;
                    case CONST:
                    case Q_CONST:
                        values[k] = new char[ODBC_MAX_STRING_LENGTH+1];
                        V_OD_erg = SQLBindCol( V_OD_hstmt,
                                               k+1,
                                               SQL_C_CHAR,
                                               values[k],
                                               ODBC_MAX_STRING_LENGTH,
                                               &V_OD_err[k] );
                        break;
                    default:
                        assert(0);
                    }
                break;
            case SQL_DECIMAL:
            case SQL_C_NUMERIC:
            case SQL_C_DOUBLE:
            case SQL_FLOAT:
            case SQL_C_FLOAT:
                switch (typeConv[k] )
                    {
                    case UR_INT:
                    case UT_INT:
                    case U_INT:
                    case DEFAULT:
                        values[k] = new (double) ();
                        V_OD_erg = SQLBindCol( V_OD_hstmt,
                                               k+1,
                                               SQL_C_DOUBLE,
                                               values[k],
                                               sizeof(SQL_C_DOUBLE),
                                               &V_OD_err[k] );
                        if( typeConv[k] == DEFAULT )
                            typeConv[k]=UR_INT;
                        break;
                    case CONST:
                    case Q_CONST:
                        values[k] = new char[ODBC_MAX_STRING_LENGTH+1];
                        V_OD_erg = SQLBindCol( V_OD_hstmt,
                                               k+1,
                                               SQL_C_CHAR,
                                               values[k],
                                               ODBC_MAX_STRING_LENGTH,
                                               &V_OD_err[k] );
                        break;
                    default:
                        assert(0);
                    }
                break;
            default:
                cerr << "Type not supported (" << vectype[k] << ")" << endl;
                SQLFreeHandle(SQL_HANDLE_STMT,V_OD_hstmt);
                odbcDisconnect();
                return false;
            }
        if ( V_OD_erg != SQL_SUCCESS  &&  V_OD_erg != SQL_SUCCESS_WITH_INFO )
            {
            cerr << "ODBC error (type column, " << V_OD_erg << ")" << endl;
            SQLFreeHandle(SQL_HANDLE_STMT,V_OD_hstmt);
            odbcDisconnect();
            return false;
            }
        } // for
    while( SQLFetch(V_OD_hstmt) != SQL_NO_DATA)
        {
        for (SQLUSMALLINT i = 0; i < numcol; i++)
            {
            if (V_OD_err[i] == SQL_NULL_DATA)
                {
                char *st = new char[10];
                strcpy(st, "null");
                rdbms_sendString (st, strlen(st),pName);
                delete[] st;
                }
            else
                {
                unsigned long val=0;
                switch ( typeConv[i] )
                    {
                    case UR_INT:
                        {
                        if( vectype[i] == SQL_C_SHORT
                            || vectype[i] == SQL_C_SSHORT
                            || vectype[i] == SQL_C_USHORT
                            || vectype[i] == SQL_C_LONG
                            || vectype[i] == SQL_C_SLONG
                            || vectype[i] == SQL_C_ULONG
                            || vectype[i] == SQL_BIGINT)
                            {
                            val = (unsigned long)labs(*(long*)values[i]);
                            }
                        else
                            if( vectype[i] == SQL_C_TYPE_TIMESTAMP
                                || vectype[i] == SQL_C_BINARY
                                || vectype[i] == SQL_C_TYPE_DATE  )
                                {
                                cerr << "Error in conversion type." << endl;
                                odbcDisconnect();
                                return false;
                                }
                        else
                            if ( vectype[i] == SQL_C_CHAR
                                 || vectype[i] == SQL_VARCHAR
                                 || vectype[i] == SQL_LONGVARCHAR)
                                val = labs(atoi((char*)values[i]));
                        else
                            val = (unsigned long)(fabs(*(double*)values[i])+0.5);
                        rdbms_sendInt(val, pName);
                        break;
                        }
                    case UT_INT:
                        if( vectype[i] == SQL_C_SHORT
                            || vectype[i] == SQL_C_SSHORT
                            || vectype[i] == SQL_C_USHORT
                            || vectype[i] == SQL_C_LONG
                            || vectype[i] == SQL_C_SLONG
                            || vectype[i] == SQL_C_ULONG
                            || vectype[i] == SQL_BIGINT)
                            {
                            val = (unsigned long)labs(*(long*)values[i]);
                            }
                        else
                            if( vectype[i] == SQL_C_TYPE_TIMESTAMP
                                || vectype[i] == SQL_C_BINARY
                                || vectype[i] == SQL_C_TYPE_DATE  )
                                {
                                cerr << "Error in conversion type." << endl;
                                odbcDisconnect();
                                return false;
                                }
                        else
                            if ( vectype[i] == SQL_C_CHAR
                                 || vectype[i] == SQL_VARCHAR
                                 || vectype[i] == SQL_LONGVARCHAR)
                                val = labs(atoi((char*)values[i]));
                        else
                            val = (unsigned long)fabs(floor(*(double*)values[i]));
                        rdbms_sendInt(val, pName);
                        break;
                    case U_INT:
                        if( vectype[i] == SQL_C_TYPE_TIMESTAMP
                            || vectype[i] == SQL_C_BINARY
                            || vectype[i] == SQL_C_TYPE_DATE )
                            {
                            cerr << "Error in conversion type." << endl;
                            odbcDisconnect();
                            return false;
                            }
                        else
                            if ( vectype[i] == SQL_C_CHAR
                                 || vectype[i] == SQL_VARCHAR
                                 || vectype[i] == SQL_LONGVARCHAR)
                                val = labs(atoi((char*)values[i]));
                        else
                            if ( vectype[i] == SQL_C_FLOAT
                                 || vectype[i] == SQL_FLOAT)
                                val = (unsigned long)fabs(floor(*(double*)values[i]));
                        else
                            val = labs(*(unsigned long*)values[i]);
                        rdbms_sendInt(val, pName);
                        break;
                    case CONST:
                        rdbms_sendConst( (char*) values[i],
                                         pName);
                        break;
                    case Q_CONST:
                        rdbms_sendString( (char*) values[i],
                                          strlen((char*) values[i]),
                                          pName);
                        break;
                    default:
                        assert(0);
                    }
                }
            }
        rdbms_sendRowEnd(pName);
        }

    for (SQLUSMALLINT y = 0; y < numcol; y++)
        {
        if ( V_OD_err[y] == SQL_NULL_DATA
             || typeConv[y] == CONST
             || typeConv[y] == Q_CONST )
            delete[] (char*)values[y];
        else
            delete (unsigned long*) values[y];
        }

    delete[] V_OD_err;
    delete[] values;
    delete vectype;
    return true;

#define pName predicate
#define type DBtype
    }

#endif

//////////////////////////////////////////////////////////////////////////////

void ODBC_EXPORT::run(
    const INTERPRET *A,
    const GINTERPRET *I ) const
    {
    if( TraceLevel >= 1 )
        cdebug << " - Exporting!";

#ifdef WITH_ODBC
    if( ! predicate  ||  ! DBtable )
        return;

    if ( ! odbcConnect(DBdns, DBuser, DBpass) )
        cerr << "Error during ODBC connect." << endl;
    else
        {
        if ( strlen(query) )
            if ( ! delete_tuple(DBtable, query) )
                cerr << "Error during ODBC deletion." << endl;
        if ( ! odbcExport( predicate,I,A,EDB,DBtable) )
            cerr << "Error during ODBC export." << endl;
        }
#else
    A=A; I=I; // Avoid compiler warnings about A and I being unused.
#endif
    }

//////////////////////////////////////////////////////////////////////////////

void odbcExport(const INTERPRET *A, const GINTERPRET *I)
    {
    for(unsigned i=0; i < ODBC_Exports.size(); i++)
        ODBC_Exports[i].run(A,I);
    }

#ifdef WITH_ODBC

void odbcImport()
    {
    for (unsigned cursor = 0; cursor < ODBC_Imports.size(); cursor++)
        {
        if ( cursor == 0 )
            // Open connection to the first data source.
            ODBC_Imports[cursor].connect();
        else // A new connection is needed only for a different data source.
            if ( strcmp( ODBC_Imports[cursor].getDBdns(),
                         ODBC_Imports[cursor-1].getDBdns()) )
                {
                SQLFreeHandle(SQL_HANDLE_STMT,V_OD_hstmt);
                odbcDisconnect();
                ODBC_Imports[cursor].connect();
                }

        if( ! ODBC_Imports[cursor].import() )
            {
            cerr << "Error during ODBC Import." << endl;
            SafeExit(1);
            }
        }

    SQLFreeHandle(SQL_HANDLE_STMT,V_OD_hstmt);
    odbcDisconnect();
    }

#endif
    
//////////////////////////////////////////////////////////////////////////////
//This function defines an ordering to execute the #import builtin.
//The imports from the same database are stored
//in a vector in adjacent position for reducing
//useless connect/disconnect operation.

void odbcAddImport( const char *dns,
                    const char *user,
                    const char *pass,
                    const char *qry,
                    const char *pred,
                    const char *type)
    {                                                 
    vector<ODBC_IMPORT>::iterator cursor;
    for (cursor = ODBC_Imports.begin();
         cursor != ODBC_Imports.end();
         cursor++)
         if ( ! strcmp((*cursor).getDBdns(), dns) )
            break;
    ODBC_Imports.insert(cursor, ODBC_IMPORT(dns,
                                            user,
                                            pass,
                                            qry,
                                            pred,
                                            type));
    }

// Local Variables:
// c-file-style: "dl"
// End:
