%{
/////////////////////////////////////////////////////////////////////////////
// scanner.l

#include "builtin.h"
#include "parser.h"
#include <stdio.h>

// We don't need unput, and it can yield a "defined but not used"-warning.
#define YY_NO_UNPUT
%}

BLANK       ([ \t]+)
NEWLINE     \15?\n
                       // some OSs use \r\n as end-of-line instead of \n

ANON_VAR    _

ID          ([A-Za-z][A-Za-z_0-9]*)
INTERNAL_ID ([A-Z]?$[A-Za-z_0-9$]*)
QUOTED_ID   (\"[^\"]*\")

NUM         ([0-9]+)


%START WANT_NEWLINE
                       // Only in state WANT_NEWLINE a newline
                       // token for the parser is generated, while
                       // normally newlines are simply skipped.

%%

	// Newline is normally skipped, except when explicitly
	// requested with WANT_NEWLINE. (e.g. by a construct that is
	// terminated by a newline.)

<WANT_NEWLINE>{NEWLINE} { 
                         parser_line++;
                         BEGIN(INITIAL);
                         return NEWLINE;
			}
{NEWLINE}		{ parser_line++;        }

","			{ return COMMA;		}
"."			{ return DOT;		}
":"                     { return COLON;         }
";"                     { return SEMICOLON;     }
"-"			|
"~"			{ return TRUE_NOT;	}
"!"                     { return EXCLAMATION;   }
"+"			{ return PLUS;		}
"*"			{ return ASTERISK;	}
"="			{ return EQUALS;	}
"|"                     { return VERTICAL_BAR;  }

"v"			{ return ID_OR_VEL;	}
"<-"			| 
":-"			{ return CONS;		}

":~"			{ return WEAK_CONS;	}

"("			{ return PARAM_OPEN;	}
")"			{ return PARAM_CLOSE;	}

"["			{ return BRACKET_OPEN;	}
"]"			{ return BRACKET_CLOSE;	}

"{"			{ return CURLY_OPEN;	}
"}"			{ return CURLY_CLOSE;	}

"=="			{ return PRED_EQUAL;	}
"<>"			|
"!="			{ return PRED_UNEQUAL;		}
"<"			{ return PRED_LESS;		}
">"			{ return PRED_GREATER;		}
"<="			{ return PRED_LESS_OR_EQ;	}
">="			{ return PRED_GREATER_OR_EQ;	}

"#any"                  { return PRED_ANY;              }
"#avg"                  { return PRED_AVG;              }
"#count"                { return PRED_COUNT;            }
"#max"                  { return PRED_MAX;              }
"#min"                  { return PRED_MIN;              }
"#sum"                  { return PRED_SUM;              }
"#times"                { return PRED_TIMES;            }

"not"                   |
"non"                   { return NOT;           }

"?"			{
                          return BOOL_QUERY;
			}

#maxint			{ return MAXINTEGER;		}

#int			{ return PRED_INT;		}
#succ			{ return PRED_SUCC;		}

^({BLANK}?)#import      {
                          BEGIN(WANT_NEWLINE); return META_IMPORT; }
^({BLANK}?)#export      {
                          BEGIN(WANT_NEWLINE); return META_EXPORT; }
^({BLANK}?)#exportQueryResult {
                          BEGIN(WANT_NEWLINE); return META_QUERY_EXPORT; }

{ANON_VAR}              { return ANON_VAR; }

{ID}			|
{QUOTED_ID}		{
			  yylval.string=new char[strlen(yytext)+1];
                          strcpy(yylval.string,yytext);
                          return ID;
			}

{INTERNAL_ID}           {
                          // IDs starting with '$' are only allowed
                          // during internal calls to the scanner
                          if(ParserStateInternal)
                              {
                              yylval.string=new char[strlen(yytext)+1];
                              strcpy(yylval.string,yytext);
                              return ID;
                              } 
                          yyerror("identifier starts with '$'");
                          parser_errors++;
                          return ERROR;
                        }


{NUM}			{ 
			  if( strlen(yytext) > NUM_LENGTH )
			      {
			      char s[1024];
			      sprintf( s,
			               "numeric constant %s is too long",
			               yytext );
			      yyerror(s);

                              parser_errors++;
			      return ERROR;
			      }
			  else
			      {
			      yylval.string=new char[strlen(yytext)+1];
			      strcpy(yylval.string,yytext);
			      }

			  return NUM; 
			}

\%.*			{ /* Skip comments */	}
{BLANK}			{ /* Skip blanks */	}


.			{ // Anything not matching one of the patterns
			  // given above is considered an error.
			  char s[1024];

			  assert(yytext[1]==0);
			  sprintf(s,"syntax error (%s)",yytext); 
			  yyerror(s);

                          parser_errors++;
                          return ERROR;
			}
