//////////////////////////////////////////////////////////////////////////////
// sql.C

#define WITH_GLOBAL_VARIABLES

#include "dl.h"
#include "sql.h"

//////////////////////////////////////////////////////////////////////////////
// Helper functions as expected by LEX/FLEX

extern "C" int sqlyywrap()    // End-of-file handler for LEX
    {
    return 1;
    }

int sqlyyerror(const char* s)
    {
    return yyerror(s);
    }

Schemata* sql_schemata;

Querynames* querynames;

Correlations* correlations;

AttributeAffiliations* attribute_affiliations;

CorrelationsReverse* correlations_reverse;

Bindings* current_bindings;

NameSpace* predicate_names;
NameSpace* variable_names;

HeadArguments* head_arguments;

int query_spec_nesting;

bool sql_strict_SQL3=false;

#include "sql_parser2.c"

//////////////////////////////////////////////////////////////////////////////

string SQL_Interesting_Predicate;

class SQLAtomPrinter : public unary_function<ATOM&,void>
    {
public:
    void operator() (const ATOM &a) const
        {
        NAMES_ITEM predname(SQL_Interesting_Predicate.c_str());

        if( a.getPredItem() == predname )
            {
            const TERMS* p=a.getParams();

            if( p )
                { 
                for(TERMS::const_iterator j=(*p).begin(); j != (*p).end(); j++)
                    {
                    cout << *j;
                    if(  (*j).getType() == TERM::Integer
                         || strlen( ((*j).getNameAsString()) ) < 8 )
                        cout << "\t";
                    cout << '\t';
                    }
                }
        
            cout << endl; 
            }
        }
    };

void SQLCallback(const INTERPRET *result)
    {
    result->foreachPositive( SQLAtomPrinter() );
    }

//////////////////////////////////////////////////////////////////////////////

/////// 
// infer_correlation_from_attribute
///////
// Purpose:
// given an attributename, determine the corresponding correlation, 
// taking into account the current correlations and the global 
// relational schemata

string infer_correlation_from_attribute(const string& attribute)
{
  typedef AttributeAffiliations::iterator AAi;
  
  string relation;
  string correlation;

  // two types of ambiguities: 
  // 1. Only one relation matches, but it is aliased to more than 
  //    one correlation.
  // 2. Several relations match, some of them might be aliased to 
  //    more than one correlation.

  bool ambiguity1=false, ambiguity2=false;

  // determine matches
  pair<AAi,AAi> r 
    = attribute_affiliations->equal_range(attribute);

  if(r.first == r.second)
    {
      // We don't know at all of such an attribute
      cerr << "Error: reference to undefined attribute '"
	   << attribute
	   << "'" << endl;
      exit(1);
    }

  // attribute is indeed affiliated to some relations

  typedef CorrelationsReverse::iterator CRi;
  pair<CRi,CRi> crr;


  // determine the first relation which is currently
  // in the active correlations

  do
    {
      // From the sequence of matching (attribute,relation)
      // pairs, take the relationname of the first pair.

      relation = (*(r.first)).second;

      // Now we have to determine whether the relation is active
      // in the current context. It is if there is at least one
      //  entry about it in correlations_reverse.

      crr = correlations_reverse->equal_range(relation);

    }
  // If the range is empty, the determined relation is inactive in
  // the current context, if there are more relation candidates
  // left, repeat the previous steps.
  while (crr.first == crr.second && !(++(r.first) == r.second));

  // At this point we've either found a candidate correlation
  // i.e. one relation which is active in the current context,
  // or no match at all on the given attribute.
  
  if(r.first == r.second)
    {
      // No active correlation matched
      cerr << "At line " << parser_line << ":" << endl;
      cerr << "Error: Attribute '" << attribute
	   << "' doesn't occur in any active relation" << endl;
      exit(2);
    }

  // Here we've found one candidate correlation (active relation).

  correlation = (*(crr.first)).second;

  // Check if there are more (i.e. whether the relation is aliased
  // more than once).

  if(!((++crr.first) == crr.second))
    {
      // ambiguity1: multiple correlation names for this relation
      ambiguity1 = true;

      SQLParseError();
      cerr << "Error: Attribute '"
	   << attribute
	   << "' is ambiguous." << endl
	   << "       There are several Correlation names " 
	   << "for a matching relation" << endl
	   << "       You'll have to qualify the attribute." << endl
	   << "       Relation:" << endl
	   << "         " << relation << endl
	   << "       Correlations:" << endl
	   << "         " << correlation << endl;
	      
      // there could be even more aliases, print them all

      do
	{
	  cerr << "         " << (*(crr.first)).second << endl;
	  crr.first++;
	}
      while(!(crr.first == crr.second));

    }

  // now check whether the attribute exists in more than one active
  // relation
	  
  ++r.first;

  while(!(r.first == r.second))
    {
      // careful: We might find a relation which is not active 
      // in the treated select -> don't overwrite already found relation

      string relation1;

      // other relation which has the specified attribute

      relation1 = (*(r.first)).second;

      // determine correlations of this relation

      crr = correlations_reverse->equal_range(relation1);

      // check whether there is an active correlation

      if(!(crr.first == crr.second))
	{
	  // ambiguity2: multiple relations have this attribute

	  ambiguity2=true;

	  if(ambiguity1)
	    {
	      // we already had a correlation ambiguity
	      // don't report an error again, just give additional info
	      cerr << "       There are even other relations " 
		   << "having the same attribute:"
		   << endl;
	    }
	  else
	    {
	      // No ambiguity occured before: Report an error.
              SQLParseError();
	      cerr << "Error: Attribute '" 
		   << attribute
		   << "' is ambiguous." << endl
		   << "       You'd better qualifiy it." << endl 
		   << "       Candidate relations/correlations are:" 
		   << endl;
	    }

	  cerr << "       Relation:" << endl
	       << "         " << relation << endl
	       << "       Correlation(s):" << endl
	       << "         " << correlation << endl
	       << "       Relation:" << endl
	       << "         " << relation1 << endl
	       << "       Correlation(s):" << endl;

	  // print out the correlations

	  do
	    {
	      cerr << "         " << (*(crr.first)).second << endl;
	      crr.first++;
	    }
	  while(!(crr.first == crr.second));

	  // there could be more relations having this attribute
	  // print them all plus their correlations

	  r.first++;
	  while(!(r.first == r.second))
	    {
	      // next potentially ambiguous relation

	      relation1 = (*(r.first)).second;

	      // but again only if it is currently active

	      crr = correlations_reverse->equal_range(relation1);

	      if(!(crr.first == crr.second))
		{
		  // There are active correlations and hance a source
		  // of ambiguity. Print the relation and its
		  // correlations.

		  cerr << "       Relation:" << endl
		       << "         " << relation1 << endl
		       << "       Correlation(s):" << endl;
		  do
		    {
		      cerr << "         " 
			   << (*(crr.first)).second << endl;
		      crr.first++;
		    }
		  while(!(crr.first == crr.second));
		}

	      // Check whether there are more

	      r.first++;
	    }
	}

      // If ambiguity2 was encountered, all potentially matching
      // relations have been dealt with (r.first==r.second already holds).

      // If not, the relation found (relation1) was inactive. In this case
      // we have to search on.

      if(ambiguity2 == false) r.first++;
    }

  // If ambiguities occured, exit the program accordingly.
   
  if(ambiguity1 == true) 
    {
      exit(ERROR_AMBIGUITY_1);
    }
  if(ambiguity2 == true) 
    {
      exit(ERROR_AMBIGUITY_2);
    }

  return correlation;
}

//////////////////////////////////////////////////////////////////////////////
// Bindings::setTo
// 
// Alter the Bindings where variable toBeSet occurs.

void Bindings::setTo(const unsigned& toBeSet, 
		     const Binding& setToThis)
{
  for(BaseType::iterator i = BaseType::begin(); 
	!(i == BaseType::end()); 
	++i)
    {
      for(Submap::iterator j = ((*i).second).begin();
	    !(j == ((*i).second).end());
	    ++j)
	{
	  // tell each Binding to change the variable toBeSet to setToThis

	  ((*j).second).setVariable(toBeSet, setToThis);
	}
    }
}

//////////////////////////////////////////////////////////////////////////////
// Binding::unify
//
// Simple unification of two Bindings

void Binding::unify(Binding& b)
{
  if(isVariable)
    {
      if(b.isVariable)
	{
	  // both are variables

	  if(!(varValue == b.varValue))
	    {
	      // the variables differ

	      // save variable for correcting other instances of it
	      unsigned tvar = this->varValue;

	      // if they're not yet equal make them equal
	      setTo(b);
	      // and correct all of the other variable instances
	      current_bindings->setTo(tvar, b);
	    }
	}
      else
	{
	  // variable =.= const

	  // save variable for correcting other instances of it
	  unsigned tvar = this->varValue;
	  
	  // bind variable
	  setTo(b);
	  // and correct all other instances
	  current_bindings->setTo(tvar, b);
	}
    }
  else
    {
      if(b.isVariable)
	{
	  // const =.= variable

	  // save variable for correcting other instances of it
	  unsigned tvar = b.varValue;

	  // bind the variable
	  b.setTo(*this);
	  // and correct all other instances
	  current_bindings->setTo(tvar, *this);
	}
      else
	{
	  // const =.= const

	  if(!(constValue == b.constValue))
	    {
	      // constants differ
	      // Clash!
              SQLParseError();
	      cerr << "Error: One condition will always be false." 
		   << endl
		   << "         The query will always be empty."
		   << endl
		   << "Error: Cannot handle this yet."
		   << endl;
	      exit(ERROR_UNIFY_BINDINGS);
	    }
	  else
	    {
	      // unifying two identical constants: 
	      // no need to do anything
	    }
	}
    }
}


///////////////////////
// Bindings::lookup returns this if no binding exists
Binding __dummy_binding;

//////////////////////////////////////////////////////////////////////////////
// Bindings::lookup
//
// Find the binding to a (correlation,attribute) 
// Return a pair: (true,b) if a binding was found, b a pointer to the binding
//                (false,__b) if no binding was found, __b is trash

pair<const bool,Binding*> Bindings::lookup(const string& correlation, 
					   const string& attribute) const
{
  BaseType::const_iterator i = BaseType::find(correlation);
  if(i == end())
    {
      // No entry for the given correlation
      return make_broken_pair(false,&__dummy_binding);
    }
  Submap::const_iterator j = ((*i).second).find(attribute);
  if(j == ((*i).second).end())
    {
      // There are entries for this correlation, but not for this attribute
      return make_broken_pair(false,&__dummy_binding);
    }

  // We found a binding, return success and a pointer to the binding

  return make_broken_pair(true, const_cast<Binding*>(&((*j).second)) );
}

//////////////////////////////////////////////////////////////////////////////
// Bindings::insert
//
// store the given binding for (correlation,attribute)
// the binding is copied

void Bindings::insert(const string& correlation, const string& attribute, 
		      Binding& b)
{
  BaseType::iterator i = BaseType::find(correlation);
  if(i == end())
    {
      // no binding yet for correlation
      // make a new attribute binding map
      
      Submap* sm = new Submap;

      (*sm)[attribute] = b;
      
      (*this)[correlation] = (*sm);
    }
  else
    {
      // there are some bindings for the correlation
      
      Submap::iterator j = ((*i).second).find(attribute);
      
      if(j == ((*i).second).end())
	{
	  // no binding yet for attribute
	  // insert (attribute,binding) into correlation's map
	    
	  ((*i).second)[attribute] = b;
	}
      else
	{
	  // (correlation,attribute) has been treated before
	  // it has some value (constant or variable) associated
	  // unify accordingly
	  // some unification warning might occur here
	    
	  ((*j).second).unify(b);
	}
    }
}

//////////////////////////////////////////////////////////////////////////////
// Bindings::print_corr_att_binding
//
// Print out the binding for a specified (correlation,attribute) pair

void Bindings::print_corr_att_binding(ostream& out, 
				      const string& correlation, 
				      const string& attribute) const
{

  // determine whether a binding exists and if it does, the associated value
  
  pair<const bool,Binding*> rv = lookup(correlation,attribute);
		
  if(rv.first)
    {
      // found a binding
      // print it out
      out << *(rv.second);
    }
  else
    {
      // couldn't find any binding
      // generate a new variable and print it
      out << variable_names->get();
    }

}

//////////////////////////////////////////////////////////////////////////////
// HeadArguments::display
//
// print out HeadArguments

void HeadArguments::display(ostream& out, 
				   const string& separator,
				   const string& leftbrace,
				   const string& rightbrace) const
{
  const_iterator i = begin();

  if(!(i == end()))
    {
      // There are Head-Arguments

      out << leftbrace;

      out << **i; // output binding

      for(++i;
	  !( i == head_arguments->end());
	  ++i)
	{
	  out << separator << **i;
	}

      out << rightbrace;
    }
  
}

//////////////////////////////////////////////////////////////////////////////
// HeadArguments::integrate
//
// Returns a head (DISJUNCTION*)
// transforms the arguments into a disjunction consisting of exactly one
// literal, and a predicatename given as parameter

DISJUNCTION* HeadArguments::integrate(const string& predicatename) const
{
  DISJUNCTION* h = new DISJUNCTION();
  
  TERMS params;

  const_iterator i = begin();

  assert(i != end()); // There should always be parameters.

  // There are Head-Arguments

  for( ; !( i == end()); ++i)
    {
      TERM* term;

      if((*i)->isVar())
	{
	  // create a variable
	  term = new TERM( (*i)->getVariable() );
	}
      else
	{
	  // create a constant (recorded in Const_Names)	 
	  term = new TERM( ((*i)->getConstant())->c_str() );
	}
      
      // insert the created stuff (will be copied)
      params.push_back(*term);
      delete term;
      
    }

  // insert the predicatename
  ATOM* atom = new ATOM(predicatename.c_str(),&params);

  // insert this atom into the DISJUNCTION
  h->add( *atom );

  // and return its pointer
  return h;
}

//////////////////////////////////////////////////////////////////////////////
// Correlations::print_correlation
//
// print out a particular correlation,

void Correlations::print_correlation(ostream& out, const string& correlation)
const
{

  const_iterator i = find(correlation);

  // The correlation has to exist!

  assert(i != end()); 

  const string& relation = (*i).second;

  // print an indentation, since this is only used for bodies (hmm)

  out << "    " << relation;

  // determine attributes to the (cor)relation

  Attributes& relation_attributes = (*sql_schemata)[relation];

  Attributes::const_iterator k = relation_attributes.begin();
	  
  if(!(k == relation_attributes.end()))
    {

      // There are attributes

      out << "(";
      {

	// Print the first binding (or new variable)
	
	current_bindings->print_corr_att_binding(out,correlation, *k);

      }

      for(++k; !(k == relation_attributes.end()); ++k)
	{
	  out << ", ";

	  // For each attribute, print the binding 
	  // (or if none exists a new variable)
	
	  current_bindings->print_corr_att_binding(out,correlation, *k);

	}
      out << ")";
    }
  
}

//////////////////////////////////////////////////////////////////////////////
// Correlations::integrate_correlation
//
// insert a particular correlation into the given body

void Correlations::integrate_correlation(const string& correlation, 
				      CONJUNCTION * const body) const
{

  const_iterator i = find(correlation);

  // The correlation has to exist!

  assert(i != end()); 

  const string& relation = (*i).second;

  // determine the attributes via the schemata

  Attributes& relation_attributes = (*sql_schemata)[relation];

  Attributes::const_iterator k = relation_attributes.begin();
	  
  assert(!(k == relation_attributes.end()));

  // There are attributes

  TERMS params;

  for(; !(k == relation_attributes.end()); ++k)
    {

      // for all attributes, determine the bindings
	
      pair<const bool,Binding*> rv 
	= current_bindings->lookup(correlation,*k);

      TERM* term;

      // and insert the binding or a new variable
		
      if(rv.first)
	{
	  // bound argument

	  if(rv.second->isVar())
	    {
	      // insert a variable
	      term = new TERM( (rv.second)->getVariable() );
	    }
	  else
	    {
	      // insert a constant
	      term = new TERM( ((rv.second)->getConstant())->c_str() );
	    }

	}
      else
	{
	  // unbound Argument
	  term = new TERM( variable_names->get_numerical() );
	}

      // now insert the binding in the new parameterlist

      params.push_back(*term);
      delete term;
       
    }

  // create a new atom (predicatename is the relationname, 
  // not the correlationname

  ATOM* atom = new ATOM(relation.c_str(), &params, PREDICATE_NAMES::typeUndef);

  // and add this new entry into the body

  body->add( LITERAL(false,*atom) );

}

//////////////////////////////////////////////////////////////////////////////
// Bindings::display
//
// Print the Bindings onto a stream - as body of a rule
//
// Alternatively, one could print all active correlations
// But then potentially some predicates are added which do not contribute
// anything [pred(_,_,_,_)], because they appear in the from clause only.

void Bindings::display(ostream& out) const
{
  // Print body according to the current bindings  

  const_iterator j = begin();

  if( j != end() )
    {
      // We have a non-empty body

      out << " :-" << endl;
      
      // now print the first correlation

      correlations->print_correlation(out,(*j).first);

      for(j++; j != current_bindings->end(); j++)
	{
	  cout << "," << endl;

	  // for every correlation, print it.
	  correlations->print_correlation(cout,(*j).first);
	}
    }
  out << "." << endl; 
}

//////////////////////////////////////////////////////////////////////////////
// Bindings::integrate
//
// Create a rulebody (CONJUNCTION) out of the bindings.
//
// NOTE: see Bindings::display

CONJUNCTION* Bindings::integrate() const
{

  CONJUNCTION* b = new CONJUNCTION();

  // Generate body according to the current bindings  

  const_iterator j = begin();

  assert(!(j == end()));

  // We have a non-empty body

  
  for(; !(j == end()); ++j)
    {
      // insert all correlations and their arguments into this body
      
      correlations->integrate_correlation((*j).first,b);

    }

  return b;

}

//////////////////////////////////////////////////////////////////////////////
// SelectList::process
//
// Do everything which needs to be done in a selectlist.

void SelectList::process() const
{

  // step through the list of select expressions
  // and call their [virtual] process procedures

  for(const_iterator sl_iter = begin();
	!(sl_iter == end());
	sl_iter++) 
    {

      // each item in the list is an SSC_Tree

      // if the top node is a RelSpec node, then it's a leaf node
      
      // this is the only thing we handle right now
      
      // *sl_iter has type SSC_Tree *
      // **sl_iter has type SSC_Tree
      // ***sl_iter has type SSC_Node *

      (***sl_iter)->process();
      
    }

}

//////////////////////////////////////////////////////////////////////////////
// SSC_RelSpec::process
//
// Do everything for a Specification of a relation in a SelectList

void SSC_RelSpec::process()
{

  // determine correlation name if it is not given

  if(correlation == "")
    {
      // 1. If the correlationname is "", we have to infer
      //    it from the attribute name.
      //    An ambiguity error may occur here.
      
      correlation 
	= infer_correlation_from_attribute(attribute);
      
    }
  
  // verify that the correlation is active

  if(! correlations->isActive(correlation))
    {
      cerr << "At line " << parser_line << ":" << endl
	   << "Error: Correlation " << correlation 
	   << " doesn't exist in the current context." << endl;
      exit(ERROR_NONEXISTING_CORRELATION);
    }
  	  
  // determine binding
  
  pair<const bool, Binding*> rv 
    = current_bindings->lookup(correlation,attribute);
  
  if(rv.first)
    {
      // Attributes in the select list should appear in the head

      head_arguments->push_back(rv.second);
    }
  else
    {
      // unbound, create new variable
      
      Binding b(variable_names->get_numerical());
      
      // record the new binding
      
      current_bindings->insert(correlation, attribute, b);

      // verify insertion (necessary, because we need a pointer to the
      //                   copied Binding for head_arguments)

      pair<const bool,Binding*> rv1
	= current_bindings->lookup(correlation, attribute);
      
      if(rv1.first)
	{
	  // Attributes in the select list should appear in the head
	  head_arguments->push_back(rv1.second);
	}
      else
	{
	  // Just to be complete.
	  // This should never occur.

          SQLParseError();
	  cerr << "Internal Error: A Binding just registered"
	       << " for a scalar value wasn't found again." << endl;
	  exit(ERROR_INTERNAL);
	}
    }
}

//////

void pretty_print_correlations(ostream& ostr)
{
  ostr << endl << "Correlations:" << endl;
  
  for(Correlations::iterator i=correlations->begin();
      !(i==correlations->end());
      i++)
    {
      ostr << "'" << (*i).first 
	   << "' <-> '" 
	   << (*i).second << "'" << endl;
    }
  ostr << "----" << endl;
}    

void pretty_print_correlations_reverse(ostream& ostr)
  {
    ostr << endl << "Correlations_reverse:" << endl;

    for(CorrelationsReverse::iterator j
	  = correlations_reverse->begin();
	!(j==correlations_reverse->end());
	j++)
      {
	ostr << "'" << (*j).first 
	     << "' <-> '" 
	     << (*j).second << "'" << endl;
      }
    ostr << "----" << endl;
  }

void pretty_print_attribute_affiliations(ostream& ostr)
  {
    ostr << endl << "Attribute Affiliations:" << endl;
    for(AttributeAffiliations::iterator h = attribute_affiliations->begin();
	!(h==attribute_affiliations->end());
	h++)
      {
	cout << "'" << (*h).first << "'"
	     << "-> '" << (*h).second << "'" << endl;
      }
    ostr << "----" << endl;
  }


//////

void print_rules(string& predicatename)
{
  if( FTraceLevel >= 1 )
    {
      cout << predicatename;

      cout << *head_arguments;

      cout << *current_bindings;

    }
}

/////////////////



void integrate_rules(string& predicatename)
{

  DISJUNCTION* h = head_arguments->integrate(predicatename);

  CONJUNCTION* b = current_bindings->integrate();

  IDB.push_back( RULE(h,b) );

}

Binding& scalar_in_conditions_proc(SSC_Traverse& t)
{
  switch((**t)->type)
    {
    case SSC_TYPE_OPERATOR:
      {
	switch(((SSC_Operator *) **t)->op)
	  {
	  case OP_AND:
	    {
	      // Fallthrough
	    }
	  case OP_EQ:
	    {
              SQLParseError();
	      cerr << "Error: Boolean operator encountered" 
		   << " where scalar was expected." << endl;
	      exit(ERROR_BOOLEAN_AS_SCALAR);	      
	    }
	  default:
	    {
              SQLParseError();
	      cerr << "Error: Operators other than = and AND "
		   << "are not supported yet." << endl;
	      exit(ERROR_NOT_YET_IMPLEMENTED);
	    }
	  }
	break;
      }
    case SSC_TYPE_RELSPEC:
      {
	string correlation = ((SSC_RelSpec *) **t)->correlation;
	string attribute = ((SSC_RelSpec *) **t)->attribute;

	if(correlation == "")
	  {
	    // correlation might be missing
	    // infer from the attribute
	    // ambiguity errors may occur here

	    correlation =
	      infer_correlation_from_attribute(attribute);
	  }

	pair<const bool, Binding*> rv
	  = current_bindings->lookup(correlation,attribute);

	if(rv.first)
	  {
	    // A binding already exists
	    return *rv.second;
	  }
	else
	  {
	    // No binding exists yet
	    Binding b(variable_names->get_numerical());
	    
	    // register binding
	    
	    current_bindings->insert(correlation, attribute, b);
	    pair<const bool, Binding*> rv1
	      = current_bindings->lookup(correlation,attribute);
	    if(rv1.first)
	      {
		return *rv1.second;
	      }
	    else
	      {
                SQLParseError();
		cerr << "Internal Error: A Binding just registered"
		     << " for a scalar value wasn't found again." << endl;
		exit(ERROR_INTERNAL);
	      }
	  }
	break;
	
      }
    default:
      {
        SQLParseError();
	cerr << "Internal Error: Undefined SSC_Tree Nodetype";
	exit(ERROR_INTERNAL);
      }
    }

}

void conditions_proc1(SSC_Traverse& t)
{
  switch((**t)->type)
    {
    case SSC_TYPE_OPERATOR:
      {
	switch(((SSC_Operator *) **t)->op)
	  {
	  case OP_AND:
	    {
	      SSC_Traverse t1=t;
	      t1.go_right();
	      t.go_left();
	      conditions_proc1(t);
	      conditions_proc1(t1);
	      break;
	    }
	  case OP_EQ:
	    {
	      SSC_Traverse t1=t;
	      t1.go_right();
	      t.go_left();
	      Binding& bl = scalar_in_conditions_proc(t);
	      Binding& br = scalar_in_conditions_proc(t1);
	      bl.unify(br);
	      break;
	    }
	  default:
	    {
              SQLParseError();
	      cerr << "Error: Operators other than = and AND"
		   << "are not supported yet.";
	      exit(ERROR_NOT_YET_IMPLEMENTED);
	    }
	  }
	break;
      }
    case SSC_TYPE_RELSPEC:
      {
        SQLParseError();
	cerr << "Error: Scalar value encountered"
	     << " where boolean was expected." << endl;
	exit(ERROR_SCALAR_AS_BOOLEAN);
      }
    default:
      {
        SQLParseError();
	cerr << "Internal Error: Undefined SSC_Tree Nodetype";
	exit(ERROR_INTERNAL);
      }
    }
}

void conditions_proc(SSC_Tree& conditions)
{

  if(*conditions != NULL)
    {
      SSC_Traverse t(&conditions);
      conditions_proc1(t);
    }
}

void newcorrelation(string& correlation, string& relation)
{
  // check whether such a correlation already exists
  if(correlations->find(correlation) == correlations->end())
    {
      // ok, no such correlation
      (*correlations)[correlation]=relation;
      correlations_reverse->insert(make_broken_pair(relation,correlation));
    }
  else
    {
      // fault: there is already a correlation with the same name
      cerr << "At Selectlist on line " << parser_line << ":" << endl;
      cerr << "Error: Correlation " << correlation 
	   << " already exists." << endl;
      if(correlation == relation)
	{
	  if((*correlations)[correlation] == correlation)
	    {
	      cerr << "       You probably have the same "
		   << "relation twice in your selectlist," << endl
		   << "       without defining aliases" 
		   << endl;
	    }
	  else
	    {
	      cerr << "       You've used this unaliased relation "
		   << "as a correlation before." << endl;
	    }
	}
      else
	{
	  if((*correlations)[correlation] == correlation)
	    {
	      cerr << "       You've used this correlation as an "
		   << "unaliased relation before." << endl;
	    }
	}	  
      exit(ERROR_AMBIGUOUS_CORRELATION);
    }
}

// Local Variables:
// c-file-style: "dl"
// End:
