//////////////////////////////////////////////////////////////////////////////
// sql.h

#ifndef SQL_H_INCLUDED
#define SQL_H_INCLUDED


#include <limits.h>

#include <list>
#include <map>
#include <vector>

#include <string>
#include <sstream>
#include <iostream>


//////////////////////////////////////////////////////////////////////////////
// The name of this function is due to historical reasons.

template <class S, class T>
inline const pair<const S, T> make_broken_pair(const S& s, const T& t)
{
  const pair<const S,T> constpair = make_pair(s,t);
  return constpair;
}

//////////////////////////////////////////////////////////////////////////////

// this defines the prefix to newly created IDB predicates
#define SQL_INTERNAL_NAME_PREFIX "sql2dl__intern"

// this defines the prefix for variables when the generated datalog rules
// are printed (otherwise variables are just numbers)
#define SQL_INTERNAL_VARNAME_PREFIX "X"

//////////////////////////////////////////////////////////////////////////////

// identifiers for different kinds of operators in SSC_Tree
enum OperatorType {OP_EQ, OP_NEQ, OP_LT, OP_GT, OP_LEQ, OP_GEQ, OP_OR, OP_AND,
		   OP_NOT, OP_PLUS, OP_MINUS, OP_TIMES, OP_FRAC, OP_UMINUS};

// error codes
enum Errors { ERROR_AMBIGUITY_1=1, 
	      ERROR_AMBIGUITY_2, 
	      ERROR_UNIFY_BINDINGS, 
	      ERROR_NOT_YET_IMPLEMENTED,
              ERROR_INTERNAL,
	      ERROR_BOOLEAN_AS_SCALAR,
	      ERROR_SCALAR_AS_BOOLEAN,
	      ERROR_AMBIGUOUS_CORRELATION,
	      ERROR_NONEXISTING_CORRELATION,
	      ERROR_NONEXISTING_RELATION,
	      ERROR_NESTING_TOO_DEEP,
	      ERROR_SQL3_STRICT
};

//////////////////////////////////////////////////////////////////////////////

inline void SQLParseError()
  {
  cerr << "In statement ending at line " << parser_line << ":" << endl;
  }

//////////////////////////////////////////////////////////////////////////////

// simple aggregate objects

// Here the attribute names to a relation are stored.
// The attributes' order is important!

typedef vector<string> Attributes;

// Here the names of the query, to which selects in this scope belong,
// are stored.
// like a stack with non-consuming pop

typedef list<string> Querynames;

// A map connecting relationnames to their attributes.

typedef map<string, Attributes, less<string> > Schemata;

// Multimap associating attributes with the relation(-name)s wherin they
// occur. One attribute can occur in several different relations, hence 
// a multimap. 

typedef multimap<string, string, less<string> > AttributeAffiliations;

// Multimap associating relationnames with correlationnames. Again, one 
// relation can have several correlationnames associated.

typedef multimap<string, string, less<string> > CorrelationsReverse;

//////////////////////////////////////////////////////////////////////////////

// not so simple objects

//////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////
// BinTree<NodeInfo>
//
// the following template describes a generic binary tree
// the parameter is the type of information stored in a node
// ATTENTION: the information is stored as a pointer
//            dereferencing also returns a pointer

template<class T>
class BinTree {
private:

  ///////
  // Node information, stored as pointer
  ///////

  T* element;

  ///////
  // Left and right subtrees
  ///////

  BinTree<T>* left;
  BinTree<T>* right;

  ///////
  // copy subtrees and node info
  ///////
  void copy(const BinTree<T>& src) 
  {
    element = src.element;
    left = new BinTree<T>();
    left = src.left;
    right = new BinTree<T>();
    right = src.right;
  }

  ///////
  // destroy subtrees
  ///////
  void destroy() 
  {
    destroy_left();
    destroy_right();    
  }

  ///////
  // destroy left subtree
  ///////
  void destroy_left() 
  {
    if(left) { delete left; }
  }

  ///////
  // destroy right subtree
  ///////
  void destroy_right() 
  {
    if(right) { delete right; }
  }


public:

  //////
  // Constructor
  ///////
  BinTree(T* el=NULL, BinTree<T>* l = NULL, 
		BinTree<T>* r = NULL) 
    : element(el), left(l), right(r) {}

  ///////
  // Copy Constructor
  ///////
  BinTree(const BinTree<T>& st) 
  {    
    copy(st);
  }

  ///////
  // Destructor
  ///////
  ~BinTree() 
  {    
    destroy();
  }


  ///////
  // Set the left subtree
  ///////
  void push_left(BinTree<T>& st) 
  {
    destroy_left();
    left = &st;
  }

  ///////
  // Set the right subtree
  ///////
  void push_right(BinTree<T>& st) 
  {
    destroy_right();
    right = &st;
  }

  ///////
  // Assignment
  ///////
  const BinTree& operator=(const BinTree<T>& st) 
  {    
    if( &st != this) {
      destroy();
      copy(st);
    }
    return *this;
  }

  ///////
  // Access to the top node's element by dereference operator
  // ATTENTION! A pointer to the info type is returned!
  ///////
  T* operator*() 
  {
    return element;
  }

  ///////
  // get the left subtree
  ///////
  BinTree<T>* lefttree() 
  {
    return left;
  }

  ///////
  // get the right subtree
  ///////
  BinTree<T>* righttree() {
    return right;
  }

};

/////////////////

template<class T>
class BinTreeTraverse {
public:

  // Constructor
  BinTreeTraverse(BinTree<T>* tree = NULL) {
    root = tree;
    current = tree;
  }

  // Copy Constructor
  BinTreeTraverse(const BinTreeTraverse<T>& btt) {
    root = btt.root;
    current = btt.current;
  }
  
  // Assignment
  const BinTreeTraverse<T>& operator=(const BinTreeTraverse<T>& btt) {
    if (&btt != this) {
      root = btt.root;
      current = btt.current;
    }

    return *this;
  }

  void go_root() {
    current = root;
  }

  void go_left() {
    if(current) {
      current = current->lefttree();
    }
  }

  void go_right() {
    if(current) {
      current = current->righttree();
    }
  }

  BinTree<T>& operator*() {
    if(current) {
      return *current;
    }
    //    throw(No_Value);

    return *root;
  }

private:
  BinTree<T>* root;
  BinTree<T>* current;
};

// In the sequel, SSC_ means Select and Search Condition

enum SSC_NodeType { SSC_TYPE_RELSPEC, SSC_TYPE_OPERATOR };

class SSC_Node {
public:
  
  SSC_NodeType type;

  virtual void process() =0;

  virtual ~SSC_Node() {}

};

class SSC_RelSpec : public SSC_Node {
public:

  SSC_RelSpec(string correl = "", string att = "") 
    : correlation(correl), attribute(att) { type=SSC_TYPE_RELSPEC;}

  string correlation;
  string attribute;

  virtual void process();

  virtual ~SSC_RelSpec() {}

private:

};

class SSC_Operator : public SSC_Node {
public:
  SSC_Operator(int o=0) 
    : op(o) { type=SSC_TYPE_OPERATOR; }

  int op;

  virtual void process() 
    {
      SQLParseError();
      cerr << "Error: Complex select lists not supported yet." << endl;
      exit(ERROR_NOT_YET_IMPLEMENTED);
    }

  virtual ~SSC_Operator() {}

};

typedef BinTree<SSC_Node> SSC_Tree;

typedef BinTreeTraverse<SSC_Node> SSC_Traverse;

class SelectList : public list<SSC_Tree*>
{
public:
  void process() const;
};

class Binding {
private:

  // Constant or Variable
  // pointer, because a union is as big as its biggest member

  union
  {
    string* constValue;
    unsigned varValue;
  };
  
  // is this a variable?
  bool isVariable;

  // change value to the given Binding
  void setTo(const Binding& setToThis) 
    {
      if(!isVariable)
	{
	  delete constValue;
	}

      if(setToThis.isVariable)
	{
	  varValue = setToThis.varValue;
	}
      else
	{
	  constValue = new string(*setToThis.constValue);
	}
      isVariable = setToThis.isVariable;
    }


public: 
  // Default Constructor (not really useful, only for a dummy binding)
  Binding() : varValue(0), isVariable(true) {}

  // Construct a constant
  Binding(const string& c) : isVariable(false) 
    {
      constValue = new string(c);
    }

  // Construct a variable
  Binding(unsigned v) : varValue(v), isVariable(true)
    { }

  // Copy constructor
  Binding(const Binding& b)
    { 
      if(b.isVariable)
	{
	  varValue = b.varValue;
	}
      else
	{
	  constValue = new string(*(b.constValue));
	}
      isVariable = b.isVariable;
      
    }

  // Destructor
  ~Binding() 
    {
      if(!isVariable)
	{
	  delete constValue;
	}
    }

  // if the Binding is bound to the specified variable, change it
  void setVariable(const unsigned& varName, const Binding& setToThis)
  {
    if(isVariable && (varValue == varName))
      {

	setTo(setToThis);

      }
  }

  // Assignment: check self-assignment!
  const Binding& operator=(const Binding& b) 
  {
    if(this != &b)
      {
	setTo(b);
      }
    return *this;
  }

  bool operator==(const Binding& b) const 
  { 
    return ((isVariable == b.isVariable) 
	    && isVariable?(varValue == b.varValue):(*constValue == *b.constValue));
  }

  // unify this binding with b
  void unify(Binding& b);

  // test whether we are a variable
  bool isVar() const { return isVariable; }

  // read variable's value
  unsigned getVariable() const
    { 
      if(isVariable) 
	{
	  return varValue;
	}
      return 0;
    }

  // get the constant
  string* getConstant() const
    {
      if(isVariable)
	{
	  return NULL;
	}
      return constValue;
    }

  
  // output onto a stream
  void display(ostream& out) const
    {
      if(isVariable)
	{
	  out << "X" << varValue;
	}
      else
	{
	  out << *constValue;
	}
    }
  
};

//////////////////////////////////////////////////////////////////////////////
// class Bindings
//
// The datastructure for bindings is not so straightforward:
// If we want isolated correlation- and attributenames,
// we need a mapping from two strings to one binding
//
// Solution: Use one map to get from a correlationname
// to another map. Use this map to get from the attributename
// to the binding-structure.

class Bindings : public map<string, 
		            map<string, Binding, less<string> >, 
		            less<string> > 
{
public:

  typedef map<string, Binding, less<string> > Submap;

  typedef map<string, Submap, less<string> > BaseType;

  void setTo(const unsigned& toBeSet, const Binding& setToThis);

  // Get the binding for (correlation, attribute)
  pair<const bool,Binding*> lookup(const string& correlation, 
				   const string& attribute) const;

  // Insert the binding for (correlation, attribute)
  void insert(const string& correlation, 
	      const string& attribute, 
	      Binding& b);

  // Print the binding corresponding to (correlation, attribute)
  void print_corr_att_binding(ostream& out,
			      const string& correlation, 
			      const string& attribute) const;

  // Print all the bindings
  void display(ostream& out) const;

  // Create a body out of the bindings
  // That means, add all correlations, which have bindings.
  // If attributes are not bound, new variables are generated
  CONJUNCTION* integrate() const;
};

//////////////////////////////////////////////////////////////////////////////
// class HeadArguments
//
// HeadArguments is used to store (SQL-)Bindings which should
// appear in the generated datalog rule's head

class HeadArguments : public vector<Binding*>
{
public:

  // Print the arguments
  void display(ostream& out, 
	       const string& separator=", ", 
	       const string& leftbrace="(",
	       const string& rightbrace=")") const;

  // Create a head out of the arguments
  // "DISJUNCTION" is in fact always one atom, 
  // its predicatename must be specified
  DISJUNCTION* integrate(const string& predicatename) const;
};

//////////////////////////////////////////////////////////////////////////////
// class Correlations
//
// A map associating correlationnames to relations.

class Correlations : public map<string, string, less<string> > 
{
public:

  // print the given correlation onto stream
  void print_correlation(ostream& out, const string& correlation) const;

  // insert the given correlation into the given body
  void integrate_correlation(const string& correlation, 
			     CONJUNCTION * const body) const;

  // is this correlation in this container?
  bool isActive(const string& correlation) const 
    {
      const_iterator cit = find(correlation);
  
      return !(cit == end());
    }


};

//////////////////////////////////////////////////////////////////////////////
// class NameSpace
//
// Set up a counter, and get unique names (if no reset occurs).
// The names can be retrieved as numbers or strings (with a custom prefix).

class NameSpace 
  {
  
  public:

    // Constructor
    // Here the prefix is set once and forever
    NameSpace(const char *prefix = "", const unsigned int c = 0) 
      : s(prefix), counter(c) {}

    // retrieve a name as string
    string get() 
      {
	// The name is the concatenation of the prefix and the value
	// of the counter as a string.
	ostringstream h;
	h << s << counter++;
	return h.str();
      }

    // back to the beginning
    void reset()
      {
	counter = 0;
      }

    // retrieve a name as uint
    unsigned int get_numerical()
      {
	return counter++;
      }
      

  private:
    string s;
    unsigned int counter;
  };

///////
// output operator for Binding
///////
inline ostream& operator<<(ostream& out, const Binding& b) 
{ 
  // just call the display function
  b.display(out); 
  // and return the stream
  return out; 
}

///////
// output operator for HeadArguments
///////
inline ostream& operator<<(ostream& out, const HeadArguments& ha) 
{ 
  // just call the display function
  ha.display(out); 
  // and return the stream
  return out; 
}

///////
// output operator for Bindings
///////
inline ostream& operator<<(ostream& out, const Bindings& bindings) 
{ 
  // just call the display function
  bindings.display(out); 
  // and return the stream
  return out; 
}

//////////////////////////////////////////////////////////////////////////////
// Global Variables

extern string SQL_Interesting_Predicate;

extern Schemata* sql_schemata;

extern Querynames* querynames;

extern Correlations* correlations;

extern AttributeAffiliations* attribute_affiliations;

extern CorrelationsReverse* correlations_reverse;

extern Bindings* current_bindings;

extern NameSpace* predicate_names;
extern NameSpace* variable_names;

extern HeadArguments* head_arguments;

extern int query_spec_nesting;

extern bool sql_strict_SQL3;

extern int yydebug; 

//////////////////////////////////////////////////////////////////////////////
// Global Functions

void SQLCallback(const INTERPRET*);

///////////
// several functions for visualising datastructures

void pretty_print_correlations(ostream& ostr = cout);

void pretty_print_correlations_reverse(ostream& ostr = cout);

void pretty_print_attribute_affiliations(ostream& ostr = cout);

///////////
// procedures used in the parser

void print_rules(string& predicatename);

void integrate_rules(string& predicatename);

void conditions_proc(SSC_Tree& conditions);

void newcorrelation(string& correlation, string& relation);

//

extern "C" FILE* sqlyyin; // Where LEX reads its input from 

// BUG-FIX: bison[1.24] fails to prototype its interfaces yylex() and yyerror().
extern "C" int sqlyylex();
extern "C" int sqlyyerror(const char*);

#endif
