//////////////////////////////////////////////////////////////////////////////
// stats.C

#include "dl.h"
#include "stats.h"

typedef set<unsigned,less<unsigned> > VARIABLES;
typedef set<TERM,less<TERM> > CONSTANTS;

//////////////////////////////////////////////////////////////////////////////
/// lpow - poor man's power function for long integers
/// @param base the base
/// @param exp the exponent
/// @return base to the exp-th power
static long lpow(long base,unsigned long exp)
{
  long result=base;

  if (!exp)
    return 1;
  while (--exp)
    result*=base;
  return result;
}

//////////////////////////////////////////////////////////////////////////////
/// adds all variables from literals in a given container to a set.
/// @param container the collection of literals
/// @param vars the set of variables to add to
template <class T>
static void addVarsTo(const T &container, VARIABLES &vars)
    {
    typename T::const_iterator l;

    for (l=container.begin(); l != container.end(); l++)
        {
        if((*l).isRegularAtom())
            {
            const TERMS *params=(*l).getParams();

            if( params )
                {
                for (TERMS::const_iterator i=params->begin(); 
                    i!=params->end(); i++)
                    {
                    if ((*i).isVar())
                        vars.insert((*i).getVar());
                    }
                }
            }//if((*l).isRegularAtom())
        }
    }

//////////////////////////////////////////////////////////////////////////////
/// computes the theoretical number of ground atoms of given type.
/// Simple grounding is accomplished by replacing the set of variables in each
/// atom with all combinations of constants.
///
/// @param type the type of atoms to consider
/// @param NConstants the size of the Herbrand Universe
/// @return the number of ground atoms of this type
unsigned tallyGroundAtoms(
    PREDICATE_NAMES::TYPE type,
    unsigned              NConstants )
    {
    unsigned count=0;

    for ( size_t i = NBuiltins; i < ATOM::Predicates.size(); i++ )
        {
        if( ATOM::Predicates.getType(i) == type )
            count+=lpow(NConstants,ATOM::Predicates.getArity(i));
        }

    return count;
    }

//////////////////////////////////////////////////////////////////////////////
/// computes the theoretical number of ground rules.
/// Simple grounding is accomplished by replacing the set of variables in each
/// rule with all combinations of constants.
/// @param r the (possibly) non-ground rules
/// @param NConstants the size of the Herbrand Universe
/// @return the number of ground rules, that this scheme would yield.
unsigned tallyGroundRules(
    const RULES& r,
    unsigned     NConstants )
    {
    unsigned count=0;

    for(RULES::const_iterator i=r.begin(); i != r.end(); i++)
        {
        VARIABLES vars;
        const CONJUNCTION *body=(*i).getBody();

        if( body ) 
            addVarsTo(*body,vars);
        addVarsTo(i->getHead(),vars);
        count+=lpow(NConstants,vars.size());
        }

    return count;
    }

//////////////////////////////////////////////////////////////////////////////
/// computes the theoretical number of ground constraints.
/// Simple grounding is accomplished by replacing the set of variables in each
/// constraint with all combinations of constants.
/// @param c the (possibly) non-ground constraints
/// @param NConstants the size of the Herbrand Universe
/// @return the number of ground constraints, that this scheme would yield.
unsigned tallyGroundConstraints(
    const CONSTRAINTS& c,
    unsigned           NConstants )
    {
    unsigned count=0;

    for( CONSTRAINTS::const_iterator i=c.begin(); i != c.end(); i++)
        {
        VARIABLES vars;

        addVarsTo(*i,vars);
        count+=lpow(NConstants,vars.size());
        }
    return count;
    }

//////////////////////////////////////////////////////////////////////////////
/// determine the Structural Size of a program, that is, the sum of literals
/// (multiple occurrences count separately) in the entire program.
unsigned tallyStructuralSize(
    const GRULES &r,
    const GCONSTRAINTS &c,
    const GWEAKCONSTRAINTS &w )
    {
    unsigned count=0;

    for( GRULES::const_iterator i=r.begin(); i != r.end(); i++)
        {
        count += i->getHead().size();

        if( i->hasBody() )
            {
            const GCONJUNCTION &c = *(i->getBody());

            count += c.size();

            // Every member of an AGGREGATESET also counts as one.
            for( GCONJUNCTION::const_iterator j=c.begin();
                 j != c.end();
                 j++ )
                if( j->isAggregate() )
                    count += (j->getAggregate().getAggregateSet()).size();
            }
        }

    for( GCONSTRAINTS::const_iterator i=c.begin(); i != c.end(); i++)
        {
        count += i->size();

        // Every member of an AGGREGATESET also counts as one.
        for( GCONJUNCTION::const_iterator j = i->begin();
                 j != i->end();
                 j++ )
                if( j->isAggregate() )
                    count += (j->getAggregate().getAggregateSet()).size();
        }

    for( GWEAKCONSTRAINTS::const_iterator i=w.begin(); i != w.end(); i++)
        count += i->size();
    
    return count;
    }

// Local Variables:
// c-file-style: "dl"
// End:
