//////////////////////////////////////////////////////////////////////////////
// timer.h

#ifndef TIMER_H
#define TIMER_H

#include <iostream>
#include <iomanip>
#ifndef assert
# include <assert.h>
#endif

#ifdef __WIN32__

#include <windows.h>

class TIMER
    {
    long started, sum;
    unsigned intervalls, empty_intervalls;
       
    long get() const;

public:
    TIMER() 
        : started(0), sum(0), intervalls(0), empty_intervalls(0)
        {
        }

    TIMER(const TIMER &t)
        : started(t.started),
          sum(t.sum),
          intervalls(t.intervalls),
          empty_intervalls(t.empty_intervalls)
        {
        }

    void operator=(const TIMER&)
        {
        assert( 0 );
        }

    void start();
    void stop();
 
    bool empty() const;

    friend ostream& operator<< (ostream&, const TIMER&);
    };

//////////////////////////////////////////////////////////////////////////////

inline void TIMER::start()
    {
    started=GetTickCount();
    }

inline void TIMER::stop()
    {
    assert(started);
    value+=(GetTickCount()-started);
    }

inline long TIMER::get() const
    {
    return value;
    }

inline bool TIMER::empty() const
    {
    return get() == 0;
    }

//////////////////////////////////////////////////////////////////////////////

inline ostream& operator<< (ostream& out, const TIMER &timer)
    {
    out << (double)timer.get() / 1000 << "s";
    return out;
    }



#else  /***** defined(__WIN32__) ********************************************/



#include <sys/time.h>

#define MICROS 1000000        /* the number of microseconds in a second */

class TIMER
    {
    struct timeval value;
    bool running;

    void add(const struct timeval &);

    const struct timeval &get() const;
public:
    TIMER()
        : running(false)
        {
        value.tv_sec = value.tv_usec = 0;
        }

    TIMER(const TIMER &t)
        : value(t.value),
          running(t.running)
        {
        }

    void operator=(const TIMER&)
        {
        assert( 0 );
        }

    void start();
    void stop();

    bool empty() const;

    double getAsDouble() const;

    TIMER operator -(const TIMER &) const;
    TIMER operator +(const TIMER &) const;

    friend ostream& operator<< (ostream&, const TIMER&);
    };

//////////////////////////////////////////////////////////////////////////////

inline void TIMER::start()
    {
    struct timeval started;

    running = true;
    if (gettimeofday(&started, NULL) < 0)
        {
        cerr << "Warning: gettimeofday failed, timing results will be wrong\n";
        return;
        }
    started.tv_sec = -started.tv_sec;
    started.tv_usec = -started.tv_usec;
    add(started);
    }

inline void TIMER::stop()
    {
    struct timeval stopped;

    assert(running);
    running = false;
    if (gettimeofday(&stopped, NULL) < 0)
        {
        cerr << "Warning: gettimeofday failed, timing results will be wrong\n";
        return;
        }
    add(stopped);
    }

inline const struct timeval& TIMER::get() const
    {
    return value;
    }

inline bool TIMER::empty() const
    {
    return value.tv_sec == 0 && value.tv_usec == 0;
    }

inline double TIMER::getAsDouble() const
    {
    return (double)value.tv_usec/MICROS + (double)value.tv_sec;
    }
 
inline void TIMER::add(const struct timeval &x)
    {
    value.tv_sec += x.tv_sec;
    value.tv_usec += x.tv_usec;
    if (value.tv_usec < 0)
        {
        value.tv_sec--;
        value.tv_usec += MICROS;
        assert(value.tv_usec >= 0);
        }
    else if (value.tv_usec >= MICROS)
        {
        value.tv_sec++;
        value.tv_usec -= MICROS;
        assert(value.tv_usec < MICROS);
        }
    }

inline TIMER TIMER::operator -(const TIMER &that) const
    {
    TIMER diff;

    diff.value.tv_sec = -that.value.tv_sec;
    diff.value.tv_usec = -that.value.tv_usec;
    diff.add(value);

    return diff;
    }

inline TIMER TIMER::operator +(const TIMER &that) const
    {
    TIMER sum=that;

    sum.add(value);

    return sum;
    }

//////////////////////////////////////////////////////////////////////////////

inline ostream& operator<< (ostream& out, const TIMER &timer)
    {
    long s, us;

    s = timer.get().tv_sec;
    us = timer.get().tv_usec;
    if (s >= 0 && us >= 0)
        out << s << '.' << setw(6) << setfill('0') << us;
    else
        out << "TIMING ERROR."; 
    return out;
    }

#endif /* defined(__WIN32__) */

#endif
