#!/bin/sh

# generates a snippet for inclusion into a gnuplot script file
#
# Usage: $0 [options] XXX-summary.txt TESTS n incr
# where options includes
#  --combine                  calculate an overall score based on first baseline
#  --baseline YYY.txt label   baseline values to be added to the separate plots
#  --skip N                   skip the first N values
#  --no-summary		      do not create a summary plot
#  --title "blah"    	      title to use for the plot(s)
#  --ylabel "foo"	      label to use for the y-axis
# XXX-summary.txt is the filename the raw data producing the plots can
#   be read from
# TESTS is supposed to be a filename that contains labels for the individual
#   tests
# N is the index in the set of sub-tests, where the next group is reached
#   by incrementing the column number by INCR.

# The --combine switch generates a plot with a weighted average number
# calculated using the first specified baseline as follows:
#   Score = Sum_n ( Data[i] / Baseline[i] ) / n
# where the sum is over all data points i = 1..n;  that means, a Score
# of 1.0 indicates the same (average) performance as the baseline.
# Note that in this mode the other baselines are not included in the plot.

if test -z "$1"; then
  exit 0
fi

# echo command
echo
echo -n "# $0 "

combine="no"
summary="yes"
BASELINE=
BASELINES=
LABELS=
TITLE=
YLABEL=
SKIP=0
while true; do
  echo -n "$1 "
  if test "$1" == "--combine"; then
    combine="yes"
  elif test "$1" == "--no-summary"; then
    summary="no"
  elif test "$1" == "--baseline"; then
    shift
    echo -n "$1 $2 "
    if test -z "$BASELINE"; then
      BASELINE="$1"
      BASELINES="$1"
      LABELS="$2"
    else
      BASELINES="$BASELINES $1"
      LABELS="$LABELS $2"
    fi
    shift
  elif test "$1" == "--skip"; then
    shift
    echo -n "$1 "
    SKIP="$1"
  elif test "$1" == "--title"; then
    shift
    echo -n "\"$1\" "
    TITLE="$1"
  elif test "$1" == "--ylabel"; then
    shift
    echo -n "\"$1\" "
    YLABEL="$1"
  else
    break
  fi
  shift
done
echo "$2 $3 $4"

TXTNAME="$1"
TESTS="`cat $2`"
N="$3"
INCR="$4"

if test -z "$BASELINE" && test "$combine" = "yes"; then
  echo "--combine needs a baseline"
  exit 1
fi

# global stuff
if ! test -z "$YLABEL"; then
  echo -e -n "set ylabel \"$YLABEL\"\n\n"
fi

# a summary plot
if test "$summary" == "yes"; then
if test "$combine" == "no"; then
field=$[1 + $SKIP]
first=1
part=0
count=0
echo -e -n "set logscale y\n"
echo -e -n "set output \"$TXTNAME-$N-$part.png\"\n"
if ! test -z "$TITLE"; then
  echo -e -n "set title \"$TITLE\"\n"
fi
echo -e -n "plot [:] [:] \"$TXTNAME\" "
echo -n > $TXTNAME-$N-$part.html
for tst in $TESTS; do
  f=$[$field + $N]
  if test $first == 0; then
    echo -e -n ", \\\\\n\t\"\""
  fi
  echo -e -n " u 1:$f title \"$tst\" w lp"
  echo -e -n "<IMG SRC=\"$TXTNAME-$tst-$f.png\">\n" >> $TXTNAME-$N-$part.html
  field=$[$field + $INCR]
  first=0
  count=$[$count + 1]
  if test $count == 32; then
    count=0
    first=1
    part=$[$part + 1]
    echo -e -n "\n"
    echo -e -n "set output \"$TXTNAME-$N-$part.png\"\n"
    echo -e -n "plot [:] [:] \"$TXTNAME\" "
    echo -n > $TXTNAME-$N-$part.html
  fi
done
echo -e -n "\nset nologscale\n\n"

# different kind of summary plot
else
field=$[1 + $SKIP]
first=1
count=0
echo -e -n "set output \"$TXTNAME-$N.png\"\n"
if ! test -z "$TITLE"; then
  echo -e -n "set title \"$TITLE weighted average relative to $BASELINE\"\n"
fi
echo -e -n "plot [:] [:] \"$TXTNAME\" u 1:("
echo -n > $TXTNAME-$N.html
for tst in $TESTS; do
  f=$[$field + $N]
  baselineval=`cat $BASELINE | cut -d ' ' -f $f | tr -d '\n'`
  if ! test "$baselineval" == "0" \
     && ! test "$baselineval" == "0.0" \
     && ! test -z "$baselineval"; then
    if test $first == 0; then
      echo -e -n "+"
    fi
    echo -e -n "\$$f/$baselineval"
    count=$[$count + 1]
    first=0
  fi
  echo -e -n "<IMG SRC=\"$TXTNAME-$tst-$f.png\">\n" >> $TXTNAME-$N.html
  field=$[$field + $INCR]
done
echo -e -n ")/$count title \"overall score\" w l\n"
fi
fi

# separate plots
field=$[1 + $SKIP]
for tst in $TESTS; do
  f=$[$field + $N]
  echo -e -n "set output \"$TXTNAME-$tst-$f.png\"\n"
  if ! test -z "$TITLE"; then
    echo -e -n "set title \"$TITLE for $tst\"\n"
  fi
  echo -e -n "plot [:] [:] \"$TXTNAME\" u 1:$f title \"$tst\" w lp"
  bslnr=1
  for baseline in $BASELINES; do
    bltitle=`echo $LABELS | cut -d ' ' -f $bslnr`
    echo -e -n ", \\\\\n\t"
    cat $baseline | cut -d ' ' -f $f | tr -d '\n'
    echo -e -n " title \"$bltitle\" w l"
    bslnr=$[$bslnr + 1]
  done
  echo -e -n "\n"
  field=$[$field + $INCR]
done

# footer
echo "set output"
echo
