/* -*- C++ -*-

   This file declares the Weaver, Job and Thread classes.

   $ Author: Mirko Boehm $
   $ Copyright: (C) 2004, Mirko Boehm $
   $ Contact: mirko@kde.org
         http://www.kde.org
         http://www.hackerbuero.org $
   $ License: LGPL with the following explicit clarification:
         This code may be linked against any version of the Qt toolkit
         from Troll Tech, Norway. $

*/

#ifndef WEAVER_H
#define WEAVER_H

extern "C" {

    #include <stdarg.h>
    #include <unistd.h>
    #include <stdio.h>

}

#include <qobject.h>
#include <qptrlist.h>
#include <qthread.h>
#include <qwaitcondition.h>
#include <qmutex.h>
#include <qevent.h>

class Thread;
class Job;

class Event : public QCustomEvent
{

    public:
    
        enum Action {
        NoAction = 0,
        Finished, /// All jobs in the queue are done.
        Suspended, /// Thread queueing halted.
        ThreadStarted,
        ThreadExiting,
        ThreadBusy,
        ThreadSuspended,
        JobStarted,
        JobFinished,
        JobSPR, /// Synchronous Process Request
        JobAPR  /// Asynchronous Process Request
};
        
    Event ( Action = NoAction, Thread * = 0, Job *job = 0);
    static const int type ();
    Thread* thread () const;
    Job* job () const;
    Action action () const;
    
    private:
        
        Action m_action;
        Thread *m_thread;
        Job *m_job;
        static const int Type;
    };

    
class Job : public QObject
{
    Q_OBJECT

    public:
        
        Job(QObject* parent=0, const char* name=0);
        virtual ~Job();
        virtual void execute(Thread*);
        virtual bool isFinished() const;
        void wakeAPR ();
        virtual void processEvent ( Event* );

    signals:

        void started ();
        void done ();
        void SPR ();
        void APR ();
    
    protected:
        
        void lock();
        void unlock();
        virtual void run () = 0;
        Thread *thread();
        virtual void setFinished(bool status);
        void triggerSPR ();
        void triggerAPR ();
        bool m_finished;
        QMutex *m_mutex;
        Thread * m_thread;
        QWaitCondition *m_wc;
};

class Weaver;

class Thread : public QThread
{
    
    public:
        
        Thread(Weaver *parent);
        ~Thread();

        void run();
        void msleep(unsigned long msec);
        const unsigned int id() const;
        void post (Event::Action, Job* = 0);

    private:
        
        Weaver *m_parent;
        const unsigned int m_id;
        static unsigned int sm_Id;
        static unsigned int makeId();

};

class Weaver : public QObject
{
    Q_OBJECT
    
    public:
        Weaver (QObject* parent=0, const char* name=0,
                int inventoryMin = 4, // minimal number of provided threads
                int inventoryMax = 32); // maximum number of provided threads
        virtual ~Weaver ();
        virtual void enqueue (Job*);
        void enqueue (QPtrList<Job> jobs);
        virtual bool dequeue (Job*);
        virtual void dequeue ();
        virtual void finish();
        virtual void suspend (bool state);
        bool isEmpty () const;
        bool isIdle () const;
        int queueLength ();
        virtual Job* applyForWork (Thread *thread, Job *previous);
        void lock ();
        void unlock ();
        void post (Event::Action, Thread* = 0, Job* = 0);
        int threads () const;
    
    signals:
        
        void finished ();
        void suspended ();
        void jobDone (Job*);

        void threadCreated (Thread *);
        void threadDestroyed (Thread *);
        void threadBusy (Thread *);
        void threadSuspended (Thread *);

    protected:
        
        void assignJobs();
        bool event ( QEvent* );
        QPtrList<Thread> m_inventory;
        QPtrList<Job> m_assignments;
        int m_active;
        int m_inventoryMin;
        int m_inventoryMax;
        QWaitCondition m_jobAvailable;
        QWaitCondition m_jobFinished;
        bool m_shuttingDown;
        bool m_running;
        bool m_suspend;
    
    private:
    
        QMutex *m_mutex;

};

#endif // defined WEAVER_H
