/*
 * HWDEP Interface for HD-audio codec
 *
 * Copyright (c) 2007 Takashi Iwai <tiwai@suse.de>
 *
 *  This driver is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This driver is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#ifndef __SOUND_HDA_HWDEP_H
#define __SOUND_HDA_HWDEP_H

#define HDA_HWDEP_VERSION	((1 << 16) | (0 << 8) | (0 << 0)) /* 1.0.0 */

/* status */
enum {
	HDA_STATUS_UNINITIALIZED,
	HDA_STATUS_RUNNING,
};

#define HDA_CODEC_NAME_LEN	64
#define HDA_CODEC_SPEC_LEN	64

/* codec information */
struct hda_codec_info {
	/* codec ids */
	u32 vendor_id;
	u32 subsystem_id;
	u32 revision_id;
	/* PCI ids */
	u16 pci_vendor;
	u16 pci_device;
	u16 pci_subvendor;
	u16 pci_subdevice;
	/* codec slot and widgets */
	u16 codec_addr;
	u16 afg;
	u16 mfg;
	u16 num_nodes;
	u16 start_nid;
	u16 reserved;
	/* set by hda-tool */
	char name[HDA_CODEC_NAME_LEN];
	/* misc info (freely used by hda-tool) */
	u32 spec[HDA_CODEC_SPEC_LEN];
};

/* verb */
#define HDA_REG_NID_SHIFT	24
#define HDA_REG_VERB_SHIFT	8
#define HDA_REG_VAL_SHIFT	0
#define HDA_VERB(nid,verb,param)	((nid)<<24 | (verb)<<8 | (param))

struct hda_verb_ioctl {
	u32 verb;	/* HDA_VERB() */
	u32 res;	/* response */
};

/* unsolicited event */
struct hda_unsol_ioctl {
	u16 nid;	/* widget nid */
	u8 tag;		/* identifier tag */
	u8 shift;	/* shift bit (usually 26) */
};

#define HDA_REG_USR0		0
#define HDA_MAX_USER_REGS	32

enum {
	HDA_CMD_TYPE_NOP,		/* no value */
	HDA_CMD_TYPE_INT,		/* INT(val) */
	HDA_CMD_TYPE_LABEL,		/* string */
	HDA_CMD_TYPE_LIST,		/* list */
	HDA_CMD_TYPE_CMDS,		/* list of commands */
	HDA_CMD_TYPE_IF,		/* (CMD(cond) CMD(yes) CMD(no) */
	HDA_CMD_TYPE_NOT,		/* CMD(A) */
	HDA_CMD_TYPE_AND,		/* (CMD(A) CMD(B)) */
	HDA_CMD_TYPE_OR,		/* (CMD(A) CMD(B)) */
	HDA_CMD_TYPE_SHIFTL,		/* (CMD(A) CMD(B)) */
	HDA_CMD_TYPE_SHIFTR,		/* (CMD(A) CMD(B)) */
	HDA_CMD_TYPE_PLUS,		/* (CMD(A) CMD(B)) */
	HDA_CMD_TYPE_MINUS,		/* (CMD(A) CMD(B)) */
	HDA_CMD_TYPE_LNOT,		/* CMD(A) */
	HDA_CMD_TYPE_LAND,		/* (CMD(A) CMD(B)...) */
	HDA_CMD_TYPE_LOR,		/* (CMD(A) CMD(B)...) */
	HDA_CMD_TYPE_GET_REG,		/* INT(id) */
	HDA_CMD_TYPE_SET_REG,		/* (INT(id) INT(val)) */
	HDA_CMD_TYPE_VERBS = 0x20,	/* write array of u32 verbs */
	HDA_CMD_TYPE_AMP_READ,		/* INT(AMP) */
	HDA_CMD_TYPE_VERB_UPDATE,	/* (VERB VAL) */
	HDA_CMD_TYPE_AMP_UPDATE,	/* (AMP VAL) */
	HDA_CMD_TYPE_CTL_NOTIFY,	/* INT(numid) */
	HDA_CMD_TYPE_SET_CHANNELS,	/* INT(channels) */
	HDA_CMD_TYPE_DELAY,		/* INT(msec) */
	HDA_CMD_TYPE_LAST,
};

struct hda_tlv {
	u16 type;
	u16 len;
	char val[0];
};

/* value for HDA_CMD_TYPE_AMP_UPDATE */
#define HDA_AMP_NID_SHIFT	24
#define HDA_AMP_CH_SHIFT	23
#define HDA_AMP_DIR_SHIFT	22
#define HDA_AMP_IDX_SHIFT	16
#define HDA_AMP_MASK_SHIFT	0
#define HDA_AMP_IDX(nid,ch,diridx,mask)					\
	(((nid) << HDA_AMP_NID_SHIFT ) |				\
	 ((ch) << HDA_AMP_CH_SHIFT) |					\
	 ((diridx) << HDA_AMP_IDX_SHIFT) |				\
	 ((mask) << HDA_AMP_MASK_SHIFT))

/* mixer type */
enum {
	HDA_MIXER_TYPE_VOLUME,
	HDA_MIXER_TYPE_SWITCH,
	HDA_MIXER_TYPE_BIND_VOLUME,
	HDA_MIXER_TYPE_BIND_SWITCH,
	HDA_MIXER_TYPE_ENUM,
	HDA_MIXER_TYPE_BOOLEAN
};

struct hda_mixer_head {
	u32 numid;		/* num id: stored in return from driver */
	u32 iface;		/* interface id */
	u32 device;		/* device number */
	u32 subdevice;		/* subdevice number */
	char name[44];		/* ASCII name */
	u32 index;		/* index of item */
	u32 type;		/* HDA_MIXER_TYPE_XXX */
};

struct hda_volsw_mixer_head {
	struct hda_mixer_head head;
	u32 value[1];
};

struct hda_enum_mixer_head {
	struct hda_mixer_head head;
	u32 cur_val;
	u32 reserved;
	struct hda_tlv tlv[0];
	
};

#define hda_bool_mixer_head	hda_enum_mixer_head

/* standard PCM */
enum {
	HDA_PCM_TYPE_ANALOG_MULTI_OUT,
	HDA_PCM_TYPE_ANALOG_IN,
	HDA_PCM_TYPE_SPDIF_OUT,
	HDA_PCM_TYPE_SPDIF_IN,
	HDA_PCM_TYPE_EXT_OUT,
	HDA_PCM_TYPE_EXT_IN,
};

struct hda_usr_pcm_info {
	u8 type;		/* PCM type: HDA_PCM_TYPE_XXX */
	u8 is_modem;		/* is modem? */
	u8 substreams;		/* number of substreams */
	u8 channels_min;	/* min number of channels */
	u8 channels_max;	/* max number of channels */
	u16 nid;		/* reference widget NID */
	u32 rates;		/* supported rates (optional) */
	u32 maxbps;		/* supported max bit per sample */
	u64 formats;		/* supported formats */
	/* assigned widgets */
	u16 assoc_nids[8];
	u16 extra_nids[8];
	/* device attributes */
	char name[32];		/* PCM name */
	u32 device;		/* device number */
};

/*
 * ioctls
 */
#define HDA_IOCTL_PVERSION		_IOR('H', 0x10, int)
#define HDA_IOCTL_VERB_WRITE		_IOWR('H', 0x11, struct hda_verb_ioctl)
#define HDA_IOCTL_GET_WCAP		_IOWR('H', 0x12, struct hda_verb_ioctl)
#define HDA_IOCTL_GET_STATUS		_IOR('H', 0x13, int)
#define HDA_IOCTL_SET_STATUS		_IOW('H', 0x14, int)
#define HDA_IOCTL_GET_CODEC_INFO	_IOR('H', 0x15, struct hda_codec_info)
#define HDA_IOCTL_SET_CODEC_INFO	_IOW('H', 0x16, struct hda_codec_info)
#define HDA_IOCTL_CMD_EXEC		_IOWR('H', 0x17, int)
#define HDA_IOCTL_ADD_MIXER		_IOWR('H', 0x20, struct hda_mixer_head)
#define HDA_IOCTL_ADD_PCM		_IOWR('H', 0x30, struct hda_usr_pcm_info)
#define HDA_IOCTL_ADD_INIT		_IOWR('H', 0x50, int)
#define HDA_IOCTL_ADD_RESUME		_IOWR('H', 0x51, int)
#define HDA_IOCTL_ADD_UNSOL		_IOWR('H', 0x52, struct hda_unsol_ioctl)
#define HDA_IOCTL_RESET			_IO('H', 0x60)

#endif
