/*
 * Convert HD-audio command sequences to TLV stream
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#ifndef __HDA_CMDS_H_INC
#define __HDA_CMDS_H_INC

#include "tlv.h"

/*
 * items for verb array
 */
struct hda_verb {
	u16 nid;
	u16 verb;
	u32 param;
};

/*
 * array of standard commands
 */
struct hda_cmds {
	int type;	/* HDA_CMD_TYPE_XXX, or -1 as terminator */
	long val;
};

/*
 * AMP verbs
 */
#define AMP_VERB(nid,ch,diridx,mask)	HDA_AMP_IDX(nid, ch, diridx, mask)
#define AMP_VERB_NULL 0

/*
 * direction value for amp controls
 */
#define HDA_AC_OUT	0			/* output */
#define HDA_AC_IN(x)	((1 << 6) | (x))	/* input with index */

/*
 * standard commands
 */
/* terminator for STD_CMDS() array */
#define STD_NULL \
	{ .type = -1, .val = 0 }
/* NOP command */
#define STD_NOP \
	{ .type = HDA_CMD_TYPE_NOP, .val = 0 }

/* Integer */
#define STD_INT(x) \
	{ .type = HDA_CMD_TYPE_INT, .val = (x) }

#define STD_SET_CHANNELS(n) \
	{ .type = HDA_CMD_TYPE_SET_CHANNELS, .val = (n) }

#define STD_DELAY(ms) \
	{ .type = HDA_CMD_TYPE_DELAY, .val = (ms) }

#define STD_NOTIFY(ctl) \
	{ .type = HDA_CMD_TYPE_CTL_NOTIFY, .val = (long) (ctl) }

/* Takes struct hda_verb array */
#define STD_VERBS(verbs) \
	{ .type = HDA_CMD_TYPE_VERBS, .val = (long) (verbs) }

/* One verb */
#define STD_VERB(nid,verb,param) \
	{ .type = HDA_CMD_TYPE_VERBS, .val = (long) & (const struct hda_verb []) \
		{ {nid, verb, param}, {} } }

/* Take a string (char pointer) */
#define STD_LABEL(label) \
	{ .type = HDA_CMD_TYPE_LABEL, .val = (long) (label) }

/* Take struct hda_cmds array, terminated by STD_NULL */
#define STD_CMDS(cmds) \
	{ .type = HDA_CMD_TYPE_CMDS, .val = (long) (cmds) }

/* Take variable-length arguments (STD_NULL is _not_ necessary) */
#define STD_CMD_LIST(cmds...) \
	{ .type = HDA_CMD_TYPE_CMDS, .val = (long) (const struct hda_cmds []) \
		{ cmds, STD_NULL } }

struct hda_cmd_if {
	struct hda_cmds cond;
	struct hda_cmds yes;
	struct hda_cmds no;
};

struct hda_cmd_two_ops {
	struct hda_cmds a;
	struct hda_cmds b;
};

/* IF (xcond) THEN (xyes) */
#define STD_IF(xcond, xyes) \
	{ .type = HDA_CMD_TYPE_IF,	\
	  .val = (long) & (const struct hda_cmd_if) \
		{ .cond = xcond, .yes = xyes, .no = STD_NOP } }
/* IF (xcond) THEN (xyes) ELSE (xno) */
#define STD_IF_ELSE(xcond, xyes, xno) \
	{ .type = HDA_CMD_TYPE_IF,	\
	  .val = (long) & (const struct hda_cmd_if) \
		{ .cond = xcond, .yes = xyes, .no = xno } }
/* bitwise not (~) */
#define STD_NOT(xcond)	\
	{ .type = HDA_CMD_TYPE_NOT, \
		.val = (long) & (const struct hda_cmds) xcond }
/* logical not (!) */
#define STD_LNOT(xcond)	\
	{ .type = HDA_CMD_TYPE_LNOT, \
		.val = (long) & (const struct hda_cmds) xcond }

/* bitwise and (&) */
#define STD_AND(val1,val2)	\
	{ .type = HDA_CMD_TYPE_AND, \
	  .val = (long) & (const struct hda_cmd_two_ops) \
		{ .a = val1, .b = val2 } }
/* bitwise or (!) */
#define STD_OR(val1,val2)	\
	{ .type = HDA_CMD_TYPE_OR, \
	  .val = (long) & (const struct hda_cmd_two_ops) \
		{ .a = val1, .b = val2 } }
/* arithmetic plus (+) */
#define STD_PLUS(val1,val2)	\
	{ .type = HDA_CMD_TYPE_PLUS, \
	  .val = (long) & (const struct hda_cmd_two_ops) \
		{ .a = val1, .b = val2 } }
/* arithmetic minus (-) */
#define STD_MINUS(val1,val2)	\
	{ .type = HDA_CMD_TYPE_MINUS, \
	  .val = (long) & (const struct hda_cmd_two_ops) \
		{ .a = val1, .b = val2 } }
/* left shift bit (<<) */
#define STD_SHIFTL(val1,val2)	\
	{ .type = HDA_CMD_TYPE_SHIFTL, \
	  .val = (long) & (const struct hda_cmd_two_ops) \
		{ .a = val1, .b = val2 } }
/* right shift bit (<<) */
#define STD_SHIFTR(val1,val2)	\
	{ .type = HDA_CMD_TYPE_SHIFTR, \
	  .val = (long) & (const struct hda_cmd_two_ops) \
		{ .a = val1, .b = val2 } }

/* logical and (&&) */
/* takes an struct hda_cmds array, terminated by STD_NULL */
#define STD_LAND(list)	\
	{ .type = HDA_CMD_TYPE_LAND, .val = (long) (list) }
/* takes variable length arguemnts (without STD_NULL) */
#define STD_LAND_LIST(cmds...)	\
	{ .type = HDA_CMD_TYPE_LAND, \
		.val = (long) (const struct hda_cmds []) { cmds, STD_NULL } }
/* logical or (||) */
/* takes an struct hda_cmds array, terminated by STD_NULL */
#define STD_LOR(list)	\
	{ .type = HDA_CMD_TYPE_LOR, .val = (long) (list) }
/* takes variable length arguemnts (without STD_NULL) */
#define STD_LOR_LIST(cmds...)	\
	{ .type = HDA_CMD_TYPE_LOR, \
		.val = (long) (const struct hda_cmds []) { cmds, STD_NULL } }

/* use two_args: a = mask, b = val */
#define STD_VERB_UPDATE(nid,verb,mask,cmd)	\
	{ .type = HDA_CMD_TYPE_VERB_UPDATE, \
	  .val = (long) & (const struct hda_cmd_two_ops)	\
		{ .a = STD_INT(HDA_VERB(nid, verb, mask)), .b = cmd } }
#define STD_AMP_UPDATE(nid,ch,diridx,mask,cmd) \
	{ .type = HDA_CMD_TYPE_AMP_UPDATE, \
	  .val = (long) & (const struct hda_cmd_two_ops)	\
		{ .a = STD_INT(AMP_VERB(nid, ch, diridx, mask)), .b = cmd } }

/* Read an amp value */
#define STD_AMP(nid,ch,diridx,mask) \
	{ .type = HDA_CMD_TYPE_AMP_READ, .val = AMP_VERB(nid,ch,diridx,mask) }

/* Macro to do mute on a stereo widget */
#define STD_STEREO_MUTE(nid, dir) \
	STD_CMD_LIST(STD_AMP_UPDATE(nid, 0, dir, 0x80, STD_INT(0x80)),\
		     STD_AMP_UPDATE(nid, 1, dir, 0x80, STD_INT(0x80)))
/* Macro to do unmute on a stereo widget */
#define STD_STEREO_UNMUTE(nid, dir) \
	STD_CMD_LIST(STD_AMP_UPDATE(nid, 0, dir, 0x80, STD_INT(0x00)),\
		     STD_AMP_UPDATE(nid, 1, dir, 0x80, STD_INT(0x00)))

/*
 * get/set user register
 */
#define STD_GET_REG(id) \
	{ .type = HDA_CMD_TYPE_GET_REG, .val = (id) }
#define STD_SET_REG(id,xval)		\
	{ .type = HDA_CMD_TYPE_SET_REG, \
		.val = (long) (const struct hda_cmds []) { \
			STD_INT(id), xval } }


/*
 * extended commands - converted to other types later
 */

enum {
	HDA_CMD_TYPE_JACK_SENSE = 0x40
};

struct hda_jack_sense {
	unsigned int nid;
	struct hda_cmds present;
	struct hda_cmds not_present;
};

#define STD_JACK_SENSE(xnid,xpres,xnot)	\
	{ .type = HDA_CMD_TYPE_JACK_SENSE, \
	  .val = (long) & (const struct hda_jack_sense) \
		{ .nid = xnid, .present = xpres, .not_present = xnot } }

/*
 * convert to TLV
 */
struct hda_tlv *hda_cmd_to_tlv(const struct hda_cmds *cmds);
struct hda_tlv *hda_cmd_seq_to_tlv(int type, const struct hda_cmds *cmds);
struct hda_tlv *hda_verbs_to_tlv(const struct hda_verb *verbs);
struct hda_tlv *hda_amps_to_tlv(const u32 *verbs);
struct hda_tlv *hda_jack_sense_to_tlv(const struct hda_jack_sense *jack);

int hda_get_cmds_len(struct hda_cmds *cmds);

/*
 * OOB execution of a command sequence
 */
int hda_exec_cmds(struct hda_cmds *cmds);

/*
 */
struct hda_verb_array {
	unsigned int num, alloc;
	struct hda_verb *verbs;
};

int hda_append_verb(struct hda_verb_array *rec, unsigned int nid,
		    unsigned int verb, unsigned int val);
int hda_append_verbs(struct hda_verb_array *rec, struct hda_verb *verbs);
void hda_clear_verb_array(struct hda_verb_array *rec);

#endif /* __HDA_CMDS_H_INC */
