/*
 * AD1988 auto-configuration module
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include <stdio.h>
#include <string.h>
#include "hda/log.h"
#include "hda/codecs-helper.h"
#include "hda/pincfg.h"

#include "ad1988-common.c"


/*
 * auto-configuration
 */

#define AD1988_PIN_CD_NID		0x18
#define AD1988_PIN_BEEP_NID		0x10

static hda_nid_t ad1988_mixer_nids[8] = {
	/* A     B     C     D     E     F     G     H */
	0x22, 0x2b, 0x2c, 0x29, 0x26, 0x2a, 0x27, 0x28
};

static inline hda_nid_t ad1988_idx_to_dac(int idx)
{
	static hda_nid_t idx_to_dac[8] = {
		/* A     B     C     D     E     F     G     H */
		0x04, 0x06, 0x05, 0x04, 0x0a, 0x06, 0x05, 0x0a
	};
	static hda_nid_t idx_to_dac_rev2[8] = {
		/* A     B     C     D     E     F     G     H */
		0x04, 0x05, 0x0a, 0x04, 0x06, 0x05, 0x0a, 0x06
	};
	if (is_rev2())
		return idx_to_dac_rev2[idx];
	else
		return idx_to_dac[idx];
}

static hda_nid_t ad1988_boost_nids[8] = {
	0x38, 0x39, 0x3a, 0x3d, 0x3c, 0x3b, 0, 0
};

static int ad1988_pin_idx(hda_nid_t nid)
{
	static hda_nid_t ad1988_io_pins[8] = {
		0x11, 0x14, 0x15, 0x12, 0x17, 0x16, 0x24, 0x25
	};
	int i;
	for (i = 0; i < ARRAY_SIZE(ad1988_io_pins); i++)
		if (ad1988_io_pins[i] == nid)
			return i;
	return 0; /* should be -1 */
}

static int ad1988_pin_to_loopback_idx(hda_nid_t nid)
{
	static int loopback_idx[8] = {
		2, 0, 1, 3, 4, 5, 1, 4
	};
	switch (nid) {
	case AD1988_PIN_CD_NID:
		return 6;
	default:
		return loopback_idx[ad1988_pin_idx(nid)];
	}
}

static int ad1988_pin_to_adc_idx(hda_nid_t nid)
{
	static int adc_idx[8] = {
		0, 1, 2, 8, 4, 3, 6, 7
	};
	switch (nid) {
	case AD1988_PIN_CD_NID:
		return 5;
	default:
		return adc_idx[ad1988_pin_idx(nid)];
	}
}

/* fill in the dac_nids table from the parsed pin configuration */
static int auto_fill_dac_nids(const struct hda_auto_pin_cfg *cfg,
			      hda_nid_t *dac_nids)
{
	int i, idx;

	/* check the pins hardwired to audio widget */
	for (i = 0; i < cfg->line_outs; i++) {
		idx = ad1988_pin_idx(cfg->line_out_pins[i]);
		dac_nids[i] = ad1988_idx_to_dac(idx);
	}
	return 0;
}

/* Add Center/LFE mixer channels */
static int add_mixer_clfe(hda_nid_t dac, hda_nid_t sw_nid,
			  struct hda_mixer_array *mix)
{
	int err;
	static u32 center_bind_sws[3];
	static u32 lfe_bind_sws[3];

	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME,
			       "Center Playback Volume",
			       HDA_COMPOSE_AMP_VAL(dac, 1, 0, HDA_OUTPUT));
	if (err < 0)
		return err;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME,
			       "LFE Playback Volume",
			       HDA_COMPOSE_AMP_VAL(dac, 2, 0, HDA_OUTPUT));
	if (err < 0)
		return err;

	center_bind_sws[0] = HDA_COMPOSE_AMP_VAL(sw_nid, 1, 0, HDA_INPUT);
	center_bind_sws[1] = HDA_COMPOSE_AMP_VAL(sw_nid, 1, 1, HDA_INPUT);
	center_bind_sws[2] = 0;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_BIND_SWITCH,
			       "Center Playback Switch",
			       (long)center_bind_sws);
	if (err < 0)
		return err;
	lfe_bind_sws[0] = HDA_COMPOSE_AMP_VAL(sw_nid, 2, 0, HDA_INPUT);
	lfe_bind_sws[1] = HDA_COMPOSE_AMP_VAL(sw_nid, 2, 1, HDA_INPUT);
	lfe_bind_sws[2] = 0;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_BIND_SWITCH,
			       "LFE Playback Switch",
			       (long)lfe_bind_sws);
	if (err < 0)
		return err;
	return 0;
}

static int add_mixer_channel(hda_nid_t dac, hda_nid_t sw_nid,
			     const char *chname, u32 *bind_sws,
			     struct hda_mixer_array *mix)
{
	int err;
	char name[32];

	sprintf(name, "%s Playback Volume", chname);
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME, name,
			       HDA_COMPOSE_AMP_VAL(dac, 3, 0, HDA_OUTPUT));
	if (err < 0)
		return err;
	sprintf(name, "%s Playback Switch", chname);
	bind_sws[0] = HDA_COMPOSE_AMP_VAL(sw_nid, 3, 0, HDA_INPUT);
	bind_sws[1] = HDA_COMPOSE_AMP_VAL(sw_nid, 3, 1, HDA_INPUT);
	bind_sws[2] = 0;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_BIND_SWITCH, name,
			       (long)bind_sws);
	if (err < 0)
		return err;
	return 0;
}

/* add playback controls from the parsed DAC table */
static int auto_create_multi_out_ctls(const struct hda_auto_pin_cfg *cfg,
				      hda_nid_t *dac_nids,
				      struct hda_mixer_array *mix)
{
	static const char *chname[4] = {
		"Front", "Surround", NULL /*CLFE*/, "Side"
	};
	hda_nid_t nid;
	int i, err;
	static u32 bind_sws[4][3];

	for (i = 0; i < cfg->line_outs; i++) {
		hda_nid_t dac = dac_nids[i];
		if (!dac)
			continue;
		nid = ad1988_mixer_nids[ad1988_pin_idx(cfg->line_out_pins[i])];
		if (i == 2)
			err = add_mixer_clfe(dac, nid, mix);
		else
			err = add_mixer_channel(dac, nid, chname[i],
						&bind_sws[i][3], mix);
		if (err < 0)
			return err;
	}
	return 0;
}

/* add playback controls for speaker and HP outputs */
static int auto_create_extra_out(hda_nid_t pin, const char *pfx,
				 hda_nid_t *extra_nids,
				 struct hda_mixer_array *mix)
{
	hda_nid_t nid;
	int idx, err, exid;
	static u32 bind_sws[2][3];

	if (!pin)
		return 0;

	idx = ad1988_pin_idx(pin);
	nid = ad1988_idx_to_dac(idx);
	/* specify the DAC as the extra output */
	if (!extra_nids[0])
		exid = 0;
	else
		exid = 1;
	extra_nids[exid] = nid;
	/* control HP volume/switch on the output mixer amp */
	err = add_mixer_channel(nid, ad1988_mixer_nids[idx], pfx,
				&bind_sws[exid][0], mix);
	if (err < 0)
		return err;
	return 0;
}

/* create input playback/capture controls for the given pin */
static int new_analog_input(hda_nid_t pin, const char *ctlname, int boost)
{
	char name[32];
	int err, idx;

	sprintf(name, "%s Playback Volume", ctlname);
	idx = ad1988_pin_to_loopback_idx(pin);
	err = hda_register_mixer(HDA_MIXER_TYPE_VOLUME, name,
				 HDA_COMPOSE_AMP_VAL(0x20, 3, idx, HDA_INPUT));
	if (err < 0)
		return err;
	sprintf(name, "%s Playback Switch", ctlname);
	err = hda_register_mixer(HDA_MIXER_TYPE_SWITCH, name,
				 HDA_COMPOSE_AMP_VAL(0x20, 3, idx, HDA_INPUT));
	if (err < 0)
		return err;
	if (boost) {
		hda_nid_t bnid;
		idx = ad1988_pin_idx(pin);
		bnid = ad1988_boost_nids[idx];
		if (bnid) {
			sprintf(name, "%s Boost Volume", ctlname);
			return hda_register_mixer(HDA_MIXER_TYPE_VOLUME, name,
						  HDA_COMPOSE_AMP_VAL(bnid, 3, idx, HDA_OUTPUT));

		}
	}
	return 0;
}

/* create playback/capture controls for input pins */
static int auto_create_analog_input_ctls(const struct hda_auto_pin_cfg *cfg,
					 struct hda_input_mux *imux,
					 struct hda_mixer_array *mix)
{
	int i, err;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		err = new_analog_input(cfg->input_pins[i],
				       hda_auto_pin_cfg_labels[i],
				       i <= AUTO_PIN_FRONT_MIC);
		if (err < 0)
			return err;
		imux->items[imux->num_items].label =
			hda_auto_pin_cfg_labels[i];
		imux->items[imux->num_items].index =
			ad1988_pin_to_adc_idx(cfg->input_pins[i]);
		imux->num_items++;
	}
	imux->items[imux->num_items].label = "Mix";
	imux->items[imux->num_items].index = 9;
	imux->num_items++;

	err = hda_append_mixer(mix, HDA_MIXER_TYPE_VOLUME,
			       "Analog Mix Playback Volume",
			       HDA_COMPOSE_AMP_VAL(0x21, 3, 0x0, HDA_OUTPUT));
	if (err < 0)
		return err;
	err = hda_append_mixer(mix, HDA_MIXER_TYPE_SWITCH,
			       "Analog Mix Playback Switch",
			       HDA_COMPOSE_AMP_VAL(0x21, 3, 0x0, HDA_OUTPUT));
	if (err < 0)
		return err;

	return 0;
}

static int auto_set_output_and_unmute(hda_nid_t nid, int pin_type, int dac_idx,
				      struct hda_verb_array *init)
{
	int err;

	/* set as output */
	err = hda_append_verb(init, nid, AC_VERB_SET_PIN_WIDGET_CONTROL,
			      pin_type);
	if (err < 0)
		return err;
	err = hda_append_verb(init, nid, AC_VERB_SET_AMP_GAIN_MUTE,
			      AMP_OUT_UNMUTE);
	if (err < 0)
		return err;
	switch (nid) {
	case 0x11: /* port-A - DAC 04 */
		err = hda_append_verb(init, 0x37, AC_VERB_SET_CONNECT_SEL,
				      0x01);
		break;
	case 0x14: /* port-B - DAC 06 */
		err = hda_append_verb(init, 0x30, AC_VERB_SET_CONNECT_SEL,
				      0x02);
		break;
	case 0x15: /* port-C - DAC 05 */
		err = hda_append_verb(init, 0x31, AC_VERB_SET_CONNECT_SEL,
				      0x00);
		break;
	case 0x17: /* port-E - DAC 0a */
		err = hda_append_verb(init, 0x32, AC_VERB_SET_CONNECT_SEL,
				      0x01);
		break;
	case 0x13: /* mono - DAC 04 */
		err = hda_append_verb(init, 0x36, AC_VERB_SET_CONNECT_SEL,
				      0x01);
		break;
	}
	return err;
}

static int auto_init_multi_out(struct hda_auto_pin_cfg *cfg,
			       struct hda_verb_array *init)
{
	int i, err;

	for (i = 0; i < cfg->line_outs; i++) {
		hda_nid_t nid = cfg->line_out_pins[i];
		err = auto_set_output_and_unmute(nid, PIN_OUT, i, init);
		if (err < 0)
			return err;
	}
	return 0;
}

static int auto_init_extra_out(struct hda_auto_pin_cfg *cfg,
			       struct hda_verb_array *init)
{
	hda_nid_t pin;
	int err;

	pin = cfg->speaker_pins[0];
	if (pin) { /* connect to front */
		err = auto_set_output_and_unmute(pin, PIN_OUT, 0, init);
		if (err < 0)
			return err;
	}
	pin = cfg->hp_pins[0];
	if (pin) { /* connect to front */
		err = auto_set_output_and_unmute(pin, PIN_HP, 0, init);
		if (err < 0)
			return err;
	}
	return 0;
}

static int auto_init_analog_input(struct hda_auto_pin_cfg *cfg,
				  struct hda_verb_array *init)
{
	int i, idx, err = 0;

	for (i = 0; i < AUTO_PIN_LAST; i++) {
		hda_nid_t nid = cfg->input_pins[i];
		if (!nid)
			continue;
		switch (nid) {
		case 0x15: /* port-C */
			err = hda_append_verb(init, 0x33,
					      AC_VERB_SET_CONNECT_SEL, 0x0);
			break;
		case 0x17: /* port-E */
			err = hda_append_verb(init, 0x34,
					      AC_VERB_SET_CONNECT_SEL, 0x0);
			break;
		}
		if (err < 0)
			return err;
		err = hda_append_verb(init, nid,
				      AC_VERB_SET_PIN_WIDGET_CONTROL,
				      i <= AUTO_PIN_FRONT_MIC ?
				      PIN_VREF80 : PIN_IN);
		if (err < 0)
			return err;
		if (nid != AD1988_PIN_CD_NID) {
			err = hda_append_verb(init, nid,
					      AC_VERB_SET_AMP_GAIN_MUTE,
					      AMP_OUT_MUTE);
			if (err < 0)
				return err;
		}
		idx = ad1988_pin_idx(nid);
		if (ad1988_boost_nids[idx]) {
			err = hda_append_verb(init, ad1988_boost_nids[idx],
					      AC_VERB_SET_AMP_GAIN_MUTE,
					      AMP_OUT_ZERO);
			if (err < 0)
				return err;
		}
	}
	return 0;
}


/*
 */

static struct hda_auto_pin_cfg autocfg;
static hda_nid_t dac_nids[5];
static hda_nid_t extra_nids[5];
static struct hda_verb_array auto_init;
static struct hda_mixer_array auto_mix;
static struct hda_input_mux auto_imux;

static struct codec_config_preset preset = {
	.init_verbs = {
		ad1988_6stack_init_verbs,
		ad1988_capture_init_verbs,
	},
	.mixers = {
		ad1988_capture_mixers,
	},
	.adc_nid = AD1988_ADC1,
	.input_mux_nid = AD1988_CAPSRC1,
};

/* parse the BIOS configuration and set up */
static int patch_ad1988_auto(const struct hda_codec_table *tbl,
			     int board_config, const char **args)
{
	int err;

	err = hda_parse_pin_def_config(&autocfg, NULL);
	if (err < 0)
		goto error;
	err = auto_fill_dac_nids(&autocfg, dac_nids);
	if (err < 0)
		goto error;
	if (!autocfg.line_outs)
		return -ENODEV; /* can't find valid BIOS pin config */

	err = auto_create_multi_out_ctls(&autocfg, dac_nids, &auto_mix);
	if (err < 0)
		goto error;
	err = auto_create_extra_out(autocfg.speaker_pins[0], "Speaker",
				    extra_nids, &auto_mix);
	if (err < 0)
		goto error;
	err = auto_create_extra_out(autocfg.hp_pins[0], "Headphone",
				    extra_nids, &auto_mix);
	if (err < 0)
		goto error;
	err = auto_create_analog_input_ctls(&autocfg, &auto_imux, &auto_mix);
	if (err < 0)
		goto error;

	err = auto_init_multi_out(&autocfg, &auto_init);
	if (err < 0)
		goto error;
	err = auto_init_extra_out(&autocfg, &auto_init);
	if (err < 0)
		goto error;
	err = auto_init_analog_input(&autocfg, &auto_init);
	if (err < 0)
		goto error;

	codec_preset_add_init_verb(&preset, auto_init.verbs);
	codec_preset_add_mixer(&preset, auto_mix.mixers);
	preset.num_dacs = autocfg.line_outs;
	preset.dac_nids = dac_nids;
	preset.extra_nids = extra_nids;
	preset.input_mux = &auto_imux;
	if (autocfg.dig_out_pin) {
		preset.dig_out_nid = AD1988_SPDIF_OUT;
		codec_preset_add_init_verb(&preset, ad1988_spdif_init_verbs);
		codec_preset_add_mixer(&preset, ad1988_spdif_out_mixers);
	}
	if (autocfg.dig_in_pin) {
		preset.dig_out_nid = AD1988_SPDIF_IN;
		codec_preset_add_mixer(&preset, ad1988_spdif_in_mixers);
	}

	err = codec_build_preset(tbl, &preset);
	if (err < 0)
		goto error;

	err = 0;	/* OK finished */

 error:
	hda_clear_verb_array(&auto_init);
	hda_clear_mixer_array(&auto_mix);
	return err; 
}

/*
 */

static struct hda_codec_table ad1988_table[] = {
	{ .id = 0x11d41988, .name = "AD1988",
	  .patch = patch_ad1988_auto },
	{ .id = 0x11d4198b, .name = "AD1988B",
	  .patch = patch_ad1988_auto },
	{ }
};

const struct hda_codec_table *patch_descriptor(void)
{
	return ad1988_table;
}
