/*
 * Conversion to TLV stream
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include "local.h"

/*
 * convert from hda_jack_sense type to TLV:
 */
struct hda_tlv *hda_jack_sense_to_tlv(const struct hda_jack_sense *jack)
{
	struct hda_cmds cmds =
		STD_IF_ELSE(STD_AND(STD_VERB(jack->nid,
					     AC_VERB_GET_PIN_SENSE, 0),
				    STD_INT(1U<<31)),
			    jack->present, jack->not_present);
	return hda_cmd_to_tlv(&cmds);
}

/*
 * convert from verb array to TLV
 *
 * the array must be terminated by null entry
 */
struct hda_tlv *hda_verbs_to_tlv(const struct hda_verb *verbs)
{
	struct hda_tlv *tlv;
	const struct hda_verb *v;
	int i, nverbs;
	char *data;

	for (nverbs = 0, v = verbs; v->nid; nverbs++, v++)
		;
	tlv = hda_alloc_tlv(HDA_CMD_TYPE_VERBS, nverbs * 4);
	if (!tlv)
		return NULL;
	data = tlv->val;
	for (i = 0, v = verbs; i < nverbs; i++, v++) {
		u32 val = HDA_VERB(v->nid, v->verb, v->param);
		memcpy(data, &val, 4);
		data += 4;
	}
	return tlv;
}

/*
 * convert from control_notify to TLV
 *
 * the argument is the control element name to notify.
 * it must have been registered before calling this function.
 */
static struct hda_tlv *hda_ctl_notify_to_tlv(const char *ctl)
{
	u32 val;

	val = hda_look_for_ctl_id(ctl);
	if (!val) {
		log_error("Cannot find control '%s'\n", ctl);
		return NULL;
	}
	return hda_make_int_tlv(HDA_CMD_TYPE_CTL_NOTIFY, val);
}

/*
 * command list to TLV
 */
struct hda_tlv *hda_cmd_seq_to_tlv(int type, const struct hda_cmds *cmds)
{
	int ncmds;
	struct hda_tlv *args[MAX_TLV_ARGS];
	const struct hda_cmds *c;
	struct hda_tlv *tlv = NULL;

	for (c = cmds, ncmds = 0; c->type >= 0; c++, ncmds++) {
		if (ncmds >= MAX_TLV_ARGS) {
			log_error("Too many arguments\n");
			goto error;
		}
		args[ncmds] = hda_cmd_to_tlv(c);
		if (!args[ncmds])
			goto error;
	}

	if (!ncmds)
		return NULL;
	if (ncmds == 1)
		/* single cmd; let's reduce the size */
		return args[0];

	tlv = hda_make_tlv_args(type, ncmds, args);
 error:
	for (; ncmds > 0; ncmds--)
		hda_free_tlv((void *)args[ncmds - 1]);
	return tlv;
}

/*
 * if-else to TLV
 */
static struct hda_tlv *hda_if_to_tlv(const struct hda_cmd_if *rec)
{
	struct hda_tlv *tlv = NULL, *cond, *yes, *no = NULL;
	cond = hda_cmd_to_tlv(&rec->cond);
	yes = hda_cmd_to_tlv(&rec->yes);
	if (cond && yes) {
		if (rec->no.type == HDA_CMD_TYPE_NOP)
			tlv = hda_make_tlv(HDA_CMD_TYPE_IF, cond, yes, NULL);
		else {
			no = hda_cmd_to_tlv(&rec->no);
			if (no)
				tlv = hda_make_tlv(HDA_CMD_TYPE_IF,
						   cond, yes, no, NULL);
		}
	}
	hda_free_tlv(no);
	hda_free_tlv(yes);
	hda_free_tlv(cond);
	return tlv;
}

/*
 * one-arg op to TLV
 */
static struct hda_tlv *hda_one_op_to_tlv(int type, const struct hda_cmds *rec)
{
	struct hda_tlv *tlv = NULL, *val;
	val = hda_cmd_to_tlv(rec);
	if (val)
		tlv = hda_make_tlv(type, val, NULL);
	hda_free_tlv(val);
	return tlv;
}

/*
 * two-args op to TLV
 */
static struct hda_tlv *hda_two_ops_to_tlv(int type,
					  const struct hda_cmd_two_ops *rec)
{
	struct hda_tlv *tlv = NULL, *val1, *val2;
	val1 = hda_cmd_to_tlv(&rec->a);
	val2 = hda_cmd_to_tlv(&rec->b);
	if (val1 && val2)
		tlv = hda_make_tlv(type, val1, val2, NULL);
	hda_free_tlv(val1);
	hda_free_tlv(val2);
	return tlv;
}

/*
 * convert from a command list to TLV
 */
struct hda_tlv *hda_cmd_to_tlv(const struct hda_cmds *c)
{
	void *argp = (void *)c->val;

	switch (c->type) {
	case HDA_CMD_TYPE_NOP:
		return hda_make_tlv(c->type, NULL);
	case HDA_CMD_TYPE_CMDS:
	case HDA_CMD_TYPE_LIST:
	case HDA_CMD_TYPE_LAND:
	case HDA_CMD_TYPE_LOR:
	case HDA_CMD_TYPE_SET_REG:
		return hda_cmd_seq_to_tlv(c->type, argp);
	case HDA_CMD_TYPE_VERBS:
		return hda_verbs_to_tlv(argp);
	case HDA_CMD_TYPE_INT:
	case HDA_CMD_TYPE_AMP_READ:
	case HDA_CMD_TYPE_SET_CHANNELS:
	case HDA_CMD_TYPE_DELAY:
	case HDA_CMD_TYPE_GET_REG:
		return hda_make_int_tlv(c->type, c->val);
	case HDA_CMD_TYPE_LABEL:
		return hda_make_str_tlv(c->type, argp);
	case HDA_CMD_TYPE_JACK_SENSE:
		return hda_jack_sense_to_tlv(argp);
	case HDA_CMD_TYPE_CTL_NOTIFY:
		return hda_ctl_notify_to_tlv(argp);
	case HDA_CMD_TYPE_IF:
		return hda_if_to_tlv(argp);
	case HDA_CMD_TYPE_NOT:
	case HDA_CMD_TYPE_LNOT:
		return hda_one_op_to_tlv(c->type, argp);
	case HDA_CMD_TYPE_AND:
	case HDA_CMD_TYPE_OR:
	case HDA_CMD_TYPE_SHIFTL:
	case HDA_CMD_TYPE_SHIFTR:
	case HDA_CMD_TYPE_PLUS:
	case HDA_CMD_TYPE_MINUS:
	case HDA_CMD_TYPE_VERB_UPDATE:
	case HDA_CMD_TYPE_AMP_UPDATE:
		return hda_two_ops_to_tlv(c->type, argp);
	default:
		return NULL;
	}
}

/*
 * get the length of the given command list
 */
int hda_get_cmds_len(struct hda_cmds *cmds)
{
	int ncmds;
	for (ncmds = 0; cmds->type >= 0; cmds++, ncmds++)
		;
	return ncmds;
}


/*
 * append a verb to hda_verb_array
 *
 * the verb array is automatically reallocated if needed
 */

#define VERB_ALLOC_SIZE		64

static struct hda_verb *alloc_new_verb(struct hda_verb_array *rec)
{
	struct hda_verb *vnew;

	if (rec->num + 1 >= rec->alloc) {
		unsigned int newsize = rec->alloc + VERB_ALLOC_SIZE;
		vnew = realloc(rec->verbs, newsize * sizeof(*vnew));
		if (!vnew)
			return NULL;
		rec->verbs = vnew;
		rec->alloc = newsize;
	}
	vnew = &rec->verbs[rec->num++];
	/* terminate the array */
	memset(&rec->verbs[rec->num], 0, sizeof(rec->verbs[rec->num]));
	return vnew;
}

int hda_append_verb(struct hda_verb_array *rec, unsigned int nid,
		    unsigned int verb, unsigned int val)
{
	struct hda_verb *vnew;

	vnew = alloc_new_verb(rec);
	if (!vnew)
		return -ENOMEM;
	vnew->nid = nid;
	vnew->verb = verb;
	vnew->param = val;
	return 0;
}

int hda_append_verbs(struct hda_verb_array *rec,
		     struct hda_verb *verbs)
{
	struct hda_verb *vnew;

	for (; verbs->nid; verbs++) {
		vnew = alloc_new_verb(rec);
		if (!vnew)
			return -ENOMEM;
		*vnew = *verbs;
	}
	return 0;
}

void hda_clear_verb_array(struct hda_verb_array *rec)
{
	free(rec->verbs);
	memset(rec, 0, sizeof(*rec));
}
