/*
 * I/O routines
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/io.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include "local.h"
#include "hda/pincfg.h"


/**
 * hda_codec_read - send a command and get the response
 * @nid: NID to send the command
 * @verb: the verb to send
 * @parm: the parameter for the verb
 *
 * Send a single command and read the corresponding response.
 *
 * Returns the obtained response value, or -1 for an error.
 */
unsigned int hda_codec_read(hda_nid_t nid, unsigned int verb,
			    unsigned int value)
{
	struct hda_verb_ioctl v;
	v.verb = HDA_VERB(nid, verb, value);
	if (hda_do_ioctl(HDA_IOCTL_VERB_WRITE, &v) < 0)
		return -1;
	return v.res;
}

/**
 * hda_get_sub_nodes - get the range of sub nodes
 * @nid: NID to parse
 * @start_id: the pointer to store the start NID
 *
 * Parse the NID and store the start NID of its sub-nodes.
 * Returns the number of sub-nodes.
 */
int hda_get_sub_nodes(hda_nid_t nid, hda_nid_t *start_id)
{
	unsigned int parm;

	parm = hda_param_read(nid, AC_PAR_NODE_COUNT);
	*start_id = (parm >> 16) & 0x7fff;
	return (int)(parm & 0x7fff);
}

/**
 * hda_get_connections - get connection list
 * @nid: NID to parse
 * @conn_list: connection list array
 * @max_conns: max. number of connections to store
 *
 * Parses the connection list of the given widget and stores the list
 * of NIDs.
 *
 * Returns the number of connections, or a negative error code.
 */
int hda_get_connections(hda_nid_t nid, hda_nid_t *conn_list, int max_conns)
{
	unsigned int parm;
	int i, conn_len, conns;
	unsigned int shift, num_elems, mask;
	hda_nid_t prev_nid;

	if (!conn_list)
		return -EINVAL;
	if (max_conns <= 0)
		return -EINVAL;

	parm = hda_param_read(nid, AC_PAR_CONNLIST_LEN);
	if (parm & AC_CLIST_LONG) {
		/* long form */
		shift = 16;
		num_elems = 2;
	} else {
		/* short form */
		shift = 8;
		num_elems = 4;
	}
	conn_len = parm & AC_CLIST_LENGTH;
	mask = (1 << (shift-1)) - 1;

	if (!conn_len)
		return 0; /* no connection */

	if (conn_len == 1) {
		/* single connection */
		parm = hda_codec_read(nid, AC_VERB_GET_CONNECT_LIST, 0);
		conn_list[0] = parm & mask;
		return 1;
	}

	/* multi connection */
	conns = 0;
	prev_nid = 0;
	for (i = 0; i < conn_len; i++) {
		int range_val;
		hda_nid_t val, n;

		if (i % num_elems == 0)
			parm = hda_codec_read(nid,
					      AC_VERB_GET_CONNECT_LIST, i);
		range_val = !!(parm & (1 << (shift-1))); /* ranges */
		val = parm & mask;
		parm >>= shift;
		if (range_val) {
			/* ranges between the previous and this one */
			if (!prev_nid || prev_nid >= val) {
				log_error("invalid dep_range_val %x:%x\n",
					   prev_nid, val);
				continue;
			}
			for (n = prev_nid + 1; n <= val; n++) {
				if (conns >= max_conns) {
					log_error("Too many connections\n");
					return -EINVAL;
				}
				conn_list[conns++] = n;
			}
		} else {
			if (conns >= max_conns) {
				log_error("Too many connections\n");
				return -EINVAL;
			}
			conn_list[conns++] = val;
		}
		prev_nid = val;
	}
	return conns;
}

/*
 * get the capability bits of the given widget ID
 */
u32 hda_get_wcaps(hda_nid_t nid)
{
	struct hda_verb_ioctl wcap = { HDA_VERB(nid, 0, 0), 0 };
	hda_do_ioctl(HDA_IOCTL_GET_WCAP, &wcap);
	return wcap.res;
}

/*
 * return whether the given pin widget is available
 * it checks the connection bits of default pin config.
 * if it's AC_JACK_PORT_NONE, the pin is unavailable.
 */
int hda_pin_available(hda_nid_t nid)
{
	unsigned int def_conf;

	def_conf = hda_codec_read(nid, AC_VERB_GET_CONFIG_DEFAULT, 0);
	return get_defcfg_connect(def_conf) != AC_JACK_PORT_NONE;
}

/*
 * INIT SEQUENCES
 */

/*
 * register init verbs
 */
static int do_register_init_verbs(struct hda_verb *verbs, unsigned int ctl)
{
	struct hda_tlv *tlv;
	int err;

	tlv = hda_verbs_to_tlv(verbs);
	if (!tlv)
		return -ENOMEM;
	err = hda_do_ioctl(ctl, tlv);
	hda_free_tlv(tlv);
	if (err < 0)
		return -ENODEV;
	return 0;
}

/* initialization verbs */
int hda_register_init_verbs(struct hda_verb *verbs)
{
	int err;
	err = do_register_init_verbs(verbs, HDA_IOCTL_ADD_INIT);
	if (err < 0)
		log_error("Cannot register init verbs\n");
	return err;
}

/* additional resume verbs */
int hda_register_resume_verbs(struct hda_verb *verbs)
{
	int err;
	err = do_register_init_verbs(verbs, HDA_IOCTL_ADD_RESUME);
	if (err < 0)
		log_error("Cannot register resume verbs\n");
	return err;
}

/*
 * register init commands
 */
static int do_register_init_cmds(struct hda_cmds *cmds, unsigned int ctl)
{
	struct hda_tlv *tlv;
	int err;

	tlv = hda_cmd_to_tlv(cmds);
	if (!tlv)
		return -ENOMEM;
	err = hda_do_ioctl(ctl, tlv);
	hda_free_tlv(tlv);
	if (err < 0)
		return -ENODEV;
	return 0;
}

int hda_register_init_cmds(struct hda_cmds *cmds)
{
	int err;
	err = do_register_init_cmds(cmds, HDA_IOCTL_ADD_INIT);
	if (err < 0)
		log_error("Cannot register init cmds\n");
	return err;
		
}

int hda_register_resume_cmds(struct hda_cmds *cmds)
{
	int err;
	err = do_register_init_cmds(cmds, HDA_IOCTL_ADD_RESUME);
	if (err < 0)
		log_error("Cannot register resume cmds\n");
	return err;
}

/*
 * UNSOLICITED EVENTS
 */

struct unsol_rec {
	struct hda_unsol_ioctl head;
	struct hda_tlv tlv[0];
};

/*
 * register commands for an unsolicited event
 */
int hda_register_unsol_cmds(hda_nid_t nid, int tag, int shift,
			    struct hda_cmds *cmds)
{
	struct hda_tlv *tlv;
	struct unsol_rec *rec;
	int err;

	tlv = hda_cmd_to_tlv(cmds);
	if (!tlv)
		return -ENOMEM;
	rec = malloc(sizeof(*rec) + hda_tlv_len(tlv));
	if (!rec) {
		hda_free_tlv(tlv);
		return -ENOMEM;
	}
	rec->head.nid = nid;
	rec->head.tag = tag;
	rec->head.shift = shift;
	memcpy(rec->tlv, tlv, hda_tlv_len(tlv));
	hda_free_tlv(tlv);
	err = hda_do_ioctl(HDA_IOCTL_ADD_UNSOL, rec);
	free(rec);
	if (err < 0) {
		log_error("Cannot register unsol event, nid = 0x%x, tag = 0x%x, shift = %d\n",
			  nid, tag, shift);
		return -ENODEV;
	}
	return 0;
}


/*
 * OOB execution of a command sequence
 */
int hda_exec_cmds(struct hda_cmds *cmds)
{
	struct hda_tlv *tlv;
	int err;

	tlv = hda_cmd_to_tlv(cmds);
	err = hda_do_ioctl(HDA_IOCTL_CMD_EXEC, tlv);
	hda_free_tlv(tlv);
	return err;
}

/*
 * PCM REGISTRATION
 */

int hda_register_pcm(struct hda_usr_pcm_info *pcm)
{
	int err;
	err = hda_do_ioctl(HDA_IOCTL_ADD_PCM, pcm);
	if (err < 0) {
		log_error("Cannot register PCM %s, type %d, device %d\n",
			  pcm->name, pcm->type, pcm->device);
		return -ENODEV;
	}
	return 0;
}

/*
 */

int hda_is_running(void)
{
	int status = 0;
	if (hda_do_ioctl(HDA_IOCTL_GET_STATUS, &status) < 0)
		return 0;
	return status;
}

int hda_set_status(int status)
{
	return hda_do_ioctl(HDA_IOCTL_SET_STATUS, &status);
}

int hda_reset(void)
{
	int err;

	err = hda_do_ioctl(HDA_IOCTL_RESET, NULL);
	if (err < 0) {
		log_error("Failed to RESET\n");
		return -EINVAL;
	}
	hda_mixer_free_all();
	return 0;
}
