/*
 * Registration of mixers
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#ifndef __HDA_MIXER_H_INC
#define __HDA_MIXER_H_INC

#include <alsa/asoundlib.h>

struct hda_std_mixer {
	struct hda_mixer_head head;
	long value;
	int init_val;
};

int hda_register_mixer(int type, const char *name, long value);
int hda_register_mixers(struct hda_std_mixer *mixer);

/*
 * input mux helper
 */
#define HDA_MAX_NUM_INPUTS	16
struct hda_input_mux_item {
	const char *label;
	unsigned int index;
};
struct hda_input_mux {
	unsigned int num_items;
	struct hda_input_mux_item items[HDA_MAX_NUM_INPUTS];
};

int hda_register_input_mux_mixer(const char *name, hda_nid_t nid,
				 struct hda_input_mux *imux);
struct hda_input_mux *hda_fixup_input_mux(hda_nid_t nid,
					  struct hda_input_mux *imux,
					  hda_nid_t *pin_nids);
void hda_input_mux_free(struct hda_input_mux *imux);

void hda_mixer_free_all(void);

/*
 * return the numid of the registered control, or 0 when not found
 */
unsigned int hda_look_for_ctl_id(const char *label);


/*
 * standard mixer macro
 */

#define HDA_COMPOSE_AMP_VAL(nid,chs,idx,dir) \
	((nid) | ((chs)<<16) | ((dir)<<18) | ((idx)<<19))
/* mono volume with index (index=0,1,...) (channel=1,2) */
#define HDA_CODEC_VOLUME_MONO_IDX(xname, xcidx, nid, channel, xindex, xdir) { \
	.head = {						\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.index = xcidx, \
		.type = HDA_MIXER_TYPE_VOLUME, \
	}, \
	.value = HDA_COMPOSE_AMP_VAL(nid, channel, xindex, xdir) }
/* stereo volume with index */
#define HDA_CODEC_VOLUME_IDX(xname, xcidx, nid, xindex, direction) \
	HDA_CODEC_VOLUME_MONO_IDX(xname, xcidx, nid, 3, xindex, direction)
/* mono volume */
#define HDA_CODEC_VOLUME_MONO(xname, nid, channel, xindex, direction) \
	HDA_CODEC_VOLUME_MONO_IDX(xname, 0, nid, channel, xindex, direction)
/* stereo volume */
#define HDA_CODEC_VOLUME(xname, nid, xindex, direction) \
	HDA_CODEC_VOLUME_MONO(xname, nid, 3, xindex, direction)

/* mono mute switch with index (index=0,1,...) (channel=1,2) */
#define HDA_CODEC_MUTE_MONO_IDX(xname, xcidx, nid, channel, xindex, xdir) { \
	.head = {						\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.index = xcidx, \
		.type = HDA_MIXER_TYPE_SWITCH, \
	}, \
	.value = HDA_COMPOSE_AMP_VAL(nid, channel, xindex, xdir) }
/* stereo mute switch with index */
#define HDA_CODEC_MUTE_IDX(xname, xcidx, nid, xindex, direction) \
	HDA_CODEC_MUTE_MONO_IDX(xname, xcidx, nid, 3, xindex, direction)
/* mono mute switch */
#define HDA_CODEC_MUTE_MONO(xname, nid, channel, xindex, direction) \
	HDA_CODEC_MUTE_MONO_IDX(xname, 0, nid, channel, xindex, direction)
/* stereo mute switch */
#define HDA_CODEC_MUTE(xname, nid, xindex, direction) \
	HDA_CODEC_MUTE_MONO(xname, nid, 3, xindex, direction)

/* mono switch binding multiple inputs */
#define HDA_BIND_MUTE_MONO(xname, nid, channel, direction) { \
	.head = {						\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_BIND_SWITCH, \
	}, \
	.value = (long) (const u32 []) {			\
		HDA_COMPOSE_AMP_VAL(nid, channel, 0, direction),	\
		HDA_COMPOSE_AMP_VAL(nid, channel, 1, direction),	\
		0 } }
/* stereo switch binding multiple inputs - just for compatibility */
#define HDA_BIND_MUTE(xname,nid,dir) \
	HDA_BIND_MUTE_MONO(xname,nid,3,dir)

/* bound volume control */
#define HDA_MIXER_BIND_VOLUME(xname, binds) { 	\
	.head = {				\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_BIND_VOLUME, \
	}, \
	.value = (long) (binds) }

/* bound switch control */
#define HDA_MIXER_BIND_SWITCH(xname, binds) { 	\
	.head = {				\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_BIND_SWITCH, \
	}, \
	.value = (long) (binds) }

/* enum type mixer */
#define HDA_MIXER_ENUM(xname, cmds) { 		\
	.head = {				\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_ENUM, \
	}, \
	.value = (long) (cmds) }

/* enum type mixer */
#define HDA_MIXER_ENUM_INIT(xname, cmds, init) {	\
	.head = {				\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_ENUM, \
	}, \
	.value = (long) (cmds), .init_val = (init) }

/* enum type mixer */
#define HDA_MIXER_BOOL(xname, cmds) {		\
	.head = {				\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_BOOLEAN, \
	}, \
	.value = (long) (cmds) }

/* enum type mixer */
#define HDA_MIXER_BOOL_WITH_INIT(xname, cmds, init) {	\
	.head = {				\
		.iface = SND_CTL_ELEM_IFACE_MIXER, \
		.name = xname, \
		.type = HDA_MIXER_TYPE_BOOLEAN, \
	}, \
	.value = (long) (cmds), .init_val = (init) }

/*
 */
struct hda_mixer_array {
	unsigned int num, alloc;
	struct hda_std_mixer *mixers;
};

int hda_append_mixer(struct hda_mixer_array *rec,
		     int type, const char *name, unsigned long val);
int hda_append_mixers(struct hda_mixer_array *rec,
		      struct hda_std_mixer *mixers);
void hda_clear_mixer_array(struct hda_mixer_array *rec);


#endif /* __HDA_MIXER_H_INC */
