/*
 * AD1884 / AD1984 module
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

/*
 * AD1884 / AD1984
 * 
 * port-B - front line/mic-in
 * port-E - aux in/out
 * port-F - aux in/out
 * port-C - rear line/mic-in
 * port-D - rear line/hp-out
 * port-A - front line/hp-out
 *
 * AD1984 = AD1884 + two digital mic-ins
 *
 * FIXME:
 * For simplicity, we share the single DAC for both HP and line-outs
 * right now.  The inidividual playbacks could be easily implemented,
 * but no build-up framework is given, so far.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include "hda/codecs-helper.h"

#define AD1884_DAC		0x04
#define AD1884_ADC		0x08	/* 0x09 */
#define AD1884_CAPSRC		0x0c	/* 0x0d */
#define AD1884_SPDIF_OUT	0x02
#define AD1984_DIG_MIC		0x05	/* 0x06 */

#define AD1884_SPDIF_PIN	0x1b

/*
 * mixers
 */
static struct hda_input_mux ad1884_capture_source = {
	.num_items = 4,
	.items = {
		{ "Front Mic", 0x0 },
		{ "Mic", 0x1 },
		{ "CD", 0x2 },
		{ "Mix", 0x3 },
	},
};
static hda_nid_t ad1884_cap_pins[] = {
	0x14, 0x15, 0x16, 0
};

static struct hda_std_mixer ad1884_base_mixers[] = {
	HDA_CODEC_VOLUME("PCM Playback Volume", 0x04, 0x0, HDA_OUTPUT),
	/* HDA_CODEC_VOLUME_IDX("PCM Playback Volume", 1, 0x03, 0x0, HDA_OUTPUT), */
	HDA_CODEC_MUTE("Headphone Playback Switch", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Front Playback Switch", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME_MONO("Mono Playback Volume", 0x13, 1, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE_MONO("Mono Playback Switch", 0x13, 1, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Front Mic Playback Volume", 0x20, 0x00, HDA_INPUT),
	HDA_CODEC_MUTE("Front Mic Playback Switch", 0x20, 0x00, HDA_INPUT),
	HDA_CODEC_VOLUME("Mic Playback Volume", 0x20, 0x01, HDA_INPUT),
	HDA_CODEC_MUTE("Mic Playback Switch", 0x20, 0x01, HDA_INPUT),
	HDA_CODEC_VOLUME("CD Playback Volume", 0x20, 0x02, HDA_INPUT),
	HDA_CODEC_MUTE("CD Playback Switch", 0x20, 0x02, HDA_INPUT),
	/*
	HDA_CODEC_VOLUME("PC Speaker Playback Volume", 0x20, 0x03, HDA_INPUT),
	HDA_CODEC_MUTE("PC Speaker Playback Switch", 0x20, 0x03, HDA_INPUT),
	HDA_CODEC_VOLUME("Digital Beep Playback Volume", 0x10, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Digital Beep Playback Switch", 0x10, 0x0, HDA_OUTPUT),
	*/
	HDA_CODEC_VOLUME("Mic Boost Volume", 0x15, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Front Mic Boost Volume", 0x14, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x0c, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x0c, 0x0, HDA_OUTPUT),
	{ } /* end */
};

/*
 * SPDIF playback route
 */
static struct hda_verb spdif_src_pcm[] = {
	{ AD1884_SPDIF_OUT, AC_VERB_SET_CONNECT_SEL, 0 },
	{ }
};
static struct hda_verb spdif_src_adc[] = {
	{ AD1884_SPDIF_OUT, AC_VERB_SET_CONNECT_SEL, 1 },
	{ }
};

static struct hda_cmds spdif_playback_src[] = {
	STD_LABEL("PCM"), STD_VERBS(spdif_src_pcm),
	STD_LABEL("ADC"), STD_VERBS(spdif_src_adc),
	STD_NULL
};
static struct hda_std_mixer ad1884_spdif_src[] = {
	HDA_MIXER_ENUM("IEC958 Playback Source", spdif_playback_src),
	{ } /* end */
};

/*
 * AD1984 digital mic
 */
static struct hda_std_mixer ad1984_dmic_mixers[] = {
	HDA_CODEC_VOLUME("Digital Mic Capture Volume", 0x05, 0x0, HDA_INPUT),
	HDA_CODEC_MUTE("Digital Mic Capture Switch", 0x05, 0x0, HDA_INPUT),
#if 0
	HDA_CODEC_VOLUME_IDX("Digital Mic Capture Volume", 1, 0x06, 0x0,
			     HDA_OUTPUT),
	HDA_CODEC_MUTE_IDX("Digital Mic Capture Switch", 1, 0x06, 0x0,
			   HDA_OUTPUT),
#endif
	{ } /* end */
};

/*
 * initialization verbs
 */
static struct hda_verb ad1884_init_verbs[] = {
	/* DACs; mute as default */
	{0x03, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	{0x04, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Port-A (HP) mixer */
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	/* Port-A pin */
	{0x11, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_HP},
	{0x11, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* HP selector - select DAC2 */
	{0x22, AC_VERB_SET_CONNECT_SEL, 0x1},
	/* Port-D (Line-out) mixer */
	{0x0a, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x0a, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	/* Port-D pin */
	{0x12, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_HP},
	{0x12, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Mono-out mixer */
	{0x1e, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)},
	{0x1e, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(1)},
	/* Mono-out pin */
	{0x13, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_HP},
	{0x13, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Mono selector */
	{0x0e, AC_VERB_SET_CONNECT_SEL, 0x1},
	/* Port-B (front mic) pin */
	{0x14, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_VREF80},
	{0x14, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Port-C (rear mic) pin */
	{0x15, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_VREF80},
	{0x15, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Analog mixer; mute as default */
	{0x20, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)},
	{0x20, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(1)},
	{0x20, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(2)},
	{0x20, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(3)},
	/* Analog Mix output amp */
	{0x21, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE | 0x1f}, /* 0dB */
	/* SPDIF output selector */
	{0x02, AC_VERB_SET_CONNECT_SEL, 0x0}, /* PCM */
	{0x1b, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE | 0x27}, /* 0dB */
	{ } /* end */
};

/*
 * Lenovo Thinkpad T61/X61
 */
static struct hda_input_mux thinkpad_capture_source = {
	.num_items = 4,
	.items = {
		{ "Mic", 0x0 },
		{ "Internal Mic", 0x1 },
		{ "Mix", 0x3 },
	},
};

static struct hda_std_mixer thinkpad_mixers[] = {
	HDA_CODEC_VOLUME("PCM Playback Volume", 0x04, 0x0, HDA_OUTPUT),
	/* HDA_CODEC_VOLUME_IDX("PCM Playback Volume", 1, 0x03, 0x0, HDA_OUTPUT), */
	HDA_CODEC_MUTE("Headphone Playback Switch", 0x11, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Speaker Playback Switch", 0x12, 0x0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Mic Playback Volume", 0x20, 0x00, HDA_INPUT),
	HDA_CODEC_MUTE("Mic Playback Switch", 0x20, 0x00, HDA_INPUT),
	HDA_CODEC_VOLUME("Internal Mic Playback Volume", 0x20, 0x01, HDA_INPUT),
	HDA_CODEC_MUTE("Internal Mic Playback Switch", 0x20, 0x01, HDA_INPUT),
	HDA_CODEC_VOLUME("Docking Mic Playback Volume", 0x20, 0x04, HDA_INPUT),
	HDA_CODEC_MUTE("Docking Mic Playback Switch", 0x20, 0x04, HDA_INPUT),
	HDA_CODEC_VOLUME("Mic Playback Volume", 0x20, 0x00, HDA_INPUT),
	HDA_CODEC_MUTE("Mic Playback Switch", 0x20, 0x00, HDA_INPUT),
	HDA_CODEC_VOLUME("Mic Boost Volume", 0x14, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Internal Mic Boost Volume", 0x15, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Docking Mic Boost Volume", 0x25, 0x0, HDA_INPUT),
	HDA_CODEC_VOLUME("Capture Volume", 0x0c, 0x0, HDA_OUTPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x0c, 0x0, HDA_OUTPUT),
	{ } /* end */
};

/* additional verbs */
static struct hda_verb thinkpad_init_verbs[] = {
	/* Port-E (docking station mic) pin */
	{0x1c, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_VREF80},
	{0x1c, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* docking mic boost */
	{0x25, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE},
	/* Analog mixer - docking mic; mute as default */
	{0x20, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(4)},
	{ } /* end */
};

/*
 * Digital Mic PCM
 */
static struct hda_usr_pcm_info pcm_dmic_in = {
	.type = HDA_PCM_TYPE_ANALOG_IN,
	.name = "AD1984 Digital Mic",
	.substreams = 1,
	.device = 2,
	.channels_min = 2,
	.channels_max = 2,
	.nid = AD1984_DIG_MIC,
	.assoc_nids = { AD1984_DIG_MIC },
};

/*
 * Presets
 */
static hda_nid_t ad1884_dac_nids[1] = { AD1884_DAC };

static struct codec_config_preset ad1884_preset = {
	.init_verbs = {
		ad1884_init_verbs,
	},
	.mixers = {
		ad1884_base_mixers, ad1884_spdif_src
	},
	.num_dacs = 1,
	.dac_nids = ad1884_dac_nids,
	.adc_nid = AD1884_ADC,
	.input_mux = &ad1884_capture_source,
	.input_mux_nid = AD1884_CAPSRC,
	.input_mux_pins = ad1884_cap_pins,
	.dig_out_nid = AD1884_SPDIF_OUT,
};

enum {
	AD1984_BASIC,
	AD1984_THINKPAD,
	AD1984_MODELS
};

static const char *ad1984_models[AD1984_MODELS] = {
	[AD1984_BASIC]		= "basic",
	[AD1984_THINKPAD]	= "thinkpad",
};

static struct codec_config_preset ad1984_presets[AD1984_MODELS] = {
	[AD1984_BASIC] = {
		.init_verbs = {
			ad1884_init_verbs,
		},
		.mixers = {
			ad1884_base_mixers,
			ad1884_spdif_src,
			ad1984_dmic_mixers,
		},
		.num_dacs = 1,
		.dac_nids = ad1884_dac_nids,
		.adc_nid = AD1884_ADC,
		.input_mux = &ad1884_capture_source,
		.input_mux_nid = AD1884_CAPSRC,
		.input_mux_pins = ad1884_cap_pins,
		.dig_out_nid = AD1884_SPDIF_OUT,
		.extra_pcms = { &pcm_dmic_in },
	},
	[AD1984_THINKPAD] = {
		.init_verbs = {
			ad1884_init_verbs, thinkpad_init_verbs,
		},
		.mixers = {
			thinkpad_mixers,
		},
		.num_dacs = 1,
		.dac_nids = ad1884_dac_nids,
		.adc_nid = AD1884_ADC,
		.input_mux = &thinkpad_capture_source,
		.input_mux_nid = AD1884_CAPSRC,
	},
};

static int patch_ad1884(const struct hda_codec_table *tbl,
			int board_config, const char **args)
{
	return codec_build_preset(tbl, &ad1884_preset);
}

static int patch_ad1984(const struct hda_codec_table *tbl,
			int board_config, const char **args)
{
	if (board_config < 0 || board_config >= AD1984_MODELS)
		board_config = 0;
	return codec_build_preset(tbl, &ad1984_presets[board_config]);
}

/*
 * entry point
 */
static struct hda_codec_table ad1884_presets[] = {
	{ .id = 0x11d41884, .vendor_name = "Analog Devices",
	  .name = "AD1884", .patch = patch_ad1884 },
	{ .id = 0x11d41984, .vendor_name = "Analog Devices",
	  .num_presets = AD1984_MODELS, .presets = ad1984_models,
	  .name = "AD1984", .patch = patch_ad1984 },
	{ }
};

const struct hda_codec_table *patch_descriptor(void)
{
	return ad1884_presets;
}
