/*
 * Sony VAIO FE/SZ/AR with STAC9872
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include "hda/codecs-helper.h"

#define VAIO_DAC	0x02
#define VAIO_HP_DAC	0x05
#define VAIO_ADC	0x08	/* 0x06 */
#define VAIO_MUX_NID	0x15

static struct hda_input_mux vaio_mux = {
	.num_items = 3,
	.items = {
		/* { "HP", 0x0 }, */
		{ "Line", 0x1 },
		{ "Mic", 0x2 },
		{ "PCM", 0x3 },
	}
};

static struct hda_verb vaio_init[] = {
	{0x0a, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_HP }, /* HP <- 0x2 */
	{0x0f, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_OUT }, /* Speaker <- 0x5 */
	{0x0d, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_VREF80 }, /* Mic? (<- 0x2) */
	{0x0e, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_IN }, /* CD */
	{0x14, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_VREF80 }, /* Mic? */
	{0x15, AC_VERB_SET_CONNECT_SEL, 0x2}, /* mic-sel: 0a,0d,14,02 */
	{0x02, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE}, /* HP */
	{0x05, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE}, /* Speaker */
	{0x09, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_MUTE(0)}, /* capture sw/vol -> 0x8 */
	{0x07, AC_VERB_SET_AMP_GAIN_MUTE, AMP_IN_UNMUTE(0)}, /* CD-in -> 0x6 */
	{0x15, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_UNMUTE}, /* Mic-in -> 0x9 */
	{}
};

static struct hda_verb vaio_ar_init[] = {
	{0x11, AC_VERB_SET_PIN_WIDGET_CONTROL, PIN_OUT }, /* Optical Out */
	{0x10, AC_VERB_SET_AMP_GAIN_MUTE, AMP_OUT_MUTE}, /* Optical Out */
	{}
};

/* bind volumes of both NID 0x02 and 0x05 */
static u32 bind_ctls[] = {
	HDA_COMPOSE_AMP_VAL(0x02, 3, 0, HDA_OUTPUT),
	HDA_COMPOSE_AMP_VAL(0x05, 3, 0, HDA_OUTPUT),
	0
};

static struct hda_std_mixer vaio_mixer[] = {
	HDA_MIXER_BIND_VOLUME("Master Playback Volume", bind_ctls),
	HDA_MIXER_BIND_SWITCH("Master Playback Switch", bind_ctls),
	/* HDA_CODEC_VOLUME("CD Capture Volume", 0x07, 0, HDA_INPUT), */
	HDA_CODEC_VOLUME("Capture Volume", 0x09, 0, HDA_INPUT),
	HDA_CODEC_MUTE("Capture Switch", 0x09, 0, HDA_INPUT),
	{}
};

static struct hda_std_mixer vaio_ar_mixer[] = {
	HDA_CODEC_MUTE("Optical Out Switch", 0x10, 0, HDA_OUTPUT),
	HDA_CODEC_VOLUME("Optical Out Volume", 0x10, 0, HDA_OUTPUT),
	{}
};

static hda_nid_t vaio_dac_nids[1] = { VAIO_DAC };

enum {
	VAIO_FESZ,
	VAIO_AR,
	VAIO_MODELS
};

static struct codec_config_preset vaio_presets[VAIO_MODELS] = {
	[VAIO_FESZ] = {
		.init_verbs = {
			vaio_init,
		},
		.mixers = {
			vaio_mixer,
		},
		.num_dacs = 1,
		.dac_nids = vaio_dac_nids,
		.adc_nid = VAIO_ADC,
		.input_mux = &vaio_mux,
		.input_mux_nid = VAIO_MUX_NID,
	},
	[VAIO_AR] = {
		.init_verbs = {
			vaio_init,
			vaio_ar_init,
		},
		.mixers = {
			vaio_mixer,
			vaio_ar_mixer,
		},
		.num_dacs = 1,
		.dac_nids = vaio_dac_nids,
		.adc_nid = VAIO_ADC,
		.input_mux = &vaio_mux,
		.input_mux_nid = VAIO_MUX_NID,
	},
};

static int patch_stac9872(const struct hda_codec_table *tbl,
			  int board_config, const char **args)
{
	int preset = VAIO_FESZ;
	if (codec_info.vendor_id == 0x88347664)
		preset = VAIO_AR; /* VAIO AR */
	return codec_build_preset(tbl, &vaio_presets[preset]);
}

/*
 * entry point
 */
static struct hda_codec_table vaio_table[] = {
	{ .id = 0x83847661, .vendor_name = "IDT", .name = "CXD9872RD/K",
	  .patch = patch_stac9872 },
	{ .id = 0x83847662, .vendor_name = "IDT", .name = "STAC9872AK",
	  .patch = patch_stac9872 },
	{ .id = 0x83847664, .vendor_name = "IDT", .name = "CXD9872AKD",
	  .patch = patch_stac9872 },
	{ } /* terminator */
};

const struct hda_codec_table *patch_descriptor(void)
{
	return vaio_table;
}
