/*
 * Mixer interface registration
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/io.h>
#include <sys/types.h>
#include <sys/fcntl.h>
#include "local.h"
#include "hda/pincfg.h"

/*
 * MIXER INTERFACE
 */

struct hda_mixer_head_list {
	struct hda_mixer_head head;
	struct hda_mixer_head_list *next;
};

static struct hda_mixer_head_list *mixer_link;

/*
 * Look for the control element numid corresponding to the given name
 * If not found, return zero.
 */
unsigned int hda_look_for_ctl_id(const char *name)
{
	struct hda_mixer_head_list *list;

	for (list = mixer_link; list; list = list->next) {
		if (!strcmp(list->head.name, name))
			return list->head.numid;
	}
	return 0; /* not found */
}

static int add_mixer_head(struct hda_mixer_head *head)
{
	struct hda_mixer_head_list *list;

	list = malloc(sizeof(*list));
	if (!list)
		return -ENOMEM;
	list->head = *head;
	list->next = mixer_link;
	mixer_link = list;
	return 0;
}

/*
 * free mixer list
 */
void hda_mixer_free_all(void)
{
	struct hda_mixer_head_list *list, *next;

	for (list = mixer_link; list; list = next) {
		next = list->next;
		free(list);
	}
	mixer_link = NULL;
}

static int add_mixer(struct hda_std_mixer *mixer)
{
	struct hda_mixer_head *head;
	int err;
	
	switch (mixer->head.type) {
	case HDA_MIXER_TYPE_VOLUME:
	case HDA_MIXER_TYPE_SWITCH: {
		struct hda_volsw_mixer_head *imix = malloc(sizeof(*imix));
		if (!imix)
			return -ENOMEM;
		imix->head = mixer->head;
		imix->value[0] = mixer->value;
		head = &imix->head;
		break;
	}
	case HDA_MIXER_TYPE_BIND_VOLUME:
	case HDA_MIXER_TYPE_BIND_SWITCH: {
		struct hda_volsw_mixer_head *imix;
		u32 *values = (u32 *)mixer->value;
		int vals;

		for (vals = 0; values[vals]; vals++)
			;
		if (!vals)
			return -EINVAL;
		imix = malloc(sizeof(*imix) + vals * 4);
		if (!imix)
			return -ENOMEM;
		imix->head = mixer->head;
		memcpy(imix->value, values, (vals + 1) * 4);
		head = &imix->head;
		break;
	}
	case HDA_MIXER_TYPE_ENUM:
	case HDA_MIXER_TYPE_BOOLEAN:
	{
		struct hda_enum_mixer_head *emix;
		struct hda_cmds *cmds = (struct hda_cmds *)mixer->value;
		struct hda_tlv *tlv = hda_cmd_seq_to_tlv(HDA_CMD_TYPE_LIST, cmds);
		int len = hda_tlv_len(tlv);
		emix = malloc(sizeof(*emix) + len);
		if (!emix)
			return -ENOMEM;
		emix->head = mixer->head;
		emix->cur_val = mixer->init_val;
		memcpy(emix->tlv, tlv, len);
		hda_free_tlv(tlv);
		head = &emix->head;
		break;
	}
	default:
		return -EINVAL;
	}

	err = hda_do_ioctl(HDA_IOCTL_ADD_MIXER, head);
	if (err < 0) {
		log_error("Cannot add mixer '%s'\n", head->name);
		if (verbose_mode > 1 &&
		    (mixer->head.type == HDA_MIXER_TYPE_ENUM ||
		     mixer->head.type == HDA_MIXER_TYPE_BOOLEAN))
			hda_tlv_dump((struct hda_tlv *)(head + 1), stderr);
		goto error;
	}
	add_mixer_head(head);
 error:
	free(head);
	return 0;
}

/*
 * register mixer arrays
 */
int hda_register_mixers(struct hda_std_mixer *mixer)
{

	int err;
	for (; *mixer->head.name; mixer++) {
		err = add_mixer(mixer);
		if (err < 0)
			return err;
	}
	return 0;
}

/*
 * register a mixer
 */
int hda_register_mixer(int type, const char *name, long value)
{
	struct hda_std_mixer mix;

	memset(&mix, 0, sizeof(mix));
	mix.head.type = type;
	mix.head.iface = SND_CTL_ELEM_IFACE_MIXER;
	strncpy(mix.head.name, name, sizeof(mix.head.name) - 1);
	mix.value = value;
	return add_mixer(&mix);
}

/*
 * register an input_mux mixer
 */
int hda_register_input_mux_mixer(const char *name, hda_nid_t nid,
				 struct hda_input_mux *imux)
{
	struct hda_tlv *cmds[imux->num_items * 2];
	struct hda_tlv *tlv;
	struct hda_verb args[2];
	struct hda_enum_mixer_head *emix;
	int i, item;
	int err = -ENOMEM;

	if (!imux->num_items) {
		log_verbose("Mux without items\n");
		return -EINVAL;
	}
	item = 0;
	memset(args, 0, sizeof(args));
	for (i = 0; i < imux->num_items; i++) {
		cmds[item] = hda_make_str_tlv(HDA_CMD_TYPE_LABEL,
					      imux->items[i].label);
		if (!cmds[item])
			goto error;
		item++;
		args[0].nid = nid;
		args[0].verb = AC_VERB_SET_CONNECT_SEL;
		args[0].param = imux->items[i].index;
		cmds[item] = hda_verbs_to_tlv(args);
		if (!cmds[item])
			goto error;
		item++;
	}
	tlv = hda_make_tlv_args(HDA_CMD_TYPE_LIST, item, cmds);
	for (i = 0; i < item; i++)
		hda_free_tlv(cmds[i]);
	if (!tlv)
		return -ENOMEM;
	emix = malloc(sizeof(*emix) + hda_tlv_len(tlv));
	if (!emix) {
		hda_free_tlv(tlv);
		return -ENOMEM;
	}
	memset(emix, 0, sizeof(*emix));
	memcpy(emix->tlv, tlv, hda_tlv_len(tlv));
	hda_free_tlv(tlv);
	emix->head.type = HDA_MIXER_TYPE_ENUM;
	emix->head.iface = SND_CTL_ELEM_IFACE_MIXER;
	strncpy(emix->head.name, name, sizeof(emix->head.name) - 1);
	err = hda_do_ioctl(HDA_IOCTL_ADD_MIXER, &emix->head);
	if (err < 0) {
		log_error("Cannot register enum mixer '%s'\n", emix->head.name);
		if (verbose_mode > 1)
			hda_tlv_dump(emix->tlv, stderr);
		return -ENODEV;
	}
	add_mixer_head(&emix->head);
	hda_free_tlv(emix);
	return 0;

 error:
	for (i = 0; i < item; i++)
		hda_free_tlv(cmds[i]);
	return err;
}


/*
 * free dynamically allocated input_mux record
 */
void hda_input_mux_free(struct hda_input_mux *mux)
{
	int i;
	for (i = 0; i < mux->num_items; i++)
		free((void *)mux->items[i].label);
	free(mux);
}

/*
 * Fix up input mux
 *
 * check the pin default configuration and reduce the non-available items
 * from the given input_mux
 *
 * return the newly created input_mux struct (malloced).  the caller has
 * to free it manually by itself.
 */

struct hda_input_mux *hda_fixup_input_mux(hda_nid_t nid,
					  struct hda_input_mux *imux,
					  hda_nid_t *pin_nids)
{
	int i;
	struct hda_input_mux *nmux;

	nmux = malloc(sizeof(*nmux));
	if (!nmux)
		return NULL;
	memset(nmux, 0, sizeof(*nmux)); 
	for (i = 0; i < imux->num_items; i++) {
		if (pin_nids[i]) {
			if (!hda_pin_available(pin_nids[i])) {
				log_verbose("Input-mux pin 0x%x for %s not available\n",
					    pin_nids[i], imux->items[i].label);
				continue;
			}
		}
		nmux->items[nmux->num_items].index = imux->items[i].index;
		nmux->items[nmux->num_items].label =
			strdup(imux->items[i].label);
		if (!nmux->items[nmux->num_items].label) {
			hda_input_mux_free(nmux);
			return NULL;
		}
		nmux->num_items++;
	}
	return nmux;
}


/*
 * mixer array operation
 */

#define MIXER_ALLOC_SIZE	64

static struct hda_std_mixer *alloc_new_mixer(struct hda_mixer_array *rec)
{
	struct hda_std_mixer *mix;

	if (rec->num + 1 >= rec->alloc) {
		unsigned int newsize = rec->alloc + MIXER_ALLOC_SIZE;
		mix = realloc(rec->mixers, newsize * sizeof(*mix));
		if (!mix)
			return NULL;
		rec->mixers = mix;
		rec->alloc = newsize;
	}
	mix = &rec->mixers[rec->num++];
	memset(&rec->mixers[rec->num], 0, sizeof(*mix)); /* terminator */
	return mix;
}

int hda_append_mixer(struct hda_mixer_array *rec,
		     int type, const char *name, unsigned long val)
{
	struct hda_std_mixer *mix;

	mix = alloc_new_mixer(rec);
	if (!mix)
		return -ENOMEM;
	memset(mix, 0, sizeof(*mix));
	mix->head.iface = SND_CTL_ELEM_IFACE_MIXER;
	strncpy(mix->head.name, name, sizeof(mix->head.name) - 1);
	mix->head.type = type;
	mix->value = val;
	return 0;
}

int hda_append_mixers(struct hda_mixer_array *rec,
		      struct hda_std_mixer *mixers)
{
	for (; mixers->head.name; mixers++) {
		struct hda_std_mixer *mix;
		mix = alloc_new_mixer(rec);
		if (!mix)
			return -ENOMEM;
		*mix = *mixers;
	}
	return 0;
}

void hda_clear_mixer_array(struct hda_mixer_array *rec)
{
	free(rec->mixers);
	memset(rec, 0, sizeof(*rec));
}
