/*
 * Dump TLV stream
 *
 * Copyright (C) 2007 Takashi Iwai
 * 
 *  This library is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 */

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include "local.h"

#define MAX_TLV_LEN	4096

static void dump(FILE *fp, int level, const char *fmt, ...)
{
	int i;
	va_list ap;

	for (i = 0; i < level; i++)
		fputs("  ", fp);

	va_start(ap, fmt);
	vfprintf(fp, fmt, ap);
	va_end(ap);
	fputc('\n', fp);
}

static int process_cmd(FILE *fp, const struct hda_tlv *cmd, int *valuep,
		       int level, int *maxlen);

static int check_tlv_len(const struct hda_tlv *tlv, int *maxlen)
{
	if (tlv->len > MAX_TLV_LEN) {
		fprintf(stderr, "Too long len %d\n", tlv->len);
		return -EINVAL;
	}
	if (tlv->len % 4 != 0) {
		fprintf(stderr, "unaligned len %d\n", tlv->len);
		return -EINVAL;
	}
	if (tlv->len + 4 > *maxlen) {
		fprintf(stderr, "child len overflow 0x%x > 0x%x, type=%d\n",
			tlv->len, *maxlen, tlv->type);
		return -EINVAL;
	}
	*maxlen -= tlv->len + 4;
	return 0;
}

static int process_num_args(FILE *fp, const struct hda_tlv *cmd,
			    int nums, int *args, int level, int len)
{
	int i, err;
	for (i = 0; i < nums; i++) {
		err = process_cmd(fp, cmd, &args[i], level, &len);
		if (err < 0)
			return err;
		cmd += cmd->len / 4 + 1;
	}
	return 0;
}

static int process_verbs(FILE *fp, u32 *verbs, int level, int maxlen)
{
	for (; maxlen > 0; maxlen -= 4) {
		u32 verb = *verbs++;
		if (!verb)
			continue;
		dump(fp, level, "VERB: NID=0x%02x, VERB=0x%04x, PARM=0x%02x",
		     verb >> HDA_REG_NID_SHIFT,
		     (verb >> HDA_REG_VERB_SHIFT) & 0xfff,
		     verb & 0xff);
	} 
	return 0;
}

static int process_amp_read(FILE *fp, const struct hda_tlv *cmd,
			    int level, int maxlen)
{
	int err, verb;

	dump(fp, level, "AMP-READ:");
	err = process_cmd(fp, cmd, &verb, level + 1, &maxlen);
	if (err < 0)
		return err;
	dump(fp, level,
	     "[AMP: NID=0x%02x, CH=%d, DIR=%d, IDX=%d, MASK=0x%02x]",
	     verb >> HDA_AMP_NID_SHIFT,
	     (verb >> HDA_AMP_CH_SHIFT) & 1,
	     (verb >> HDA_AMP_DIR_SHIFT) & 1,
	     (verb >> HDA_AMP_IDX_SHIFT) & 0xf,
	     (verb >> HDA_AMP_MASK_SHIFT) & 0xff);
	return 0;
}

static int process_verb_update(FILE *fp, const struct hda_tlv *cmd, int *valuep,
			       int level, int maxlen)
{
	int val[2], err;
	dump(fp, level, "VERB-SINGLE:");
	err = process_num_args(fp, cmd, 2, val, level + 1, maxlen);
	if (err < 0)
		return err;
	dump(fp, level, "[VERB: NID=0x%02x, VERB=0x%04x, MASK=0x%02x, PARM=0x%02x]",
	     val[0] >> HDA_REG_NID_SHIFT,
	     (val[1] >> HDA_REG_VERB_SHIFT) & 0xfff,
	     val[1] & 0xff,
	     val[2]);
	if (valuep)
		*valuep = val[2];
	return 0;
}

static int process_amp_update(FILE *fp, const struct hda_tlv *cmd,
			      int level, int maxlen)
{
	int val[2], err;
	dump(fp, level, "AMP-SINGLE:");
	err = process_num_args(fp, cmd, 2, val, level + 1, maxlen);
	if (err < 0)
		return err;
	dump(fp, level,
	     "[AMP: NID=0x%02x, CH=%d, DIR=%d, IDX=%d, MASK=0x%02x, VAL=0x%02x]",
	     val[0] >> HDA_AMP_NID_SHIFT,
	     (val[0] >> HDA_AMP_CH_SHIFT) & 1,
	     (val[0] >> HDA_AMP_DIR_SHIFT) & 1,
	     (val[0] >> HDA_AMP_IDX_SHIFT) & 0xf,
	     (val[0] >> HDA_AMP_MASK_SHIFT) & 0xff,
	     val[1]);
	return 0;
}

static int process_ctl_notify(FILE *fp, u32 val, int level)
{
	dump(fp, level, "NOTIFY: ID=%d", val);
	return 0;
}

static int process_if(FILE *fp, const struct hda_tlv *cmd,
		      int level, int maxlen)
			      
{
	int err;

	dump(fp, level, "IF:");
	dump(fp, level + 1, "[IF COND]");
	err = process_cmd(fp, cmd, NULL, level + 1, &maxlen);
	if (err < 0)
		return err;
	if (!maxlen)
		return -EINVAL;
	cmd += cmd->len / 4 + 1;
	dump(fp, level + 1, "[IF-THEN]");
	process_cmd(fp, cmd, NULL, level + 1, &maxlen);
	if (!maxlen)
		return 0;
	cmd += cmd->len / 4 + 1;
	dump(fp, level + 1, "[IF-ELSE]");
	return process_cmd(fp, cmd, NULL, level + 1, &maxlen);
}

static int process_one_op(int type, FILE *fp, const struct hda_tlv *cmd,
			  int *valuep, int level, int maxlen)
{
	const char *label;
	int err;

	switch (type) {
	case HDA_CMD_TYPE_NOT:	label = "B-NOT"; break;
	case HDA_CMD_TYPE_LNOT:	label = "L-NOT"; break;
	default: return 0;
	}

	dump(fp, level, "%s:", label);
	err = process_cmd(fp, cmd, valuep, level + 1, &maxlen);
	if (err < 0)
		return err;
	if (!valuep)
		return 0;
	switch (type) {
	case HDA_CMD_TYPE_NOT:	*valuep = ~*valuep; break;
	case HDA_CMD_TYPE_LNOT:	*valuep = !*valuep; break;
	}
	return 0;
}

static int process_two_ops(int type, FILE *fp, const struct hda_tlv *cmd,
			   int *valuep, int level, int maxlen)
{
	const char *label;
	int val[2], err;

	switch (type) {
	case HDA_CMD_TYPE_AND:	label = "B-AND"; break;
	case HDA_CMD_TYPE_OR:	label = "B-OR"; break;
	case HDA_CMD_TYPE_SHIFTL: label = "SHIFTL"; break;
	case HDA_CMD_TYPE_SHIFTR: label = "SHIFTR"; break;
	case HDA_CMD_TYPE_PLUS:	label = "PLUS"; break;
	case HDA_CMD_TYPE_MINUS: label = "MINUS"; break;
	case HDA_CMD_TYPE_LAND:	label = "L-AND"; break;
	case HDA_CMD_TYPE_LOR:	label = "L-OR"; break;
	default: return 0;
	}

	dump(fp, level, "%s:", label);
	err = process_num_args(fp, cmd, 2, val, level + 1, maxlen);
	if (err < 0)
		return err;
	if (!valuep)
		return 0;

	switch (type) {
	case HDA_CMD_TYPE_AND:	*valuep = val[0] & val[1]; break;
	case HDA_CMD_TYPE_OR:	*valuep = val[0] | val[1]; break;
	case HDA_CMD_TYPE_SHIFTL: *valuep = val[0] << val[1]; break;
	case HDA_CMD_TYPE_SHIFTR: *valuep = val[0] >> val[1]; break;
	case HDA_CMD_TYPE_PLUS:	*valuep = val[0] + val[1]; break;
	case HDA_CMD_TYPE_MINUS: *valuep = val[0] - val[1]; break;
	case HDA_CMD_TYPE_LAND:	*valuep = val[0] && val[1]; break;
	case HDA_CMD_TYPE_LOR:	*valuep = val[0] || val[1]; break;
	}
	return 0;
}

static int process_set_channels(FILE *fp, int val, int level)
{
	dump(fp, level, "SET_CHANNELS: CHS=%d", val);
	return 0;
}

static int process_cmds(int type, FILE *fp, const struct hda_tlv *cmd,
			int *valuep, int level, int len)
{
	const char *label;
	int err;

	switch (type) {
	case HDA_CMD_TYPE_CMDS:	label = "CMDS"; break;
	case HDA_CMD_TYPE_LIST:	label = "LIST"; break;
	case HDA_CMD_TYPE_LAND:	label = "L-AND"; break;
	case HDA_CMD_TYPE_LOR:	label = "L-OR"; break;
	default: return 0;
	}
	dump(fp, level, "%s:", label);
	while (len > 0) {
		err = process_cmd(fp, cmd, valuep, level + 1, &len);
		if (err < 0)
			return err;
		cmd += cmd->len / 4 + 1;
	}
	return 0;
}

static int process_cmd(FILE *fp, const struct hda_tlv *cmd, int *valuep,
		       int level, int *maxlen)
{
	int len, type;
	u32 *val;

	if (!cmd)
		return 0;
	if (check_tlv_len(cmd, maxlen))
		return -EINVAL;
	len = cmd->len;
	type = cmd->type;
	cmd++;
	val = (u32 *)cmd;

	if (type == HDA_CMD_TYPE_NOP) {
		dump(fp, level, "NOP:");
		return 0;
	}
	if (len < 4)
		return -EINVAL;

	switch (type) {
	case HDA_CMD_TYPE_INT:
		dump(fp, level, "INT: %d", *val);
		if (valuep)
			*valuep = *val;
		return 0;
	case HDA_CMD_TYPE_DELAY:
		dump(fp, level, "DELAY: %d", *val);
		return 0;
	case HDA_CMD_TYPE_LIST:
	case HDA_CMD_TYPE_CMDS:
	case HDA_CMD_TYPE_LAND:
	case HDA_CMD_TYPE_LOR:
		return process_cmds(type, fp, cmd, valuep, level, len);
	case HDA_CMD_TYPE_VERBS:
		return process_verbs(fp, val, level, len);
	case HDA_CMD_TYPE_AMP_READ:
		return process_amp_read(fp, cmd, level, len);
	case HDA_CMD_TYPE_VERB_UPDATE:
		return process_verb_update(fp, cmd, valuep, level, len);
	case HDA_CMD_TYPE_AMP_UPDATE:
		return process_amp_update(fp, cmd, level, len);
	case HDA_CMD_TYPE_CTL_NOTIFY:
		return process_ctl_notify(fp, *val, level);
	case HDA_CMD_TYPE_SET_CHANNELS:
		return process_set_channels(fp, *val, level);
	case HDA_CMD_TYPE_LABEL:
		dump(fp, level, "LABEL: %s", (char *)cmd);
		return 0;
	case HDA_CMD_TYPE_IF:
		return process_if(fp, cmd, level, len);
	case HDA_CMD_TYPE_NOT:
	case HDA_CMD_TYPE_LNOT:
		return process_one_op(type, fp, cmd, valuep, level, len);
	case HDA_CMD_TYPE_AND:
	case HDA_CMD_TYPE_OR:
	case HDA_CMD_TYPE_PLUS:
	case HDA_CMD_TYPE_MINUS:
	case HDA_CMD_TYPE_SHIFTL:
	case HDA_CMD_TYPE_SHIFTR:
		return process_two_ops(type, fp, cmd, valuep, level, len);
	}
	return -EINVAL;
}

int hda_tlv_dump(const struct hda_tlv *cmd, FILE *fp)
{
	int maxlen = cmd->len + 4;
	return process_cmd(fp, cmd, NULL, 0, &maxlen);
}
