/*
 * invoking periodical RTC interrupts and report the stack pointers
 * 
 * for i386 only!
 */

#include <linux/config.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/rtc.h>
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/time.h>
#include <linux/sched.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/kallsyms.h>
#include <asm/uaccess.h>
#include "latencytest.h"


static spinlock_t my_lock = SPIN_LOCK_UNLOCKED;

static int rtc_freq = 1024;
static rtc_task_t rtc_task;
static int count, irq_count;

static wait_queue_head_t my_sleep;

static struct latency_test_info irq_info;
static struct latency_test_info saved_info;

static int opened;
static int rtc_running;

static void my_interrupt(void *private_data)
{
	unsigned long stack;

	spin_lock(&my_lock);
	count++;
	if (count < irq_count)
		return;
	count = 0;
	if (irq_info.processed < MAX_PROC_CNTS) {
		int i;
		unsigned long *dump = &irq_info.stacks[irq_info.processed][0];
		unsigned short idx;
		unsigned short *posp = &irq_info.stack_pos[irq_info.processed][0];
		unsigned long *stackp = &stack + 1;
		// rdtscll(irq_info.irq_time[irq_info.processed]);
		i = 0;
		idx = 0;
		while (! kstack_end(stackp)) {
			unsigned long addr, check;
			addr = *stackp++;
			idx++;
			/* if i could use kernel_text_address()... */
			check = addr & 0xf0000000;
			if (check == 0xc0000000 || check == 0xd0000000) {
				*dump++ = addr;
				*posp++ = idx;
				if (++i >= MAX_STACK)
					break;
			}
		}
		memcpy(&irq_info.comm[irq_info.processed][0], current->comm, 16);
	}
	irq_info.processed++;
	wake_up(&my_sleep);
	spin_unlock(&my_lock);
}


static int my_start(void)
{
	if (irq_count) {
		rtc_control(&rtc_task, RTC_IRQP_SET, rtc_freq);
		rtc_control(&rtc_task, RTC_PIE_ON, 0);
		rtc_running = 1;
		return 0;
	}
	return -EINVAL;
}

static void my_stop(void)
{
	if (rtc_running)
		rtc_control(&rtc_task, RTC_PIE_OFF, 0);
	rtc_running = 0;
}

static int my_open(struct inode *inode, struct file *file)
{
	int err;

	if (opened)
		return -EBUSY;

	irq_count = 0;
	count = 0;
	memset(&irq_info, 0, sizeof(irq_info));
	init_waitqueue_head(&my_sleep);
	rtc_task.func = my_interrupt;
	err = rtc_register(&rtc_task);
	if (err < 0)
		return err;

	opened = 1;
	return 0;
}

static int my_release(struct inode *inode, struct file *file)
{
	if (opened) {
		my_stop();
		rtc_unregister(&rtc_task);
		opened = 0;
	}
	return 0;
}

static int my_read(unsigned long arg)
{
	wait_queue_t wait;

	init_waitqueue_entry(&wait, current);
	spin_lock_irq(&my_lock);
	add_wait_queue(&my_sleep, &wait);
	while (! irq_info.processed) {
		if (signal_pending(current)) {
			remove_wait_queue(&my_sleep, &wait);
			spin_unlock_irq(&my_lock);
			return -EINTR;
		}
		set_current_state(TASK_INTERRUPTIBLE);
		spin_unlock_irq(&my_lock);
		schedule();
		spin_lock_irq(&my_lock);
	}
	remove_wait_queue(&my_sleep, &wait);
	memcpy(&saved_info, &irq_info, sizeof(irq_info));
	memset(&irq_info, 0, sizeof(irq_info));
	spin_unlock_irq(&my_lock);

	// rdtscll(saved_info.read_time);
	if (copy_to_user((void*)arg, &saved_info, sizeof(saved_info)))
		return -EFAULT;
	return 0;
}

static int get_symbol(unsigned long arg)
{
	struct ksym_lookup_info info;
	char *modname;
	const char *name;

	if (copy_from_user(&info, (void *)arg, sizeof(info)))
		return -EFAULT;
	*info.name = 0;
	name = kallsyms_lookup(info.addr, &info.size, &info.offset, &modname, info.name);
	if (name && name != info.name)
		strncpy(info.name, name, sizeof(info.name));
	if (copy_to_user((void *)arg, &info, sizeof(info)))
		return -EFAULT;
	return 0;
}

static int my_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	int i;
	switch (cmd) {
	case LAT_TEST_FREQ:
		for (i = 0; arg > (1 << i); i++)
			;
		if (arg != (1 << i))
			return -EINVAL;
		rtc_freq = arg;
		return 0;
	case LAT_TEST_COUNT:
		irq_count = arg;
		return 0;
	case LAT_TEST_START:
		return my_start();
	case LAT_TEST_STOP:
		my_stop();
		return 0;
	case LAT_TEST_READ:
		return my_read(arg);
	case LAT_TEST_GETSYMBOL:
		return get_symbol(arg);
	}
	printk(KERN_ERR "irqtest: unknown ioctl cmd=0x%x\n", cmd);
	return -EINVAL;
}

static struct file_operations my_fops = {
	.owner		= THIS_MODULE,
	.ioctl		= my_ioctl,
	.open		= my_open,
	.release	= my_release,
};

#define IRQTEST_MAJOR	35

static int __init irq_test_init(void)
{
	if (register_chrdev(IRQTEST_MAJOR, "irqtest", &my_fops)) {
		printk(KERN_ERR "cannot register device\n");
		return -ENODEV;
	}
	printk(KERN_ERR "irq-test registered\n");
	return 0;
}

static void __exit irq_test_exit(void)
{
	unregister_chrdev(IRQTEST_MAJOR, "irqtest");
	printk(KERN_ERR "irq-test de-registered\n");
}

module_init(irq_test_init);
module_exit(irq_test_exit);

MODULE_LICENSE("GPL");
