/*
 * madhouse.c - a wacky insane wrapper for libmad
 *
 * Copyright (c) 2009 Takashi Iwai <tiwai@suse.de>
 *
 * Partly contains the code from libmad-0.15.2b
 *
 * libmad - MPEG audio decoder library
 * Copyright (C) 2000-2004 Underbit Technologies, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dlfcn.h>
#include <errno.h>
#include "mad.h"

static void *maddl;

#ifndef MADLIBNAME
#define MADLIBNAME	"/usr/lib64/libmad.so.0"
#endif

static char *madlibpath;

static void maddl_update(void)
{
	if (!maddl)
		maddl = dlopen(madlibpath ? madlibpath : MADLIBNAME,
			       RTLD_LAZY);
}

int madhouse_is_available(void)
{
	maddl_update();
	return maddl ? 1 : 0;
}

int madhouse_set_path(const char *path)
{
	if (!path || strcmp(madlibpath, path)) {
		free(madlibpath);
		madlibpath = NULL;
	}
	if (!madlibpath && path) {
		madlibpath = strdup(path);
		if (!madlibpath)
			return -ENOMEM;
	}
	return 0;
}

/* clean up dl handle */
static void __attribute__((destructor)) madhouse_exist(void)
{
	if (maddl)
		dlclose(maddl);
	free(madlibpath);
}

static void *get_sym(const char *sym)
{
	maddl_update();
	if (!maddl)
		return NULL;
	return dlsym(maddl, sym);
}

/* define a function pointer to its real function in libmad.so */
#define DEFINE_FUNC(me)				\
	static typeof(me) *func;		\
	if (!func)				\
		func = get_sym(#me);

/* copied from version.c */

char const mad_version[]   = "MPEG Audio Decoder " MAD_VERSION;
char const mad_copyright[] = "Copyright (C) " MAD_PUBLISHYEAR " " MAD_AUTHOR;
char const mad_author[]    = MAD_AUTHOR " <" MAD_EMAIL ">";

char const mad_build[] = ""
# if defined(DEBUG)
  "DEBUG "
# elif defined(NDEBUG)
  "NDEBUG "
# endif

# if defined(EXPERIMENTAL)
  "EXPERIMENTAL "
# endif

# if defined(FPM_64BIT)
  "FPM_64BIT "
# elif defined(FPM_INTEL)
  "FPM_INTEL "
# elif defined(FPM_ARM)
  "FPM_ARM "
# elif defined(FPM_MIPS)
  "FPM_MIPS "
# elif defined(FPM_SPARC)
  "FPM_SPARC "
# elif defined(FPM_PPC)
  "FPM_PPC "
# elif defined(FPM_DEFAULT)
  "FPM_DEFAULT "
# endif

# if defined(ASO_IMDCT)
  "ASO_IMDCT "
# endif
# if defined(ASO_INTERLEAVE1)
  "ASO_INTERLEAVE1 "
# endif
# if defined(ASO_INTERLEAVE2)
  "ASO_INTERLEAVE2 "
# endif
# if defined(ASO_ZEROCHECK)
  "ASO_ZEROCHECK "
# endif

# if defined(OPT_SPEED)
  "OPT_SPEED "
# elif defined(OPT_ACCURACY)
  "OPT_ACCURACY "
# endif

# if defined(OPT_SSO)
  "OPT_SSO "
# endif

# if defined(OPT_DCTO)  /* never defined here */
  "OPT_DCTO "
# endif

# if defined(OPT_STRICT)
  "OPT_STRICT "
# endif
;

/* copied from fixed.c */

mad_fixed_t mad_f_abs(mad_fixed_t x)
{
  return x < 0 ? -x : x;
}

mad_fixed_t mad_f_div(mad_fixed_t x, mad_fixed_t y)
{
  mad_fixed_t q, r;
  unsigned int bits;

  q = mad_f_abs(x / y);

  if (x < 0) {
    x = -x;
    y = -y;
  }

  r = x % y;

  if (y < 0) {
    x = -x;
    y = -y;
  }

  if (q > mad_f_intpart(MAD_F_MAX) &&
      !(q == -mad_f_intpart(MAD_F_MIN) && r == 0 && (x < 0) != (y < 0)))
    return 0;

  for (bits = MAD_F_FRACBITS; bits && r; --bits) {
    q <<= 1, r <<= 1;
    if (r >= y)
      r -= y, ++q;
  }

  /* round */
  if (2 * r >= y)
    ++q;

  /* fix sign */
  if ((x < 0) != (y < 0))
    q = -q;

  return q << bits;
}

/* bit.c */
/* wrappers */

void mad_bit_init(struct mad_bitptr *bit, unsigned char const *data)
{
	DEFINE_FUNC(mad_bit_init);
	if (!func)
		return;
	func(bit, data);
}

unsigned int mad_bit_length(struct mad_bitptr const *a,
			    struct mad_bitptr const *b)
{
	DEFINE_FUNC(mad_bit_length);
	if (!func)
		return 0;
	return func(a, b);
}

unsigned char const *mad_bit_nextbyte(struct mad_bitptr const *bit)
{
	DEFINE_FUNC(mad_bit_nextbyte);
	if (!func)
		return NULL;
	return func(bit);
}

void mad_bit_skip(struct mad_bitptr *bit, unsigned int val)
{
	DEFINE_FUNC(mad_bit_skip);
	if (!func)
		return;
	func(bit, val);
}

unsigned long mad_bit_read(struct mad_bitptr *bit, unsigned int pos)
{
	DEFINE_FUNC(mad_bit_read);
	if (!func)
		return 0;
	return func(bit, pos);
}

void mad_bit_write(struct mad_bitptr *bit, unsigned int pos, unsigned long val)
{
	DEFINE_FUNC(mad_bit_write);
	if (!func)
		return;
	func(bit, pos, val);
}

unsigned short mad_bit_crc(struct mad_bitptr bit, unsigned int pos,
			   unsigned short val)
{
	DEFINE_FUNC(mad_bit_crc);
	if (!func)
		return 0;
	return func(bit, pos, val);
}

/* timer.c */

mad_timer_t const mad_timer_zero = { 0, 0 };

int mad_timer_compare(mad_timer_t a, mad_timer_t b)
{
	DEFINE_FUNC(mad_timer_compare);
	if (!func)
		return 0;
	return func(a, b);
}

void mad_timer_negate(mad_timer_t *t)
{
	DEFINE_FUNC(mad_timer_negate);
	if (!func)
		return;
	func(t);
}

mad_timer_t mad_timer_abs(mad_timer_t t)
{
	DEFINE_FUNC(mad_timer_abs);
	if (!func)
		return t;
	return func(t);
}

void mad_timer_set(mad_timer_t *t, unsigned long a, unsigned long b,
		   unsigned long c)
{
	DEFINE_FUNC(mad_timer_set);
	if (!func)
		return;
	return func(t, a, b, c);
}

void mad_timer_add(mad_timer_t *a, mad_timer_t b)
{
	DEFINE_FUNC(mad_timer_add);
	if (!func)
		return;
	return func(a, b);
}

void mad_timer_multiply(mad_timer_t *t, signed long a)
{
	DEFINE_FUNC(mad_timer_multiply);
	if (!func)
		return;
	return func(t, a);
}

signed long mad_timer_count(mad_timer_t t, enum mad_units c)
{
	DEFINE_FUNC(mad_timer_count);
	if (!func)
		return 0;
	return func(t, c);
}

unsigned long mad_timer_fraction(mad_timer_t t, unsigned long c)
{
	DEFINE_FUNC(mad_timer_fraction);
	if (!func)
		return 0;
	return func(t, c);
}

void mad_timer_string(mad_timer_t t, char * a, char const * b,
		      enum mad_units c, enum mad_units d, unsigned long e)
{
	DEFINE_FUNC(mad_timer_string);
	if (!func)
		return;
	func(t, a, b, c, d, e);
}

/* stream.c */

void mad_stream_init(struct mad_stream *s)
{
	DEFINE_FUNC(mad_stream_init);
	if (!func)
		return;
	func(s);
}

void mad_stream_finish(struct mad_stream *s)
{
	DEFINE_FUNC(mad_stream_finish);
	if (!func)
		return;
	func(s);
}

void mad_stream_buffer(struct mad_stream *s,
		       unsigned char const *a, unsigned long b)
{
	DEFINE_FUNC(mad_stream_buffer);
	if (!func)
		return;
	func(s, a, b);
}

void mad_stream_skip(struct mad_stream *s, unsigned long a)
{
	DEFINE_FUNC(mad_stream_skip);
	if (!func)
		return;
	func(s, a);
}

int mad_stream_sync(struct mad_stream *s)
{
	DEFINE_FUNC(mad_stream_sync);
	if (!func)
		return -1;
	return func(s);
}

char const *mad_stream_errorstr(struct mad_stream const *s)
{
	DEFINE_FUNC(mad_stream_errorstr);
	if (!func)
		return NULL;
	return func(s);
}

/* frame.c */

void mad_header_init(struct mad_header *h)
{
	DEFINE_FUNC(mad_header_init);
	if (!func)
		return;
	func(h);
}

int mad_header_decode(struct mad_header *h, struct mad_stream *s)
{
	DEFINE_FUNC(mad_header_decode);
	if (!func)
		return -1;
	return func(h, s);
}

void mad_frame_init(struct mad_frame *f)
{
	DEFINE_FUNC(mad_frame_init);
	if (!func)
		return;
	return func(f);
}

void mad_frame_finish(struct mad_frame *f)
{
	DEFINE_FUNC(mad_frame_finish);
	if (!func)
		return;
	return func(f);
}

int mad_frame_decode(struct mad_frame *f, struct mad_stream *s)
{
	DEFINE_FUNC(mad_frame_decode);
	if (!func)
		return -1;
	return func(f, s);
}

void mad_frame_mute(struct mad_frame *f)
{
	DEFINE_FUNC(mad_frame_mute);
	if (!func)
		return;
	func(f);
}

/* synth.c */

void mad_synth_init(struct mad_synth *s)
{
	DEFINE_FUNC(mad_synth_init);
	if (!func)
		return;
	func(s);
}

void mad_synth_mute(struct mad_synth *s)
{
	DEFINE_FUNC(mad_synth_mute);
	if (!func)
		return;
	func(s);
}

void mad_synth_frame(struct mad_synth *s, struct mad_frame const *f)
{
	DEFINE_FUNC(mad_synth_frame);
	if (!func)
		return;
	func(s, f);
}

/* decoder.c */

void mad_decoder_init(struct mad_decoder *decoder, void *data,
		      enum mad_flow (*input_func)(void *,
						  struct mad_stream *),
		      enum mad_flow (*header_func)(void *,
						   struct mad_header const *),
		      enum mad_flow (*filter_func)(void *,
						   struct mad_stream const *,
						   struct mad_frame *),
		      enum mad_flow (*output_func)(void *,
						   struct mad_header const *,
						   struct mad_pcm *),
		      enum mad_flow (*error_func)(void *,
						  struct mad_stream *,
						  struct mad_frame *),
		      enum mad_flow (*message_func)(void *,
						    void *, unsigned int *))
{
	DEFINE_FUNC(mad_decoder_init);
	if (!func)
		return;
	func(decoder, data, input_func, header_func, filter_func,
	     output_func, error_func, message_func);
}

int mad_decoder_finish(struct mad_decoder *decoder)
{
	DEFINE_FUNC(mad_decoder_finish);
	if (!func)
		return -1;
	return func(decoder);
}

int mad_decoder_run(struct mad_decoder *decoder, enum mad_decoder_mode mode)
{
	DEFINE_FUNC(mad_decoder_run);
	if (!func)
		return -1;
	return func(decoder, mode);
}

int mad_decoder_message(struct mad_decoder *decoder, void *data, unsigned int *msg)
{
	DEFINE_FUNC(mad_decoder_message);
	if (!func)
		return -1;
	return func(decoder, data, msg);
}
