/*
 * zio.c	Provide an streamable interface to gziped/bzip2ed files
 *
 * Copyright 2004 Werner Fink, 2004 SuSE LINUX AG, Germany.
 * Copyright 2006 Werner Fink, 2006 SuSE Products GmbH, Germany.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Author:	Werner Fink <werner@suse.de>
 */

#include "zioP.h"
#include "zio.h"
#include "lzw.h"

static ssize_t   zread(void *cookie, char *buf, size_t count)
{
    return (ssize_t)gzread((gzFile)cookie, (voidp)buf, count);
}

static ssize_t   zwrite(void *cookie, const char *buf, size_t count)
{
    return (ssize_t)gzwrite((gzFile)cookie, (const voidp)buf, count);
}

static int       zseek(void *cookie, off_t offset, int whence)
{
    return (int)gzseek((gzFile)cookie, (z_off_t)(offset), whence);
}

static int       zclose(void *cookie)
{
    int status = gzclose((gzFile)cookie);
    return (status >= 0) ? 0 : EOF;
}

__extension__
static cookie_io_functions_t ioz = {
    .read  = (cookie_read_function_t*) &zread,
    .seek  = (cookie_seek_function_t*) &zseek,
    .write = (cookie_write_function_t*)&zwrite,
    .close = (cookie_close_function_t*)&zclose,
};

static ssize_t   bzread(void *cookie, char *buf, size_t count)
{
    return (ssize_t)BZ2_bzread((BZFILE*)cookie, (voidp)buf, count);
}

static ssize_t   bzwrite(void *cookie, const char *buf, size_t count)
{
    return (ssize_t)BZ2_bzwrite((BZFILE*)cookie, (const voidp)buf, count);
}

static int       bzseek(void *cookie, off_t offset, int whence)
{
    errno = ESPIPE;
    return -1;
}

static int       bzclose(void *cookie)
{
    int status = BZ2_bzflush((BZFILE*)cookie);
    BZ2_bzclose((BZFILE*)cookie);
    return (status >= 0) ? 0 : EOF;
}

__extension__
static cookie_io_functions_t iobz = {
    .read  = (cookie_read_function_t*) &bzread,
    .seek  = (cookie_seek_function_t*) &bzseek,
    .write = (cookie_write_function_t*)&bzwrite,
    .close = (cookie_close_function_t*)&bzclose,
};

static ssize_t   lzwread(void *cookie, char *buf, size_t count)
{
    return readlzw((LZW_t*)cookie, buf, count);
}

static ssize_t   lzwwrite(void *cookie, const char *buf, size_t count)
{
    errno = ENOTSUP;
    return -1;
}

static int       lzwseek(void *cookie, off_t offset, int whence)
{
    errno = ESPIPE;
    return -1;
}

static int       lzwclose(void *cookie)
{
    closelzw((LZW_t*)cookie);
    return 0;
}

__extension__
static cookie_io_functions_t iolzw = {
    .read  = (cookie_read_function_t*) &lzwread,
    .seek  = (cookie_seek_function_t*) &lzwseek,
    .write = (cookie_write_function_t*)&lzwwrite,
    .close = (cookie_close_function_t*)&lzwclose,
};

FILE * fzopen(const char * path, const char * mode)
{
    FILE * ret = (FILE *)0;
    char * check = (char*)0, * ext = (char*)0;
    size_t n = 0, len;
    unsigned int i;
    char what = 'n';

    if (!mode || !(n = strlen(mode))) {
	errno = EINVAL;
	goto out;
    }

    if (!(check = (char*)malloc(n*sizeof(char))))
	goto out;

    /* No append mode possible */
    switch (*mode) {
	case 'r': check[0] = 'r'; break;
	case 'w': check[0] = 'w'; break;
	default:  errno = EINVAL; goto out;
    }

    for (i = 1; i < n; i++) {
	/* We can only open for reading OR writing but NOT for both */
	switch (mode[i]) {
	    case '\0': break;
	    case '+': errno = EINVAL; goto out;
	    case 'b': case 'x': check[i] = mode[i]; continue;
	    /* Ingore switches for gzopen() */
	    case 'f': case 'h': check[i] = '\0';    continue;
	    default:		check[i] = '\0';    continue;
	}
	break;
    }

    if (!path || !(len = strlen(path))) {
	errno = EINVAL;
	goto out;
    }

    if      (strcmp(path + len - 2, ".z"  ) == 0)
	what = 'z';
    else if (strcmp(path + len - 3, ".gz" ) == 0)
	what = 'g';
    else if (strcmp(path + len - 3, ".Z"  ) == 0)
	what = 'Z';
    else if (strcmp(path + len - 4, ".bz2") == 0)
	what = 'b';

    if (what == 'n' && *check == 'r') {
	int olderr, fd;
	struct stat st;
	char m[3];
	ext = malloc(sizeof(char)*(len + 4 + 1));
	if (!ext)
	    goto out;
	strcpy(ext, path);

	olderr = errno;
	if (stat(strcat(ext, ".gz"),  &st) == 0) {
	    what = 'g';
	    goto skip;
	}
	*(ext + len) = '\0';
	if (stat(strcat(ext, ".bz2"), &st) == 0) {
	    what = 'b';
	    goto skip;
	}
	*(ext + len) = '\0';
	if (stat(strcat(ext, ".z"),   &st) == 0) {
	    what = 'z';
	    goto skip;
	}
	*(ext + len) = '\0';
	if (stat(strcat(ext, ".Z"),   &st) == 0) {
	    what = 'Z';
	    goto skip;
	}
	*(ext + len) = '\0';

	if ((fd = open(ext, O_RDONLY|O_NOCTTY)) < 0)
	    goto skip;
	if (read(fd, m, sizeof(m)) == sizeof(m)) {
	    if (m[0] == '\037' && m[1] == '\213')
		what = 'g';
	    if (m[0] == '\037' && m[1] == '\235')
		what = 'Z';
	    if (m[0] == '\037' && m[1] == '\236')
		what = 'z';
	    else if (m[0] == 'B' && m[1] == 'Z' && m[2] == 'h')
		what = 'b';
	}
	close(fd);

    skip:
	errno = olderr;
    } else
	ext = (char *)path;

    switch (what) {
    case 'g':
    case 'z':		/* Is this correct? Old gzip magic */
	{
	    gzFile cookie = (gzFile)0;

	    if (&gzopen == NULL) {
		errno = ENOSYS;
		goto out;
	    }

	    if (!(cookie = gzopen(ext, mode))) {
		if (!errno)
		    errno = ENOMEM;
		goto out;
	    }

	    if (!(ret = fopencookie(cookie, check, ioz))) {
		gzclose(cookie);
		errno = EINVAL;
		goto out;
	    }
#ifndef LIBIO_IS_FIXED
	    if (ret->_fileno < 0)
		ret->_fileno = 0;
#endif
	}
	break;
    case 'Z':
	{
	    LZW_t *cookie;
	    if (*mode != 'r') {
		errno = ENOTSUP;
		goto out;
	    }

	    if (!(cookie = openlzw(ext, mode))) {
		if (!errno)
		    errno = ENOMEM;
		goto out;
	    }

	    if (!(ret = fopencookie(cookie, check, iolzw))) {
		closelzw(cookie);
		errno = EINVAL;
		goto out;
	    }
#ifndef LIBIO_IS_FIXED
	    if (ret->_fileno < 0)
		ret->_fileno = 0;
#endif
	}
	break;
    case 'b':
	{
	    BZFILE* cookie = (BZFILE*)0;

	    if (&BZ2_bzopen == NULL) {
		errno = ENOSYS;
		goto out;
	    }

	    if (!(cookie = BZ2_bzopen(ext, mode))) {
		if (!errno)
		    errno = ENOMEM;
		goto out;
	    }

	    if (!(ret = fopencookie(cookie, check, iobz))) {
		BZ2_bzclose(cookie);
		errno = EINVAL;
		goto out;
	    }
#ifndef LIBIO_IS_FIXED
	    if (ret->_fileno < 0)
		ret->_fileno = 0;
#endif
	}
	break;
    default:
	ret = fopen(ext, mode);
	break;
    }

out:
    if (check)
	free(check);
    if (ext && ext != path)
	free(ext);

    return ret;
}

FILE * fdzopen(int fildes, const char * mode, const char *what)
{
    FILE * ret = (FILE *)0;
    char * check = (char*)0;
    size_t n = 0;
    unsigned int i;

    if (!mode || !(n = strlen(mode))) {
	errno = EINVAL;
	goto out;
    }

    if (!(check = (char*)malloc(n*sizeof(char))))
	goto out;

    /* No append mode possible */
    switch (*mode) {
	case 'r': check[0] = 'r'; break;
	case 'w': check[0] = 'w'; break;
	default:  errno = EINVAL; goto out;
    }

    for (i = 1; i < n; i++) {
	/* We can only open for reading OR writing but NOT for both */
	switch (mode[i]) {
	    case '\0': break;
	    case '+': errno = EINVAL; goto out;
	    case 'b': case 'x': check[i] = mode[i]; continue;
	    /* Ingore switches for gzopen() */
	    case 'f': case 'h': check[i] = '\0';    continue;
	    default:		check[i] = '\0';    continue;
	}
	break;
    }

    switch (*what) {
    case 'g':
    case 'z':		/* Is this correct? Old gzip magic */
	{
	    gzFile cookie = (gzFile)0;

	    if (&gzdopen == NULL) {
		errno = ENOSYS;
		goto out;
	    }

	    if (!(cookie = gzdopen(fildes, mode))) {
		if (!errno)
		    errno = ENOMEM;
		goto out;
	    }

	    if (!(ret = fopencookie(cookie, check, ioz))) {
		gzclose(cookie);
		errno = EINVAL;
		goto out;
	    }
#ifndef LIBIO_IS_FIXED
	    if (ret->_fileno < 0)
		ret->_fileno = 0;
#endif
	}
	break;
    case 'Z':
	{
	    LZW_t *cookie;
	    if (*mode != 'r') {
		errno = ENOTSUP;
		goto out;
	    }

	    if (!(cookie = dopenlzw(fildes, mode))) {
		if (!errno)
		    errno = ENOMEM;
		goto out;
	    }

	    if (!(ret = fopencookie(cookie, check, iolzw))) {
		closelzw(cookie);
		errno = EINVAL;
		goto out;
	    }
#ifndef LIBIO_IS_FIXED
	    if (ret->_fileno < 0)
		ret->_fileno = 0;
#endif
	}
	break;
    case 'b':
	{
	    BZFILE* cookie = (BZFILE*)0;

	    if (&BZ2_bzdopen == NULL) {
		errno = ENOSYS;
		goto out;
	    }

	    if (!(cookie = BZ2_bzdopen(fildes, mode))) {
		if (!errno)
		    errno = ENOMEM;
		goto out;
	    }

	    if (!(ret = fopencookie(cookie, check, iobz))) {
		BZ2_bzclose(cookie);
		errno = EINVAL;
		goto out;
	    }
#ifndef LIBIO_IS_FIXED
	    if (ret->_fileno < 0)
		ret->_fileno = 0;
#endif
	}
	break;
    default:
	ret = fdopen(fildes, mode);
	break;
    }

out:
    if (check)
	free(check);
    return ret;
}
