/*P:demo1
NAME: A problem with singularity
\cindex{Neumann boundary condition}
\cindex{singularity}
\cindex{crack}
\subsection{Formulation}

   Let us now consider the Laplace operator in a non-convex
   domain. The domain is a disk with a missing part, thus
   there exists a re-entrant corner (see Figure, on the left).
   The angle associated to the re-entrant corner is denoted by 
   $\omega$ and whe suppose 
   $\pi < \omega \leq 2\pi$.
   We assume homogeneous Dirichlet conditions on the boundaries
   corresponding to the radius, and denoted by $\Gamma_0$.
   The rest of the boundary is related to the envelop
   boundary and denoted by $\Gamma_e$.

\begin{figure}[H]
    \centerline{\hspace{2cm}\includegraphics{crack-fig.pdf}}
    \caption{The domain of computation:
		(a) with a crack;
		(b) by using the $Ox_1$ symetry.}
\end{figure}
 
   The problem expresses:

   {\it find $u$ defined in $\Omega$ such that}
   $$
     \vbox{\halign{$#\ $&$\ #\ $&$\ #$\cr
       -\Delta u & = & 0 {\rm \ in \ } \Omega  \ \cr
       u         & = & 0 {\rm \ on \ } \Gamma_0 \cr
       u         & = & g {\rm \ on \ } \Gamma_e \cr
       }}
   $$ 
   where $g$ is expressed by the use of polar coordinates: 
   $$
       g (r,\theta) = r^\alpha {\rm sin}(\alpha \theta)
   $$
   and
   $$
       \alpha = \frac{\pi}{\omega}
   $$
   The parameter $\alpha$ is the intensity of
   the singularity at $x=0$ (see e.g.~\cite{grisvard-86}).
   This problem is convenient since the exact solution
   is known: $u = r^\alpha {\rm sin}(\alpha \theta)$.
   The problem for $\omega = 2\pi$ is related to a crack
   (see Figure, on the right) and the intensity is $\alpha=1/2$.
   Notes that there is a symmetry axis, and the domain
   can be reduced to $y>0$.
   The symmetry axis is associated to a homogeneous Neumann
   condition for $x<0$, and the corresponding boundary is 
   denoted by $\Gamma_s$.
   For the problem reduced to the half domain, the boundary 
   conditions changes from homogeneous Neumann to homogeneous Dirichlet,
   while the half domain is convex.

   The problem can be implemented as the example of the previous
   sections. In order to put Dirichlet condition on the 
   boundary domains \code{"crack"} and \code{"envelop"}, we
   block the related degrees of freedom:
   \begin{verbatim}
	Vh.block ("envelop");
        Vh.block ("crack");
   \end{verbatim}
   while degrees of freedom on the \code{"symmetry"} domain,
   associated to homogeneous Neumann condition, are unknown.
END:
*/

//<crack:
// usage: crach <geo> [-diff|-exact]
#include "rheolef/rheolef.h"
using namespace rheolef;
using namespace std;

Float alpha = 0.5; // the singularity at r=0

Float u_crack(const point& x)
{
    Float r = sqrt(x[0]*x[0] + x[1]*x[1]);
    if (1 + r == Float(1)) return 0;
    Float theta = atan2(x[1], x[0]);  // in ]-pi,pi]
    return pow(r,alpha)*sin(alpha*theta);
}
int main(int argc, char**argv)
{
    geo omega(argv[1]);
    space Vh (omega, argv[2]);
    Vh.block ("envelop");
    Vh.block ("crack");
    form a(Vh, Vh,"grad_grad");
    ssk<Float> fact = ldlt(a.uu);
    field u = interpolate(Vh, u_crack);
    field uh (Vh);
    uh.b = u.b;
    uh.u = fact.solve(- (a.ub*uh.b));
    field eh = u - uh;
    form m(Vh, Vh, "mass");
    cerr << "error_l2       " << sqrt(m(eh, eh))  << endl;
    cerr << "error_infinity " << eh.max_abs()  << endl;

    // H1 error
    string approx_grad = (strcmp(argv[2],"P1") == 0) ? "P0" : "P1d";
    space Th (omega, approx_grad);
    form d_dx0(Vh, Th, "d_dx0");
    form d_dx1(Vh, Th, "d_dx1");
    form inv_mt (Th, Th, "inv_mass");
    field deh_dx0 = inv_mt*(d_dx0*eh);
    field deh_dx1 = inv_mt*(d_dx1*eh);
    form mt (Th, Th, "mass");
    cerr << "error_h1       " << sqrt(mt(deh_dx0, deh_dx0)
				     +mt(deh_dx1, deh_dx1))
	 << endl;

    int digits10 = numeric_limits<Float>::digits10;
    cout << setprecision(digits10);
    if (argc == 2) {
        cout << uh;
    } else {
        if (strcmp(argv[2],"-exact") == 0) { cout << u; }
        else if (strcmp(argv[2],"-diff") == 0) { cout << u-uh; }
        else { cout << uh; }
    }
    return 0;
}
//>crack:

/*P:demo1
ANALYSIS:
  \begin{figure}[H]
    \begin{center}
      \begin{tabular}{cc}
        \includegraphics[scale=0.9]{crack-P1-uniform.pdf}
        &
        \includegraphics[scale=0.9]{crack-P2-uniform.pdf}
      \end{tabular}
    \end{center}
   \caption{Error analysis for a domain with a crack.}
    \label{fig-demo2-err}
  \end{figure}

   The figure plots the error in $L^2$ and $L^\infty$ norms
   for both $P_1$ (on the left) and $P_2$ (on the right) by using
   a familly of unifom meshes.
   First, remarks that the order of convergence is not improved by
   using $P_2$ elements. This case is different from the situation 
   when the solution was regular.
   More precisely, the following error estimates holds:
  $$
      \Vert u - u_h \Vert_{0,2,\Omega} \approx {\cal O}(h^{{\rm min}(\alpha+{1 \over 2},k+1)})  
  $$
  $$
      \Vert u - u_h \Vert_{0,\infty,\Omega} \approx O(h^{{\rm min}(\alpha,k+1)})  
  $$
  The finite element mesh adaptation is an efficient way
  to restaure the optimal convergence rate versus the mesh size, as we will
  see in the next paragraph.

END:
*/
