## @file
# Routines for generating Pcd Database
#
# Copyright (c) 2013, Intel Corporation. All rights reserved.<BR>
# This program and the accompanying materials
# are licensed and made available under the terms and conditions of the BSD License
# which accompanies this distribution.  The full text of the license may be found at
# http://opensource.org/licenses/bsd-license.php
#
# THE PROGRAM IS DISTRIBUTED UNDER THE BSD LICENSE ON AN "AS IS" BASIS,
# WITHOUT WARRANTIES OR REPRESENTATIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED.
#
from StringIO import StringIO
from Common.Misc import *
from Common.String import StringToArray
from struct import pack

DATABASE_VERSION = 4

gPcdDatabaseAutoGenC = TemplateString("""
//
// External PCD database debug information
//
#if 0
${PHASE}_PCD_DATABASE_INIT g${PHASE}PcdDbInit = {
${BEGIN}  { ${INIT_VALUE_UINT64} }, /*  ${INIT_CNAME_DECL_UINT64}_${INIT_GUID_DECL_UINT64}[${INIT_NUMSKUS_DECL_UINT64}] */
${END}
${BEGIN}  ${VARDEF_VALUE_UINT64}, /* ${VARDEF_CNAME_UINT64}_${VARDEF_GUID_UINT64}_VariableDefault_${VARDEF_SKUID_UINT64} */
${END}
${BEGIN}  { ${INIT_VALUE_UINT32} }, /*  ${INIT_CNAME_DECL_UINT32}_${INIT_GUID_DECL_UINT32}[${INIT_NUMSKUS_DECL_UINT32}] */
${END}
${BEGIN}  ${VARDEF_VALUE_UINT32}, /* ${VARDEF_CNAME_UINT32}_${VARDEF_GUID_UINT32}_VariableDefault_${VARDEF_SKUID_UINT32} */
${END}
  /* VPD */
${BEGIN}  { ${VPD_HEAD_VALUE} }, /* ${VPD_HEAD_CNAME_DECL}_${VPD_HEAD_GUID_DECL}[${VPD_HEAD_NUMSKUS_DECL}] */
${END}
  /* ExMapTable */
  {
${BEGIN}    { ${EXMAPPING_TABLE_EXTOKEN}, ${EXMAPPING_TABLE_LOCAL_TOKEN}, ${EXMAPPING_TABLE_GUID_INDEX} },
${END}
  },
  /* LocalTokenNumberTable */
  {
${BEGIN}    offsetof(${PHASE}_PCD_DATABASE, ${TOKEN_INIT}.${TOKEN_CNAME}_${TOKEN_GUID}${VARDEF_HEADER}) | ${TOKEN_TYPE},
${END}
  },
  /* GuidTable */
  {
${BEGIN}    ${GUID_STRUCTURE},
${END}
  },
${BEGIN}  { ${STRING_HEAD_VALUE} }, /* ${STRING_HEAD_CNAME_DECL}_${STRING_HEAD_GUID_DECL}[${STRING_HEAD_NUMSKUS_DECL}] */
${END}
${BEGIN}  /* ${VARIABLE_HEAD_CNAME_DECL}_${VARIABLE_HEAD_GUID_DECL}_Variable_Header[${VARIABLE_HEAD_NUMSKUS_DECL}] */
  {
    ${VARIABLE_HEAD_VALUE}
  },
${END}
/* SkuHead */
  {
  ${BEGIN} offsetof (${PHASE}_PCD_DATABASE, ${TOKEN_INIT}.${TOKEN_CNAME}_${TOKEN_GUID}${VARDEF_HEADER}) | ${TOKEN_TYPE}, /* */
           offsetof (${PHASE}_PCD_DATABASE, ${TOKEN_INIT}.SkuHead)  /* */
  ${END}
  },
 /* StringTable */
${BEGIN}  ${STRING_TABLE_VALUE}, /* ${STRING_TABLE_CNAME}_${STRING_TABLE_GUID} */
${END}
  /* SizeTable */
  {
${BEGIN}    ${SIZE_TABLE_MAXIMUM_LENGTH}, ${SIZE_TABLE_CURRENT_LENGTH}, /* ${SIZE_TABLE_CNAME}_${SIZE_TABLE_GUID} */
${END}
  },
${BEGIN}  { ${INIT_VALUE_UINT16} }, /*  ${INIT_CNAME_DECL_UINT16}_${INIT_GUID_DECL_UINT16}[${INIT_NUMSKUS_DECL_UINT16}] */
${END}
${BEGIN}  ${VARDEF_VALUE_UINT16}, /* ${VARDEF_CNAME_UINT16}_${VARDEF_GUID_UINT16}_VariableDefault_${VARDEF_SKUID_UINT16} */
${END}
${BEGIN}  { ${INIT_VALUE_UINT8} }, /*  ${INIT_CNAME_DECL_UINT8}_${INIT_GUID_DECL_UINT8}[${INIT_NUMSKUS_DECL_UINT8}] */
${END}
${BEGIN}  ${VARDEF_VALUE_UINT8}, /* ${VARDEF_CNAME_UINT8}_${VARDEF_GUID_UINT8}_VariableDefault_${VARDEF_SKUID_UINT8} */
${END}
${BEGIN}  { ${INIT_VALUE_BOOLEAN} }, /*  ${INIT_CNAME_DECL_BOOLEAN}_${INIT_GUID_DECL_BOOLEAN}[${INIT_NUMSKUS_DECL_BOOLEAN}] */
${END}
${BEGIN}  ${VARDEF_VALUE_BOOLEAN}, /* ${VARDEF_CNAME_BOOLEAN}_${VARDEF_GUID_BOOLEAN}_VariableDefault_${VARDEF_SKUID_BOOLEAN} */
${END}
  /* SkuIdTable */
  { ${BEGIN}${SKUID_VALUE}, ${END} },
  ${SYSTEM_SKU_ID_VALUE}
};
#endif
""")

## Mapping between PCD driver type and EFI phase
gPcdPhaseMap = {
    "PEI_PCD_DRIVER"    :   "PEI",
    "DXE_PCD_DRIVER"    :   "DXE"
}

gPcdDatabaseAutoGenH = TemplateString("""
#define PCD_${PHASE}_SERVICE_DRIVER_VERSION         ${SERVICE_DRIVER_VERSION}

//
// External PCD database debug information
//
#if 0
#define ${PHASE}_GUID_TABLE_SIZE                ${GUID_TABLE_SIZE}
#define ${PHASE}_STRING_TABLE_SIZE              ${STRING_TABLE_SIZE}
#define ${PHASE}_SKUID_TABLE_SIZE               ${SKUID_TABLE_SIZE}
#define ${PHASE}_LOCAL_TOKEN_NUMBER_TABLE_SIZE  ${LOCAL_TOKEN_NUMBER_TABLE_SIZE}
#define ${PHASE}_LOCAL_TOKEN_NUMBER             ${LOCAL_TOKEN_NUMBER}
#define ${PHASE}_EXMAPPING_TABLE_SIZE           ${EXMAPPING_TABLE_SIZE}
#define ${PHASE}_EX_TOKEN_NUMBER                ${EX_TOKEN_NUMBER}
#define ${PHASE}_SIZE_TABLE_SIZE                ${SIZE_TABLE_SIZE}
#define ${PHASE}_SKU_HEAD_SIZE                  ${SKU_HEAD_SIZE}
#define ${PHASE}_GUID_TABLE_EMPTY               ${GUID_TABLE_EMPTY}
#define ${PHASE}_STRING_TABLE_EMPTY             ${STRING_TABLE_EMPTY}
#define ${PHASE}_SKUID_TABLE_EMPTY              ${SKUID_TABLE_EMPTY}
#define ${PHASE}_DATABASE_EMPTY                 ${DATABASE_EMPTY}
#define ${PHASE}_EXMAP_TABLE_EMPTY              ${EXMAP_TABLE_EMPTY}

typedef struct {
${BEGIN}  UINT64             ${INIT_CNAME_DECL_UINT64}_${INIT_GUID_DECL_UINT64}[${INIT_NUMSKUS_DECL_UINT64}];
${END}
${BEGIN}  UINT64             ${VARDEF_CNAME_UINT64}_${VARDEF_GUID_UINT64}_VariableDefault_${VARDEF_SKUID_UINT64};
${END}
${BEGIN}  UINT32             ${INIT_CNAME_DECL_UINT32}_${INIT_GUID_DECL_UINT32}[${INIT_NUMSKUS_DECL_UINT32}];
${END}
${BEGIN}  UINT32             ${VARDEF_CNAME_UINT32}_${VARDEF_GUID_UINT32}_VariableDefault_${VARDEF_SKUID_UINT32};
${END}
${BEGIN}  VPD_HEAD           ${VPD_HEAD_CNAME_DECL}_${VPD_HEAD_GUID_DECL}[${VPD_HEAD_NUMSKUS_DECL}];
${END}
  DYNAMICEX_MAPPING  ExMapTable[${PHASE}_EXMAPPING_TABLE_SIZE];
  UINT32             LocalTokenNumberTable[${PHASE}_LOCAL_TOKEN_NUMBER_TABLE_SIZE];
  GUID               GuidTable[${PHASE}_GUID_TABLE_SIZE];
${BEGIN}  STRING_HEAD        ${STRING_HEAD_CNAME_DECL}_${STRING_HEAD_GUID_DECL}[${STRING_HEAD_NUMSKUS_DECL}];
${END}
${BEGIN}  VARIABLE_HEAD      ${VARIABLE_HEAD_CNAME_DECL}_${VARIABLE_HEAD_GUID_DECL}_Variable_Header[${VARIABLE_HEAD_NUMSKUS_DECL}];
${END}
${BEGIN}  SKU_HEAD           SkuHead[${PHASE}_SKU_HEAD_SIZE];
${END}
${BEGIN}  UINT8              StringTable${STRING_TABLE_INDEX}[${STRING_TABLE_LENGTH}]; /* ${STRING_TABLE_CNAME}_${STRING_TABLE_GUID} */
${END}
  SIZE_INFO          SizeTable[${PHASE}_SIZE_TABLE_SIZE];
${BEGIN}  UINT16             ${INIT_CNAME_DECL_UINT16}_${INIT_GUID_DECL_UINT16}[${INIT_NUMSKUS_DECL_UINT16}];
${END}
${BEGIN}  UINT16             ${VARDEF_CNAME_UINT16}_${VARDEF_GUID_UINT16}_VariableDefault_${VARDEF_SKUID_UINT16};
${END}
${BEGIN}  UINT8              ${INIT_CNAME_DECL_UINT8}_${INIT_GUID_DECL_UINT8}[${INIT_NUMSKUS_DECL_UINT8}];
${END}
${BEGIN}  UINT8              ${VARDEF_CNAME_UINT8}_${VARDEF_GUID_UINT8}_VariableDefault_${VARDEF_SKUID_UINT8};
${END}
${BEGIN}  BOOLEAN            ${INIT_CNAME_DECL_BOOLEAN}_${INIT_GUID_DECL_BOOLEAN}[${INIT_NUMSKUS_DECL_BOOLEAN}];
${END}
${BEGIN}  BOOLEAN            ${VARDEF_CNAME_BOOLEAN}_${VARDEF_GUID_BOOLEAN}_VariableDefault_${VARDEF_SKUID_BOOLEAN};
${END}
  UINT8              SkuIdTable[${PHASE}_SKUID_TABLE_SIZE];
${SYSTEM_SKU_ID}
} ${PHASE}_PCD_DATABASE_INIT;

typedef struct {
${PCD_DATABASE_UNINIT_EMPTY}
${BEGIN}  UINT64   ${UNINIT_CNAME_DECL_UINT64}_${UNINIT_GUID_DECL_UINT64}[${UNINIT_NUMSKUS_DECL_UINT64}];
${END}
${BEGIN}  UINT32   ${UNINIT_CNAME_DECL_UINT32}_${UNINIT_GUID_DECL_UINT32}[${UNINIT_NUMSKUS_DECL_UINT32}];
${END}
${BEGIN}  UINT16   ${UNINIT_CNAME_DECL_UINT16}_${UNINIT_GUID_DECL_UINT16}[${UNINIT_NUMSKUS_DECL_UINT16}];
${END}
${BEGIN}  UINT8    ${UNINIT_CNAME_DECL_UINT8}_${UNINIT_GUID_DECL_UINT8}[${UNINIT_NUMSKUS_DECL_UINT8}];
${END}
${BEGIN}  BOOLEAN  ${UNINIT_CNAME_DECL_BOOLEAN}_${UNINIT_GUID_DECL_BOOLEAN}[${UNINIT_NUMSKUS_DECL_BOOLEAN}];
${END}
} ${PHASE}_PCD_DATABASE_UNINIT;

typedef struct {
  //GUID                  Signature;  // PcdDataBaseGuid
  //UINT32                Length;
  //UINT32                UninitDataBaseSize;// Total size for PCD those default value with 0.
  //TABLE_OFFSET          LocalTokenNumberTableOffset;
  //TABLE_OFFSET          ExMapTableOffset;
  //TABLE_OFFSET          GuidTableOffset;
  //TABLE_OFFSET          StringTableOffset;
  //TABLE_OFFSET          SizeTableOffset;
  //TABLE_OFFSET          SkuIdTableOffset;
  //UINT16                LocalTokenCount;  // LOCAL_TOKEN_NUMBER for all
  //UINT16                ExTokenCount;     // EX_TOKEN_NUMBER for DynamicEx
  //UINT16                GuidTableCount;   // The Number of Guid in GuidTable
  //SKU_ID                SystemSkuId;      // Current SkuId value.
  //UINT8                 Pad;
  ${PHASE}_PCD_DATABASE_INIT    Init;
  ${PHASE}_PCD_DATABASE_UNINIT  Uninit;
} ${PHASE}_PCD_DATABASE;

#define ${PHASE}_NEX_TOKEN_NUMBER (${PHASE}_LOCAL_TOKEN_NUMBER - ${PHASE}_EX_TOKEN_NUMBER)
#endif
""")


gEmptyPcdDatabaseAutoGenC = TemplateString("""
//
// External PCD database debug information
//
#if 0
${PHASE}_PCD_DATABASE_INIT g${PHASE}PcdDbInit = {
  /* ExMapTable */
  {
    {0, 0, 0}
  },
  /* LocalTokenNumberTable */
  {
    0
  },
  /* GuidTable */
  {
    {0x00000000, 0x0000, 0x0000, {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00}}
  },
  /* StringTable */
  { 0 },
  /* SkuHead */
  {
    0, 0
  },
  /* SizeTable */
  {
    0, 0
  },
  /* SkuIdTable */
  { 0 },
  ${SYSTEM_SKU_ID_VALUE}
};
#endif
""")

## PackGuid
#
# Pack the GUID value in C structure format into data array
#
# @param GuidStructureValue:   The GUID value in C structure format
#
# @retval Buffer:  a data array contains the Guid
#
def PackGuid(GuidStructureValue):
    GuidString = GuidStructureStringToGuidString(GuidStructureValue)
    Guid = GuidString.split('-')
    Buffer = pack('=LHHBBBBBBBB',
                int(Guid[0], 16),
                int(Guid[1], 16),
                int(Guid[2], 16),
                int(Guid[3][-4:-2], 16),
                int(Guid[3][-2:], 16),
                int(Guid[4][-12:-10], 16),
                int(Guid[4][-10:-8], 16),
                int(Guid[4][-8:-6], 16),
                int(Guid[4][-6:-4], 16),
                int(Guid[4][-4:-2], 16),
                int(Guid[4][-2:], 16)
                )
    return Buffer

def toHex(s):
    lst = []
    for ch in s:
        hv = hex(ord(ch)).replace('0x', ' ')
        if len(hv) == 1:
            hv = '0'+hv
        lst.append(hv)
    if lst:
        return reduce(lambda x,y:x+y, lst)
    else:
        return 'empty'
## DbItemList
#
#  The class holds the Pcd database items. ItemSize if not zero should match the item datum type in the C structure.
#  When the structure is changed, remember to check the ItemSize and the related  PackStr in PackData()
#  RawDataList is the RawData that may need some kind of calculation or transformation,
#  the DataList corresponds to the data that need to be written to database. If DataList is not present, then RawDataList
#  will be written to the database.
#
class DbItemList:
    def __init__(self, ItemSize, DataList=None, RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        self.ItemSize = ItemSize
        self.DataList = DataList
        self.RawDataList = RawDataList
        self.ListSize = 0

    def GetInterOffset(self, Index):
        Offset = 0
        if self.ItemSize == 0:
            #
            # Variable length, need to calculate one by one
            #
            assert(Index < len(self.RawDataList))
            for ItemIndex in xrange(Index):
                Offset += len(self.RawDataList[ItemIndex])
        else:
            for Datas in self.RawDataList:
                Offset = self.ItemSize * Index

        return Offset

    def GetListSize(self):
        if self.ListSize:
            return self.ListSize
        if len(self.RawDataList) == 0:
            self.ListSize = 0
            return self.ListSize
        if self.ItemSize == 0:
            self.ListSize = self.GetInterOffset(len(self.RawDataList) - 1) + len(self.RawDataList[len(self.RawDataList)-1])
        else:
            self.ListSize = self.ItemSize * len(self.RawDataList)
        return self.ListSize

    def PackData(self):
        if self.ItemSize == 8:
            PackStr = "=Q"
        elif self.ItemSize == 4:
            PackStr = "=L"
        elif self.ItemSize == 2:
            PackStr = "=H"
        elif self.ItemSize == 1:
            PackStr = "=B"
        elif self.ItemSize == 0:
            PackStr = "=B"
        elif self.ItemSize == 16:
            # pack Guid
            PackStr = ''
        else:
            # should not reach here
            assert(False)

        Buffer = ''
        for Datas in self.RawDataList:
            if type(Datas) in (list, tuple):
                for Data in Datas:
                    if PackStr:
                        Buffer += pack(PackStr, GetIntegerValue(Data))
                    else:
                        Buffer += PackGuid(Data)
            else:
                if PackStr:
                    Buffer += pack(PackStr, GetIntegerValue(Datas))
                else:
                    Buffer += PackGuid(Datas)

        return Buffer

## DbExMapTblItemList
#
#  The class holds the ExMap table
#
class DbExMapTblItemList (DbItemList):
    def __init__(self, ItemSize, DataList=None, RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        DbItemList.__init__(self, ItemSize, DataList, RawDataList)
    def PackData(self):
        Buffer = ''
        PackStr = "=LHH"
        for Datas in self.RawDataList:
            Buffer += pack(PackStr,
                           GetIntegerValue(Datas[0]),
                           GetIntegerValue(Datas[1]),
                           GetIntegerValue(Datas[2]))
        return Buffer

## DbComItemList
#
# The DbComItemList is a special kind of DbItemList in case that the size of the List can not be computed by the
# ItemSize multiply the ItemCount.
#
class DbComItemList (DbItemList):
    def __init__(self, ItemSize, DataList=None, RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        DbItemList.__init__(self, ItemSize, DataList, RawDataList)
    def GetInterOffset(self, Index):
        Offset = 0
        if self.ItemSize == 0:
            #
            # Variable length, need to calculte one by one
            # The only variable table is stringtable, it is not Composite item, should not reach here
            #
            assert(False)
        else:
            assert(Index < len(self.RawDataList))
            for ItemIndex in xrange(Index):
                Offset += len(self.RawDataList[ItemIndex]) * self.ItemSize

        return Offset

    def GetListSize(self):
        if self.ListSize:
            return self.ListSize
        if self.ItemSize == 0:
            assert(False)
        else:
            if len(self.RawDataList) == 0:
                self.ListSize = 0
            else:
                self.ListSize = self.GetInterOffset(len(self.RawDataList) - 1) + len(self.RawDataList[len(self.RawDataList)-1]) * self.ItemSize

        return self.ListSize

    def PackData(self):
        if self.ItemSize == 8:
            PackStr = "=Q"
        elif self.ItemSize == 4:
            PackStr = "=L"
        elif self.ItemSize == 2:
            PackStr = "=H"
        elif self.ItemSize == 1:
            PackStr = "=B"
        elif self.ItemSize == 0:
            PackStr = "=B"
        else:
            assert(False)

        Buffer = ''
        for DataList in self.RawDataList:
            for Data in DataList:
                if type(Data) in (list, tuple):
                    for SingleData in Data:
                        Buffer += pack(PackStr, GetIntegerValue(SingleData))
                else:
                    Buffer += pack(PackStr, GetIntegerValue(Data))

        return Buffer

## DbVariableTableItemList
#
#  The class holds the Variable header value table
#
class DbVariableTableItemList (DbComItemList):
    def __init__(self, ItemSize, DataList=None, RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        DbComItemList.__init__(self, ItemSize, DataList, RawDataList)
    def PackData(self):
        PackStr = "=LLHH"
        Buffer = ''
        for DataList in self.RawDataList:
            for Data in DataList:
                Buffer += pack(PackStr,
                               GetIntegerValue(Data[0]),
                               GetIntegerValue(Data[1]),
                               GetIntegerValue(Data[2]),
                               GetIntegerValue(Data[3]))
        return Buffer

class DbStringHeadTableItemList(DbItemList):
    def __init__(self,ItemSize,DataList=None,RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        DbItemList.__init__(self, ItemSize, DataList, RawDataList)

    def GetInterOffset(self, Index):
        Offset = 0
        if self.ItemSize == 0:
            #
            # Variable length, need to calculate one by one
            #
            assert(Index < len(self.RawDataList))
            for ItemIndex in xrange(Index):
                Offset += len(self.RawDataList[ItemIndex])
        else:
            for innerIndex in range(Index):
                if type(self.RawDataList[innerIndex]) in (list, tuple):
                    Offset += len(self.RawDataList[innerIndex]) * self.ItemSize
                else:
                    Offset += self.ItemSize

        return Offset

    def GetListSize(self):
        if self.ListSize:
            return self.ListSize
        if len(self.RawDataList) == 0:
            self.ListSize = 0
            return self.ListSize
        if self.ItemSize == 0:
            self.ListSize = self.GetInterOffset(len(self.RawDataList) - 1) + len(self.RawDataList[len(self.RawDataList)-1])
        else:
            for Datas in self.RawDataList:
                if type(Datas) in (list, tuple):
                    self.ListSize += len(Datas) * self.ItemSize
                else:
                    self.ListSize += self.ItemSize
        return self.ListSize

## DbSkuHeadTableItemList
#
#  The class holds the Sku header value table
#
class DbSkuHeadTableItemList (DbItemList):
    def __init__(self, ItemSize, DataList=None, RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        DbItemList.__init__(self, ItemSize, DataList, RawDataList)
    def PackData(self):
        PackStr = "=LL"
        Buffer = ''
        for Data in self.RawDataList:
            Buffer += pack(PackStr,
                           GetIntegerValue(Data[0]),
                           GetIntegerValue(Data[1]))
        return Buffer

## DbSizeTableItemList
#
#  The class holds the size table
#
class DbSizeTableItemList (DbItemList):
    def __init__(self, ItemSize, DataList=None, RawDataList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        DbItemList.__init__(self, ItemSize, DataList, RawDataList)
    def GetListSize(self):
        length = 0
        for Data in self.RawDataList:
            length += (1 + len(Data[1]))
        return length * self.ItemSize
    def PackData(self):
        PackStr = "=H"
        Buffer = ''
        for Data in self.RawDataList:
            Buffer += pack(PackStr,
                           GetIntegerValue(Data[0]))
            for subData in Data[1]:
                Buffer += pack(PackStr,
                           GetIntegerValue(subData))
        return Buffer

## DbStringItemList
#
#  The class holds the string table
#
class DbStringItemList (DbComItemList):
    def __init__(self, ItemSize, DataList=None, RawDataList=None, LenList=None):
        if DataList is None:
            DataList = []
        if RawDataList is None:
            RawDataList = []
        if LenList is None:
            LenList = []

        assert(len(RawDataList) == len(LenList))
        DataList = []
        # adjust DataList according to the LenList
        for Index in xrange(len(RawDataList)):
            Len = LenList[Index]
            RawDatas = RawDataList[Index]
            assert(Len >= len(RawDatas))
            ActualDatas = []
            for i in xrange(len(RawDatas)):
                ActualDatas.append(RawDatas[i])
            for i in xrange(len(RawDatas), Len):
                ActualDatas.append(0)
            DataList.append(ActualDatas)
        self.LenList = LenList
        DbComItemList.__init__(self, ItemSize, DataList, RawDataList)
    def GetInterOffset(self, Index):
        Offset = 0

        assert(Index < len(self.LenList))
        for ItemIndex in xrange(Index):
            Offset += self.LenList[ItemIndex]

        return Offset

    def GetListSize(self):
        if self.ListSize:
            return self.ListSize

        if len(self.LenList) == 0:
            self.ListSize = 0
        else:
            self.ListSize = self.GetInterOffset(len(self.LenList) - 1) + self.LenList[len(self.LenList)-1]

        return self.ListSize

    def PackData(self):
        self.RawDataList = self.DataList
        return DbComItemList.PackData(self)



##  Find the index in two list where the item matches the key separately
#
#   @param      Key1   The key used to search the List1
#   @param      List1  The list that Key1 will be searched
#   @param      Key2   The key used to search the List2
#   @param      List2  The list that Key2 will be searched
#
#   @retval     Index  The position inside the list where list1[Index] == Key1 and list2[Index] == Key2
#
def GetMatchedIndex(Key1, List1, Key2, List2):
    StartPos = 0
    while StartPos < len(List1):
        Index = List1.index(Key1, StartPos)
        if List2[Index] == Key2:
            return Index
        else:
            StartPos = Index + 1

    return -1


##  Get the integer value from string like "14U" or integer like 2
#
#   @param      Input   The object that may be either a integer value or a string
#
#   @retval     Value    The integer value that the input represents
#
def GetIntegerValue(Input):
    if type(Input) in (int, long):
        return Input
    String = Input
    if String.endswith("U"):
        String = String[:-1]
    if String.endswith("ULL"):
        String = String[:-3]
    if String.endswith("LL"):
        String = String[:-2]

    if String.startswith("0x") or String.startswith("0X"):
        return int(String, 16)
    elif String == '':
        return 0
    else:
        return int(String)


## convert StringArray like {0x36, 0x00, 0x34, 0x00, 0x21, 0x00, 0x36, 0x00, 0x34, 0x00, 0x00, 0x00}
# to List like [0x36, 0x00, 0x34, 0x00, 0x21, 0x00, 0x36, 0x00, 0x34, 0x00, 0x00, 0x00]
#
#   @param      StringArray A string array like {0x36, 0x00, 0x34, 0x00, 0x21, 0x00, 0x36, 0x00, 0x34, 0x00, 0x00, 0x00}
#
#   @retval                 A list object of integer items
#
def StringArrayToList(StringArray):
    StringArray = StringArray[1:-1]
    StringArray = '[' + StringArray + ']'
    return eval(StringArray)


## Convert TokenType String like  "PCD_DATUM_TYPE_UINT32 | PCD_TYPE_HII" to TokenType value
#
#   @param      TokenType  A TokenType string like "PCD_DATUM_TYPE_UINT32 | PCD_TYPE_HII"
#
#   @retval                A integer representation of the TokenType
#
def GetTokenTypeValue(TokenType):
    TokenTypeDict = {
        "PCD_TYPE_SHIFT":28,
        "PCD_TYPE_DATA":(0x0 << 28),
        "PCD_TYPE_HII":(0x8 << 28),
        "PCD_TYPE_VPD":(0x4 << 28),
        "PCD_TYPE_SKU_ENABLED":(0x2 << 28),
        "PCD_TYPE_STRING":(0x1 << 28),

        "PCD_DATUM_TYPE_SHIFT":24,
        "PCD_DATUM_TYPE_POINTER":(0x0 << 24),
        "PCD_DATUM_TYPE_UINT8":(0x1 << 24),
        "PCD_DATUM_TYPE_UINT16":(0x2 << 24),
        "PCD_DATUM_TYPE_UINT32":(0x4 << 24),
        "PCD_DATUM_TYPE_UINT64":(0x8 << 24),

        "PCD_DATUM_TYPE_SHIFT2":20,
        "PCD_DATUM_TYPE_UINT8_BOOLEAN":(0x1 << 20 | 0x1 << 24),
        }
    return eval(TokenType, TokenTypeDict)

## construct the external Pcd database using data from Dict
#
#   @param      Dict  A dictionary contains Pcd related tables
#
#   @retval     Buffer A byte stream of the Pcd database
#
def BuildExDataBase(Dict):
    # init Db items
    InitValueUint64 = Dict['INIT_DB_VALUE_UINT64']
    DbInitValueUint64 = DbComItemList(8, RawDataList = InitValueUint64)
    VardefValueUint64 = Dict['VARDEF_DB_VALUE_UINT64']
    DbVardefValueUint64 = DbItemList(8, RawDataList = VardefValueUint64)
    InitValueUint32 = Dict['INIT_DB_VALUE_UINT32']
    DbInitValueUint32 = DbComItemList(4, RawDataList = InitValueUint32)
    VardefValueUint32 = Dict['VARDEF_DB_VALUE_UINT32']
    DbVardefValueUint32 = DbItemList(4, RawDataList = VardefValueUint32)
    VpdHeadValue = Dict['VPD_DB_VALUE']
    DbVpdHeadValue = DbComItemList(4, RawDataList = VpdHeadValue)
    ExMapTable = zip(Dict['EXMAPPING_TABLE_EXTOKEN'], Dict['EXMAPPING_TABLE_LOCAL_TOKEN'], Dict['EXMAPPING_TABLE_GUID_INDEX'])
    DbExMapTable = DbExMapTblItemList(8, RawDataList = ExMapTable)
    LocalTokenNumberTable = Dict['LOCAL_TOKEN_NUMBER_DB_VALUE']
    DbLocalTokenNumberTable = DbItemList(4, RawDataList = LocalTokenNumberTable)
    GuidTable = Dict['GUID_STRUCTURE']
    DbGuidTable = DbItemList(16, RawDataList = GuidTable)
    StringHeadValue = Dict['STRING_DB_VALUE']
    # DbItemList to DbStringHeadTableItemList
    DbStringHeadValue = DbStringHeadTableItemList(4, RawDataList = StringHeadValue)
    VariableTable = Dict['VARIABLE_DB_VALUE']
    DbVariableTable = DbVariableTableItemList(12, RawDataList = VariableTable)
    NumberOfSkuEnabledPcd = GetIntegerValue(Dict['SKU_HEAD_SIZE'])
    Dict['SKUHEAD_TABLE_VALUE'] = [(0,0) for i in xrange(NumberOfSkuEnabledPcd)]
    SkuTable = Dict['SKUHEAD_TABLE_VALUE']  # Generated later
    DbSkuTable = DbSkuHeadTableItemList(8, RawDataList = SkuTable)
    Dict['STRING_TABLE_DB_VALUE'] = [StringArrayToList(x) for x in Dict['STRING_TABLE_VALUE']]

    StringTableValue = Dict['STRING_TABLE_DB_VALUE']
    # when calcute the offset, should use StringTableLen instead of StringTableValue, as string maxium len may be different with actual len
    StringTableLen = Dict['STRING_TABLE_LENGTH']
    DbStringTableLen = DbStringItemList(0, RawDataList = StringTableValue, LenList = StringTableLen)


    PcdTokenTable = Dict['PCD_TOKENSPACE']
    PcdTokenLen = Dict['PCD_TOKENSPACE_LENGTH']
    PcdTokenTableValue = [StringArrayToList(x) for x in Dict['PCD_TOKENSPACE']]
    DbPcdTokenTable = DbStringItemList(0, RawDataList = PcdTokenTableValue, LenList = PcdTokenLen)

    PcdCNameTable = Dict['PCD_CNAME']
    PcdCNameLen = Dict['PCD_CNAME_LENGTH']
    PcdCNameTableValue = [StringArrayToList(x) for x in Dict['PCD_CNAME']]
    DbPcdCNameTable = DbStringItemList(0, RawDataList = PcdCNameTableValue, LenList = PcdCNameLen)

    PcdNameOffsetTable = Dict['PCD_NAME_OFFSET']
    DbPcdNameOffsetTable = DbItemList(4,RawDataList = PcdNameOffsetTable)

    SizeTableValue = zip(Dict['SIZE_TABLE_MAXIMUM_LENGTH'], Dict['SIZE_TABLE_CURRENT_LENGTH'])
    DbSizeTableValue = DbSizeTableItemList(2, RawDataList = SizeTableValue)
    InitValueUint16 = Dict['INIT_DB_VALUE_UINT16']
    DbInitValueUint16 = DbComItemList(2, RawDataList = InitValueUint16)
    VardefValueUint16 = Dict['VARDEF_DB_VALUE_UINT16']
    DbVardefValueUint16 = DbItemList(2, RawDataList = VardefValueUint16)
    InitValueUint8 = Dict['INIT_DB_VALUE_UINT8']
    DbInitValueUint8 = DbComItemList(1, RawDataList = InitValueUint8)
    VardefValueUint8 = Dict['VARDEF_DB_VALUE_UINT8']
    DbVardefValueUint8 = DbItemList(1, RawDataList = VardefValueUint8)
    InitValueBoolean = Dict['INIT_DB_VALUE_BOOLEAN']
    DbInitValueBoolean = DbComItemList(1, RawDataList = InitValueBoolean)
    VardefValueBoolean = Dict['VARDEF_DB_VALUE_BOOLEAN']
    DbVardefValueBoolean = DbItemList(1, RawDataList = VardefValueBoolean)
    SkuidValue = Dict['SKUID_VALUE']
    DbSkuidValue = DbItemList(1, RawDataList = SkuidValue)
    SkuIndexValue = Dict['SKU_INDEX_VALUE']
    DbSkuIndexValue = DbItemList(0,RawDataList = SkuIndexValue)

    # Unit Db Items
    UnInitValueUint64 = Dict['UNINIT_GUID_DECL_UINT64']
    DbUnInitValueUint64 = DbItemList(8, RawDataList = UnInitValueUint64)
    UnInitValueUint32 = Dict['UNINIT_GUID_DECL_UINT32']
    DbUnInitValueUint32 = DbItemList(4, RawDataList = UnInitValueUint32)
    UnInitValueUint16 = Dict['UNINIT_GUID_DECL_UINT16']
    DbUnInitValueUint16 = DbItemList(2, RawDataList = UnInitValueUint16)
    UnInitValueUint8 = Dict['UNINIT_GUID_DECL_UINT8']
    DbUnInitValueUint8 = DbItemList(1, RawDataList = UnInitValueUint8)
    UnInitValueBoolean = Dict['UNINIT_GUID_DECL_BOOLEAN']
    DbUnInitValueBoolean = DbItemList(1, RawDataList = UnInitValueBoolean)
    PcdTokenNumberMap = Dict['PCD_ORDER_TOKEN_NUMBER_MAP']

    DbNameTotle = ["InitValueUint64", "VardefValueUint64", "InitValueUint32", "VardefValueUint32", "VpdHeadValue", "ExMapTable",
               "LocalTokenNumberTable", "GuidTable", "StringHeadValue",  "PcdNameOffsetTable","VariableTable","SkuTable", "StringTableLen", "PcdTokenTable", "PcdCNameTable",
               "SizeTableValue", "InitValueUint16", "VardefValueUint16", "InitValueUint8", "VardefValueUint8", "InitValueBoolean",
               "VardefValueBoolean", "SkuidValue", "SkuIndexValue","UnInitValueUint64", "UnInitValueUint32", "UnInitValueUint16", "UnInitValueUint8", "UnInitValueBoolean"]

    DbTotal = [InitValueUint64, VardefValueUint64, InitValueUint32, VardefValueUint32, VpdHeadValue, ExMapTable,
               LocalTokenNumberTable, GuidTable, StringHeadValue,  PcdNameOffsetTable,VariableTable,SkuTable, StringTableLen, PcdTokenTable,PcdCNameTable,
               SizeTableValue, InitValueUint16, VardefValueUint16,InitValueUint8, VardefValueUint8, InitValueBoolean,
               VardefValueBoolean, SkuidValue, SkuIndexValue, UnInitValueUint64, UnInitValueUint32, UnInitValueUint16, UnInitValueUint8, UnInitValueBoolean]
    DbItemTotal = [DbInitValueUint64, DbVardefValueUint64, DbInitValueUint32, DbVardefValueUint32, DbVpdHeadValue, DbExMapTable,
               DbLocalTokenNumberTable, DbGuidTable, DbStringHeadValue,  DbPcdNameOffsetTable,DbVariableTable,DbSkuTable, DbStringTableLen, DbPcdTokenTable, DbPcdCNameTable,
               DbSizeTableValue, DbInitValueUint16, DbVardefValueUint16,DbInitValueUint8, DbVardefValueUint8, DbInitValueBoolean,
               DbVardefValueBoolean, DbSkuidValue, DbSkuIndexValue, DbUnInitValueUint64, DbUnInitValueUint32, DbUnInitValueUint16, DbUnInitValueUint8, DbUnInitValueBoolean]

    # SkuidValue is the last table in the init table items
    InitTableNum = DbTotal.index(SkuidValue) + 1 + 1 # +1 is for SkuIndexValue table
    # The FixedHeader length of the PCD_DATABASE_INIT, from Signature to Pad
    FixedHeaderLen = 64

    # Get offset of SkuId table in the database
    SkuIdTableOffset = FixedHeaderLen
    for DbIndex in xrange(len(DbTotal)):
        if DbTotal[DbIndex] is SkuidValue:
            break
        SkuIdTableOffset += DbItemTotal[DbIndex].GetListSize()


    # Get offset of SkuValue table in the database
    SkuTableOffset = FixedHeaderLen
    for DbIndex in xrange(len(DbTotal)):
        if DbTotal[DbIndex] is SkuTable:
            break
        SkuTableOffset += DbItemTotal[DbIndex].GetListSize()
    PcdTokenTableDbOffset = FixedHeaderLen
    for DbIndex in xrange(len(DbTotal)):
        if DbTotal[DbIndex] is PcdTokenTable:
            break
        PcdTokenTableDbOffset += DbItemTotal[DbIndex].GetListSize()

    PcdCNameTableDbOffset = FixedHeaderLen
    for DbIndex in xrange(len(DbTotal)):
        if DbTotal[DbIndex] is PcdCNameTable:
            break
        PcdCNameTableDbOffset += DbItemTotal[DbIndex].GetListSize()
    # Fix up the LocalTokenNumberTable, SkuHeader table
    SkuHeaderIndex = 0
    if len(Dict['SKU_INDEX_VALUE']) > 0:
        SkuIndexIndexTable = [(0) for i in xrange(len(Dict['SKU_INDEX_VALUE']))]
        SkuIndexIndexTable[0] = 0  #Dict['SKU_INDEX_VALUE'][0][0]
        for i in range(1,len(Dict['SKU_INDEX_VALUE'])):
            SkuIndexIndexTable[i] = SkuIndexIndexTable[i-1]+Dict['SKU_INDEX_VALUE'][i-1][0] + 1
    for (LocalTokenNumberTableIndex, (Offset, Table)) in enumerate(LocalTokenNumberTable):
        DbIndex = 0
        DbOffset = FixedHeaderLen
        for DbIndex in xrange(len(DbTotal)):
            if DbTotal[DbIndex] is Table:
                DbOffset += DbItemTotal[DbIndex].GetInterOffset(Offset)
                break
            DbOffset += DbItemTotal[DbIndex].GetListSize()
        else:
            assert(False)

        TokenTypeValue = Dict['TOKEN_TYPE'][LocalTokenNumberTableIndex]
        TokenTypeValue = GetTokenTypeValue(TokenTypeValue)
        LocalTokenNumberTable[LocalTokenNumberTableIndex] = DbOffset|int(TokenTypeValue)
        # if PCD_TYPE_SKU_ENABLED, then we need to fix up the SkuTable

        SkuIndexTabalOffset = SkuIdTableOffset + Dict['SKUID_VALUE'][0] + 1
        if (TokenTypeValue & (0x2 << 28)):
            SkuTable[SkuHeaderIndex] = (DbOffset|int(TokenTypeValue & ~(0x2<<28)), SkuIndexTabalOffset + SkuIndexIndexTable[PcdTokenNumberMap[LocalTokenNumberTableIndex]])
            LocalTokenNumberTable[LocalTokenNumberTableIndex] = (SkuTableOffset + SkuHeaderIndex * 8) | int(TokenTypeValue)
            SkuHeaderIndex += 1


    if SkuHeaderIndex == 0:
        SkuHeaderIndex = 1
    assert(SkuHeaderIndex == NumberOfSkuEnabledPcd)

    # resolve variable table offset
    for VariableEntries in VariableTable:
        skuindex = 0
        for VariableEntryPerSku in VariableEntries:
            (VariableHeadGuidIndex, VariableHeadStringIndex, SKUVariableOffset, VariableOffset, VariableRefTable) = VariableEntryPerSku[:]
            DbIndex = 0
            DbOffset = FixedHeaderLen
            for DbIndex in xrange(len(DbTotal)):
                if DbTotal[DbIndex] is VariableRefTable:
                    DbOffset += DbItemTotal[DbIndex].GetInterOffset(VariableOffset)
                    break
                DbOffset += DbItemTotal[DbIndex].GetListSize()
            else:
                assert(False)
            if isinstance(VariableRefTable[0],list):
                DbOffset += skuindex * 4
            skuindex += 1
            if DbIndex >= InitTableNum:
                assert(False)

            VariableEntryPerSku[:] = (VariableHeadStringIndex, DbOffset, VariableHeadGuidIndex, SKUVariableOffset)

    # calculate various table offset now
    DbTotalLength = FixedHeaderLen
    for DbIndex in xrange(len(DbItemTotal)):
        if DbItemTotal[DbIndex] is DbLocalTokenNumberTable:
            LocalTokenNumberTableOffset = DbTotalLength
        elif DbItemTotal[DbIndex] is DbExMapTable:
            ExMapTableOffset = DbTotalLength
        elif DbItemTotal[DbIndex] is DbGuidTable:
            GuidTableOffset = DbTotalLength
        elif DbItemTotal[DbIndex] is DbStringTableLen:
            StringTableOffset = DbTotalLength
        elif DbItemTotal[DbIndex] is DbSizeTableValue:
            SizeTableOffset = DbTotalLength
        elif DbItemTotal[DbIndex] is DbSkuidValue:
            SkuIdTableOffset = DbTotalLength
        elif DbItemTotal[DbIndex] is DbPcdNameOffsetTable:
            DbPcdNameOffset = DbTotalLength

        DbTotalLength += DbItemTotal[DbIndex].GetListSize()
    if not Dict['PCD_INFO_FLAG']:
        DbPcdNameOffset  = 0
    LocalTokenCount = GetIntegerValue(Dict['LOCAL_TOKEN_NUMBER'])
    ExTokenCount = GetIntegerValue(Dict['EX_TOKEN_NUMBER'])
    GuidTableCount = GetIntegerValue(Dict['GUID_TABLE_SIZE'])
    SystemSkuId = GetIntegerValue(Dict['SYSTEM_SKU_ID_VALUE'])
    Pad = 0xDA

    UninitDataBaseSize  = 0
    for Item in (DbUnInitValueUint64, DbUnInitValueUint32, DbUnInitValueUint16, DbUnInitValueUint8, DbUnInitValueBoolean):
        UninitDataBaseSize += Item.GetListSize()

    # Construct the database buffer
    Guid = "{0x3c7d193c, 0x682c, 0x4c14, 0xa6, 0x8f, 0x55, 0x2d, 0xea, 0x4f, 0x43, 0x7e}"
    Guid = StringArrayToList(Guid)
    Buffer = pack('=LHHBBBBBBBB',
                Guid[0],
                Guid[1],
                Guid[2],
                Guid[3],
                Guid[4],
                Guid[5],
                Guid[6],
                Guid[7],
                Guid[8],
                Guid[9],
                Guid[10],
                )

    b = pack("=L", DATABASE_VERSION)
    Buffer += b

    b = pack('=L', DbTotalLength - UninitDataBaseSize)

    Buffer += b
    b = pack('=L', UninitDataBaseSize)

    Buffer += b
    b = pack('=L', LocalTokenNumberTableOffset)

    Buffer += b
    b = pack('=L', ExMapTableOffset)

    Buffer += b
    b = pack('=L', GuidTableOffset)

    Buffer += b
    b = pack('=L', StringTableOffset)

    Buffer += b
    b = pack('=L', SizeTableOffset)

    Buffer += b
    b = pack('=L', SkuIdTableOffset)

    Buffer += b
    b = pack('=L', DbPcdNameOffset)

    Buffer += b
    b = pack('=H', LocalTokenCount)

    Buffer += b
    b = pack('=H', ExTokenCount)

    Buffer += b
    b = pack('=H', GuidTableCount)

    Buffer += b
    b = pack('=B', SystemSkuId)

    Buffer += b
    b = pack('=B', Pad)

    Buffer += b

    Index = 0
    for Item in DbItemTotal:
        Index +=1
        b = Item.PackData()
        Buffer += b
        if Index == InitTableNum:
            break
    return Buffer

## Create code for PCD database
#
#   @param      Info        The ModuleAutoGen object
#   @param      AutoGenC    The TemplateString object for C code
#   @param      AutoGenH    The TemplateString object for header file
#
def CreatePcdDatabaseCode (Info, AutoGenC, AutoGenH):
    if Info.PcdIsDriver == "":
        return
    if Info.PcdIsDriver not in gPcdPhaseMap:
        EdkLogger.error("build", AUTOGEN_ERROR, "Not supported PcdIsDriver type:%s" % Info.PcdIsDriver,
                        ExtraData="[%s]" % str(Info))

    AdditionalAutoGenH, AdditionalAutoGenC, PcdDbBuffer = CreatePcdDatabasePhaseSpecificAutoGen (Info.PlatformInfo, 'PEI')
    AutoGenH.Append(AdditionalAutoGenH.String)

    Phase = gPcdPhaseMap[Info.PcdIsDriver]
    if Phase == 'PEI':
        AutoGenC.Append(AdditionalAutoGenC.String)

    if Phase == 'DXE':
        AdditionalAutoGenH, AdditionalAutoGenC, PcdDbBuffer = CreatePcdDatabasePhaseSpecificAutoGen (Info.PlatformInfo, Phase)
        AutoGenH.Append(AdditionalAutoGenH.String)
        AutoGenC.Append(AdditionalAutoGenC.String)

    if Info.IsBinaryModule:
        DbFileName = os.path.join(Info.PlatformInfo.BuildDir, "FV", Phase + "PcdDataBase.raw")
    else:
        DbFileName = os.path.join(Info.OutputDir, Phase + "PcdDataBase.raw")
    DbFile = StringIO()
    DbFile.write(PcdDbBuffer)
    Changed = SaveFileOnChange(DbFileName, DbFile.getvalue(), True)

## Create PCD database in DXE or PEI phase
#
#   @param      Platform    The platform object
#   @retval     tuple       Two TemplateString objects for C code and header file,
#                           respectively
#
def CreatePcdDatabasePhaseSpecificAutoGen (Platform, Phase):
    AutoGenC = TemplateString()
    AutoGenH = TemplateString()

    Dict = {
        'PHASE'                         : Phase,
        'SERVICE_DRIVER_VERSION'        : DATABASE_VERSION,
        'GUID_TABLE_SIZE'               : '1U',
        'STRING_TABLE_SIZE'             : '1U',
        'SKUID_TABLE_SIZE'              : '1U',
        'LOCAL_TOKEN_NUMBER_TABLE_SIZE' : '0U',
        'LOCAL_TOKEN_NUMBER'            : '0U',
        'EXMAPPING_TABLE_SIZE'          : '1U',
        'EX_TOKEN_NUMBER'               : '0U',
        'SIZE_TABLE_SIZE'               : '2U',
        'SKU_HEAD_SIZE'                 : '1U',
        'GUID_TABLE_EMPTY'              : 'TRUE',
        'STRING_TABLE_EMPTY'            : 'TRUE',
        'SKUID_TABLE_EMPTY'             : 'TRUE',
        'DATABASE_EMPTY'                : 'TRUE',
        'EXMAP_TABLE_EMPTY'             : 'TRUE',
        'PCD_DATABASE_UNINIT_EMPTY'     : '  UINT8  dummy; /* PCD_DATABASE_UNINIT is emptry */',
        'SYSTEM_SKU_ID'                 : '  SKU_ID             SystemSkuId;',
        'SYSTEM_SKU_ID_VALUE'           : '0U'
    }


    SkuObj = SkuClass(Platform.Platform.SkuName,Platform.Platform.SkuIds)
    Dict['SYSTEM_SKU_ID_VALUE'] = Platform.Platform.SkuIds[SkuObj.SystemSkuId]

    Dict['PCD_INFO_FLAG'] = Platform.Platform.PcdInfoFlag

    for DatumType in ['UINT64','UINT32','UINT16','UINT8','BOOLEAN', "VOID*"]:
        Dict['VARDEF_CNAME_' + DatumType] = []
        Dict['VARDEF_GUID_' + DatumType]  = []
        Dict['VARDEF_SKUID_' + DatumType] = []
        Dict['VARDEF_VALUE_' + DatumType] = []
        Dict['VARDEF_DB_VALUE_' + DatumType] = []
        for Init in ['INIT','UNINIT']:
            Dict[Init+'_CNAME_DECL_' + DatumType]   = []
            Dict[Init+'_GUID_DECL_' + DatumType]    = []
            Dict[Init+'_NUMSKUS_DECL_' + DatumType] = []
            Dict[Init+'_VALUE_' + DatumType]        = []
            Dict[Init+'_DB_VALUE_'+DatumType] = []

    for Type in ['STRING_HEAD','VPD_HEAD','VARIABLE_HEAD']:
        Dict[Type + '_CNAME_DECL']   = []
        Dict[Type + '_GUID_DECL']    = []
        Dict[Type + '_NUMSKUS_DECL'] = []
        Dict[Type + '_VALUE'] = []

    Dict['STRING_DB_VALUE'] = []
    Dict['VPD_DB_VALUE'] = []
    Dict['VARIABLE_DB_VALUE'] = []

    Dict['STRING_TABLE_INDEX'] = []
    Dict['STRING_TABLE_LENGTH']  = []
    Dict['STRING_TABLE_CNAME'] = []
    Dict['STRING_TABLE_GUID']  = []
    Dict['STRING_TABLE_VALUE'] = []
    Dict['STRING_TABLE_DB_VALUE'] = []

    Dict['SIZE_TABLE_CNAME'] = []
    Dict['SIZE_TABLE_GUID']  = []
    Dict['SIZE_TABLE_CURRENT_LENGTH']  = []
    Dict['SIZE_TABLE_MAXIMUM_LENGTH']  = []

    Dict['EXMAPPING_TABLE_EXTOKEN'] = []
    Dict['EXMAPPING_TABLE_LOCAL_TOKEN'] = []
    Dict['EXMAPPING_TABLE_GUID_INDEX'] = []

    Dict['GUID_STRUCTURE'] = []
    Dict['SKUID_VALUE'] = [0] # init Dict length
    Dict['VARDEF_HEADER'] = []

    Dict['LOCAL_TOKEN_NUMBER_DB_VALUE'] = []
    Dict['VARIABLE_DB_VALUE'] = []
    Dict['SKUHEAD_TABLE_VALUE'] = []
    Dict['SKU_INDEX_VALUE'] = []

    Dict['PCD_TOKENSPACE'] = []
    Dict['PCD_CNAME'] = []
    Dict['PCD_TOKENSPACE_LENGTH'] = []
    Dict['PCD_CNAME_LENGTH'] = []
    Dict['PCD_TOKENSPACE_OFFSET'] = []
    Dict['PCD_CNAME_OFFSET'] = []
    Dict['PCD_TOKENSPACE_MAP'] = []
    Dict['PCD_NAME_OFFSET'] = []

    Dict['PCD_ORDER_TOKEN_NUMBER_MAP'] = {}
    PCD_STRING_INDEX_MAP = {}

    StringTableIndex = 0
    StringTableSize = 0
    NumberOfLocalTokens = 0
    NumberOfPeiLocalTokens = 0
    NumberOfDxeLocalTokens = 0
    NumberOfExTokens = 0
    NumberOfSizeItems = 0
    NumberOfSkuEnabledPcd = 0
    GuidList = []
    i = 0
    ReorderedDynPcdList = GetOrderedDynamicPcdList(Platform.DynamicPcdList, Platform.PcdTokenNumber)
    for Pcd in ReorderedDynPcdList:
        VoidStarTypeCurrSize = []
        i += 1
        CName = Pcd.TokenCName
        TokenSpaceGuidCName = Pcd.TokenSpaceGuidCName

        EdkLogger.debug(EdkLogger.DEBUG_3, "PCD: %s %s (%s : %s)" % (CName, TokenSpaceGuidCName, Pcd.Phase, Phase))

        if Pcd.Phase == 'PEI':
            NumberOfPeiLocalTokens += 1
        if Pcd.Phase == 'DXE':
            NumberOfDxeLocalTokens += 1
        if Pcd.Phase != Phase:
            continue

        #
        # TODO: need GetGuidValue() definition
        #
        TokenSpaceGuidStructure = Pcd.TokenSpaceGuidValue
        TokenSpaceGuid = GuidStructureStringToGuidValueName(TokenSpaceGuidStructure)
        if Pcd.Type in gDynamicExPcd:
            if TokenSpaceGuid not in GuidList:
                GuidList += [TokenSpaceGuid]
                Dict['GUID_STRUCTURE'].append(TokenSpaceGuidStructure)
            NumberOfExTokens += 1

        ValueList = []
        DbValueList = []
        StringHeadOffsetList = []
        StringDbOffsetList = []
        VpdHeadOffsetList = []
        VpdDbOffsetList = []
        VariableHeadValueList = []
        VariableDbValueList = []
        Pcd.InitString = 'UNINIT'

        if Pcd.DatumType == 'VOID*':
            if Pcd.Type not in ["DynamicVpd", "DynamicExVpd"]:
                Pcd.TokenTypeList = ['PCD_TYPE_STRING']
            else:
                Pcd.TokenTypeList = []
        elif Pcd.DatumType == 'BOOLEAN':
            Pcd.TokenTypeList = ['PCD_DATUM_TYPE_UINT8_BOOLEAN']
        else:
            Pcd.TokenTypeList = ['PCD_DATUM_TYPE_' + Pcd.DatumType]

        if len(Pcd.SkuInfoList) > 1:
            Pcd.TokenTypeList += ['PCD_TYPE_SKU_ENABLED']
            NumberOfSkuEnabledPcd += 1

        SkuIndexTableTmp = []
        SkuIndexTableTmp.append(0)
        SkuIdIndex = 1
        VariableHeadList = []
        for SkuName in Pcd.SkuInfoList:
            Sku = Pcd.SkuInfoList[SkuName]
            SkuId = Sku.SkuId
            if SkuId == None or SkuId == '':
                continue

            if (SkuId + 'U') not in Dict['SKUID_VALUE']:
                Dict['SKUID_VALUE'].append(SkuId + 'U')

            SkuIndexTableTmp.append(SkuId+'U')
            SkuIdIndex += 1

            if len(Sku.VariableName) > 0:
                Pcd.TokenTypeList += ['PCD_TYPE_HII']
                Pcd.InitString = 'INIT'
                # Store all variable names of one HII PCD under different SKU to stringTable
                # and calculate the VariableHeadStringIndex
                if SkuIdIndex - 2 == 0:
                    for SkuName2 in Pcd.SkuInfoList:
                        SkuInfo = Pcd.SkuInfoList[SkuName2]
                        if SkuInfo.SkuId == None or SkuInfo.SkuId == '':
                            continue
                        VariableNameStructure = StringToArray(SkuInfo.VariableName)
                        if VariableNameStructure not in Dict['STRING_TABLE_VALUE']:
                            Dict['STRING_TABLE_CNAME'].append(CName)
                            Dict['STRING_TABLE_GUID'].append(TokenSpaceGuid)
                            if StringTableIndex == 0:
                                Dict['STRING_TABLE_INDEX'].append('')
                            else:
                                Dict['STRING_TABLE_INDEX'].append('_%d' % StringTableIndex)
                            VarNameSize = len(VariableNameStructure.replace(',',' ').split())
                            Dict['STRING_TABLE_LENGTH'].append(VarNameSize )
                            Dict['STRING_TABLE_VALUE'].append(VariableNameStructure)
                            StringHeadOffsetList.append(str(StringTableSize) + 'U')
                            VarStringDbOffsetList = []
                            VarStringDbOffsetList.append(StringTableSize)
                            Dict['STRING_DB_VALUE'].append(VarStringDbOffsetList)
                            StringTableIndex += 1
                            StringTableSize += len(VariableNameStructure.replace(',',' ').split())
                        VariableHeadStringIndex = 0
                        for Index in range(Dict['STRING_TABLE_VALUE'].index(VariableNameStructure)):
                            VariableHeadStringIndex += Dict['STRING_TABLE_LENGTH'][Index]
                        VariableHeadList.append(VariableHeadStringIndex)

                VariableHeadStringIndex = VariableHeadList[SkuIdIndex - 2]
                # store VariableGuid to GuidTable and get the VariableHeadGuidIndex
                VariableGuidStructure = Sku.VariableGuidValue
                VariableGuid = GuidStructureStringToGuidValueName(VariableGuidStructure)

                if VariableGuid not in GuidList:
                    GuidList += [VariableGuid]
                    Dict['GUID_STRUCTURE'].append(VariableGuidStructure)
                VariableHeadGuidIndex = GuidList.index(VariableGuid)

                if "PCD_TYPE_STRING" in Pcd.TokenTypeList:
                    VariableHeadValueList.append('%dU, offsetof(%s_PCD_DATABASE, Init.%s_%s), %dU, %sU' %
                                                 (VariableHeadStringIndex, Phase, CName, TokenSpaceGuid,
                                                 VariableHeadGuidIndex, Sku.VariableOffset))
                else:
                    VariableHeadValueList.append('%dU, offsetof(%s_PCD_DATABASE, Init.%s_%s_VariableDefault_%s), %dU, %sU' %
                                                 (VariableHeadStringIndex, Phase, CName, TokenSpaceGuid, SkuIdIndex,
                                                 VariableHeadGuidIndex, Sku.VariableOffset))
                Dict['VARDEF_CNAME_'+Pcd.DatumType].append(CName)
                Dict['VARDEF_GUID_'+Pcd.DatumType].append(TokenSpaceGuid)
                Dict['VARDEF_SKUID_'+Pcd.DatumType].append(SkuIdIndex)
                if "PCD_TYPE_STRING" in  Pcd.TokenTypeList:
                    Dict['VARDEF_VALUE_' + Pcd.DatumType].append("%s_%s[%d]" % (Pcd.TokenCName, TokenSpaceGuid, SkuIdIndex))
                else:
                    #
                    # ULL (for UINT64) or U(other integer type) should be append to avoid
                    # warning under linux building environment.
                    #
                    Dict['VARDEF_DB_VALUE_'+Pcd.DatumType].append(Sku.HiiDefaultValue)

                    if Pcd.DatumType == "UINT64":
                        Dict['VARDEF_VALUE_'+Pcd.DatumType].append(Sku.HiiDefaultValue + "ULL")
                    elif Pcd.DatumType in ("UINT32", "UINT16", "UINT8"):
                        Dict['VARDEF_VALUE_'+Pcd.DatumType].append(Sku.HiiDefaultValue + "U")
                    elif Pcd.DatumType == "BOOLEAN":
                        if eval(Sku.HiiDefaultValue) in [1,0]:
                            Dict['VARDEF_VALUE_'+Pcd.DatumType].append(str(eval(Sku.HiiDefaultValue)) + "U")
                    else:
                        Dict['VARDEF_VALUE_'+Pcd.DatumType].append(Sku.HiiDefaultValue)

                # construct the VariableHeader value
                if "PCD_TYPE_STRING" in Pcd.TokenTypeList:
                    VariableHeadValueList.append('%dU, %dU, %sU, offsetof(%s_PCD_DATABASE, Init.%s_%s)' %
                                                 (VariableHeadGuidIndex, VariableHeadStringIndex, Sku.VariableOffset,
                                                  Phase, CName, TokenSpaceGuid))
                    # the Pcd default value will be filled later on
                    VariableOffset = len(Dict['STRING_DB_VALUE'])
                    VariableRefTable = Dict['STRING_DB_VALUE']
                else:
                    VariableHeadValueList.append('%dU, %dU, %sU, offsetof(%s_PCD_DATABASE, Init.%s_%s_VariableDefault_%s)' %
                                                 (VariableHeadGuidIndex, VariableHeadStringIndex, Sku.VariableOffset,
                                                  Phase, CName, TokenSpaceGuid, SkuIdIndex))
                    # the Pcd default value was filled before
                    VariableOffset = len(Dict['VARDEF_DB_VALUE_' + Pcd.DatumType]) - 1
                    VariableRefTable = Dict['VARDEF_DB_VALUE_' + Pcd.DatumType]
                VariableDbValueList.append([VariableHeadGuidIndex, VariableHeadStringIndex, Sku.VariableOffset, VariableOffset, VariableRefTable])

            elif Sku.VpdOffset != '':
                Pcd.TokenTypeList += ['PCD_TYPE_VPD']
                Pcd.InitString = 'INIT'
                VpdHeadOffsetList.append(str(Sku.VpdOffset) + 'U')
                VpdDbOffsetList.append(Sku.VpdOffset)
                # Also add the VOID* string of VPD PCD to SizeTable
                if Pcd.DatumType == 'VOID*':
                    NumberOfSizeItems += 1
                    # For VPD type of PCD, its current size is equal to its MAX size.
                    VoidStarTypeCurrSize = [str(Pcd.MaxDatumSize) + 'U']
                continue

            if Pcd.DatumType == 'VOID*':
                Pcd.TokenTypeList += ['PCD_TYPE_STRING']
                Pcd.InitString = 'INIT'
                if Sku.HiiDefaultValue != '' and Sku.DefaultValue == '':
                    Sku.DefaultValue = Sku.HiiDefaultValue
                if Sku.DefaultValue != '':
                    NumberOfSizeItems += 1
                    Dict['STRING_TABLE_CNAME'].append(CName)
                    Dict['STRING_TABLE_GUID'].append(TokenSpaceGuid)

                    if StringTableIndex == 0:
                        Dict['STRING_TABLE_INDEX'].append('')
                    else:
                        Dict['STRING_TABLE_INDEX'].append('_%d' % StringTableIndex)
                    if Sku.DefaultValue[0] == 'L':
                        DefaultValueBinStructure = StringToArray(Sku.DefaultValue)
                        Size = len(DefaultValueBinStructure.replace(',',' ').split())
                        Dict['STRING_TABLE_VALUE'].append(DefaultValueBinStructure)
                    elif Sku.DefaultValue[0] == '"':
                        DefaultValueBinStructure = StringToArray(Sku.DefaultValue)
                        Size = len(Sku.DefaultValue) -2 + 1
                        Dict['STRING_TABLE_VALUE'].append(DefaultValueBinStructure)
                    elif Sku.DefaultValue[0] == '{':
                        DefaultValueBinStructure = StringToArray(Sku.DefaultValue)
                        Size = len(Sku.DefaultValue.split(","))
                        Dict['STRING_TABLE_VALUE'].append(DefaultValueBinStructure)

                    StringHeadOffsetList.append(str(StringTableSize) + 'U')
                    StringDbOffsetList.append(StringTableSize)
                    if Pcd.MaxDatumSize != '':
                        MaxDatumSize = int(Pcd.MaxDatumSize, 0)
                        if MaxDatumSize < Size:
                            EdkLogger.error("build", AUTOGEN_ERROR,
                                            "The maximum size of VOID* type PCD '%s.%s' is less than its actual size occupied." % (Pcd.TokenSpaceGuidCName, Pcd.TokenCName),
                                            ExtraData="[%s]" % str(Platform))
                    else:
                        MaxDatumSize = Size
                    StringTabLen = MaxDatumSize
                    if StringTabLen % 2:
                        StringTabLen += 1
                    if Sku.VpdOffset == '':
                        VoidStarTypeCurrSize.append(str(Size) + 'U')
                    Dict['STRING_TABLE_LENGTH'].append(StringTabLen)
                    StringTableIndex += 1
                    StringTableSize += (StringTabLen)
            else:
                if "PCD_TYPE_HII" not in Pcd.TokenTypeList:
                    Pcd.TokenTypeList += ['PCD_TYPE_DATA']
                    if Sku.DefaultValue == 'TRUE':
                        Pcd.InitString = 'INIT'
                    else:
                        if int(Sku.DefaultValue, 0) != 0:
                            Pcd.InitString = 'INIT'
                #
                # For UNIT64 type PCD's value, ULL should be append to avoid
                # warning under linux building environment.
                #
                if Pcd.DatumType == "UINT64":
                    ValueList.append(Sku.DefaultValue + "ULL")
                elif Pcd.DatumType in ("UINT32", "UINT16", "UINT8"):
                    ValueList.append(Sku.DefaultValue + "U")
                elif Pcd.DatumType == "BOOLEAN":
                    if Sku.DefaultValue in ["1", "0"]:
                        ValueList.append(Sku.DefaultValue + "U")
                else:
                    ValueList.append(Sku.DefaultValue)

                DbValueList.append(Sku.DefaultValue)

        Pcd.TokenTypeList = list(set(Pcd.TokenTypeList))
        if Pcd.DatumType == 'VOID*':
            Dict['SIZE_TABLE_CNAME'].append(CName)
            Dict['SIZE_TABLE_GUID'].append(TokenSpaceGuid)
            Dict['SIZE_TABLE_MAXIMUM_LENGTH'].append(str(Pcd.MaxDatumSize) + 'U')
            Dict['SIZE_TABLE_CURRENT_LENGTH'].append(VoidStarTypeCurrSize)


        SkuIndexTableTmp[0] = len(SkuIndexTableTmp) - 1
        if len(Pcd.SkuInfoList) > 1:
            Dict['SKU_INDEX_VALUE'].append(SkuIndexTableTmp)

        if 'PCD_TYPE_HII' in Pcd.TokenTypeList:
            Dict['VARIABLE_HEAD_CNAME_DECL'].append(CName)
            Dict['VARIABLE_HEAD_GUID_DECL'].append(TokenSpaceGuid)
            Dict['VARIABLE_HEAD_NUMSKUS_DECL'].append(len(Pcd.SkuInfoList))
            Dict['VARIABLE_HEAD_VALUE'].append('{ %s }\n' % ' },\n    { '.join(VariableHeadValueList))
            Dict['VARDEF_HEADER'].append('_Variable_Header')
            Dict['VARIABLE_DB_VALUE'].append(VariableDbValueList)
        else:
            Dict['VARDEF_HEADER'].append('')
        if 'PCD_TYPE_VPD' in Pcd.TokenTypeList:
            Dict['VPD_HEAD_CNAME_DECL'].append(CName)
            Dict['VPD_HEAD_GUID_DECL'].append(TokenSpaceGuid)
            Dict['VPD_HEAD_NUMSKUS_DECL'].append(len(Pcd.SkuInfoList))
            Dict['VPD_HEAD_VALUE'].append('{ %s }' % ' }, { '.join(VpdHeadOffsetList))
            Dict['VPD_DB_VALUE'].append(VpdDbOffsetList)
        if 'PCD_TYPE_STRING' in Pcd.TokenTypeList:
            Dict['STRING_HEAD_CNAME_DECL'].append(CName)
            Dict['STRING_HEAD_GUID_DECL'].append(TokenSpaceGuid)
            Dict['STRING_HEAD_NUMSKUS_DECL'].append(len(Pcd.SkuInfoList))
            Dict['STRING_HEAD_VALUE'].append(', '.join(StringHeadOffsetList))
            Dict['STRING_DB_VALUE'].append(StringDbOffsetList)
            PCD_STRING_INDEX_MAP[len(Dict['STRING_HEAD_CNAME_DECL']) -1 ] = len(Dict['STRING_DB_VALUE']) -1
        if 'PCD_TYPE_DATA' in Pcd.TokenTypeList:
            Dict[Pcd.InitString+'_CNAME_DECL_'+Pcd.DatumType].append(CName)
            Dict[Pcd.InitString+'_GUID_DECL_'+Pcd.DatumType].append(TokenSpaceGuid)
            Dict[Pcd.InitString+'_NUMSKUS_DECL_'+Pcd.DatumType].append(len(Pcd.SkuInfoList))
            if Pcd.InitString == 'UNINIT':
                Dict['PCD_DATABASE_UNINIT_EMPTY'] = ''
            else:
                Dict[Pcd.InitString+'_VALUE_'+Pcd.DatumType].append(', '.join(ValueList))
                Dict[Pcd.InitString+'_DB_VALUE_'+Pcd.DatumType].append(DbValueList)

    if Phase == 'PEI':
        NumberOfLocalTokens = NumberOfPeiLocalTokens
    if Phase == 'DXE':
        NumberOfLocalTokens = NumberOfDxeLocalTokens

    Dict['TOKEN_INIT']       = ['' for x in range(NumberOfLocalTokens)]
    Dict['TOKEN_CNAME']      = ['' for x in range(NumberOfLocalTokens)]
    Dict['TOKEN_GUID']       = ['' for x in range(NumberOfLocalTokens)]
    Dict['TOKEN_TYPE']       = ['' for x in range(NumberOfLocalTokens)]
    Dict['LOCAL_TOKEN_NUMBER_DB_VALUE'] = ['' for x in range(NumberOfLocalTokens)]
    Dict['PCD_CNAME']        = ['' for x in range(NumberOfLocalTokens)]
    Dict['PCD_TOKENSPACE_MAP'] = ['' for x in range(NumberOfLocalTokens)]
    Dict['PCD_CNAME_LENGTH'] = [0 for x in range(NumberOfLocalTokens)]
    SkuEnablePcdIndex = 0
    for Pcd in ReorderedDynPcdList:
        CName = Pcd.TokenCName
        TokenSpaceGuidCName = Pcd.TokenSpaceGuidCName
        if Pcd.Phase != Phase:
            continue

        TokenSpaceGuid = GuidStructureStringToGuidValueName(Pcd.TokenSpaceGuidValue) #(Platform.PackageList, TokenSpaceGuidCName))
        GeneratedTokenNumber = Platform.PcdTokenNumber[CName, TokenSpaceGuidCName] - 1
        if Phase == 'DXE':
            GeneratedTokenNumber -= NumberOfPeiLocalTokens

        if len(Pcd.SkuInfoList) > 1:
            Dict['PCD_ORDER_TOKEN_NUMBER_MAP'][GeneratedTokenNumber] = SkuEnablePcdIndex
            SkuEnablePcdIndex += 1
        EdkLogger.debug(EdkLogger.DEBUG_1, "PCD = %s.%s" % (CName, TokenSpaceGuidCName))
        EdkLogger.debug(EdkLogger.DEBUG_1, "phase = %s" % Phase)
        EdkLogger.debug(EdkLogger.DEBUG_1, "GeneratedTokenNumber = %s" % str(GeneratedTokenNumber))

        #
        # following four Dict items hold the information for LocalTokenNumberTable
        #
        Dict['TOKEN_INIT'][GeneratedTokenNumber] = 'Init'
        if Pcd.InitString == 'UNINIT':
            Dict['TOKEN_INIT'][GeneratedTokenNumber] = 'Uninit'

        Dict['TOKEN_CNAME'][GeneratedTokenNumber] = CName
        Dict['TOKEN_GUID'][GeneratedTokenNumber] = TokenSpaceGuid
        Dict['TOKEN_TYPE'][GeneratedTokenNumber] = ' | '.join(Pcd.TokenTypeList)

        if Platform.Platform.PcdInfoFlag:
            TokenSpaceGuidCNameArray = StringToArray('"' + TokenSpaceGuidCName + '"' )
            if TokenSpaceGuidCNameArray not in Dict['PCD_TOKENSPACE']:
                Dict['PCD_TOKENSPACE'].append(TokenSpaceGuidCNameArray)
                Dict['PCD_TOKENSPACE_LENGTH'].append( len(TokenSpaceGuidCNameArray.split(",")) )
            Dict['PCD_TOKENSPACE_MAP'][GeneratedTokenNumber] = Dict['PCD_TOKENSPACE'].index(TokenSpaceGuidCNameArray)
            CNameBinArray = StringToArray('"' + CName + '"' )
            Dict['PCD_CNAME'][GeneratedTokenNumber] = CNameBinArray

            Dict['PCD_CNAME_LENGTH'][GeneratedTokenNumber] = len(CNameBinArray.split(","))


        Pcd.TokenTypeList = list(set(Pcd.TokenTypeList))

        # search the Offset and Table, used by LocalTokenNumberTableOffset
        if 'PCD_TYPE_HII' in Pcd.TokenTypeList:
            # Find index by CName, TokenSpaceGuid
            Offset = GetMatchedIndex(CName, Dict['VARIABLE_HEAD_CNAME_DECL'], TokenSpaceGuid, Dict['VARIABLE_HEAD_GUID_DECL'])
            assert(Offset != -1)
            Table = Dict['VARIABLE_DB_VALUE']
        if 'PCD_TYPE_VPD' in Pcd.TokenTypeList:
            Offset = GetMatchedIndex(CName, Dict['VPD_HEAD_CNAME_DECL'], TokenSpaceGuid, Dict['VPD_HEAD_GUID_DECL'])
            assert(Offset != -1)
            Table = Dict['VPD_DB_VALUE']
        if 'PCD_TYPE_STRING' in Pcd.TokenTypeList and 'PCD_TYPE_HII' not in Pcd.TokenTypeList:
            # Find index by CName, TokenSpaceGuid
            Offset = GetMatchedIndex(CName, Dict['STRING_HEAD_CNAME_DECL'], TokenSpaceGuid, Dict['STRING_HEAD_GUID_DECL'])
            Offset = PCD_STRING_INDEX_MAP[Offset]
            assert(Offset != -1)
            Table = Dict['STRING_DB_VALUE']
        if 'PCD_TYPE_DATA' in Pcd.TokenTypeList:
            # need to store whether it is in init table or not
            Offset = GetMatchedIndex(CName, Dict[Pcd.InitString+'_CNAME_DECL_'+Pcd.DatumType], TokenSpaceGuid, Dict[Pcd.InitString+'_GUID_DECL_'+Pcd.DatumType])
            assert(Offset != -1)
            if Pcd.InitString == 'UNINIT':
                Table =  Dict[Pcd.InitString+'_GUID_DECL_'+Pcd.DatumType]
            else:
                Table = Dict[Pcd.InitString+'_DB_VALUE_'+Pcd.DatumType]
        Dict['LOCAL_TOKEN_NUMBER_DB_VALUE'][GeneratedTokenNumber] = (Offset, Table)

        #
        # Update VARDEF_HEADER
        #
        if 'PCD_TYPE_HII' in Pcd.TokenTypeList:
            Dict['VARDEF_HEADER'][GeneratedTokenNumber] = '_Variable_Header'
        else:
            Dict['VARDEF_HEADER'][GeneratedTokenNumber] = ''


        if Pcd.Type in gDynamicExPcd:

            if Phase == 'DXE':
                GeneratedTokenNumber += NumberOfPeiLocalTokens
            #
            # Per, PCD architecture specification, PCD Token Number is 1 based and 0 is defined as invalid token number.
            # For each EX type PCD, a PCD Token Number is assigned. When the
            # PCD Driver/PEIM map EX_GUID and EX_TOKEN_NUMBER to the PCD Token Number,
            # the non-EX Protocol/PPI interface can be called to get/set the value. This assumption is made by
            # Pcd Driver/PEIM in MdeModulePkg.
            # Therefore, 1 is added to GeneratedTokenNumber to generate a PCD Token Number before being inserted
            # to the EXMAPPING_TABLE.
            #


            Dict['EXMAPPING_TABLE_EXTOKEN'].append(str(Pcd.TokenValue) + 'U')
            Dict['EXMAPPING_TABLE_LOCAL_TOKEN'].append(str(GeneratedTokenNumber + 1) + 'U')
            Dict['EXMAPPING_TABLE_GUID_INDEX'].append(str(GuidList.index(TokenSpaceGuid)) + 'U')

    if Platform.Platform.PcdInfoFlag:
        for index in range(len(Dict['PCD_TOKENSPACE_MAP'])):
            TokenSpaceIndex = StringTableSize
            for i in range(Dict['PCD_TOKENSPACE_MAP'][index]):
                TokenSpaceIndex += Dict['PCD_TOKENSPACE_LENGTH'][i]
            Dict['PCD_TOKENSPACE_OFFSET'].append(TokenSpaceIndex)
        for index in range(len(Dict['PCD_TOKENSPACE'])):
            StringTableSize += Dict['PCD_TOKENSPACE_LENGTH'][index]
            StringTableIndex += 1
        for index in range(len(Dict['PCD_CNAME'])):
            Dict['PCD_CNAME_OFFSET'].append(StringTableSize)
            Dict['PCD_NAME_OFFSET'].append(Dict['PCD_TOKENSPACE_OFFSET'][index])
            Dict['PCD_NAME_OFFSET'].append(StringTableSize)
            StringTableSize += Dict['PCD_CNAME_LENGTH'][index]
            StringTableIndex += 1
    if GuidList != []:
        Dict['GUID_TABLE_EMPTY'] = 'FALSE'
        Dict['GUID_TABLE_SIZE'] = str(len(GuidList)) + 'U'
    else:
        Dict['GUID_STRUCTURE'] = [GuidStringToGuidStructureString('00000000-0000-0000-0000-000000000000')]

    if StringTableIndex == 0:
        Dict['STRING_TABLE_INDEX'].append('')
        Dict['STRING_TABLE_LENGTH'].append(1)
        Dict['STRING_TABLE_CNAME'].append('')
        Dict['STRING_TABLE_GUID'].append('')
        Dict['STRING_TABLE_VALUE'].append('{ 0 }')
    else:
        Dict['STRING_TABLE_EMPTY'] = 'FALSE'
        Dict['STRING_TABLE_SIZE'] = str(StringTableSize) + 'U'

    if Dict['SIZE_TABLE_CNAME'] == []:
        Dict['SIZE_TABLE_CNAME'].append('')
        Dict['SIZE_TABLE_GUID'].append('')
        Dict['SIZE_TABLE_CURRENT_LENGTH'].append(['0U'])
        Dict['SIZE_TABLE_MAXIMUM_LENGTH'].append('0U')

    if NumberOfLocalTokens != 0:
        Dict['DATABASE_EMPTY']                = 'FALSE'
        Dict['LOCAL_TOKEN_NUMBER_TABLE_SIZE'] = NumberOfLocalTokens
        Dict['LOCAL_TOKEN_NUMBER']            = NumberOfLocalTokens

    if NumberOfExTokens != 0:
        Dict['EXMAP_TABLE_EMPTY']    = 'FALSE'
        Dict['EXMAPPING_TABLE_SIZE'] = str(NumberOfExTokens) + 'U'
        Dict['EX_TOKEN_NUMBER']      = str(NumberOfExTokens) + 'U'
    else:
        Dict['EXMAPPING_TABLE_EXTOKEN'].append('0U')
        Dict['EXMAPPING_TABLE_LOCAL_TOKEN'].append('0U')
        Dict['EXMAPPING_TABLE_GUID_INDEX'].append('0U')

    if NumberOfSizeItems != 0:
        Dict['SIZE_TABLE_SIZE'] = str(NumberOfSizeItems * 2) + 'U'

    if NumberOfSkuEnabledPcd != 0:
        Dict['SKU_HEAD_SIZE'] = str(NumberOfSkuEnabledPcd) + 'U'

    Dict['SKUID_VALUE'][0] = len(Dict['SKUID_VALUE']) - 1

    AutoGenH.Append(gPcdDatabaseAutoGenH.Replace(Dict))
    if NumberOfLocalTokens == 0:
        AutoGenC.Append(gEmptyPcdDatabaseAutoGenC.Replace(Dict))
    else:
        #
        # Update Size Table to the right order, it should be same with LocalTokenNumberTable
        #
        SizeCNameTempList = []
        SizeGuidTempList = []
        SizeCurLenTempList = []
        SizeMaxLenTempList = []
        ReOrderFlag = True

        if len(Dict['SIZE_TABLE_CNAME']) == 1:
            if not (Dict['SIZE_TABLE_CNAME'][0] and Dict['SIZE_TABLE_GUID'][0]):
                ReOrderFlag = False

        if ReOrderFlag:
            for Count in range(len(Dict['TOKEN_CNAME'])):
                for Count1 in range(len(Dict['SIZE_TABLE_CNAME'])):
                    if Dict['TOKEN_CNAME'][Count] == Dict['SIZE_TABLE_CNAME'][Count1] and \
                        Dict['TOKEN_GUID'][Count] == Dict['SIZE_TABLE_GUID'][Count1]:
                        SizeCNameTempList.append(Dict['SIZE_TABLE_CNAME'][Count1])
                        SizeGuidTempList.append(Dict['SIZE_TABLE_GUID'][Count1])
                        SizeCurLenTempList.append(Dict['SIZE_TABLE_CURRENT_LENGTH'][Count1])
                        SizeMaxLenTempList.append(Dict['SIZE_TABLE_MAXIMUM_LENGTH'][Count1])

            for Count in range(len(Dict['SIZE_TABLE_CNAME'])):
                Dict['SIZE_TABLE_CNAME'][Count] = SizeCNameTempList[Count]
                Dict['SIZE_TABLE_GUID'][Count] = SizeGuidTempList[Count]
                Dict['SIZE_TABLE_CURRENT_LENGTH'][Count] = SizeCurLenTempList[Count]
                Dict['SIZE_TABLE_MAXIMUM_LENGTH'][Count] = SizeMaxLenTempList[Count]

        AutoGenC.Append(gPcdDatabaseAutoGenC.Replace(Dict))

    Buffer = BuildExDataBase(Dict)
    return AutoGenH, AutoGenC, Buffer

def GetOrderedDynamicPcdList(DynamicPcdList, PcdTokenNumberList):
    ReorderedDyPcdList = [None for i in range(len(DynamicPcdList))]
    for Pcd in DynamicPcdList:
        if (Pcd.TokenCName, Pcd.TokenSpaceGuidCName) in PcdTokenNumberList:
            ReorderedDyPcdList[PcdTokenNumberList[Pcd.TokenCName, Pcd.TokenSpaceGuidCName]-1] = Pcd
    return ReorderedDyPcdList

