/*
 * This recursion engine takes its roots from map-macro [1] and Cloak [2], with a few improvements:
 *
 *  - It can do many more expansions (roughly 1024 * 16 or 2^14).
 *
 *  - The expansion chain is linear: `ML99_PRIV_REC_0` invokes `ML99_PRIV_REC_1`, `ML99_PRIV_REC_1`
 * invokes `ML99_PRIV_REC_2`, and so on.
 *
 *  - If a given metaprogram does not require more expansions, then it will stop expanding. I.e.,
 * perform only as many expansions as needed. This is controlled by `ML99_PRIV_REC_NEXT`: if
 * `choice` is `0stop`, then just terminate the expansion chain.
 *
 *  - The last expander `ML99_PRIV_REC_1023` really results in a deferred `ML99_PRIV_REC_0`, not to
 * make it painted blue. Then, in `ML99_PRIV_REC_UNROLL_AUX`, this `ML99_PRIV_REC_0` is expanded
 * once again 16 times.
 *
 *  - It requires recursive macros to be written in CPS, continuation-passing style [3]. This is
 * controlled by `ML99_PRIV_REC_CONTINUE`: the `k` parameter stands for "continuation". `k` must
 * eventually expand to yet another `ML99_PRIV_REC_CONTINUE`. Also, there is a special continuation
 * called `ML99_PRIV_REC_STOP` -- it terminates the engine.
 *
 * The minimal usage example is located at `tests/eval/rec.c`.
 *
 * [1]: https://github.com/swansontec/map-macro
 * [2]: https://github.com/pfultz2/Cloak/wiki/C-Preprocessor-tricks,-tips,-and-idioms#recursion
 * [3]: https://en.wikipedia.org/wiki/Continuation-passing_style
 */

#ifndef ML99_EVAL_REC_H
#define ML99_EVAL_REC_H

#define ML99_PRIV_REC_CONTINUE(k)      0continue, ML99_PRIV_REC_DEFER(k##_HOOK)()
#define ML99_PRIV_REC_STOP(_k_cx, ...) 0stop, __VA_ARGS__
#define ML99_PRIV_REC_STOP_HOOK()      ML99_PRIV_REC_STOP

#define ML99_PRIV_REC_DEFER(op) op ML99_PRIV_REC_EMPTY
#define ML99_PRIV_REC_EMPTY
#define ML99_PRIV_REC_EXPAND(...) __VA_ARGS__

#define ML99_PRIV_REC_NEXT(next_lvl, choice)   ML99_PRIV_REC_NEXT_##choice(next_lvl)
#define ML99_PRIV_REC_NEXT_0continue(next_lvl) ML99_PRIV_REC_##next_lvl
#define ML99_PRIV_REC_NEXT_0stop(_next_lvl)    ML99_PRIV_REC_HALT

#define ML99_PRIV_REC_HALT(...) __VA_ARGS__

#define ML99_PRIV_REC_UNROLL(...) ML99_PRIV_REC_UNROLL_AUX(__VA_ARGS__)

// clang-format off
#define ML99_PRIV_REC_UNROLL_AUX(choice, ...) \
    /* Approximately 1024 * 16 reduction steps. */ \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
    ML99_PRIV_REC_EXPAND( \
        ML99_PRIV_REC_NEXT(0, choice)(__VA_ARGS__) \
    ))))))))))))))))
// clang-format on

#define ML99_PRIV_REC_0(choice, ...)    ML99_PRIV_REC_NEXT(1, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1(choice, ...)    ML99_PRIV_REC_NEXT(2, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_2(choice, ...)    ML99_PRIV_REC_NEXT(3, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_3(choice, ...)    ML99_PRIV_REC_NEXT(4, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_4(choice, ...)    ML99_PRIV_REC_NEXT(5, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_5(choice, ...)    ML99_PRIV_REC_NEXT(6, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_6(choice, ...)    ML99_PRIV_REC_NEXT(7, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_7(choice, ...)    ML99_PRIV_REC_NEXT(8, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_8(choice, ...)    ML99_PRIV_REC_NEXT(9, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_9(choice, ...)    ML99_PRIV_REC_NEXT(10, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_10(choice, ...)   ML99_PRIV_REC_NEXT(11, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_11(choice, ...)   ML99_PRIV_REC_NEXT(12, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_12(choice, ...)   ML99_PRIV_REC_NEXT(13, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_13(choice, ...)   ML99_PRIV_REC_NEXT(14, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_14(choice, ...)   ML99_PRIV_REC_NEXT(15, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_15(choice, ...)   ML99_PRIV_REC_NEXT(16, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_16(choice, ...)   ML99_PRIV_REC_NEXT(17, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_17(choice, ...)   ML99_PRIV_REC_NEXT(18, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_18(choice, ...)   ML99_PRIV_REC_NEXT(19, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_19(choice, ...)   ML99_PRIV_REC_NEXT(20, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_20(choice, ...)   ML99_PRIV_REC_NEXT(21, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_21(choice, ...)   ML99_PRIV_REC_NEXT(22, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_22(choice, ...)   ML99_PRIV_REC_NEXT(23, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_23(choice, ...)   ML99_PRIV_REC_NEXT(24, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_24(choice, ...)   ML99_PRIV_REC_NEXT(25, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_25(choice, ...)   ML99_PRIV_REC_NEXT(26, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_26(choice, ...)   ML99_PRIV_REC_NEXT(27, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_27(choice, ...)   ML99_PRIV_REC_NEXT(28, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_28(choice, ...)   ML99_PRIV_REC_NEXT(29, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_29(choice, ...)   ML99_PRIV_REC_NEXT(30, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_30(choice, ...)   ML99_PRIV_REC_NEXT(31, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_31(choice, ...)   ML99_PRIV_REC_NEXT(32, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_32(choice, ...)   ML99_PRIV_REC_NEXT(33, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_33(choice, ...)   ML99_PRIV_REC_NEXT(34, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_34(choice, ...)   ML99_PRIV_REC_NEXT(35, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_35(choice, ...)   ML99_PRIV_REC_NEXT(36, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_36(choice, ...)   ML99_PRIV_REC_NEXT(37, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_37(choice, ...)   ML99_PRIV_REC_NEXT(38, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_38(choice, ...)   ML99_PRIV_REC_NEXT(39, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_39(choice, ...)   ML99_PRIV_REC_NEXT(40, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_40(choice, ...)   ML99_PRIV_REC_NEXT(41, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_41(choice, ...)   ML99_PRIV_REC_NEXT(42, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_42(choice, ...)   ML99_PRIV_REC_NEXT(43, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_43(choice, ...)   ML99_PRIV_REC_NEXT(44, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_44(choice, ...)   ML99_PRIV_REC_NEXT(45, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_45(choice, ...)   ML99_PRIV_REC_NEXT(46, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_46(choice, ...)   ML99_PRIV_REC_NEXT(47, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_47(choice, ...)   ML99_PRIV_REC_NEXT(48, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_48(choice, ...)   ML99_PRIV_REC_NEXT(49, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_49(choice, ...)   ML99_PRIV_REC_NEXT(50, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_50(choice, ...)   ML99_PRIV_REC_NEXT(51, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_51(choice, ...)   ML99_PRIV_REC_NEXT(52, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_52(choice, ...)   ML99_PRIV_REC_NEXT(53, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_53(choice, ...)   ML99_PRIV_REC_NEXT(54, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_54(choice, ...)   ML99_PRIV_REC_NEXT(55, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_55(choice, ...)   ML99_PRIV_REC_NEXT(56, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_56(choice, ...)   ML99_PRIV_REC_NEXT(57, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_57(choice, ...)   ML99_PRIV_REC_NEXT(58, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_58(choice, ...)   ML99_PRIV_REC_NEXT(59, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_59(choice, ...)   ML99_PRIV_REC_NEXT(60, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_60(choice, ...)   ML99_PRIV_REC_NEXT(61, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_61(choice, ...)   ML99_PRIV_REC_NEXT(62, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_62(choice, ...)   ML99_PRIV_REC_NEXT(63, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_63(choice, ...)   ML99_PRIV_REC_NEXT(64, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_64(choice, ...)   ML99_PRIV_REC_NEXT(65, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_65(choice, ...)   ML99_PRIV_REC_NEXT(66, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_66(choice, ...)   ML99_PRIV_REC_NEXT(67, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_67(choice, ...)   ML99_PRIV_REC_NEXT(68, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_68(choice, ...)   ML99_PRIV_REC_NEXT(69, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_69(choice, ...)   ML99_PRIV_REC_NEXT(70, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_70(choice, ...)   ML99_PRIV_REC_NEXT(71, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_71(choice, ...)   ML99_PRIV_REC_NEXT(72, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_72(choice, ...)   ML99_PRIV_REC_NEXT(73, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_73(choice, ...)   ML99_PRIV_REC_NEXT(74, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_74(choice, ...)   ML99_PRIV_REC_NEXT(75, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_75(choice, ...)   ML99_PRIV_REC_NEXT(76, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_76(choice, ...)   ML99_PRIV_REC_NEXT(77, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_77(choice, ...)   ML99_PRIV_REC_NEXT(78, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_78(choice, ...)   ML99_PRIV_REC_NEXT(79, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_79(choice, ...)   ML99_PRIV_REC_NEXT(80, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_80(choice, ...)   ML99_PRIV_REC_NEXT(81, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_81(choice, ...)   ML99_PRIV_REC_NEXT(82, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_82(choice, ...)   ML99_PRIV_REC_NEXT(83, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_83(choice, ...)   ML99_PRIV_REC_NEXT(84, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_84(choice, ...)   ML99_PRIV_REC_NEXT(85, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_85(choice, ...)   ML99_PRIV_REC_NEXT(86, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_86(choice, ...)   ML99_PRIV_REC_NEXT(87, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_87(choice, ...)   ML99_PRIV_REC_NEXT(88, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_88(choice, ...)   ML99_PRIV_REC_NEXT(89, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_89(choice, ...)   ML99_PRIV_REC_NEXT(90, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_90(choice, ...)   ML99_PRIV_REC_NEXT(91, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_91(choice, ...)   ML99_PRIV_REC_NEXT(92, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_92(choice, ...)   ML99_PRIV_REC_NEXT(93, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_93(choice, ...)   ML99_PRIV_REC_NEXT(94, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_94(choice, ...)   ML99_PRIV_REC_NEXT(95, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_95(choice, ...)   ML99_PRIV_REC_NEXT(96, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_96(choice, ...)   ML99_PRIV_REC_NEXT(97, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_97(choice, ...)   ML99_PRIV_REC_NEXT(98, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_98(choice, ...)   ML99_PRIV_REC_NEXT(99, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_99(choice, ...)   ML99_PRIV_REC_NEXT(100, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_100(choice, ...)  ML99_PRIV_REC_NEXT(101, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_101(choice, ...)  ML99_PRIV_REC_NEXT(102, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_102(choice, ...)  ML99_PRIV_REC_NEXT(103, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_103(choice, ...)  ML99_PRIV_REC_NEXT(104, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_104(choice, ...)  ML99_PRIV_REC_NEXT(105, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_105(choice, ...)  ML99_PRIV_REC_NEXT(106, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_106(choice, ...)  ML99_PRIV_REC_NEXT(107, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_107(choice, ...)  ML99_PRIV_REC_NEXT(108, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_108(choice, ...)  ML99_PRIV_REC_NEXT(109, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_109(choice, ...)  ML99_PRIV_REC_NEXT(110, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_110(choice, ...)  ML99_PRIV_REC_NEXT(111, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_111(choice, ...)  ML99_PRIV_REC_NEXT(112, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_112(choice, ...)  ML99_PRIV_REC_NEXT(113, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_113(choice, ...)  ML99_PRIV_REC_NEXT(114, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_114(choice, ...)  ML99_PRIV_REC_NEXT(115, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_115(choice, ...)  ML99_PRIV_REC_NEXT(116, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_116(choice, ...)  ML99_PRIV_REC_NEXT(117, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_117(choice, ...)  ML99_PRIV_REC_NEXT(118, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_118(choice, ...)  ML99_PRIV_REC_NEXT(119, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_119(choice, ...)  ML99_PRIV_REC_NEXT(120, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_120(choice, ...)  ML99_PRIV_REC_NEXT(121, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_121(choice, ...)  ML99_PRIV_REC_NEXT(122, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_122(choice, ...)  ML99_PRIV_REC_NEXT(123, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_123(choice, ...)  ML99_PRIV_REC_NEXT(124, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_124(choice, ...)  ML99_PRIV_REC_NEXT(125, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_125(choice, ...)  ML99_PRIV_REC_NEXT(126, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_126(choice, ...)  ML99_PRIV_REC_NEXT(127, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_127(choice, ...)  ML99_PRIV_REC_NEXT(128, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_128(choice, ...)  ML99_PRIV_REC_NEXT(129, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_129(choice, ...)  ML99_PRIV_REC_NEXT(130, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_130(choice, ...)  ML99_PRIV_REC_NEXT(131, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_131(choice, ...)  ML99_PRIV_REC_NEXT(132, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_132(choice, ...)  ML99_PRIV_REC_NEXT(133, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_133(choice, ...)  ML99_PRIV_REC_NEXT(134, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_134(choice, ...)  ML99_PRIV_REC_NEXT(135, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_135(choice, ...)  ML99_PRIV_REC_NEXT(136, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_136(choice, ...)  ML99_PRIV_REC_NEXT(137, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_137(choice, ...)  ML99_PRIV_REC_NEXT(138, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_138(choice, ...)  ML99_PRIV_REC_NEXT(139, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_139(choice, ...)  ML99_PRIV_REC_NEXT(140, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_140(choice, ...)  ML99_PRIV_REC_NEXT(141, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_141(choice, ...)  ML99_PRIV_REC_NEXT(142, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_142(choice, ...)  ML99_PRIV_REC_NEXT(143, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_143(choice, ...)  ML99_PRIV_REC_NEXT(144, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_144(choice, ...)  ML99_PRIV_REC_NEXT(145, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_145(choice, ...)  ML99_PRIV_REC_NEXT(146, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_146(choice, ...)  ML99_PRIV_REC_NEXT(147, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_147(choice, ...)  ML99_PRIV_REC_NEXT(148, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_148(choice, ...)  ML99_PRIV_REC_NEXT(149, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_149(choice, ...)  ML99_PRIV_REC_NEXT(150, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_150(choice, ...)  ML99_PRIV_REC_NEXT(151, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_151(choice, ...)  ML99_PRIV_REC_NEXT(152, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_152(choice, ...)  ML99_PRIV_REC_NEXT(153, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_153(choice, ...)  ML99_PRIV_REC_NEXT(154, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_154(choice, ...)  ML99_PRIV_REC_NEXT(155, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_155(choice, ...)  ML99_PRIV_REC_NEXT(156, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_156(choice, ...)  ML99_PRIV_REC_NEXT(157, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_157(choice, ...)  ML99_PRIV_REC_NEXT(158, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_158(choice, ...)  ML99_PRIV_REC_NEXT(159, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_159(choice, ...)  ML99_PRIV_REC_NEXT(160, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_160(choice, ...)  ML99_PRIV_REC_NEXT(161, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_161(choice, ...)  ML99_PRIV_REC_NEXT(162, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_162(choice, ...)  ML99_PRIV_REC_NEXT(163, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_163(choice, ...)  ML99_PRIV_REC_NEXT(164, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_164(choice, ...)  ML99_PRIV_REC_NEXT(165, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_165(choice, ...)  ML99_PRIV_REC_NEXT(166, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_166(choice, ...)  ML99_PRIV_REC_NEXT(167, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_167(choice, ...)  ML99_PRIV_REC_NEXT(168, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_168(choice, ...)  ML99_PRIV_REC_NEXT(169, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_169(choice, ...)  ML99_PRIV_REC_NEXT(170, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_170(choice, ...)  ML99_PRIV_REC_NEXT(171, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_171(choice, ...)  ML99_PRIV_REC_NEXT(172, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_172(choice, ...)  ML99_PRIV_REC_NEXT(173, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_173(choice, ...)  ML99_PRIV_REC_NEXT(174, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_174(choice, ...)  ML99_PRIV_REC_NEXT(175, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_175(choice, ...)  ML99_PRIV_REC_NEXT(176, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_176(choice, ...)  ML99_PRIV_REC_NEXT(177, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_177(choice, ...)  ML99_PRIV_REC_NEXT(178, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_178(choice, ...)  ML99_PRIV_REC_NEXT(179, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_179(choice, ...)  ML99_PRIV_REC_NEXT(180, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_180(choice, ...)  ML99_PRIV_REC_NEXT(181, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_181(choice, ...)  ML99_PRIV_REC_NEXT(182, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_182(choice, ...)  ML99_PRIV_REC_NEXT(183, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_183(choice, ...)  ML99_PRIV_REC_NEXT(184, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_184(choice, ...)  ML99_PRIV_REC_NEXT(185, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_185(choice, ...)  ML99_PRIV_REC_NEXT(186, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_186(choice, ...)  ML99_PRIV_REC_NEXT(187, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_187(choice, ...)  ML99_PRIV_REC_NEXT(188, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_188(choice, ...)  ML99_PRIV_REC_NEXT(189, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_189(choice, ...)  ML99_PRIV_REC_NEXT(190, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_190(choice, ...)  ML99_PRIV_REC_NEXT(191, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_191(choice, ...)  ML99_PRIV_REC_NEXT(192, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_192(choice, ...)  ML99_PRIV_REC_NEXT(193, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_193(choice, ...)  ML99_PRIV_REC_NEXT(194, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_194(choice, ...)  ML99_PRIV_REC_NEXT(195, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_195(choice, ...)  ML99_PRIV_REC_NEXT(196, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_196(choice, ...)  ML99_PRIV_REC_NEXT(197, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_197(choice, ...)  ML99_PRIV_REC_NEXT(198, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_198(choice, ...)  ML99_PRIV_REC_NEXT(199, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_199(choice, ...)  ML99_PRIV_REC_NEXT(200, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_200(choice, ...)  ML99_PRIV_REC_NEXT(201, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_201(choice, ...)  ML99_PRIV_REC_NEXT(202, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_202(choice, ...)  ML99_PRIV_REC_NEXT(203, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_203(choice, ...)  ML99_PRIV_REC_NEXT(204, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_204(choice, ...)  ML99_PRIV_REC_NEXT(205, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_205(choice, ...)  ML99_PRIV_REC_NEXT(206, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_206(choice, ...)  ML99_PRIV_REC_NEXT(207, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_207(choice, ...)  ML99_PRIV_REC_NEXT(208, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_208(choice, ...)  ML99_PRIV_REC_NEXT(209, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_209(choice, ...)  ML99_PRIV_REC_NEXT(210, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_210(choice, ...)  ML99_PRIV_REC_NEXT(211, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_211(choice, ...)  ML99_PRIV_REC_NEXT(212, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_212(choice, ...)  ML99_PRIV_REC_NEXT(213, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_213(choice, ...)  ML99_PRIV_REC_NEXT(214, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_214(choice, ...)  ML99_PRIV_REC_NEXT(215, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_215(choice, ...)  ML99_PRIV_REC_NEXT(216, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_216(choice, ...)  ML99_PRIV_REC_NEXT(217, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_217(choice, ...)  ML99_PRIV_REC_NEXT(218, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_218(choice, ...)  ML99_PRIV_REC_NEXT(219, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_219(choice, ...)  ML99_PRIV_REC_NEXT(220, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_220(choice, ...)  ML99_PRIV_REC_NEXT(221, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_221(choice, ...)  ML99_PRIV_REC_NEXT(222, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_222(choice, ...)  ML99_PRIV_REC_NEXT(223, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_223(choice, ...)  ML99_PRIV_REC_NEXT(224, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_224(choice, ...)  ML99_PRIV_REC_NEXT(225, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_225(choice, ...)  ML99_PRIV_REC_NEXT(226, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_226(choice, ...)  ML99_PRIV_REC_NEXT(227, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_227(choice, ...)  ML99_PRIV_REC_NEXT(228, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_228(choice, ...)  ML99_PRIV_REC_NEXT(229, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_229(choice, ...)  ML99_PRIV_REC_NEXT(230, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_230(choice, ...)  ML99_PRIV_REC_NEXT(231, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_231(choice, ...)  ML99_PRIV_REC_NEXT(232, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_232(choice, ...)  ML99_PRIV_REC_NEXT(233, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_233(choice, ...)  ML99_PRIV_REC_NEXT(234, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_234(choice, ...)  ML99_PRIV_REC_NEXT(235, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_235(choice, ...)  ML99_PRIV_REC_NEXT(236, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_236(choice, ...)  ML99_PRIV_REC_NEXT(237, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_237(choice, ...)  ML99_PRIV_REC_NEXT(238, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_238(choice, ...)  ML99_PRIV_REC_NEXT(239, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_239(choice, ...)  ML99_PRIV_REC_NEXT(240, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_240(choice, ...)  ML99_PRIV_REC_NEXT(241, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_241(choice, ...)  ML99_PRIV_REC_NEXT(242, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_242(choice, ...)  ML99_PRIV_REC_NEXT(243, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_243(choice, ...)  ML99_PRIV_REC_NEXT(244, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_244(choice, ...)  ML99_PRIV_REC_NEXT(245, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_245(choice, ...)  ML99_PRIV_REC_NEXT(246, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_246(choice, ...)  ML99_PRIV_REC_NEXT(247, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_247(choice, ...)  ML99_PRIV_REC_NEXT(248, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_248(choice, ...)  ML99_PRIV_REC_NEXT(249, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_249(choice, ...)  ML99_PRIV_REC_NEXT(250, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_250(choice, ...)  ML99_PRIV_REC_NEXT(251, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_251(choice, ...)  ML99_PRIV_REC_NEXT(252, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_252(choice, ...)  ML99_PRIV_REC_NEXT(253, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_253(choice, ...)  ML99_PRIV_REC_NEXT(254, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_254(choice, ...)  ML99_PRIV_REC_NEXT(255, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_255(choice, ...)  ML99_PRIV_REC_NEXT(256, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_256(choice, ...)  ML99_PRIV_REC_NEXT(257, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_257(choice, ...)  ML99_PRIV_REC_NEXT(258, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_258(choice, ...)  ML99_PRIV_REC_NEXT(259, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_259(choice, ...)  ML99_PRIV_REC_NEXT(260, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_260(choice, ...)  ML99_PRIV_REC_NEXT(261, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_261(choice, ...)  ML99_PRIV_REC_NEXT(262, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_262(choice, ...)  ML99_PRIV_REC_NEXT(263, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_263(choice, ...)  ML99_PRIV_REC_NEXT(264, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_264(choice, ...)  ML99_PRIV_REC_NEXT(265, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_265(choice, ...)  ML99_PRIV_REC_NEXT(266, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_266(choice, ...)  ML99_PRIV_REC_NEXT(267, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_267(choice, ...)  ML99_PRIV_REC_NEXT(268, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_268(choice, ...)  ML99_PRIV_REC_NEXT(269, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_269(choice, ...)  ML99_PRIV_REC_NEXT(270, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_270(choice, ...)  ML99_PRIV_REC_NEXT(271, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_271(choice, ...)  ML99_PRIV_REC_NEXT(272, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_272(choice, ...)  ML99_PRIV_REC_NEXT(273, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_273(choice, ...)  ML99_PRIV_REC_NEXT(274, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_274(choice, ...)  ML99_PRIV_REC_NEXT(275, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_275(choice, ...)  ML99_PRIV_REC_NEXT(276, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_276(choice, ...)  ML99_PRIV_REC_NEXT(277, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_277(choice, ...)  ML99_PRIV_REC_NEXT(278, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_278(choice, ...)  ML99_PRIV_REC_NEXT(279, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_279(choice, ...)  ML99_PRIV_REC_NEXT(280, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_280(choice, ...)  ML99_PRIV_REC_NEXT(281, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_281(choice, ...)  ML99_PRIV_REC_NEXT(282, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_282(choice, ...)  ML99_PRIV_REC_NEXT(283, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_283(choice, ...)  ML99_PRIV_REC_NEXT(284, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_284(choice, ...)  ML99_PRIV_REC_NEXT(285, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_285(choice, ...)  ML99_PRIV_REC_NEXT(286, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_286(choice, ...)  ML99_PRIV_REC_NEXT(287, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_287(choice, ...)  ML99_PRIV_REC_NEXT(288, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_288(choice, ...)  ML99_PRIV_REC_NEXT(289, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_289(choice, ...)  ML99_PRIV_REC_NEXT(290, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_290(choice, ...)  ML99_PRIV_REC_NEXT(291, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_291(choice, ...)  ML99_PRIV_REC_NEXT(292, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_292(choice, ...)  ML99_PRIV_REC_NEXT(293, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_293(choice, ...)  ML99_PRIV_REC_NEXT(294, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_294(choice, ...)  ML99_PRIV_REC_NEXT(295, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_295(choice, ...)  ML99_PRIV_REC_NEXT(296, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_296(choice, ...)  ML99_PRIV_REC_NEXT(297, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_297(choice, ...)  ML99_PRIV_REC_NEXT(298, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_298(choice, ...)  ML99_PRIV_REC_NEXT(299, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_299(choice, ...)  ML99_PRIV_REC_NEXT(300, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_300(choice, ...)  ML99_PRIV_REC_NEXT(301, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_301(choice, ...)  ML99_PRIV_REC_NEXT(302, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_302(choice, ...)  ML99_PRIV_REC_NEXT(303, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_303(choice, ...)  ML99_PRIV_REC_NEXT(304, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_304(choice, ...)  ML99_PRIV_REC_NEXT(305, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_305(choice, ...)  ML99_PRIV_REC_NEXT(306, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_306(choice, ...)  ML99_PRIV_REC_NEXT(307, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_307(choice, ...)  ML99_PRIV_REC_NEXT(308, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_308(choice, ...)  ML99_PRIV_REC_NEXT(309, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_309(choice, ...)  ML99_PRIV_REC_NEXT(310, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_310(choice, ...)  ML99_PRIV_REC_NEXT(311, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_311(choice, ...)  ML99_PRIV_REC_NEXT(312, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_312(choice, ...)  ML99_PRIV_REC_NEXT(313, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_313(choice, ...)  ML99_PRIV_REC_NEXT(314, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_314(choice, ...)  ML99_PRIV_REC_NEXT(315, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_315(choice, ...)  ML99_PRIV_REC_NEXT(316, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_316(choice, ...)  ML99_PRIV_REC_NEXT(317, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_317(choice, ...)  ML99_PRIV_REC_NEXT(318, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_318(choice, ...)  ML99_PRIV_REC_NEXT(319, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_319(choice, ...)  ML99_PRIV_REC_NEXT(320, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_320(choice, ...)  ML99_PRIV_REC_NEXT(321, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_321(choice, ...)  ML99_PRIV_REC_NEXT(322, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_322(choice, ...)  ML99_PRIV_REC_NEXT(323, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_323(choice, ...)  ML99_PRIV_REC_NEXT(324, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_324(choice, ...)  ML99_PRIV_REC_NEXT(325, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_325(choice, ...)  ML99_PRIV_REC_NEXT(326, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_326(choice, ...)  ML99_PRIV_REC_NEXT(327, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_327(choice, ...)  ML99_PRIV_REC_NEXT(328, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_328(choice, ...)  ML99_PRIV_REC_NEXT(329, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_329(choice, ...)  ML99_PRIV_REC_NEXT(330, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_330(choice, ...)  ML99_PRIV_REC_NEXT(331, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_331(choice, ...)  ML99_PRIV_REC_NEXT(332, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_332(choice, ...)  ML99_PRIV_REC_NEXT(333, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_333(choice, ...)  ML99_PRIV_REC_NEXT(334, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_334(choice, ...)  ML99_PRIV_REC_NEXT(335, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_335(choice, ...)  ML99_PRIV_REC_NEXT(336, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_336(choice, ...)  ML99_PRIV_REC_NEXT(337, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_337(choice, ...)  ML99_PRIV_REC_NEXT(338, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_338(choice, ...)  ML99_PRIV_REC_NEXT(339, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_339(choice, ...)  ML99_PRIV_REC_NEXT(340, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_340(choice, ...)  ML99_PRIV_REC_NEXT(341, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_341(choice, ...)  ML99_PRIV_REC_NEXT(342, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_342(choice, ...)  ML99_PRIV_REC_NEXT(343, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_343(choice, ...)  ML99_PRIV_REC_NEXT(344, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_344(choice, ...)  ML99_PRIV_REC_NEXT(345, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_345(choice, ...)  ML99_PRIV_REC_NEXT(346, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_346(choice, ...)  ML99_PRIV_REC_NEXT(347, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_347(choice, ...)  ML99_PRIV_REC_NEXT(348, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_348(choice, ...)  ML99_PRIV_REC_NEXT(349, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_349(choice, ...)  ML99_PRIV_REC_NEXT(350, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_350(choice, ...)  ML99_PRIV_REC_NEXT(351, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_351(choice, ...)  ML99_PRIV_REC_NEXT(352, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_352(choice, ...)  ML99_PRIV_REC_NEXT(353, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_353(choice, ...)  ML99_PRIV_REC_NEXT(354, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_354(choice, ...)  ML99_PRIV_REC_NEXT(355, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_355(choice, ...)  ML99_PRIV_REC_NEXT(356, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_356(choice, ...)  ML99_PRIV_REC_NEXT(357, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_357(choice, ...)  ML99_PRIV_REC_NEXT(358, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_358(choice, ...)  ML99_PRIV_REC_NEXT(359, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_359(choice, ...)  ML99_PRIV_REC_NEXT(360, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_360(choice, ...)  ML99_PRIV_REC_NEXT(361, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_361(choice, ...)  ML99_PRIV_REC_NEXT(362, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_362(choice, ...)  ML99_PRIV_REC_NEXT(363, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_363(choice, ...)  ML99_PRIV_REC_NEXT(364, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_364(choice, ...)  ML99_PRIV_REC_NEXT(365, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_365(choice, ...)  ML99_PRIV_REC_NEXT(366, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_366(choice, ...)  ML99_PRIV_REC_NEXT(367, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_367(choice, ...)  ML99_PRIV_REC_NEXT(368, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_368(choice, ...)  ML99_PRIV_REC_NEXT(369, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_369(choice, ...)  ML99_PRIV_REC_NEXT(370, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_370(choice, ...)  ML99_PRIV_REC_NEXT(371, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_371(choice, ...)  ML99_PRIV_REC_NEXT(372, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_372(choice, ...)  ML99_PRIV_REC_NEXT(373, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_373(choice, ...)  ML99_PRIV_REC_NEXT(374, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_374(choice, ...)  ML99_PRIV_REC_NEXT(375, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_375(choice, ...)  ML99_PRIV_REC_NEXT(376, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_376(choice, ...)  ML99_PRIV_REC_NEXT(377, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_377(choice, ...)  ML99_PRIV_REC_NEXT(378, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_378(choice, ...)  ML99_PRIV_REC_NEXT(379, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_379(choice, ...)  ML99_PRIV_REC_NEXT(380, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_380(choice, ...)  ML99_PRIV_REC_NEXT(381, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_381(choice, ...)  ML99_PRIV_REC_NEXT(382, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_382(choice, ...)  ML99_PRIV_REC_NEXT(383, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_383(choice, ...)  ML99_PRIV_REC_NEXT(384, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_384(choice, ...)  ML99_PRIV_REC_NEXT(385, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_385(choice, ...)  ML99_PRIV_REC_NEXT(386, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_386(choice, ...)  ML99_PRIV_REC_NEXT(387, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_387(choice, ...)  ML99_PRIV_REC_NEXT(388, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_388(choice, ...)  ML99_PRIV_REC_NEXT(389, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_389(choice, ...)  ML99_PRIV_REC_NEXT(390, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_390(choice, ...)  ML99_PRIV_REC_NEXT(391, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_391(choice, ...)  ML99_PRIV_REC_NEXT(392, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_392(choice, ...)  ML99_PRIV_REC_NEXT(393, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_393(choice, ...)  ML99_PRIV_REC_NEXT(394, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_394(choice, ...)  ML99_PRIV_REC_NEXT(395, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_395(choice, ...)  ML99_PRIV_REC_NEXT(396, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_396(choice, ...)  ML99_PRIV_REC_NEXT(397, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_397(choice, ...)  ML99_PRIV_REC_NEXT(398, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_398(choice, ...)  ML99_PRIV_REC_NEXT(399, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_399(choice, ...)  ML99_PRIV_REC_NEXT(400, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_400(choice, ...)  ML99_PRIV_REC_NEXT(401, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_401(choice, ...)  ML99_PRIV_REC_NEXT(402, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_402(choice, ...)  ML99_PRIV_REC_NEXT(403, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_403(choice, ...)  ML99_PRIV_REC_NEXT(404, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_404(choice, ...)  ML99_PRIV_REC_NEXT(405, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_405(choice, ...)  ML99_PRIV_REC_NEXT(406, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_406(choice, ...)  ML99_PRIV_REC_NEXT(407, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_407(choice, ...)  ML99_PRIV_REC_NEXT(408, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_408(choice, ...)  ML99_PRIV_REC_NEXT(409, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_409(choice, ...)  ML99_PRIV_REC_NEXT(410, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_410(choice, ...)  ML99_PRIV_REC_NEXT(411, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_411(choice, ...)  ML99_PRIV_REC_NEXT(412, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_412(choice, ...)  ML99_PRIV_REC_NEXT(413, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_413(choice, ...)  ML99_PRIV_REC_NEXT(414, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_414(choice, ...)  ML99_PRIV_REC_NEXT(415, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_415(choice, ...)  ML99_PRIV_REC_NEXT(416, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_416(choice, ...)  ML99_PRIV_REC_NEXT(417, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_417(choice, ...)  ML99_PRIV_REC_NEXT(418, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_418(choice, ...)  ML99_PRIV_REC_NEXT(419, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_419(choice, ...)  ML99_PRIV_REC_NEXT(420, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_420(choice, ...)  ML99_PRIV_REC_NEXT(421, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_421(choice, ...)  ML99_PRIV_REC_NEXT(422, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_422(choice, ...)  ML99_PRIV_REC_NEXT(423, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_423(choice, ...)  ML99_PRIV_REC_NEXT(424, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_424(choice, ...)  ML99_PRIV_REC_NEXT(425, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_425(choice, ...)  ML99_PRIV_REC_NEXT(426, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_426(choice, ...)  ML99_PRIV_REC_NEXT(427, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_427(choice, ...)  ML99_PRIV_REC_NEXT(428, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_428(choice, ...)  ML99_PRIV_REC_NEXT(429, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_429(choice, ...)  ML99_PRIV_REC_NEXT(430, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_430(choice, ...)  ML99_PRIV_REC_NEXT(431, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_431(choice, ...)  ML99_PRIV_REC_NEXT(432, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_432(choice, ...)  ML99_PRIV_REC_NEXT(433, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_433(choice, ...)  ML99_PRIV_REC_NEXT(434, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_434(choice, ...)  ML99_PRIV_REC_NEXT(435, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_435(choice, ...)  ML99_PRIV_REC_NEXT(436, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_436(choice, ...)  ML99_PRIV_REC_NEXT(437, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_437(choice, ...)  ML99_PRIV_REC_NEXT(438, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_438(choice, ...)  ML99_PRIV_REC_NEXT(439, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_439(choice, ...)  ML99_PRIV_REC_NEXT(440, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_440(choice, ...)  ML99_PRIV_REC_NEXT(441, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_441(choice, ...)  ML99_PRIV_REC_NEXT(442, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_442(choice, ...)  ML99_PRIV_REC_NEXT(443, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_443(choice, ...)  ML99_PRIV_REC_NEXT(444, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_444(choice, ...)  ML99_PRIV_REC_NEXT(445, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_445(choice, ...)  ML99_PRIV_REC_NEXT(446, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_446(choice, ...)  ML99_PRIV_REC_NEXT(447, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_447(choice, ...)  ML99_PRIV_REC_NEXT(448, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_448(choice, ...)  ML99_PRIV_REC_NEXT(449, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_449(choice, ...)  ML99_PRIV_REC_NEXT(450, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_450(choice, ...)  ML99_PRIV_REC_NEXT(451, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_451(choice, ...)  ML99_PRIV_REC_NEXT(452, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_452(choice, ...)  ML99_PRIV_REC_NEXT(453, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_453(choice, ...)  ML99_PRIV_REC_NEXT(454, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_454(choice, ...)  ML99_PRIV_REC_NEXT(455, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_455(choice, ...)  ML99_PRIV_REC_NEXT(456, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_456(choice, ...)  ML99_PRIV_REC_NEXT(457, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_457(choice, ...)  ML99_PRIV_REC_NEXT(458, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_458(choice, ...)  ML99_PRIV_REC_NEXT(459, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_459(choice, ...)  ML99_PRIV_REC_NEXT(460, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_460(choice, ...)  ML99_PRIV_REC_NEXT(461, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_461(choice, ...)  ML99_PRIV_REC_NEXT(462, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_462(choice, ...)  ML99_PRIV_REC_NEXT(463, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_463(choice, ...)  ML99_PRIV_REC_NEXT(464, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_464(choice, ...)  ML99_PRIV_REC_NEXT(465, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_465(choice, ...)  ML99_PRIV_REC_NEXT(466, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_466(choice, ...)  ML99_PRIV_REC_NEXT(467, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_467(choice, ...)  ML99_PRIV_REC_NEXT(468, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_468(choice, ...)  ML99_PRIV_REC_NEXT(469, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_469(choice, ...)  ML99_PRIV_REC_NEXT(470, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_470(choice, ...)  ML99_PRIV_REC_NEXT(471, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_471(choice, ...)  ML99_PRIV_REC_NEXT(472, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_472(choice, ...)  ML99_PRIV_REC_NEXT(473, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_473(choice, ...)  ML99_PRIV_REC_NEXT(474, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_474(choice, ...)  ML99_PRIV_REC_NEXT(475, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_475(choice, ...)  ML99_PRIV_REC_NEXT(476, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_476(choice, ...)  ML99_PRIV_REC_NEXT(477, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_477(choice, ...)  ML99_PRIV_REC_NEXT(478, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_478(choice, ...)  ML99_PRIV_REC_NEXT(479, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_479(choice, ...)  ML99_PRIV_REC_NEXT(480, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_480(choice, ...)  ML99_PRIV_REC_NEXT(481, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_481(choice, ...)  ML99_PRIV_REC_NEXT(482, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_482(choice, ...)  ML99_PRIV_REC_NEXT(483, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_483(choice, ...)  ML99_PRIV_REC_NEXT(484, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_484(choice, ...)  ML99_PRIV_REC_NEXT(485, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_485(choice, ...)  ML99_PRIV_REC_NEXT(486, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_486(choice, ...)  ML99_PRIV_REC_NEXT(487, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_487(choice, ...)  ML99_PRIV_REC_NEXT(488, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_488(choice, ...)  ML99_PRIV_REC_NEXT(489, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_489(choice, ...)  ML99_PRIV_REC_NEXT(490, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_490(choice, ...)  ML99_PRIV_REC_NEXT(491, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_491(choice, ...)  ML99_PRIV_REC_NEXT(492, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_492(choice, ...)  ML99_PRIV_REC_NEXT(493, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_493(choice, ...)  ML99_PRIV_REC_NEXT(494, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_494(choice, ...)  ML99_PRIV_REC_NEXT(495, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_495(choice, ...)  ML99_PRIV_REC_NEXT(496, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_496(choice, ...)  ML99_PRIV_REC_NEXT(497, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_497(choice, ...)  ML99_PRIV_REC_NEXT(498, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_498(choice, ...)  ML99_PRIV_REC_NEXT(499, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_499(choice, ...)  ML99_PRIV_REC_NEXT(500, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_500(choice, ...)  ML99_PRIV_REC_NEXT(501, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_501(choice, ...)  ML99_PRIV_REC_NEXT(502, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_502(choice, ...)  ML99_PRIV_REC_NEXT(503, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_503(choice, ...)  ML99_PRIV_REC_NEXT(504, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_504(choice, ...)  ML99_PRIV_REC_NEXT(505, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_505(choice, ...)  ML99_PRIV_REC_NEXT(506, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_506(choice, ...)  ML99_PRIV_REC_NEXT(507, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_507(choice, ...)  ML99_PRIV_REC_NEXT(508, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_508(choice, ...)  ML99_PRIV_REC_NEXT(509, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_509(choice, ...)  ML99_PRIV_REC_NEXT(510, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_510(choice, ...)  ML99_PRIV_REC_NEXT(511, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_511(choice, ...)  ML99_PRIV_REC_NEXT(512, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_512(choice, ...)  ML99_PRIV_REC_NEXT(513, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_513(choice, ...)  ML99_PRIV_REC_NEXT(514, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_514(choice, ...)  ML99_PRIV_REC_NEXT(515, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_515(choice, ...)  ML99_PRIV_REC_NEXT(516, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_516(choice, ...)  ML99_PRIV_REC_NEXT(517, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_517(choice, ...)  ML99_PRIV_REC_NEXT(518, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_518(choice, ...)  ML99_PRIV_REC_NEXT(519, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_519(choice, ...)  ML99_PRIV_REC_NEXT(520, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_520(choice, ...)  ML99_PRIV_REC_NEXT(521, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_521(choice, ...)  ML99_PRIV_REC_NEXT(522, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_522(choice, ...)  ML99_PRIV_REC_NEXT(523, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_523(choice, ...)  ML99_PRIV_REC_NEXT(524, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_524(choice, ...)  ML99_PRIV_REC_NEXT(525, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_525(choice, ...)  ML99_PRIV_REC_NEXT(526, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_526(choice, ...)  ML99_PRIV_REC_NEXT(527, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_527(choice, ...)  ML99_PRIV_REC_NEXT(528, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_528(choice, ...)  ML99_PRIV_REC_NEXT(529, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_529(choice, ...)  ML99_PRIV_REC_NEXT(530, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_530(choice, ...)  ML99_PRIV_REC_NEXT(531, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_531(choice, ...)  ML99_PRIV_REC_NEXT(532, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_532(choice, ...)  ML99_PRIV_REC_NEXT(533, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_533(choice, ...)  ML99_PRIV_REC_NEXT(534, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_534(choice, ...)  ML99_PRIV_REC_NEXT(535, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_535(choice, ...)  ML99_PRIV_REC_NEXT(536, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_536(choice, ...)  ML99_PRIV_REC_NEXT(537, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_537(choice, ...)  ML99_PRIV_REC_NEXT(538, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_538(choice, ...)  ML99_PRIV_REC_NEXT(539, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_539(choice, ...)  ML99_PRIV_REC_NEXT(540, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_540(choice, ...)  ML99_PRIV_REC_NEXT(541, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_541(choice, ...)  ML99_PRIV_REC_NEXT(542, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_542(choice, ...)  ML99_PRIV_REC_NEXT(543, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_543(choice, ...)  ML99_PRIV_REC_NEXT(544, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_544(choice, ...)  ML99_PRIV_REC_NEXT(545, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_545(choice, ...)  ML99_PRIV_REC_NEXT(546, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_546(choice, ...)  ML99_PRIV_REC_NEXT(547, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_547(choice, ...)  ML99_PRIV_REC_NEXT(548, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_548(choice, ...)  ML99_PRIV_REC_NEXT(549, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_549(choice, ...)  ML99_PRIV_REC_NEXT(550, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_550(choice, ...)  ML99_PRIV_REC_NEXT(551, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_551(choice, ...)  ML99_PRIV_REC_NEXT(552, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_552(choice, ...)  ML99_PRIV_REC_NEXT(553, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_553(choice, ...)  ML99_PRIV_REC_NEXT(554, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_554(choice, ...)  ML99_PRIV_REC_NEXT(555, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_555(choice, ...)  ML99_PRIV_REC_NEXT(556, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_556(choice, ...)  ML99_PRIV_REC_NEXT(557, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_557(choice, ...)  ML99_PRIV_REC_NEXT(558, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_558(choice, ...)  ML99_PRIV_REC_NEXT(559, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_559(choice, ...)  ML99_PRIV_REC_NEXT(560, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_560(choice, ...)  ML99_PRIV_REC_NEXT(561, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_561(choice, ...)  ML99_PRIV_REC_NEXT(562, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_562(choice, ...)  ML99_PRIV_REC_NEXT(563, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_563(choice, ...)  ML99_PRIV_REC_NEXT(564, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_564(choice, ...)  ML99_PRIV_REC_NEXT(565, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_565(choice, ...)  ML99_PRIV_REC_NEXT(566, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_566(choice, ...)  ML99_PRIV_REC_NEXT(567, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_567(choice, ...)  ML99_PRIV_REC_NEXT(568, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_568(choice, ...)  ML99_PRIV_REC_NEXT(569, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_569(choice, ...)  ML99_PRIV_REC_NEXT(570, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_570(choice, ...)  ML99_PRIV_REC_NEXT(571, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_571(choice, ...)  ML99_PRIV_REC_NEXT(572, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_572(choice, ...)  ML99_PRIV_REC_NEXT(573, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_573(choice, ...)  ML99_PRIV_REC_NEXT(574, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_574(choice, ...)  ML99_PRIV_REC_NEXT(575, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_575(choice, ...)  ML99_PRIV_REC_NEXT(576, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_576(choice, ...)  ML99_PRIV_REC_NEXT(577, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_577(choice, ...)  ML99_PRIV_REC_NEXT(578, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_578(choice, ...)  ML99_PRIV_REC_NEXT(579, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_579(choice, ...)  ML99_PRIV_REC_NEXT(580, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_580(choice, ...)  ML99_PRIV_REC_NEXT(581, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_581(choice, ...)  ML99_PRIV_REC_NEXT(582, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_582(choice, ...)  ML99_PRIV_REC_NEXT(583, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_583(choice, ...)  ML99_PRIV_REC_NEXT(584, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_584(choice, ...)  ML99_PRIV_REC_NEXT(585, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_585(choice, ...)  ML99_PRIV_REC_NEXT(586, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_586(choice, ...)  ML99_PRIV_REC_NEXT(587, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_587(choice, ...)  ML99_PRIV_REC_NEXT(588, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_588(choice, ...)  ML99_PRIV_REC_NEXT(589, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_589(choice, ...)  ML99_PRIV_REC_NEXT(590, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_590(choice, ...)  ML99_PRIV_REC_NEXT(591, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_591(choice, ...)  ML99_PRIV_REC_NEXT(592, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_592(choice, ...)  ML99_PRIV_REC_NEXT(593, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_593(choice, ...)  ML99_PRIV_REC_NEXT(594, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_594(choice, ...)  ML99_PRIV_REC_NEXT(595, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_595(choice, ...)  ML99_PRIV_REC_NEXT(596, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_596(choice, ...)  ML99_PRIV_REC_NEXT(597, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_597(choice, ...)  ML99_PRIV_REC_NEXT(598, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_598(choice, ...)  ML99_PRIV_REC_NEXT(599, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_599(choice, ...)  ML99_PRIV_REC_NEXT(600, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_600(choice, ...)  ML99_PRIV_REC_NEXT(601, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_601(choice, ...)  ML99_PRIV_REC_NEXT(602, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_602(choice, ...)  ML99_PRIV_REC_NEXT(603, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_603(choice, ...)  ML99_PRIV_REC_NEXT(604, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_604(choice, ...)  ML99_PRIV_REC_NEXT(605, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_605(choice, ...)  ML99_PRIV_REC_NEXT(606, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_606(choice, ...)  ML99_PRIV_REC_NEXT(607, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_607(choice, ...)  ML99_PRIV_REC_NEXT(608, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_608(choice, ...)  ML99_PRIV_REC_NEXT(609, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_609(choice, ...)  ML99_PRIV_REC_NEXT(610, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_610(choice, ...)  ML99_PRIV_REC_NEXT(611, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_611(choice, ...)  ML99_PRIV_REC_NEXT(612, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_612(choice, ...)  ML99_PRIV_REC_NEXT(613, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_613(choice, ...)  ML99_PRIV_REC_NEXT(614, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_614(choice, ...)  ML99_PRIV_REC_NEXT(615, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_615(choice, ...)  ML99_PRIV_REC_NEXT(616, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_616(choice, ...)  ML99_PRIV_REC_NEXT(617, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_617(choice, ...)  ML99_PRIV_REC_NEXT(618, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_618(choice, ...)  ML99_PRIV_REC_NEXT(619, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_619(choice, ...)  ML99_PRIV_REC_NEXT(620, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_620(choice, ...)  ML99_PRIV_REC_NEXT(621, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_621(choice, ...)  ML99_PRIV_REC_NEXT(622, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_622(choice, ...)  ML99_PRIV_REC_NEXT(623, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_623(choice, ...)  ML99_PRIV_REC_NEXT(624, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_624(choice, ...)  ML99_PRIV_REC_NEXT(625, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_625(choice, ...)  ML99_PRIV_REC_NEXT(626, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_626(choice, ...)  ML99_PRIV_REC_NEXT(627, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_627(choice, ...)  ML99_PRIV_REC_NEXT(628, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_628(choice, ...)  ML99_PRIV_REC_NEXT(629, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_629(choice, ...)  ML99_PRIV_REC_NEXT(630, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_630(choice, ...)  ML99_PRIV_REC_NEXT(631, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_631(choice, ...)  ML99_PRIV_REC_NEXT(632, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_632(choice, ...)  ML99_PRIV_REC_NEXT(633, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_633(choice, ...)  ML99_PRIV_REC_NEXT(634, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_634(choice, ...)  ML99_PRIV_REC_NEXT(635, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_635(choice, ...)  ML99_PRIV_REC_NEXT(636, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_636(choice, ...)  ML99_PRIV_REC_NEXT(637, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_637(choice, ...)  ML99_PRIV_REC_NEXT(638, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_638(choice, ...)  ML99_PRIV_REC_NEXT(639, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_639(choice, ...)  ML99_PRIV_REC_NEXT(640, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_640(choice, ...)  ML99_PRIV_REC_NEXT(641, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_641(choice, ...)  ML99_PRIV_REC_NEXT(642, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_642(choice, ...)  ML99_PRIV_REC_NEXT(643, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_643(choice, ...)  ML99_PRIV_REC_NEXT(644, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_644(choice, ...)  ML99_PRIV_REC_NEXT(645, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_645(choice, ...)  ML99_PRIV_REC_NEXT(646, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_646(choice, ...)  ML99_PRIV_REC_NEXT(647, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_647(choice, ...)  ML99_PRIV_REC_NEXT(648, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_648(choice, ...)  ML99_PRIV_REC_NEXT(649, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_649(choice, ...)  ML99_PRIV_REC_NEXT(650, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_650(choice, ...)  ML99_PRIV_REC_NEXT(651, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_651(choice, ...)  ML99_PRIV_REC_NEXT(652, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_652(choice, ...)  ML99_PRIV_REC_NEXT(653, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_653(choice, ...)  ML99_PRIV_REC_NEXT(654, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_654(choice, ...)  ML99_PRIV_REC_NEXT(655, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_655(choice, ...)  ML99_PRIV_REC_NEXT(656, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_656(choice, ...)  ML99_PRIV_REC_NEXT(657, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_657(choice, ...)  ML99_PRIV_REC_NEXT(658, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_658(choice, ...)  ML99_PRIV_REC_NEXT(659, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_659(choice, ...)  ML99_PRIV_REC_NEXT(660, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_660(choice, ...)  ML99_PRIV_REC_NEXT(661, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_661(choice, ...)  ML99_PRIV_REC_NEXT(662, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_662(choice, ...)  ML99_PRIV_REC_NEXT(663, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_663(choice, ...)  ML99_PRIV_REC_NEXT(664, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_664(choice, ...)  ML99_PRIV_REC_NEXT(665, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_665(choice, ...)  ML99_PRIV_REC_NEXT(666, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_666(choice, ...)  ML99_PRIV_REC_NEXT(667, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_667(choice, ...)  ML99_PRIV_REC_NEXT(668, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_668(choice, ...)  ML99_PRIV_REC_NEXT(669, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_669(choice, ...)  ML99_PRIV_REC_NEXT(670, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_670(choice, ...)  ML99_PRIV_REC_NEXT(671, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_671(choice, ...)  ML99_PRIV_REC_NEXT(672, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_672(choice, ...)  ML99_PRIV_REC_NEXT(673, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_673(choice, ...)  ML99_PRIV_REC_NEXT(674, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_674(choice, ...)  ML99_PRIV_REC_NEXT(675, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_675(choice, ...)  ML99_PRIV_REC_NEXT(676, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_676(choice, ...)  ML99_PRIV_REC_NEXT(677, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_677(choice, ...)  ML99_PRIV_REC_NEXT(678, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_678(choice, ...)  ML99_PRIV_REC_NEXT(679, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_679(choice, ...)  ML99_PRIV_REC_NEXT(680, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_680(choice, ...)  ML99_PRIV_REC_NEXT(681, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_681(choice, ...)  ML99_PRIV_REC_NEXT(682, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_682(choice, ...)  ML99_PRIV_REC_NEXT(683, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_683(choice, ...)  ML99_PRIV_REC_NEXT(684, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_684(choice, ...)  ML99_PRIV_REC_NEXT(685, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_685(choice, ...)  ML99_PRIV_REC_NEXT(686, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_686(choice, ...)  ML99_PRIV_REC_NEXT(687, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_687(choice, ...)  ML99_PRIV_REC_NEXT(688, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_688(choice, ...)  ML99_PRIV_REC_NEXT(689, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_689(choice, ...)  ML99_PRIV_REC_NEXT(690, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_690(choice, ...)  ML99_PRIV_REC_NEXT(691, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_691(choice, ...)  ML99_PRIV_REC_NEXT(692, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_692(choice, ...)  ML99_PRIV_REC_NEXT(693, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_693(choice, ...)  ML99_PRIV_REC_NEXT(694, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_694(choice, ...)  ML99_PRIV_REC_NEXT(695, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_695(choice, ...)  ML99_PRIV_REC_NEXT(696, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_696(choice, ...)  ML99_PRIV_REC_NEXT(697, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_697(choice, ...)  ML99_PRIV_REC_NEXT(698, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_698(choice, ...)  ML99_PRIV_REC_NEXT(699, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_699(choice, ...)  ML99_PRIV_REC_NEXT(700, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_700(choice, ...)  ML99_PRIV_REC_NEXT(701, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_701(choice, ...)  ML99_PRIV_REC_NEXT(702, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_702(choice, ...)  ML99_PRIV_REC_NEXT(703, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_703(choice, ...)  ML99_PRIV_REC_NEXT(704, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_704(choice, ...)  ML99_PRIV_REC_NEXT(705, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_705(choice, ...)  ML99_PRIV_REC_NEXT(706, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_706(choice, ...)  ML99_PRIV_REC_NEXT(707, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_707(choice, ...)  ML99_PRIV_REC_NEXT(708, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_708(choice, ...)  ML99_PRIV_REC_NEXT(709, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_709(choice, ...)  ML99_PRIV_REC_NEXT(710, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_710(choice, ...)  ML99_PRIV_REC_NEXT(711, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_711(choice, ...)  ML99_PRIV_REC_NEXT(712, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_712(choice, ...)  ML99_PRIV_REC_NEXT(713, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_713(choice, ...)  ML99_PRIV_REC_NEXT(714, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_714(choice, ...)  ML99_PRIV_REC_NEXT(715, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_715(choice, ...)  ML99_PRIV_REC_NEXT(716, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_716(choice, ...)  ML99_PRIV_REC_NEXT(717, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_717(choice, ...)  ML99_PRIV_REC_NEXT(718, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_718(choice, ...)  ML99_PRIV_REC_NEXT(719, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_719(choice, ...)  ML99_PRIV_REC_NEXT(720, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_720(choice, ...)  ML99_PRIV_REC_NEXT(721, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_721(choice, ...)  ML99_PRIV_REC_NEXT(722, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_722(choice, ...)  ML99_PRIV_REC_NEXT(723, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_723(choice, ...)  ML99_PRIV_REC_NEXT(724, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_724(choice, ...)  ML99_PRIV_REC_NEXT(725, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_725(choice, ...)  ML99_PRIV_REC_NEXT(726, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_726(choice, ...)  ML99_PRIV_REC_NEXT(727, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_727(choice, ...)  ML99_PRIV_REC_NEXT(728, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_728(choice, ...)  ML99_PRIV_REC_NEXT(729, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_729(choice, ...)  ML99_PRIV_REC_NEXT(730, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_730(choice, ...)  ML99_PRIV_REC_NEXT(731, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_731(choice, ...)  ML99_PRIV_REC_NEXT(732, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_732(choice, ...)  ML99_PRIV_REC_NEXT(733, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_733(choice, ...)  ML99_PRIV_REC_NEXT(734, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_734(choice, ...)  ML99_PRIV_REC_NEXT(735, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_735(choice, ...)  ML99_PRIV_REC_NEXT(736, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_736(choice, ...)  ML99_PRIV_REC_NEXT(737, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_737(choice, ...)  ML99_PRIV_REC_NEXT(738, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_738(choice, ...)  ML99_PRIV_REC_NEXT(739, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_739(choice, ...)  ML99_PRIV_REC_NEXT(740, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_740(choice, ...)  ML99_PRIV_REC_NEXT(741, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_741(choice, ...)  ML99_PRIV_REC_NEXT(742, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_742(choice, ...)  ML99_PRIV_REC_NEXT(743, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_743(choice, ...)  ML99_PRIV_REC_NEXT(744, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_744(choice, ...)  ML99_PRIV_REC_NEXT(745, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_745(choice, ...)  ML99_PRIV_REC_NEXT(746, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_746(choice, ...)  ML99_PRIV_REC_NEXT(747, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_747(choice, ...)  ML99_PRIV_REC_NEXT(748, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_748(choice, ...)  ML99_PRIV_REC_NEXT(749, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_749(choice, ...)  ML99_PRIV_REC_NEXT(750, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_750(choice, ...)  ML99_PRIV_REC_NEXT(751, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_751(choice, ...)  ML99_PRIV_REC_NEXT(752, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_752(choice, ...)  ML99_PRIV_REC_NEXT(753, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_753(choice, ...)  ML99_PRIV_REC_NEXT(754, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_754(choice, ...)  ML99_PRIV_REC_NEXT(755, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_755(choice, ...)  ML99_PRIV_REC_NEXT(756, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_756(choice, ...)  ML99_PRIV_REC_NEXT(757, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_757(choice, ...)  ML99_PRIV_REC_NEXT(758, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_758(choice, ...)  ML99_PRIV_REC_NEXT(759, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_759(choice, ...)  ML99_PRIV_REC_NEXT(760, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_760(choice, ...)  ML99_PRIV_REC_NEXT(761, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_761(choice, ...)  ML99_PRIV_REC_NEXT(762, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_762(choice, ...)  ML99_PRIV_REC_NEXT(763, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_763(choice, ...)  ML99_PRIV_REC_NEXT(764, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_764(choice, ...)  ML99_PRIV_REC_NEXT(765, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_765(choice, ...)  ML99_PRIV_REC_NEXT(766, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_766(choice, ...)  ML99_PRIV_REC_NEXT(767, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_767(choice, ...)  ML99_PRIV_REC_NEXT(768, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_768(choice, ...)  ML99_PRIV_REC_NEXT(769, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_769(choice, ...)  ML99_PRIV_REC_NEXT(770, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_770(choice, ...)  ML99_PRIV_REC_NEXT(771, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_771(choice, ...)  ML99_PRIV_REC_NEXT(772, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_772(choice, ...)  ML99_PRIV_REC_NEXT(773, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_773(choice, ...)  ML99_PRIV_REC_NEXT(774, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_774(choice, ...)  ML99_PRIV_REC_NEXT(775, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_775(choice, ...)  ML99_PRIV_REC_NEXT(776, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_776(choice, ...)  ML99_PRIV_REC_NEXT(777, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_777(choice, ...)  ML99_PRIV_REC_NEXT(778, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_778(choice, ...)  ML99_PRIV_REC_NEXT(779, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_779(choice, ...)  ML99_PRIV_REC_NEXT(780, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_780(choice, ...)  ML99_PRIV_REC_NEXT(781, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_781(choice, ...)  ML99_PRIV_REC_NEXT(782, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_782(choice, ...)  ML99_PRIV_REC_NEXT(783, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_783(choice, ...)  ML99_PRIV_REC_NEXT(784, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_784(choice, ...)  ML99_PRIV_REC_NEXT(785, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_785(choice, ...)  ML99_PRIV_REC_NEXT(786, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_786(choice, ...)  ML99_PRIV_REC_NEXT(787, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_787(choice, ...)  ML99_PRIV_REC_NEXT(788, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_788(choice, ...)  ML99_PRIV_REC_NEXT(789, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_789(choice, ...)  ML99_PRIV_REC_NEXT(790, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_790(choice, ...)  ML99_PRIV_REC_NEXT(791, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_791(choice, ...)  ML99_PRIV_REC_NEXT(792, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_792(choice, ...)  ML99_PRIV_REC_NEXT(793, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_793(choice, ...)  ML99_PRIV_REC_NEXT(794, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_794(choice, ...)  ML99_PRIV_REC_NEXT(795, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_795(choice, ...)  ML99_PRIV_REC_NEXT(796, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_796(choice, ...)  ML99_PRIV_REC_NEXT(797, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_797(choice, ...)  ML99_PRIV_REC_NEXT(798, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_798(choice, ...)  ML99_PRIV_REC_NEXT(799, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_799(choice, ...)  ML99_PRIV_REC_NEXT(800, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_800(choice, ...)  ML99_PRIV_REC_NEXT(801, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_801(choice, ...)  ML99_PRIV_REC_NEXT(802, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_802(choice, ...)  ML99_PRIV_REC_NEXT(803, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_803(choice, ...)  ML99_PRIV_REC_NEXT(804, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_804(choice, ...)  ML99_PRIV_REC_NEXT(805, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_805(choice, ...)  ML99_PRIV_REC_NEXT(806, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_806(choice, ...)  ML99_PRIV_REC_NEXT(807, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_807(choice, ...)  ML99_PRIV_REC_NEXT(808, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_808(choice, ...)  ML99_PRIV_REC_NEXT(809, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_809(choice, ...)  ML99_PRIV_REC_NEXT(810, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_810(choice, ...)  ML99_PRIV_REC_NEXT(811, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_811(choice, ...)  ML99_PRIV_REC_NEXT(812, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_812(choice, ...)  ML99_PRIV_REC_NEXT(813, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_813(choice, ...)  ML99_PRIV_REC_NEXT(814, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_814(choice, ...)  ML99_PRIV_REC_NEXT(815, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_815(choice, ...)  ML99_PRIV_REC_NEXT(816, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_816(choice, ...)  ML99_PRIV_REC_NEXT(817, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_817(choice, ...)  ML99_PRIV_REC_NEXT(818, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_818(choice, ...)  ML99_PRIV_REC_NEXT(819, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_819(choice, ...)  ML99_PRIV_REC_NEXT(820, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_820(choice, ...)  ML99_PRIV_REC_NEXT(821, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_821(choice, ...)  ML99_PRIV_REC_NEXT(822, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_822(choice, ...)  ML99_PRIV_REC_NEXT(823, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_823(choice, ...)  ML99_PRIV_REC_NEXT(824, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_824(choice, ...)  ML99_PRIV_REC_NEXT(825, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_825(choice, ...)  ML99_PRIV_REC_NEXT(826, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_826(choice, ...)  ML99_PRIV_REC_NEXT(827, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_827(choice, ...)  ML99_PRIV_REC_NEXT(828, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_828(choice, ...)  ML99_PRIV_REC_NEXT(829, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_829(choice, ...)  ML99_PRIV_REC_NEXT(830, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_830(choice, ...)  ML99_PRIV_REC_NEXT(831, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_831(choice, ...)  ML99_PRIV_REC_NEXT(832, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_832(choice, ...)  ML99_PRIV_REC_NEXT(833, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_833(choice, ...)  ML99_PRIV_REC_NEXT(834, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_834(choice, ...)  ML99_PRIV_REC_NEXT(835, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_835(choice, ...)  ML99_PRIV_REC_NEXT(836, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_836(choice, ...)  ML99_PRIV_REC_NEXT(837, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_837(choice, ...)  ML99_PRIV_REC_NEXT(838, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_838(choice, ...)  ML99_PRIV_REC_NEXT(839, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_839(choice, ...)  ML99_PRIV_REC_NEXT(840, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_840(choice, ...)  ML99_PRIV_REC_NEXT(841, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_841(choice, ...)  ML99_PRIV_REC_NEXT(842, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_842(choice, ...)  ML99_PRIV_REC_NEXT(843, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_843(choice, ...)  ML99_PRIV_REC_NEXT(844, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_844(choice, ...)  ML99_PRIV_REC_NEXT(845, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_845(choice, ...)  ML99_PRIV_REC_NEXT(846, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_846(choice, ...)  ML99_PRIV_REC_NEXT(847, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_847(choice, ...)  ML99_PRIV_REC_NEXT(848, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_848(choice, ...)  ML99_PRIV_REC_NEXT(849, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_849(choice, ...)  ML99_PRIV_REC_NEXT(850, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_850(choice, ...)  ML99_PRIV_REC_NEXT(851, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_851(choice, ...)  ML99_PRIV_REC_NEXT(852, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_852(choice, ...)  ML99_PRIV_REC_NEXT(853, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_853(choice, ...)  ML99_PRIV_REC_NEXT(854, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_854(choice, ...)  ML99_PRIV_REC_NEXT(855, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_855(choice, ...)  ML99_PRIV_REC_NEXT(856, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_856(choice, ...)  ML99_PRIV_REC_NEXT(857, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_857(choice, ...)  ML99_PRIV_REC_NEXT(858, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_858(choice, ...)  ML99_PRIV_REC_NEXT(859, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_859(choice, ...)  ML99_PRIV_REC_NEXT(860, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_860(choice, ...)  ML99_PRIV_REC_NEXT(861, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_861(choice, ...)  ML99_PRIV_REC_NEXT(862, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_862(choice, ...)  ML99_PRIV_REC_NEXT(863, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_863(choice, ...)  ML99_PRIV_REC_NEXT(864, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_864(choice, ...)  ML99_PRIV_REC_NEXT(865, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_865(choice, ...)  ML99_PRIV_REC_NEXT(866, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_866(choice, ...)  ML99_PRIV_REC_NEXT(867, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_867(choice, ...)  ML99_PRIV_REC_NEXT(868, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_868(choice, ...)  ML99_PRIV_REC_NEXT(869, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_869(choice, ...)  ML99_PRIV_REC_NEXT(870, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_870(choice, ...)  ML99_PRIV_REC_NEXT(871, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_871(choice, ...)  ML99_PRIV_REC_NEXT(872, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_872(choice, ...)  ML99_PRIV_REC_NEXT(873, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_873(choice, ...)  ML99_PRIV_REC_NEXT(874, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_874(choice, ...)  ML99_PRIV_REC_NEXT(875, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_875(choice, ...)  ML99_PRIV_REC_NEXT(876, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_876(choice, ...)  ML99_PRIV_REC_NEXT(877, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_877(choice, ...)  ML99_PRIV_REC_NEXT(878, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_878(choice, ...)  ML99_PRIV_REC_NEXT(879, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_879(choice, ...)  ML99_PRIV_REC_NEXT(880, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_880(choice, ...)  ML99_PRIV_REC_NEXT(881, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_881(choice, ...)  ML99_PRIV_REC_NEXT(882, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_882(choice, ...)  ML99_PRIV_REC_NEXT(883, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_883(choice, ...)  ML99_PRIV_REC_NEXT(884, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_884(choice, ...)  ML99_PRIV_REC_NEXT(885, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_885(choice, ...)  ML99_PRIV_REC_NEXT(886, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_886(choice, ...)  ML99_PRIV_REC_NEXT(887, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_887(choice, ...)  ML99_PRIV_REC_NEXT(888, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_888(choice, ...)  ML99_PRIV_REC_NEXT(889, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_889(choice, ...)  ML99_PRIV_REC_NEXT(890, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_890(choice, ...)  ML99_PRIV_REC_NEXT(891, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_891(choice, ...)  ML99_PRIV_REC_NEXT(892, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_892(choice, ...)  ML99_PRIV_REC_NEXT(893, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_893(choice, ...)  ML99_PRIV_REC_NEXT(894, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_894(choice, ...)  ML99_PRIV_REC_NEXT(895, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_895(choice, ...)  ML99_PRIV_REC_NEXT(896, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_896(choice, ...)  ML99_PRIV_REC_NEXT(897, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_897(choice, ...)  ML99_PRIV_REC_NEXT(898, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_898(choice, ...)  ML99_PRIV_REC_NEXT(899, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_899(choice, ...)  ML99_PRIV_REC_NEXT(900, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_900(choice, ...)  ML99_PRIV_REC_NEXT(901, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_901(choice, ...)  ML99_PRIV_REC_NEXT(902, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_902(choice, ...)  ML99_PRIV_REC_NEXT(903, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_903(choice, ...)  ML99_PRIV_REC_NEXT(904, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_904(choice, ...)  ML99_PRIV_REC_NEXT(905, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_905(choice, ...)  ML99_PRIV_REC_NEXT(906, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_906(choice, ...)  ML99_PRIV_REC_NEXT(907, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_907(choice, ...)  ML99_PRIV_REC_NEXT(908, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_908(choice, ...)  ML99_PRIV_REC_NEXT(909, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_909(choice, ...)  ML99_PRIV_REC_NEXT(910, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_910(choice, ...)  ML99_PRIV_REC_NEXT(911, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_911(choice, ...)  ML99_PRIV_REC_NEXT(912, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_912(choice, ...)  ML99_PRIV_REC_NEXT(913, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_913(choice, ...)  ML99_PRIV_REC_NEXT(914, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_914(choice, ...)  ML99_PRIV_REC_NEXT(915, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_915(choice, ...)  ML99_PRIV_REC_NEXT(916, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_916(choice, ...)  ML99_PRIV_REC_NEXT(917, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_917(choice, ...)  ML99_PRIV_REC_NEXT(918, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_918(choice, ...)  ML99_PRIV_REC_NEXT(919, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_919(choice, ...)  ML99_PRIV_REC_NEXT(920, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_920(choice, ...)  ML99_PRIV_REC_NEXT(921, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_921(choice, ...)  ML99_PRIV_REC_NEXT(922, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_922(choice, ...)  ML99_PRIV_REC_NEXT(923, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_923(choice, ...)  ML99_PRIV_REC_NEXT(924, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_924(choice, ...)  ML99_PRIV_REC_NEXT(925, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_925(choice, ...)  ML99_PRIV_REC_NEXT(926, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_926(choice, ...)  ML99_PRIV_REC_NEXT(927, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_927(choice, ...)  ML99_PRIV_REC_NEXT(928, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_928(choice, ...)  ML99_PRIV_REC_NEXT(929, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_929(choice, ...)  ML99_PRIV_REC_NEXT(930, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_930(choice, ...)  ML99_PRIV_REC_NEXT(931, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_931(choice, ...)  ML99_PRIV_REC_NEXT(932, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_932(choice, ...)  ML99_PRIV_REC_NEXT(933, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_933(choice, ...)  ML99_PRIV_REC_NEXT(934, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_934(choice, ...)  ML99_PRIV_REC_NEXT(935, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_935(choice, ...)  ML99_PRIV_REC_NEXT(936, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_936(choice, ...)  ML99_PRIV_REC_NEXT(937, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_937(choice, ...)  ML99_PRIV_REC_NEXT(938, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_938(choice, ...)  ML99_PRIV_REC_NEXT(939, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_939(choice, ...)  ML99_PRIV_REC_NEXT(940, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_940(choice, ...)  ML99_PRIV_REC_NEXT(941, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_941(choice, ...)  ML99_PRIV_REC_NEXT(942, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_942(choice, ...)  ML99_PRIV_REC_NEXT(943, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_943(choice, ...)  ML99_PRIV_REC_NEXT(944, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_944(choice, ...)  ML99_PRIV_REC_NEXT(945, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_945(choice, ...)  ML99_PRIV_REC_NEXT(946, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_946(choice, ...)  ML99_PRIV_REC_NEXT(947, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_947(choice, ...)  ML99_PRIV_REC_NEXT(948, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_948(choice, ...)  ML99_PRIV_REC_NEXT(949, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_949(choice, ...)  ML99_PRIV_REC_NEXT(950, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_950(choice, ...)  ML99_PRIV_REC_NEXT(951, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_951(choice, ...)  ML99_PRIV_REC_NEXT(952, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_952(choice, ...)  ML99_PRIV_REC_NEXT(953, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_953(choice, ...)  ML99_PRIV_REC_NEXT(954, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_954(choice, ...)  ML99_PRIV_REC_NEXT(955, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_955(choice, ...)  ML99_PRIV_REC_NEXT(956, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_956(choice, ...)  ML99_PRIV_REC_NEXT(957, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_957(choice, ...)  ML99_PRIV_REC_NEXT(958, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_958(choice, ...)  ML99_PRIV_REC_NEXT(959, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_959(choice, ...)  ML99_PRIV_REC_NEXT(960, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_960(choice, ...)  ML99_PRIV_REC_NEXT(961, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_961(choice, ...)  ML99_PRIV_REC_NEXT(962, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_962(choice, ...)  ML99_PRIV_REC_NEXT(963, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_963(choice, ...)  ML99_PRIV_REC_NEXT(964, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_964(choice, ...)  ML99_PRIV_REC_NEXT(965, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_965(choice, ...)  ML99_PRIV_REC_NEXT(966, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_966(choice, ...)  ML99_PRIV_REC_NEXT(967, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_967(choice, ...)  ML99_PRIV_REC_NEXT(968, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_968(choice, ...)  ML99_PRIV_REC_NEXT(969, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_969(choice, ...)  ML99_PRIV_REC_NEXT(970, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_970(choice, ...)  ML99_PRIV_REC_NEXT(971, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_971(choice, ...)  ML99_PRIV_REC_NEXT(972, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_972(choice, ...)  ML99_PRIV_REC_NEXT(973, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_973(choice, ...)  ML99_PRIV_REC_NEXT(974, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_974(choice, ...)  ML99_PRIV_REC_NEXT(975, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_975(choice, ...)  ML99_PRIV_REC_NEXT(976, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_976(choice, ...)  ML99_PRIV_REC_NEXT(977, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_977(choice, ...)  ML99_PRIV_REC_NEXT(978, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_978(choice, ...)  ML99_PRIV_REC_NEXT(979, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_979(choice, ...)  ML99_PRIV_REC_NEXT(980, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_980(choice, ...)  ML99_PRIV_REC_NEXT(981, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_981(choice, ...)  ML99_PRIV_REC_NEXT(982, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_982(choice, ...)  ML99_PRIV_REC_NEXT(983, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_983(choice, ...)  ML99_PRIV_REC_NEXT(984, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_984(choice, ...)  ML99_PRIV_REC_NEXT(985, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_985(choice, ...)  ML99_PRIV_REC_NEXT(986, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_986(choice, ...)  ML99_PRIV_REC_NEXT(987, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_987(choice, ...)  ML99_PRIV_REC_NEXT(988, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_988(choice, ...)  ML99_PRIV_REC_NEXT(989, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_989(choice, ...)  ML99_PRIV_REC_NEXT(990, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_990(choice, ...)  ML99_PRIV_REC_NEXT(991, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_991(choice, ...)  ML99_PRIV_REC_NEXT(992, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_992(choice, ...)  ML99_PRIV_REC_NEXT(993, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_993(choice, ...)  ML99_PRIV_REC_NEXT(994, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_994(choice, ...)  ML99_PRIV_REC_NEXT(995, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_995(choice, ...)  ML99_PRIV_REC_NEXT(996, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_996(choice, ...)  ML99_PRIV_REC_NEXT(997, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_997(choice, ...)  ML99_PRIV_REC_NEXT(998, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_998(choice, ...)  ML99_PRIV_REC_NEXT(999, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_999(choice, ...)  ML99_PRIV_REC_NEXT(1000, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1000(choice, ...) ML99_PRIV_REC_NEXT(1001, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1001(choice, ...) ML99_PRIV_REC_NEXT(1002, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1002(choice, ...) ML99_PRIV_REC_NEXT(1003, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1003(choice, ...) ML99_PRIV_REC_NEXT(1004, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1004(choice, ...) ML99_PRIV_REC_NEXT(1005, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1005(choice, ...) ML99_PRIV_REC_NEXT(1006, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1006(choice, ...) ML99_PRIV_REC_NEXT(1007, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1007(choice, ...) ML99_PRIV_REC_NEXT(1008, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1008(choice, ...) ML99_PRIV_REC_NEXT(1009, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1009(choice, ...) ML99_PRIV_REC_NEXT(1010, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1010(choice, ...) ML99_PRIV_REC_NEXT(1011, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1011(choice, ...) ML99_PRIV_REC_NEXT(1012, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1012(choice, ...) ML99_PRIV_REC_NEXT(1013, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1013(choice, ...) ML99_PRIV_REC_NEXT(1014, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1014(choice, ...) ML99_PRIV_REC_NEXT(1015, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1015(choice, ...) ML99_PRIV_REC_NEXT(1016, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1016(choice, ...) ML99_PRIV_REC_NEXT(1017, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1017(choice, ...) ML99_PRIV_REC_NEXT(1018, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1018(choice, ...) ML99_PRIV_REC_NEXT(1019, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1019(choice, ...) ML99_PRIV_REC_NEXT(1020, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1020(choice, ...) ML99_PRIV_REC_NEXT(1021, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1021(choice, ...) ML99_PRIV_REC_NEXT(1022, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1022(choice, ...) ML99_PRIV_REC_NEXT(1023, choice)(__VA_ARGS__)
#define ML99_PRIV_REC_1023              ML99_PRIV_REC_DEFER(ML99_PRIV_REC_0_HOOK)()

#define ML99_PRIV_REC_0_HOOK() ML99_PRIV_REC_0

#endif // ML99_EVAL_REC_H
