/*
 * @(#)DefaultScreenScraper.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.uicapture.v1;

import java.awt.image.BufferedImage;

import java.io.File;
import java.io.IOException;

import java.util.Iterator;

import javax.imageio.ImageIO;

import org.apache.log4j.Logger;


/**
 * Writes images to files using the <tt>javax.imageio</tt> package.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   Jan 7, 2002
 */
public class DefaultScreenScraper implements IScreenScraper
{
    private static final Logger LOG = Logger.getLogger( DefaultScreenScraper.class );
    private String writerFormatName = null;
    
    private static final boolean DEBUG = true;
    
    
    //-------------------------------------------------------------------------
    // Constructors
    
    
    /**
     * Find the first writer format name we come across.
     *
     * @exception IllegalArgumentException thrown if there are no formats
     *		supported.
     */
    public DefaultScreenScraper()
    {
        String[] names = ImageIO.getWriterFormatNames();
        if (names == null || names.length <= 0)
        {
            throw new IllegalArgumentException(
                "No writer format names supported." );
        }
        
        this.writerFormatName = names[0];
        
        for (int i = 0; i < names.length; ++i)
        {
            LOG.debug( "Writer format found: "+names[i] );
        }
        
        assertFormatName();
    }
    
    
    /**
     * Use the given writerFormatName to write images to disk.
     *
     * @exception IllegalArgumentException thrown if the given name is not
     *		supported.
     */
    public DefaultScreenScraper( String writerFormatName )
    {
        this.writerFormatName = writerFormatName;
        
        assertFormatName();
    }
    
    
    //-------------------------------------------------------------------------
    // Public methods
    
    

    /**
     * Write the given image to the given file.
     *
     * @param image The screen image to write to disk.
     * @param file The File to save the image as.
     * @exception IOException thrown if there was a problem saving the image to
     *		the file.
     * @exception IllegalArgumentException if any parameter is <tt>null</tt>.
     * @exception IllegalStateException if no appropriate writer could be found.
     */
    public void writeImageToFile( BufferedImage image, File file )
            throws IOException
    {
        boolean result = ImageIO.write( image, this.writerFormatName, file );
        if (!result)
        {
            throw new IllegalStateException(
                "No appropriate writer was found." );
        }
    }
    
    
    
    /**
     * Discover the file extention for images created by this scraper.
     * This does not include a '.'.
     *
     * @return the extention for files this scraper writes.
     */
    public String getFileExtention()
    {
        return this.writerFormatName;
    }
    
    
    
    //-------------------------------------------------------------------------
    // Protected methods
    
    
    /**
     * Discover if the current format name is supported.
     *
     * @exception IllegalArgumentException thrown if the current name is not
     *		supported.
     */
    protected void assertFormatName()
    {
        Iterator iter = ImageIO.getImageWritersByFormatName(
            this.writerFormatName );
        if (iter == null || !iter.hasNext())
        {
            throw new IllegalArgumentException(
                "Image Format " + this.writerFormatName +
                " is not supported in the current runtime system." );
        }
    }
}

