"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addTranslation = addTranslation;
exports.getDefaultLocale = getDefaultLocale;
exports.getFormats = getFormats;
exports.getLocale = getLocale;
exports.getRegisteredLocales = getRegisteredLocales;
exports.getTranslation = getTranslation;
exports.init = init;
exports.load = load;
exports.normalizeLocale = normalizeLocale;
exports.setDefaultLocale = setDefaultLocale;
exports.setFormats = setFormats;
exports.setLocale = setLocale;
exports.translate = translate;
var _intlFormatCache = _interopRequireDefault(require("intl-format-cache"));
var _intlMessageformat = _interopRequireDefault(require("intl-messageformat"));
var _intlRelativeformat = _interopRequireDefault(require("intl-relativeformat"));
var _formats = require("./formats");
var _helper = require("./helper");
var _pseudo_locale = require("./pseudo_locale");
require("./locales.js");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

// Add all locale data to `IntlMessageFormat`.

const EN_LOCALE = 'en';
const translationsForLocale = {};
const getMessageFormat = (0, _intlFormatCache.default)(_intlMessageformat.default);

/* A locale code is made of several components:
 *    * lang: The two- and three-letter lower-case language code follows the ISO 639-1 and ISO 639-2/3 standards, respectively.
 *    * script: The optional four-letter title-case code follows the ISO 15924 standard for representing writing systems.
 *    * region: The two-letter upper-case region code follows the ISO 3166-1 alpha-2 standard.
 *
 * Ref: https://www.rfc-editor.org/rfc/rfc5646.txt
 * Note: While case carries no distinction with locale codes, proper formatting is recommended.
 */
const localeParser = /^(?<lang>[a-z]{2,3})(?:-(?<script>[a-z]{4}))?(?:-(?<region>[a-z]{2}|[0-9]{3}))?(?:[_@\-].*)?$/i;
let defaultLocale = EN_LOCALE;
let currentLocale = EN_LOCALE;
let formats = _formats.formats;
_intlMessageformat.default.defaultLocale = defaultLocale;
_intlRelativeformat.default.defaultLocale = defaultLocale;

/**
 * Returns message by the given message id.
 * @param id - path to the message
 */
function getMessageById(id) {
  const translation = getTranslation();
  return translation.messages ? translation.messages[id] : undefined;
}

/**
 * Normalizes locale to make it consistent with IntlMessageFormat locales
 * @param locale
 */
function normalizeLocale(locale) {
  var _localeParser$exec;
  const {
    lang,
    script,
    region
  } = ((_localeParser$exec = localeParser.exec(locale)) === null || _localeParser$exec === void 0 ? void 0 : _localeParser$exec.groups) || {};
  // If parsing failed or the language code was not extracted, return the locale
  if (!lang) return locale;
  const parts = [lang.toLowerCase()];
  if (script) parts.push(script[0].toUpperCase() + script.slice(1).toLowerCase());
  if (region) parts.push(region.toUpperCase());
  return parts.join('-');
}

/**
 * Provides a way to register translations with the engine
 * @param newTranslation
 * @param [locale = messages.locale]
 */
function addTranslation(newTranslation, locale = newTranslation.locale) {
  if (!locale || !(0, _helper.isString)(locale)) {
    throw new Error('[I18n] A `locale` must be a non-empty string to add messages.');
  }
  if (newTranslation.locale && newTranslation.locale !== locale) {
    throw new Error('[I18n] A `locale` in the translation object is different from the one provided as a second argument.');
  }
  const normalizedLocale = normalizeLocale(locale);
  const existingTranslation = translationsForLocale[normalizedLocale] || {
    messages: {}
  };
  translationsForLocale[normalizedLocale] = {
    formats: newTranslation.formats || existingTranslation.formats,
    locale: newTranslation.locale || existingTranslation.locale,
    messages: {
      ...existingTranslation.messages,
      ...newTranslation.messages
    }
  };
}

/**
 * Returns messages for the current language
 */
function getTranslation() {
  return translationsForLocale[currentLocale] || {
    messages: {}
  };
}

/**
 * Tells the engine which language to use by given language key
 * @param locale
 */
function setLocale(locale) {
  if (!locale || !(0, _helper.isString)(locale)) {
    throw new Error('[I18n] A `locale` must be a non-empty string.');
  }
  currentLocale = normalizeLocale(locale);
}

/**
 * Returns the current locale
 */
function getLocale() {
  return currentLocale;
}

/**
 * Tells the library which language to fallback when missing translations
 * @param locale
 */
function setDefaultLocale(locale) {
  if (!locale || !(0, _helper.isString)(locale)) {
    throw new Error('[I18n] A `locale` must be a non-empty string.');
  }
  defaultLocale = normalizeLocale(locale);
  _intlMessageformat.default.defaultLocale = defaultLocale;
  _intlRelativeformat.default.defaultLocale = defaultLocale;
}
function getDefaultLocale() {
  return defaultLocale;
}

/**
 * Supplies a set of options to the underlying formatter
 * [Default format options used as the prototype of the formats]
 * {@link https://github.com/yahoo/intl-messageformat/blob/master/src/core.js#L62}
 * These are used when constructing the internal Intl.NumberFormat
 * and Intl.DateTimeFormat instances.
 * @param newFormats
 * @param [newFormats.number]
 * @param [newFormats.date]
 * @param [newFormats.time]
 */
function setFormats(newFormats) {
  if (!(0, _helper.isObject)(newFormats) || !(0, _helper.hasValues)(newFormats)) {
    throw new Error('[I18n] A `formats` must be a non-empty object.');
  }
  formats = (0, _helper.mergeAll)(formats, newFormats);
}

/**
 * Returns current formats
 */
function getFormats() {
  return formats;
}

/**
 * Returns array of locales having translations
 */
function getRegisteredLocales() {
  return Object.keys(translationsForLocale);
}
/**
 * Translate message by id
 * @param id - translation id to be translated
 * @param [options]
 * @param [options.values] - values to pass into translation
 * @param [options.defaultMessage] - will be used unless translation was successful
 */
function translate(id, {
  values = {},
  defaultMessage
}) {
  const shouldUsePseudoLocale = (0, _pseudo_locale.isPseudoLocale)(currentLocale);
  if (!id || !(0, _helper.isString)(id)) {
    throw new Error('[I18n] An `id` must be a non-empty string to translate a message.');
  }
  const message = shouldUsePseudoLocale ? defaultMessage : getMessageById(id);
  if (!message && !defaultMessage) {
    throw new Error(`[I18n] Cannot format message: "${id}". Default message must be provided.`);
  }
  if (message) {
    try {
      // We should call `format` even for messages without any value references
      // to let it handle escaped curly braces `\\{` that are the part of the text itself
      // and not value reference boundaries.
      const formattedMessage = getMessageFormat(message, getLocale(), getFormats()).format(values);
      return shouldUsePseudoLocale ? (0, _pseudo_locale.translateUsingPseudoLocale)(formattedMessage) : formattedMessage;
    } catch (e) {
      throw new Error(`[I18n] Error formatting message: "${id}" for locale: "${getLocale()}".\n${e}`);
    }
  }
  try {
    const msg = getMessageFormat(defaultMessage, getDefaultLocale(), getFormats());
    return msg.format(values);
  } catch (e) {
    throw new Error(`[I18n] Error formatting the default message for: "${id}".\n${e}`);
  }
}

/**
 * Initializes the engine
 * @param newTranslation
 */
function init(newTranslation) {
  if (!newTranslation) {
    return;
  }
  addTranslation(newTranslation);
  if (newTranslation.locale) {
    setLocale(newTranslation.locale);
  }
  if (newTranslation.formats) {
    setFormats(newTranslation.formats);
  }
}

/**
 * Loads JSON with translations from the specified URL and initializes i18n engine with them.
 * @param translationsUrl URL pointing to the JSON bundle with translations.
 */
async function load(translationsUrl) {
  // Once this package is integrated into core OpenSearch Dashboards we should switch to an abstraction
  // around `fetch` provided by the platform, e.g. `kfetch`.
  const response = await fetch(translationsUrl, {
    credentials: 'same-origin'
  });
  if (response.status >= 300) {
    throw new Error(`Translations request failed with status code: ${response.status}`);
  }
  const data = await response.json();
  if (data.warning) {
    // Store the warning to be displayed after core system setup
    window.__i18nWarning = data.warning;
  }
  init(data.translations);
}