/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.services;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.protobuf.ByteString;
import com.google.protobuf.Duration;
import com.google.protobuf.MessageLite;
import com.google.protobuf.util.Durations;
import com.google.protobuf.util.Timestamps;
import io.grpc.Attributes;
import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.Context;
import io.grpc.Deadline;
import io.grpc.ForwardingClientCall;
import io.grpc.ForwardingClientCallListener;
import io.grpc.ForwardingServerCall;
import io.grpc.ForwardingServerCallListener;
import io.grpc.Grpc;
import io.grpc.InternalMetadata;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import io.grpc.binarylog.v1.Address;
import io.grpc.binarylog.v1.ClientHeader;
import io.grpc.binarylog.v1.GrpcLogEntry;
import io.grpc.binarylog.v1.Message;
import io.grpc.binarylog.v1.Metadata;
import io.grpc.binarylog.v1.ServerHeader;
import io.grpc.binarylog.v1.Trailer;
import io.grpc.services.BinaryLogProvider;
import io.grpc.services.BinaryLogSink;
import io.grpc.services.InetAddressUtil;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
final class BinlogHelper {
    private static final Logger logger = Logger.getLogger(BinlogHelper.class.getName());
    static final Metadata.Key<byte[]> STATUS_DETAILS_KEY = Metadata.Key.of((String)"grpc-status-details-bin", (Metadata.BinaryMarshaller)Metadata.BINARY_BYTE_MARSHALLER);
    @VisibleForTesting
    final SinkWriter writer;
    private static final Set<String> NEVER_INCLUDED_METADATA = new HashSet<String>(Collections.singletonList(STATUS_DETAILS_KEY.name()));
    private static final Set<String> ALWAYS_INCLUDED_METADATA = new HashSet<String>(Collections.singletonList("grpc-trace-bin"));

    @VisibleForTesting
    BinlogHelper(SinkWriter writer) {
        this.writer = writer;
    }

    static SocketAddress getPeerSocket(Attributes streamAttributes) {
        return (SocketAddress)streamAttributes.get(Grpc.TRANSPORT_ATTR_REMOTE_ADDR);
    }

    private static Deadline min(@Nullable Deadline deadline0, @Nullable Deadline deadline1) {
        if (deadline0 == null) {
            return deadline1;
        }
        if (deadline1 == null) {
            return deadline0;
        }
        return deadline0.minimum(deadline1);
    }

    public ClientInterceptor getClientInterceptor(final long callId) {
        return new ClientInterceptor(){
            boolean trailersOnlyResponse = true;

            public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(final MethodDescriptor<ReqT, RespT> method, CallOptions callOptions, Channel next) {
                final AtomicLong seq = new AtomicLong(1L);
                final String methodName = method.getFullMethodName();
                final String authority = next.authority();
                final Deadline deadline = BinlogHelper.min(callOptions.getDeadline(), Context.current().getDeadline());
                return new ForwardingClientCall.SimpleForwardingClientCall<ReqT, RespT>(next.newCall(method, callOptions)){

                    public void start(ClientCall.Listener<RespT> responseListener, Metadata headers) {
                        Duration timeout = deadline == null ? null : Durations.fromNanos((long)deadline.timeRemaining(TimeUnit.NANOSECONDS));
                        BinlogHelper.this.writer.logClientHeader(seq.getAndIncrement(), methodName, authority, timeout, headers, GrpcLogEntry.Logger.LOGGER_CLIENT, callId, null);
                        ForwardingClientCallListener.SimpleForwardingClientCallListener wListener = new ForwardingClientCallListener.SimpleForwardingClientCallListener<RespT>(responseListener){

                            public void onMessage(RespT message) {
                                BinlogHelper.this.writer.logRpcMessage(seq.getAndIncrement(), GrpcLogEntry.EventType.EVENT_TYPE_SERVER_MESSAGE, method.getResponseMarshaller(), message, GrpcLogEntry.Logger.LOGGER_CLIENT, callId);
                                super.onMessage(message);
                            }

                            public void onHeaders(Metadata headers) {
                                trailersOnlyResponse = false;
                                BinlogHelper.this.writer.logServerHeader(seq.getAndIncrement(), headers, GrpcLogEntry.Logger.LOGGER_CLIENT, callId, BinlogHelper.getPeerSocket(this.getAttributes()));
                                super.onHeaders(headers);
                            }

                            public void onClose(Status status, Metadata trailers) {
                                SocketAddress peer = trailersOnlyResponse ? BinlogHelper.getPeerSocket(this.getAttributes()) : null;
                                BinlogHelper.this.writer.logTrailer(seq.getAndIncrement(), status, trailers, GrpcLogEntry.Logger.LOGGER_CLIENT, callId, peer);
                                super.onClose(status, trailers);
                            }
                        };
                        super.start((ClientCall.Listener)wListener, headers);
                    }

                    public void sendMessage(ReqT message) {
                        BinlogHelper.this.writer.logRpcMessage(seq.getAndIncrement(), GrpcLogEntry.EventType.EVENT_TYPE_CLIENT_MESSAGE, method.getRequestMarshaller(), message, GrpcLogEntry.Logger.LOGGER_CLIENT, callId);
                        super.sendMessage(message);
                    }

                    public void halfClose() {
                        BinlogHelper.this.writer.logHalfClose(seq.getAndIncrement(), GrpcLogEntry.Logger.LOGGER_CLIENT, callId);
                        super.halfClose();
                    }

                    public void cancel(String message, Throwable cause) {
                        BinlogHelper.this.writer.logCancel(seq.getAndIncrement(), GrpcLogEntry.Logger.LOGGER_CLIENT, callId);
                        super.cancel(message, cause);
                    }
                };
            }
        };
    }

    public ServerInterceptor getServerInterceptor(final long callId) {
        return new ServerInterceptor(){

            public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(final ServerCall<ReqT, RespT> call, Metadata headers, ServerCallHandler<ReqT, RespT> next) {
                final AtomicLong seq = new AtomicLong(1L);
                SocketAddress peer = BinlogHelper.getPeerSocket(call.getAttributes());
                String methodName = call.getMethodDescriptor().getFullMethodName();
                String authority = call.getAuthority();
                Deadline deadline = Context.current().getDeadline();
                Duration timeout = deadline == null ? null : Durations.fromNanos((long)deadline.timeRemaining(TimeUnit.NANOSECONDS));
                BinlogHelper.this.writer.logClientHeader(seq.getAndIncrement(), methodName, authority, timeout, headers, GrpcLogEntry.Logger.LOGGER_SERVER, callId, peer);
                ForwardingServerCall.SimpleForwardingServerCall wCall = new ForwardingServerCall.SimpleForwardingServerCall<ReqT, RespT>(call){

                    public void sendMessage(RespT message) {
                        BinlogHelper.this.writer.logRpcMessage(seq.getAndIncrement(), GrpcLogEntry.EventType.EVENT_TYPE_SERVER_MESSAGE, call.getMethodDescriptor().getResponseMarshaller(), message, GrpcLogEntry.Logger.LOGGER_SERVER, callId);
                        super.sendMessage(message);
                    }

                    public void sendHeaders(Metadata headers) {
                        BinlogHelper.this.writer.logServerHeader(seq.getAndIncrement(), headers, GrpcLogEntry.Logger.LOGGER_SERVER, callId, null);
                        super.sendHeaders(headers);
                    }

                    public void close(Status status, Metadata trailers) {
                        BinlogHelper.this.writer.logTrailer(seq.getAndIncrement(), status, trailers, GrpcLogEntry.Logger.LOGGER_SERVER, callId, null);
                        super.close(status, trailers);
                    }
                };
                return new ForwardingServerCallListener.SimpleForwardingServerCallListener<ReqT>(next.startCall((ServerCall)wCall, headers)){

                    public void onMessage(ReqT message) {
                        BinlogHelper.this.writer.logRpcMessage(seq.getAndIncrement(), GrpcLogEntry.EventType.EVENT_TYPE_CLIENT_MESSAGE, call.getMethodDescriptor().getRequestMarshaller(), message, GrpcLogEntry.Logger.LOGGER_SERVER, callId);
                        super.onMessage(message);
                    }

                    public void onHalfClose() {
                        BinlogHelper.this.writer.logHalfClose(seq.getAndIncrement(), GrpcLogEntry.Logger.LOGGER_SERVER, callId);
                        super.onHalfClose();
                    }

                    public void onCancel() {
                        BinlogHelper.this.writer.logCancel(seq.getAndIncrement(), GrpcLogEntry.Logger.LOGGER_SERVER, callId);
                        super.onCancel();
                    }
                };
            }
        };
    }

    @VisibleForTesting
    static Address socketToProto(SocketAddress address) {
        Preconditions.checkNotNull((Object)address, (Object)"address");
        Address.Builder builder = Address.newBuilder();
        if (address instanceof InetSocketAddress) {
            InetAddress inetAddress = ((InetSocketAddress)address).getAddress();
            if (inetAddress instanceof Inet4Address) {
                builder.setType(Address.Type.TYPE_IPV4).setAddress(InetAddressUtil.toAddrString(inetAddress));
            } else if (inetAddress instanceof Inet6Address) {
                builder.setType(Address.Type.TYPE_IPV6).setAddress(InetAddressUtil.toAddrString(inetAddress));
            } else {
                logger.log(Level.SEVERE, "unknown type of InetSocketAddress: {}", address);
                builder.setAddress(address.toString());
            }
            builder.setIpPort(((InetSocketAddress)address).getPort());
        } else if (address.getClass().getName().equals("io.netty.channel.unix.DomainSocketAddress")) {
            builder.setType(Address.Type.TYPE_UNIX).setAddress(address.toString());
        } else {
            builder.setType(Address.Type.TYPE_UNKNOWN).setAddress(address.toString());
        }
        return builder.build();
    }

    @VisibleForTesting
    static MaybeTruncated<Metadata.Builder> createMetadataProto(Metadata metadata, int maxHeaderBytes) {
        Preconditions.checkNotNull((Object)metadata, (Object)"metadata");
        Preconditions.checkArgument((maxHeaderBytes >= 0 ? 1 : 0) != 0, (Object)"maxHeaderBytes must be non negative");
        Metadata.Builder metaBuilder = io.grpc.binarylog.v1.Metadata.newBuilder();
        byte[][] serialized = InternalMetadata.serialize((Metadata)metadata);
        boolean truncated = false;
        if (serialized != null) {
            int curBytes = 0;
            for (int i = 0; i < serialized.length; i += 2) {
                String key = new String(serialized[i], Charsets.UTF_8);
                byte[] value = serialized[i + 1];
                if (NEVER_INCLUDED_METADATA.contains(key)) continue;
                boolean forceInclude = ALWAYS_INCLUDED_METADATA.contains(key);
                int bytesAfterAdd = curBytes + key.length() + value.length;
                if (!forceInclude && bytesAfterAdd > maxHeaderBytes) {
                    truncated = true;
                    continue;
                }
                metaBuilder.addEntryBuilder().setKey(key).setValue(ByteString.copyFrom((byte[])value));
                if (forceInclude) continue;
                curBytes = bytesAfterAdd;
            }
        }
        return new MaybeTruncated<Metadata.Builder>(metaBuilder, truncated);
    }

    @VisibleForTesting
    static MaybeTruncated<Message.Builder> createMessageProto(byte[] message, int maxMessageBytes) {
        Preconditions.checkNotNull((Object)message, (Object)"message");
        Preconditions.checkArgument((maxMessageBytes >= 0 ? 1 : 0) != 0, (Object)"maxMessageBytes must be non negative");
        Message.Builder msgBuilder = Message.newBuilder().setLength(message.length);
        if (maxMessageBytes > 0) {
            int desiredBytes = Math.min(maxMessageBytes, message.length);
            msgBuilder.setData(ByteString.copyFrom((byte[])message, (int)0, (int)desiredBytes));
        }
        return new MaybeTruncated<Message.Builder>(msgBuilder, maxMessageBytes < message.length);
    }

    static final class MaybeTruncated<T> {
        T proto;
        boolean truncated;

        private MaybeTruncated(T proto, boolean truncated) {
            this.proto = proto;
            this.truncated = truncated;
        }
    }

    static final class FactoryImpl
    implements Factory {
        private final BinlogHelper globalLog;
        private final Map<String, BinlogHelper> perServiceLogs;
        private final Map<String, BinlogHelper> perMethodLogs;
        private final Set<String> blacklistedMethods;

        @VisibleForTesting
        FactoryImpl(BinaryLogSink sink, String configurationString) {
            Preconditions.checkNotNull((Object)sink, (Object)"sink");
            BinlogHelper globalLog = null;
            HashMap<String, BinlogHelper> perServiceLogs = new HashMap<String, BinlogHelper>();
            HashMap<String, BinlogHelper> perMethodLogs = new HashMap<String, BinlogHelper>();
            HashSet<String> blacklistedMethods = new HashSet<String>();
            if (configurationString != null && configurationString.length() > 0) {
                for (String configuration : Splitter.on((char)',').split((CharSequence)configurationString)) {
                    String binlogOptionStr;
                    String methodOrSvc;
                    int leftCurly = configuration.indexOf(123);
                    if (leftCurly == -1) {
                        methodOrSvc = configuration;
                        binlogOptionStr = null;
                    } else {
                        int rightCurly = configuration.indexOf(125, leftCurly);
                        if (rightCurly != configuration.length() - 1) {
                            throw new IllegalArgumentException("Illegal log config pattern: " + configuration);
                        }
                        methodOrSvc = configuration.substring(0, leftCurly);
                        binlogOptionStr = configuration.substring(leftCurly + 1, configuration.length() - 1);
                    }
                    if (methodOrSvc.isEmpty()) {
                        throw new IllegalArgumentException("Illegal log config pattern: " + configuration);
                    }
                    if (methodOrSvc.equals("*")) {
                        Preconditions.checkState((globalLog == null ? 1 : 0) != 0, (Object)("Duplicate entry, this is fatal: " + configuration));
                        globalLog = FactoryImpl.createBinaryLog(sink, binlogOptionStr);
                        logger.log(Level.INFO, "Global binlog: {0}", binlogOptionStr);
                        continue;
                    }
                    if (FactoryImpl.isServiceGlob(methodOrSvc)) {
                        String service = MethodDescriptor.extractFullServiceName((String)methodOrSvc);
                        Preconditions.checkState((!perServiceLogs.containsKey(service) ? 1 : 0) != 0, (Object)("Duplicate entry, this is fatal: " + configuration));
                        perServiceLogs.put(service, FactoryImpl.createBinaryLog(sink, binlogOptionStr));
                        logger.log(Level.INFO, "Service binlog: service={0} config={1}", new Object[]{service, binlogOptionStr});
                        continue;
                    }
                    if (methodOrSvc.startsWith("-")) {
                        String blacklistedMethod = methodOrSvc.substring(1);
                        if (blacklistedMethod.length() == 0) continue;
                        Preconditions.checkState((!blacklistedMethods.contains(blacklistedMethod) ? 1 : 0) != 0, (Object)("Duplicate entry, this is fatal: " + configuration));
                        Preconditions.checkState((!perMethodLogs.containsKey(blacklistedMethod) ? 1 : 0) != 0, (Object)("Duplicate entry, this is fatal: " + configuration));
                        blacklistedMethods.add(blacklistedMethod);
                        continue;
                    }
                    Preconditions.checkState((!perMethodLogs.containsKey(methodOrSvc) ? 1 : 0) != 0, (Object)("Duplicate entry, this is fatal: " + configuration));
                    Preconditions.checkState((!blacklistedMethods.contains(methodOrSvc) ? 1 : 0) != 0, (Object)("Duplicate entry, this method was blacklisted: " + configuration));
                    perMethodLogs.put(methodOrSvc, FactoryImpl.createBinaryLog(sink, binlogOptionStr));
                    logger.log(Level.INFO, "Method binlog: method={0} config={1}", new Object[]{methodOrSvc, binlogOptionStr});
                }
            }
            this.globalLog = globalLog;
            this.perServiceLogs = Collections.unmodifiableMap(perServiceLogs);
            this.perMethodLogs = Collections.unmodifiableMap(perMethodLogs);
            this.blacklistedMethods = Collections.unmodifiableSet(blacklistedMethods);
        }

        @Override
        public BinlogHelper getLog(String fullMethodName) {
            if (this.blacklistedMethods.contains(fullMethodName)) {
                return null;
            }
            BinlogHelper methodLog = this.perMethodLogs.get(fullMethodName);
            if (methodLog != null) {
                return methodLog;
            }
            BinlogHelper serviceLog = this.perServiceLogs.get(MethodDescriptor.extractFullServiceName((String)fullMethodName));
            if (serviceLog != null) {
                return serviceLog;
            }
            return this.globalLog;
        }

        @Nullable
        @VisibleForTesting
        static BinlogHelper createBinaryLog(BinaryLogSink sink, @Nullable String logConfig) {
            if (logConfig == null) {
                return new BinlogHelper(new SinkWriterImpl(sink, TimeProvider.SYSTEM_TIME_PROVIDER, Integer.MAX_VALUE, Integer.MAX_VALUE));
            }
            try {
                int maxMsgBytes;
                int maxHeaderBytes;
                String[] parts = logConfig.split(";", 2);
                if (parts.length == 2) {
                    if (!parts[0].startsWith("h") || !parts[1].startsWith("m")) {
                        throw new IllegalArgumentException("Illegal log config pattern");
                    }
                    maxHeaderBytes = FactoryImpl.optionalInt(parts[0].substring(1));
                    maxMsgBytes = FactoryImpl.optionalInt(parts[1].substring(1));
                } else if (parts[0].startsWith("h")) {
                    maxHeaderBytes = FactoryImpl.optionalInt(parts[0].substring(1));
                    maxMsgBytes = 0;
                } else if (parts[0].startsWith("m")) {
                    maxHeaderBytes = 0;
                    maxMsgBytes = FactoryImpl.optionalInt(parts[0].substring(1));
                } else {
                    throw new IllegalArgumentException("Illegal log config pattern");
                }
                return new BinlogHelper(new SinkWriterImpl(sink, TimeProvider.SYSTEM_TIME_PROVIDER, maxHeaderBytes, maxMsgBytes));
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Illegal log config pattern");
            }
        }

        static String checkDigits(String s) {
            for (int i = 0; i < s.length(); ++i) {
                char c = s.charAt(i);
                if (c >= '0' && '9' >= c) continue;
                throw new IllegalArgumentException("Illegal log config pattern");
            }
            return s;
        }

        static int optionalInt(String s) {
            if (s.isEmpty()) {
                return Integer.MAX_VALUE;
            }
            if (!s.startsWith(":")) {
                throw new IllegalArgumentException("Illegal log config pattern");
            }
            s = FactoryImpl.checkDigits(s.substring(1));
            return Integer.parseInt(s);
        }

        static boolean isServiceGlob(String input) {
            return input.endsWith("/*");
        }
    }

    static interface Factory {
        @Nullable
        public BinlogHelper getLog(String var1);
    }

    static interface TimeProvider {
        public static final TimeProvider SYSTEM_TIME_PROVIDER = new TimeProvider(){

            @Override
            public long currentTimeNanos() {
                return TimeUnit.MILLISECONDS.toNanos(System.currentTimeMillis());
            }
        };

        public long currentTimeNanos();
    }

    static abstract class SinkWriter {
        SinkWriter() {
        }

        abstract void logClientHeader(long var1, String var3, @Nullable String var4, @Nullable Duration var5, Metadata var6, GrpcLogEntry.Logger var7, long var8, @Nullable SocketAddress var10);

        abstract void logServerHeader(long var1, Metadata var3, GrpcLogEntry.Logger var4, long var5, @Nullable SocketAddress var7);

        abstract void logTrailer(long var1, Status var3, Metadata var4, GrpcLogEntry.Logger var5, long var6, @Nullable SocketAddress var8);

        abstract <T> void logRpcMessage(long var1, GrpcLogEntry.EventType var3, MethodDescriptor.Marshaller<T> var4, T var5, GrpcLogEntry.Logger var6, long var7);

        abstract void logHalfClose(long var1, GrpcLogEntry.Logger var3, long var4);

        abstract void logCancel(long var1, GrpcLogEntry.Logger var3, long var4);

        abstract int getMaxHeaderBytes();

        abstract int getMaxMessageBytes();
    }

    static final class SinkWriterImpl
    extends SinkWriter {
        private final BinaryLogSink sink;
        private TimeProvider timeProvider;
        private final int maxHeaderBytes;
        private final int maxMessageBytes;

        SinkWriterImpl(BinaryLogSink sink, TimeProvider timeProvider, int maxHeaderBytes, int maxMessageBytes) {
            this.sink = sink;
            this.timeProvider = timeProvider;
            this.maxHeaderBytes = maxHeaderBytes;
            this.maxMessageBytes = maxMessageBytes;
        }

        GrpcLogEntry.Builder newTimestampedBuilder() {
            long epochNanos = this.timeProvider.currentTimeNanos();
            return GrpcLogEntry.newBuilder().setTimestamp(Timestamps.fromNanos((long)epochNanos));
        }

        @Override
        void logClientHeader(long seq, String methodName, @Nullable String authority, @Nullable Duration timeout, Metadata metadata, GrpcLogEntry.Logger logger, long callId, @Nullable SocketAddress peerAddress) {
            Preconditions.checkArgument((methodName != null ? 1 : 0) != 0, (Object)"methodName can not be null");
            Preconditions.checkArgument((!methodName.startsWith("/") ? 1 : 0) != 0, (Object)"in grpc-java method names should not have a leading '/'. However this class will add one to be consistent with language agnostic conventions.");
            Preconditions.checkArgument((peerAddress == null || logger == GrpcLogEntry.Logger.LOGGER_SERVER ? 1 : 0) != 0, (Object)"peerSocket can only be specified for server");
            MaybeTruncated<Metadata.Builder> pair = BinlogHelper.createMetadataProto(metadata, this.maxHeaderBytes);
            ClientHeader.Builder clientHeaderBuilder = ClientHeader.newBuilder().setMetadata((Metadata.Builder)pair.proto).setMethodName("/" + methodName);
            if (timeout != null) {
                clientHeaderBuilder.setTimeout(timeout);
            }
            if (authority != null) {
                clientHeaderBuilder.setAuthority(authority);
            }
            GrpcLogEntry.Builder entryBuilder = this.newTimestampedBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.EventType.EVENT_TYPE_CLIENT_HEADER).setClientHeader(clientHeaderBuilder).setPayloadTruncated(pair.truncated).setLogger(logger).setCallId(callId);
            if (peerAddress != null) {
                entryBuilder.setPeer(BinlogHelper.socketToProto(peerAddress));
            }
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        void logServerHeader(long seq, Metadata metadata, GrpcLogEntry.Logger logger, long callId, @Nullable SocketAddress peerAddress) {
            Preconditions.checkArgument((peerAddress == null || logger == GrpcLogEntry.Logger.LOGGER_CLIENT ? 1 : 0) != 0, (Object)"peerSocket can only be specified for client");
            MaybeTruncated<Metadata.Builder> pair = BinlogHelper.createMetadataProto(metadata, this.maxHeaderBytes);
            GrpcLogEntry.Builder entryBuilder = this.newTimestampedBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.EventType.EVENT_TYPE_SERVER_HEADER).setServerHeader(ServerHeader.newBuilder().setMetadata((Metadata.Builder)pair.proto)).setPayloadTruncated(pair.truncated).setLogger(logger).setCallId(callId);
            if (peerAddress != null) {
                entryBuilder.setPeer(BinlogHelper.socketToProto(peerAddress));
            }
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        void logTrailer(long seq, Status status, Metadata metadata, GrpcLogEntry.Logger logger, long callId, @Nullable SocketAddress peerAddress) {
            byte[] statusDetailBytes;
            Preconditions.checkArgument((peerAddress == null || logger == GrpcLogEntry.Logger.LOGGER_CLIENT ? 1 : 0) != 0, (Object)"peerSocket can only be specified for client");
            MaybeTruncated<Metadata.Builder> pair = BinlogHelper.createMetadataProto(metadata, this.maxHeaderBytes);
            Trailer.Builder trailerBuilder = Trailer.newBuilder().setStatusCode(status.getCode().value()).setMetadata((Metadata.Builder)pair.proto);
            String statusDescription = status.getDescription();
            if (statusDescription != null) {
                trailerBuilder.setStatusMessage(statusDescription);
            }
            if ((statusDetailBytes = (byte[])metadata.get(STATUS_DETAILS_KEY)) != null) {
                trailerBuilder.setStatusDetails(ByteString.copyFrom((byte[])statusDetailBytes));
            }
            GrpcLogEntry.Builder entryBuilder = this.newTimestampedBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.EventType.EVENT_TYPE_SERVER_TRAILER).setTrailer(trailerBuilder).setPayloadTruncated(pair.truncated).setLogger(logger).setCallId(callId);
            if (peerAddress != null) {
                entryBuilder.setPeer(BinlogHelper.socketToProto(peerAddress));
            }
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        <T> void logRpcMessage(long seq, GrpcLogEntry.EventType eventType, MethodDescriptor.Marshaller<T> marshaller, T message, GrpcLogEntry.Logger logger, long callId) {
            Preconditions.checkArgument((eventType == GrpcLogEntry.EventType.EVENT_TYPE_CLIENT_MESSAGE || eventType == GrpcLogEntry.EventType.EVENT_TYPE_SERVER_MESSAGE ? 1 : 0) != 0, (Object)"event type must correspond to client message or server message");
            if (marshaller != BinaryLogProvider.BYTEARRAY_MARSHALLER) {
                throw new IllegalStateException("Expected the BinaryLog's ByteArrayMarshaller");
            }
            MaybeTruncated<Message.Builder> pair = BinlogHelper.createMessageProto((byte[])message, this.maxMessageBytes);
            GrpcLogEntry.Builder entryBuilder = this.newTimestampedBuilder().setSequenceIdWithinCall(seq).setType(eventType).setMessage((Message.Builder)pair.proto).setPayloadTruncated(pair.truncated).setLogger(logger).setCallId(callId);
            this.sink.write((MessageLite)entryBuilder.build());
        }

        @Override
        void logHalfClose(long seq, GrpcLogEntry.Logger logger, long callId) {
            this.sink.write((MessageLite)this.newTimestampedBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.EventType.EVENT_TYPE_CLIENT_HALF_CLOSE).setLogger(logger).setCallId(callId).build());
        }

        @Override
        void logCancel(long seq, GrpcLogEntry.Logger logger, long callId) {
            this.sink.write((MessageLite)this.newTimestampedBuilder().setSequenceIdWithinCall(seq).setType(GrpcLogEntry.EventType.EVENT_TYPE_CANCEL).setLogger(logger).setCallId(callId).build());
        }

        @Override
        int getMaxHeaderBytes() {
            return this.maxHeaderBytes;
        }

        @Override
        int getMaxMessageBytes() {
            return this.maxMessageBytes;
        }
    }
}

