/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie.datainteg;

import com.google.common.collect.ImmutableSortedMap;
import io.reactivex.rxjava3.core.Flowable;
import io.reactivex.rxjava3.core.Maybe;
import io.reactivex.rxjava3.core.Scheduler;
import io.reactivex.rxjava3.core.Single;
import io.reactivex.rxjava3.disposables.Disposable;
import java.io.IOException;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.bookie.LedgerStorage;
import org.apache.bookkeeper.bookie.datainteg.DataIntegrityCheck;
import org.apache.bookkeeper.bookie.datainteg.EntryCopier;
import org.apache.bookkeeper.bookie.datainteg.Events;
import org.apache.bookkeeper.bookie.datainteg.MetadataAsyncIterator;
import org.apache.bookkeeper.bookie.datainteg.WriteSets;
import org.apache.bookkeeper.client.BKException;
import org.apache.bookkeeper.client.BookKeeperAdmin;
import org.apache.bookkeeper.client.api.LedgerMetadata;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.bookkeeper.meta.LedgerManager;
import org.apache.bookkeeper.net.BookieId;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DataIntegrityCheckImpl
implements DataIntegrityCheck {
    private static final Logger log = LoggerFactory.getLogger(DataIntegrityCheckImpl.class);
    private static final int MAX_INFLIGHT = 300;
    private static final int MAX_ENTRIES_INFLIGHT = 3000;
    private static final int ZK_TIMEOUT_S = 30;
    private final BookieId bookieId;
    private final LedgerManager ledgerManager;
    private final LedgerStorage ledgerStorage;
    private final EntryCopier entryCopier;
    private final BookKeeperAdmin admin;
    private final Scheduler scheduler;
    private final AtomicReference<Map<Long, LedgerMetadata>> ledgersCacheRef = new AtomicReference<Object>(null);
    private CompletableFuture<Void> preBootFuture;

    public DataIntegrityCheckImpl(BookieId bookieId, LedgerManager ledgerManager, LedgerStorage ledgerStorage, EntryCopier entryCopier, BookKeeperAdmin admin, Scheduler scheduler) {
        this.bookieId = bookieId;
        this.ledgerManager = ledgerManager;
        this.ledgerStorage = ledgerStorage;
        this.entryCopier = entryCopier;
        this.admin = admin;
        this.scheduler = scheduler;
    }

    @Override
    public synchronized CompletableFuture<Void> runPreBootCheck(String reason) {
        if (this.preBootFuture == null) {
            this.preBootFuture = this.runPreBootSequence(reason);
        }
        return this.preBootFuture;
    }

    private CompletableFuture<Void> runPreBootSequence(String reason) {
        String runId = UUID.randomUUID().toString();
        log.info("Event: {}, RunId: {}, Reason: {}", new Object[]{Events.PREBOOT_START, runId, reason});
        try {
            this.ledgerStorage.setStorageStateFlag(LedgerStorage.StorageState.NEEDS_INTEGRITY_CHECK);
        }
        catch (IOException ioe) {
            log.error("Event: {}, RunId: {}", new Object[]{Events.PREBOOT_ERROR, runId, ioe});
            return FutureUtils.exception((Throwable)ioe);
        }
        MetadataAsyncIterator iter = new MetadataAsyncIterator(this.scheduler, this.ledgerManager, 300, 30, TimeUnit.SECONDS);
        CompletableFuture<Void> promise = new CompletableFuture<Void>();
        ConcurrentSkipListMap ledgersCache = new ConcurrentSkipListMap(Comparator.naturalOrder().reversed());
        iter.forEach((ledgerId, metadata) -> {
            if (DataIntegrityCheckImpl.ensemblesContainBookie(metadata, this.bookieId)) {
                ledgersCache.put(ledgerId, metadata);
                try {
                    if (!this.ledgerStorage.ledgerExists((long)ledgerId)) {
                        this.ledgerStorage.setMasterKey((long)ledgerId, new byte[0]);
                    }
                }
                catch (IOException ioe) {
                    log.error("Event: {}, RunId: {}, LedgerId: {}", new Object[]{Events.ENSURE_LEDGER_ERROR, runId, ledgerId, ioe});
                    return FutureUtils.exception((Throwable)ioe);
                }
            }
            return this.processPreBoot((long)ledgerId, (LedgerMetadata)metadata, runId);
        }).whenComplete((ignore, exception) -> {
            if (exception != null) {
                log.error("Event: {}, runId: {}", new Object[]{Events.PREBOOT_ERROR, runId, exception});
                promise.completeExceptionally((Throwable)exception);
            } else {
                try {
                    this.ledgerStorage.flush();
                    this.updateMetadataCache(ledgersCache);
                    log.info("Event: {}, runId: {}, processed: {}", new Object[]{Events.PREBOOT_END, runId, ledgersCache.size()});
                    promise.complete(null);
                }
                catch (Throwable t) {
                    log.error("Event: {}, runId: {}", new Object[]{Events.PREBOOT_ERROR, runId, t});
                    promise.completeExceptionally(t);
                }
            }
        });
        return promise;
    }

    @Override
    public boolean needsFullCheck() throws IOException {
        return this.ledgerStorage.getStorageStateFlags().contains((Object)LedgerStorage.StorageState.NEEDS_INTEGRITY_CHECK);
    }

    @Override
    public CompletableFuture<Void> runFullCheck() {
        String runId = UUID.randomUUID().toString();
        log.info("Event: {}, runId: {}", (Object)Events.FULL_CHECK_INIT, (Object)runId);
        return ((CompletableFuture)this.getCachedOrReadMetadata(runId).thenCompose(ledgers -> {
            log.info("Event: {}, runId: {}, ledgerCount: {}", new Object[]{Events.FULL_CHECK_START, runId, ledgers.size()});
            return this.checkAndRecoverLedgers((Map<Long, LedgerMetadata>)ledgers, runId).thenApply(resolved -> {
                for (LedgerResult r2 : resolved) {
                    if (r2.isMissing() || r2.isOK()) {
                        ledgers.remove(r2.getLedgerId());
                        continue;
                    }
                    if (!r2.isError()) continue;
                    ledgers.put(r2.getLedgerId(), r2.getMetadata());
                }
                Optional<Throwable> firstError = resolved.stream().filter(r -> r.isError()).map(r -> r.getThrowable()).findFirst();
                if (firstError.isPresent()) {
                    log.error("Event: {}, runId: {}, ok: {}, error: {}, missing: {}, ledgersToRetry: {}", new Object[]{Events.FULL_CHECK_END, runId, resolved.stream().filter(r -> r.isOK()).count(), resolved.stream().filter(r -> r.isError()).count(), resolved.stream().filter(r -> r.isMissing()).count(), ledgers.size(), firstError.get()});
                } else {
                    log.info("Event: {}, runId: {}, ok: {}, error: 0, missing: {}, ledgersToRetry: {}", new Object[]{Events.FULL_CHECK_END, runId, resolved.stream().filter(r -> r.isOK()).count(), resolved.stream().filter(r -> r.isMissing()).count(), ledgers.size()});
                }
                return ledgers;
            });
        })).thenCompose(ledgers -> {
            CompletableFuture promise = new CompletableFuture();
            try {
                this.ledgerStorage.flush();
                if (ledgers.isEmpty()) {
                    log.info("Event: {}, runId: {}", (Object)Events.CLEAR_INTEGCHECK_FLAG, (Object)runId);
                    this.ledgerStorage.clearStorageStateFlag(LedgerStorage.StorageState.NEEDS_INTEGRITY_CHECK);
                }
                this.updateMetadataCache((Map<Long, LedgerMetadata>)ledgers);
                log.info("Event: {}, runId: {}", (Object)Events.FULL_CHECK_COMPLETE, (Object)runId);
                promise.complete(null);
            }
            catch (IOException ioe) {
                log.error("Event: {}, runId: {}", new Object[]{Events.FULL_CHECK_ERROR, runId, ioe});
                promise.completeExceptionally(ioe);
            }
            return promise;
        });
    }

    void updateMetadataCache(Map<Long, LedgerMetadata> ledgers) {
        this.ledgersCacheRef.set(ledgers);
    }

    CompletableFuture<Map<Long, LedgerMetadata>> getCachedOrReadMetadata(String runId) {
        Map<Long, LedgerMetadata> map = this.ledgersCacheRef.get();
        if (map != null) {
            log.info("Event: {}, runId: {}, ledgerCount: {}", new Object[]{Events.USE_CACHED_METADATA, runId, map.size()});
            return CompletableFuture.completedFuture(map);
        }
        log.info("Event: {}, runId: {}", (Object)Events.REFRESH_METADATA, (Object)runId);
        MetadataAsyncIterator iter = new MetadataAsyncIterator(this.scheduler, this.ledgerManager, 300, 30, TimeUnit.SECONDS);
        ConcurrentSkipListMap ledgersCache = new ConcurrentSkipListMap(Comparator.naturalOrder().reversed());
        return iter.forEach((ledgerId, metadata) -> {
            if (DataIntegrityCheckImpl.ensemblesContainBookie(metadata, this.bookieId)) {
                ledgersCache.put(ledgerId, metadata);
            }
            return CompletableFuture.completedFuture(null);
        }).thenApply(ignore -> {
            this.updateMetadataCache(ledgersCache);
            return ledgersCache;
        });
    }

    private CompletableFuture<Void> processPreBoot(long ledgerId, LedgerMetadata metadata, String runId) {
        Map.Entry<Long, ? extends List<BookieId>> lastEnsemble = metadata.getAllEnsembles().lastEntry();
        CompletableFuture<Void> promise = new CompletableFuture<Void>();
        if (lastEnsemble == null) {
            log.error("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.INVALID_METADATA, runId, metadata, ledgerId});
            promise.completeExceptionally(new IllegalStateException(String.format("All metadata must have at least one ensemble, %d does not", ledgerId)));
            return promise;
        }
        if (!metadata.isClosed() && lastEnsemble.getValue().contains(this.bookieId)) {
            try {
                log.info("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.MARK_LIMBO, runId, metadata, ledgerId});
                this.ledgerStorage.setLimboState(ledgerId);
                this.ledgerStorage.setFenced(ledgerId);
                promise.complete(null);
            }
            catch (IOException ioe) {
                log.info("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.LIMBO_OR_FENCE_ERROR, runId, metadata, ledgerId, ioe});
                promise.completeExceptionally(ioe);
            }
        } else {
            promise.complete(null);
        }
        return promise;
    }

    CompletableFuture<Set<LedgerResult>> checkAndRecoverLedgers(Map<Long, LedgerMetadata> ledgers, String runId) {
        CompletableFuture<Set<LedgerResult>> promise = new CompletableFuture<Set<LedgerResult>>();
        Disposable disposable = Flowable.fromIterable(ledgers.entrySet()).subscribeOn(this.scheduler, false).flatMapSingle(mapEntry -> {
            long ledgerId = (Long)mapEntry.getKey();
            LedgerMetadata originalMetadata = (LedgerMetadata)mapEntry.getValue();
            return this.recoverLedgerIfInLimbo(ledgerId, (LedgerMetadata)mapEntry.getValue(), runId).map(newMetadata -> LedgerResult.ok(ledgerId, newMetadata)).onErrorReturn(t -> LedgerResult.error(ledgerId, originalMetadata, t)).defaultIfEmpty((Object)LedgerResult.missing(ledgerId)).flatMap(res -> {
                try {
                    if (res.isOK()) {
                        this.ledgerStorage.clearLimboState(ledgerId);
                    }
                    return Single.just((Object)res);
                }
                catch (IOException ioe) {
                    return Single.just((Object)LedgerResult.error(res.getLedgerId(), res.getMetadata(), ioe));
                }
            });
        }, true, 300).flatMapSingle(res -> {
            if (res.isOK()) {
                return this.checkAndRecoverLedgerEntries(res.getLedgerId(), res.getMetadata(), runId).map(ignore -> LedgerResult.ok(res.getLedgerId(), res.getMetadata())).onErrorReturn(t -> LedgerResult.error(res.getLedgerId(), res.getMetadata(), t));
            }
            return Single.just((Object)res);
        }, true, 1).collect(Collectors.toSet()).subscribe(resolved -> promise.complete((Set<LedgerResult>)resolved), throwable -> promise.completeExceptionally((Throwable)throwable));
        promise.whenComplete((result, ex) -> disposable.dispose());
        return promise;
    }

    Maybe<LedgerMetadata> recoverLedgerIfInLimbo(long ledgerId, LedgerMetadata origMetadata, String runId) {
        try {
            if (!this.ledgerStorage.ledgerExists(ledgerId)) {
                this.ledgerStorage.setMasterKey(ledgerId, new byte[0]);
            }
            if (this.ledgerStorage.hasLimboState(ledgerId)) {
                log.info("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.RECOVER_LIMBO_LEDGER, runId, origMetadata, ledgerId});
                return this.recoverLedger(ledgerId, runId).toMaybe().onErrorResumeNext(t -> {
                    if (t instanceof BKException.BKNoSuchLedgerExistsOnMetadataServerException) {
                        log.info("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.RECOVER_LIMBO_LEDGER_MISSING, runId, origMetadata, ledgerId});
                        return Maybe.empty();
                    }
                    log.info("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.RECOVER_LIMBO_LEDGER_ERROR, runId, origMetadata, ledgerId});
                    return Maybe.error((Throwable)t);
                });
            }
            return Maybe.just((Object)origMetadata);
        }
        catch (IOException ioe) {
            return Maybe.error((Throwable)ioe);
        }
    }

    Single<LedgerMetadata> recoverLedger(long ledgerId, String runId) {
        return Single.create(emitter -> this.admin.asyncOpenLedger(ledgerId, (rc, handle, ctx) -> {
            if (rc != 0) {
                emitter.onError((Throwable)BKException.create(rc));
            } else {
                LedgerMetadata metadata = handle.getLedgerMetadata();
                handle.closeAsync().whenComplete((ignore, exception) -> {
                    if (exception != null) {
                        log.warn("Event: {}, runId: {}, ledger: {}", new Object[]{Events.RECOVER_LIMBO_LEDGER_CLOSE_ERROR, runId, ledgerId, exception});
                    }
                });
                emitter.onSuccess((Object)metadata);
            }
        }, null));
    }

    Single<Long> checkAndRecoverLedgerEntries(long ledgerId, LedgerMetadata metadata, String runId) {
        EntryCopier.Batch batch;
        WriteSets writeSets = new WriteSets(metadata.getEnsembleSize(), metadata.getWriteQuorumSize());
        NavigableMap bookieIndices = (NavigableMap)metadata.getAllEnsembles().entrySet().stream().collect(ImmutableSortedMap.toImmutableSortedMap(Comparator.naturalOrder(), e -> (Long)e.getKey(), e -> ((List)e.getValue()).indexOf(this.bookieId)));
        long lastKnownEntry = metadata.isClosed() ? metadata.getLastEntryId() : metadata.getAllEnsembles().lastEntry().getKey() - 1L;
        if (lastKnownEntry < 0L) {
            return Single.just((Object)ledgerId);
        }
        try {
            batch = this.entryCopier.newBatch(ledgerId, metadata);
        }
        catch (IOException ioe) {
            return Single.error((Throwable)ioe);
        }
        AtomicLong byteCount = new AtomicLong(0L);
        AtomicInteger count = new AtomicInteger(0);
        AtomicInteger errorCount = new AtomicInteger(0);
        AtomicReference<Object> firstError = new AtomicReference<Object>(null);
        log.info("Event: {}, runId: {}, metadata: {}, ledger: {}", new Object[]{Events.LEDGER_CHECK_AND_COPY_START, runId, metadata, ledgerId});
        return Flowable.rangeLong((long)0L, (long)(lastKnownEntry + 1L)).subscribeOn(this.scheduler, false).flatMapMaybe(entryId -> this.maybeCopyEntry(writeSets, bookieIndices, ledgerId, (long)entryId, batch).doOnError(t -> {
            firstError.compareAndSet(null, t);
            errorCount.incrementAndGet();
        }), true, 3000).doOnNext(bytes -> {
            byteCount.addAndGet((long)bytes);
            count.incrementAndGet();
        }).count().doOnTerminate(() -> {
            if (firstError.get() != null) {
                log.warn("Event: {}, runId: {}, metadata: {}, ledger: {}, entries: {}, bytes: {}, errors: {}", new Object[]{Events.LEDGER_CHECK_AND_COPY_END, runId, metadata, ledgerId, count.get(), byteCount.get(), firstError.get()});
            } else {
                log.info("Event: {}, runId: {}, metadata: {}, ledger: {}, entries: {}, bytes: {}, errors: 0", new Object[]{Events.LEDGER_CHECK_AND_COPY_END, runId, metadata, ledgerId, count.get(), byteCount.get()});
            }
        }).map(ignore -> ledgerId);
    }

    Maybe<Long> maybeCopyEntry(WriteSets writeSets, NavigableMap<Long, Integer> bookieIndices, long ledgerId, long entryId, EntryCopier.Batch batch) {
        try {
            if (this.isEntryMissing(writeSets, bookieIndices, ledgerId, entryId)) {
                return Maybe.fromCompletionStage(batch.copyFromAvailable(entryId));
            }
            return Maybe.empty();
        }
        catch (IOException | BookieException ioe) {
            return Maybe.error((Throwable)ioe);
        }
    }

    boolean isEntryMissing(WriteSets writeSets, NavigableMap<Long, Integer> bookieIndices, long ledgerId, long entryId) throws IOException, BookieException {
        int bookieIndexForEntry = bookieIndices.floorEntry(entryId).getValue();
        if (bookieIndexForEntry < 0) {
            return false;
        }
        return writeSets.getForEntry(entryId).contains((Object)bookieIndexForEntry) && !this.ledgerStorage.entryExists(ledgerId, entryId);
    }

    static boolean ensemblesContainBookie(LedgerMetadata metadata, BookieId bookieId) {
        return metadata.getAllEnsembles().values().stream().anyMatch(ensemble -> ensemble.contains(bookieId));
    }

    static class LedgerResult {
        private final State state;
        private final long ledgerId;
        private final LedgerMetadata metadata;
        private final Throwable throwable;

        static LedgerResult missing(long ledgerId) {
            return new LedgerResult(State.MISSING, ledgerId, null, null);
        }

        static LedgerResult ok(long ledgerId, LedgerMetadata metadata) {
            return new LedgerResult(State.OK, ledgerId, metadata, null);
        }

        static LedgerResult error(long ledgerId, LedgerMetadata metadata, Throwable t) {
            return new LedgerResult(State.ERROR, ledgerId, metadata, t);
        }

        private LedgerResult(State state, long ledgerId, LedgerMetadata metadata, Throwable throwable) {
            this.state = state;
            this.ledgerId = ledgerId;
            this.metadata = metadata;
            this.throwable = throwable;
        }

        boolean isMissing() {
            return this.state == State.MISSING;
        }

        boolean isOK() {
            return this.state == State.OK;
        }

        boolean isError() {
            return this.state == State.ERROR;
        }

        long getLedgerId() {
            return this.ledgerId;
        }

        LedgerMetadata getMetadata() {
            return this.metadata;
        }

        Throwable getThrowable() {
            return this.throwable;
        }

        static enum State {
            MISSING,
            ERROR,
            OK;

        }
    }
}

