/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.benchmark;

import edu.umd.cs.findbugs.annotations.SuppressFBWarnings;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.bookkeeper.client.BookKeeper;
import org.apache.bookkeeper.client.LedgerEntry;
import org.apache.bookkeeper.client.LedgerHandle;
import org.apache.bookkeeper.conf.ClientConfiguration;
import org.apache.bookkeeper.zookeeper.ZooKeeperClient;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.PosixParser;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BenchReadThroughputLatency {
    static final Logger LOG = LoggerFactory.getLogger(BenchReadThroughputLatency.class);
    private static final Pattern LEDGER_PATTERN = Pattern.compile("L([0-9]+)$");
    private static final Comparator<String> ZK_LEDGER_COMPARE = new Comparator<String>(){

        @Override
        public int compare(String o1, String o2) {
            try {
                Matcher m1 = LEDGER_PATTERN.matcher(o1);
                Matcher m2 = LEDGER_PATTERN.matcher(o2);
                if (m1.find() && m2.find()) {
                    return Integer.parseInt(m1.group(1)) - Integer.parseInt(m2.group(1));
                }
                return o1.compareTo(o2);
            }
            catch (Throwable t) {
                return o1.compareTo(o2);
            }
        }
    };

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void readLedger(ClientConfiguration conf, long ledgerId, byte[] passwd) {
        LOG.info("Reading ledger {}", (Object)ledgerId);
        BookKeeper bk = null;
        long time = 0L;
        long entriesRead = 0L;
        long lastRead = 0L;
        int nochange = 0;
        long absoluteLimit = 5000000L;
        LedgerHandle lh = null;
        try {
            bk = new BookKeeper(conf);
            while (true) {
                long lastConfirmed;
                if ((lastConfirmed = Math.min((lh = bk.openLedgerNoRecovery(ledgerId, BookKeeper.DigestType.CRC32, passwd)).getLastAddConfirmed(), absoluteLimit)) == lastRead) {
                    if (++nochange == 10) {
                        break;
                    }
                    Thread.sleep(1000L);
                    continue;
                }
                nochange = 0;
                long starttime = System.nanoTime();
                while (lastRead < lastConfirmed) {
                    long nextLimit = lastRead + 100000L;
                    long readTo = Math.min(nextLimit, lastConfirmed);
                    Enumeration entries = lh.readEntries(lastRead + 1L, readTo);
                    lastRead = readTo;
                    while (entries.hasMoreElements()) {
                        LedgerEntry e = (LedgerEntry)entries.nextElement();
                        if (++entriesRead % 10000L != 0L) continue;
                        LOG.info("{} entries read", (Object)entriesRead);
                    }
                }
                long endtime = System.nanoTime();
                time += endtime - starttime;
                lh.close();
                lh = null;
                Thread.sleep(1000L);
            }
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            LOG.error("Exception in reader", (Throwable)e);
        }
        finally {
            LOG.info("Read {} in {}ms", (Object)entriesRead, (Object)(time / 1000L / 1000L));
            try {
                if (lh != null) {
                    lh.close();
                }
                if (bk != null) {
                    bk.close();
                }
            }
            catch (Exception e) {
                LOG.error("Exception closing stuff", (Throwable)e);
            }
        }
    }

    private static void usage(Options options) {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp("BenchReadThroughputLatency <options>", options);
    }

    @SuppressFBWarnings(value={"RCN_REDUNDANT_NULLCHECK_WOULD_HAVE_BEEN_A_NPE"})
    public static void main(String[] args) throws Exception {
        Options options = new Options();
        options.addOption("ledger", true, "Ledger to read. If empty, read all ledgers which come available.  Cannot be used with -listen");
        options.addOption("listen", true, "Listen for creation of <arg> ledgers, and read each one fully");
        options.addOption("password", true, "Password used to access ledgers (default 'benchPasswd')");
        options.addOption("zookeeper", true, "Zookeeper ensemble, default \"localhost:2181\"");
        options.addOption("sockettimeout", true, "Socket timeout for bookkeeper client. In seconds. Default 5");
        options.addOption("useV2", false, "Whether use V2 protocol to read ledgers from the bookie server.");
        options.addOption("help", false, "This message");
        PosixParser parser = new PosixParser();
        CommandLine cmd = parser.parse(options, args);
        if (cmd.hasOption("help")) {
            BenchReadThroughputLatency.usage(options);
            System.exit(-1);
        }
        String servers = cmd.getOptionValue("zookeeper", "localhost:2181");
        final byte[] passwd = cmd.getOptionValue("password", "benchPasswd").getBytes(StandardCharsets.UTF_8);
        int sockTimeout = Integer.parseInt(cmd.getOptionValue("sockettimeout", "5"));
        if (cmd.hasOption("ledger") && cmd.hasOption("listen")) {
            LOG.error("Cannot used -ledger and -listen together");
            BenchReadThroughputLatency.usage(options);
            System.exit(-1);
        }
        final AtomicInteger ledger = new AtomicInteger(0);
        final AtomicInteger numLedgers = new AtomicInteger(0);
        if (cmd.hasOption("ledger")) {
            ledger.set(Integer.parseInt(cmd.getOptionValue("ledger")));
        } else if (cmd.hasOption("listen")) {
            numLedgers.set(Integer.parseInt(cmd.getOptionValue("listen")));
        } else {
            LOG.error("You must use -ledger or -listen");
            BenchReadThroughputLatency.usage(options);
            System.exit(-1);
        }
        final CountDownLatch shutdownLatch = new CountDownLatch(1);
        final String nodepath = String.format("/ledgers/L%010d", ledger.get());
        final ClientConfiguration conf = new ClientConfiguration();
        conf.setReadTimeout(sockTimeout).setZkServers(servers);
        if (cmd.hasOption("useV2")) {
            conf.setUseV2WireProtocol(true);
        }
        try (final ZooKeeperClient zk = ZooKeeperClient.newBuilder().connectString(servers).sessionTimeoutMs(3000).build();){
            final HashSet processedLedgers = new HashSet();
            zk.register(new Watcher(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                public void process(WatchedEvent event) {
                    try {
                        if (event.getType() == Watcher.Event.EventType.NodeCreated && event.getPath().equals(nodepath)) {
                            BenchReadThroughputLatency.readLedger(conf, ledger.get(), passwd);
                            shutdownLatch.countDown();
                        } else if (event.getType() == Watcher.Event.EventType.NodeChildrenChanged) {
                            if (numLedgers.get() < 0) {
                                return;
                            }
                            List children = zk.getChildren("/ledgers", true);
                            ArrayList<String> ledgers = new ArrayList<String>();
                            for (String child : children) {
                                if (!LEDGER_PATTERN.matcher(child).find()) continue;
                                ledgers.add(child);
                            }
                            for (String ledger2 : ledgers) {
                                Set set = processedLedgers;
                                synchronized (set) {
                                    if (processedLedgers.contains(ledger2)) {
                                        continue;
                                    }
                                    Matcher m = LEDGER_PATTERN.matcher(ledger2);
                                    if (m.find()) {
                                        int ledgersLeft = numLedgers.decrementAndGet();
                                        final Long ledgerId = Long.valueOf(m.group(1));
                                        processedLedgers.add(ledger2);
                                        Thread t = new Thread(){

                                            @Override
                                            public void run() {
                                                BenchReadThroughputLatency.readLedger(conf, ledgerId, passwd);
                                            }
                                        };
                                        t.start();
                                        if (ledgersLeft <= 0) {
                                            shutdownLatch.countDown();
                                        }
                                    } else {
                                        LOG.error("Cant file ledger id in {}", (Object)ledger2);
                                    }
                                }
                            }
                        } else {
                            LOG.warn("Unknown event {}", (Object)event);
                        }
                    }
                    catch (Exception e) {
                        LOG.error("Exception in watcher", (Throwable)e);
                    }
                }
            });
            if (ledger.get() != 0) {
                if (zk.exists(nodepath, true) != null) {
                    BenchReadThroughputLatency.readLedger(conf, ledger.get(), passwd);
                    shutdownLatch.countDown();
                } else {
                    LOG.info("Watching for creation of" + nodepath);
                }
            } else {
                zk.getChildren("/ledgers", true);
            }
            shutdownLatch.await();
            LOG.info("Shutting down");
        }
    }
}

