/**
 * Copyright (c) 2010-2012, Mark Czotter, Istvan Rath and Daniel Varro
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-v20.html.
 * 
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.viatra.query.patternlanguage.emf.jvmmodel;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.inject.Inject;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.TreeIterator;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EDataType;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.viatra.query.patternlanguage.emf.helper.PatternLanguageHelper;
import org.eclipse.viatra.query.patternlanguage.emf.services.EMFPatternLanguageGrammarAccess;
import org.eclipse.viatra.query.patternlanguage.emf.types.BottomTypeKey;
import org.eclipse.viatra.query.patternlanguage.emf.types.ITypeInferrer;
import org.eclipse.viatra.query.patternlanguage.emf.util.EMFPatternLanguageGeneratorConfig;
import org.eclipse.viatra.query.patternlanguage.emf.util.IErrorFeedback;
import org.eclipse.viatra.query.patternlanguage.emf.validation.IssueCodes;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Pattern;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternBody;
import org.eclipse.viatra.query.patternlanguage.emf.vql.PatternModel;
import org.eclipse.viatra.query.patternlanguage.emf.vql.Variable;
import org.eclipse.viatra.query.runtime.api.GenericPatternMatch;
import org.eclipse.viatra.query.runtime.api.GenericPatternMatcher;
import org.eclipse.viatra.query.runtime.api.impl.BaseGeneratedEMFQuerySpecification;
import org.eclipse.viatra.query.runtime.api.impl.BaseGeneratedEMFQuerySpecificationWithGenericMatcher;
import org.eclipse.viatra.query.runtime.api.impl.BaseMatcher;
import org.eclipse.viatra.query.runtime.api.impl.BasePatternMatch;
import org.eclipse.viatra.query.runtime.emf.types.EClassTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EClassUnscopedTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.emf.types.EDataTypeInSlotsKey;
import org.eclipse.viatra.query.runtime.emf.types.EStructuralFeatureInstancesKey;
import org.eclipse.viatra.query.runtime.matchers.context.IInputKey;
import org.eclipse.viatra.query.runtime.matchers.context.common.JavaTransitiveInstancesKey;
import org.eclipse.viatra.query.runtime.matchers.planning.QueryProcessingException;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtend2.lib.StringConcatenationClient;
import org.eclipse.xtext.EcoreUtil2;
import org.eclipse.xtext.TerminalRule;
import org.eclipse.xtext.common.types.JvmDeclaredType;
import org.eclipse.xtext.common.types.JvmType;
import org.eclipse.xtext.common.types.JvmTypeReference;
import org.eclipse.xtext.common.types.util.TypeReferences;
import org.eclipse.xtext.diagnostics.Severity;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.INode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.xbase.XExpression;
import org.eclipse.xtext.xbase.XFeatureCall;
import org.eclipse.xtext.xbase.jvmmodel.IJvmModelAssociations;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.IteratorExtensions;
import org.eclipse.xtext.xbase.lib.StringExtensions;

/**
 * Utility class for the EMFPatternLanguageJvmModelInferrer.
 * 
 * @author Mark Czotter
 * @since 2.0
 */
@SuppressWarnings("all")
public class EMFPatternLanguageJvmModelInferrerUtil {
  @Inject
  @Extension
  private TypeReferences _typeReferences;
  
  @Inject
  private Logger logger;
  
  @Inject
  private ITypeInferrer typeInferrer;
  
  @Inject
  private IJvmModelAssociations associations;
  
  @Inject
  private EMFPatternLanguageGrammarAccess grammar;
  
  @Inject
  private IErrorFeedback feedback;
  
  private static final String MATCH_POSTFIX = "Match";
  
  private static final String MATCHER_POSTFIX = "Matcher";
  
  private static final String PROCESSOR_POSTFIX = "Processor";
  
  private static final String SPECIFICATION_POSTFIX = "QuerySpecification";
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final Pattern pattern) {
    String name = pattern.getName();
    boolean _contains = name.contains(".");
    if (_contains) {
      int _lastIndexOf = name.lastIndexOf(".");
      int _plus = (_lastIndexOf + 1);
      return name.substring(_plus);
    }
    return name;
  }
  
  public boolean validClassName(final String simpleName) {
    return (Character.isJavaIdentifierStart(simpleName.charAt(0)) && 
      IterableExtensions.<Character>forall(((Iterable<Character>)Conversions.doWrapArray(simpleName.toCharArray())), ((Function1<Character, Boolean>) (Character it) -> {
        return Boolean.valueOf(Character.isJavaIdentifierPart((it).charValue()));
      })));
  }
  
  public String modelFileName(final EObject object) {
    final String name = PatternLanguageHelper.getModelFileName(object);
    if (((name != null) && (!this.validClassName(name)))) {
      this.feedback.reportErrorNoLocation(object, 
        String.format("The file name %s is not a valid Java type name. Please, rename the file!", name), 
        IssueCodes.OTHER_ISSUE, Severity.ERROR, IErrorFeedback.JVMINFERENCE_ERROR_TYPE);
    }
    return name;
  }
  
  /**
   * @since 2.0
   */
  public String modelFileQualifiedName(final Pattern pattern) {
    StringConcatenation _builder = new StringConcatenation();
    String _packageName = this.getPackageName(pattern);
    _builder.append(_packageName);
    _builder.append(".");
    String _firstUpper = StringExtensions.toFirstUpper(this.modelFileName(pattern));
    _builder.append(_firstUpper);
    return _builder.toString();
  }
  
  /**
   * Returns the QuerySpecificationClass name based on the Pattern's name
   * @since 2.0
   */
  public String querySpecificationClassName(final Pattern pattern, final EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy strategy) {
    String _xblockexpression = null;
    {
      String name = pattern.getName();
      boolean _contains = name.contains(".");
      if (_contains) {
        name = this.realPatternName(pattern);
      }
      String _xifexpression = null;
      if ((strategy == EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.SEPARATE_CLASS)) {
        String _firstUpper = StringExtensions.toFirstUpper(name);
        _xifexpression = (_firstUpper + EMFPatternLanguageJvmModelInferrerUtil.SPECIFICATION_POSTFIX);
      } else {
        return StringExtensions.toFirstUpper(name);
      }
      _xblockexpression = _xifexpression;
    }
    return _xblockexpression;
  }
  
  /**
   * Returns the holder class name based on the Pattern's name
   */
  public String querySpecificationHolderClassName(final Pattern pattern) {
    return "LazyHolder";
  }
  
  /**
   * Returns the PQuery class name based on the Pattern's name
   */
  public String querySpecificationPQueryClassName(final Pattern pattern) {
    return "GeneratedPQuery";
  }
  
  /**
   * Returns the MatcherClass name based on the Pattern's name
   * @since 2.0
   */
  public String matcherClassName(final Pattern pattern, final EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy strategy) {
    String _xifexpression = null;
    if ((strategy == EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.NESTED_CLASS)) {
      return EMFPatternLanguageJvmModelInferrerUtil.MATCHER_POSTFIX;
    } else {
      String _xblockexpression = null;
      {
        String name = pattern.getName();
        boolean _contains = name.contains(".");
        if (_contains) {
          name = this.realPatternName(pattern);
        }
        String _firstUpper = StringExtensions.toFirstUpper(name);
        _xblockexpression = (_firstUpper + EMFPatternLanguageJvmModelInferrerUtil.MATCHER_POSTFIX);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  /**
   * Returns the MatchClass name based on the Pattern's name
   * @since 2.0
   */
  public String matchClassName(final Pattern pattern, final EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy strategy) {
    String _xifexpression = null;
    if ((strategy == EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.NESTED_CLASS)) {
      return EMFPatternLanguageJvmModelInferrerUtil.MATCH_POSTFIX;
    } else {
      String _xblockexpression = null;
      {
        String name = pattern.getName();
        boolean _contains = name.contains(".");
        if (_contains) {
          name = this.realPatternName(pattern);
        }
        String _firstUpper = StringExtensions.toFirstUpper(name);
        _xblockexpression = (_firstUpper + EMFPatternLanguageJvmModelInferrerUtil.MATCH_POSTFIX);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  public String matchImmutableInnerClassName(final Pattern pattern) {
    return "Immutable";
  }
  
  public String matchMutableInnerClassName(final Pattern pattern) {
    return "Mutable";
  }
  
  /**
   * Returns the ProcessorClass name based on the Pattern's name
   * @since 2.0
   */
  public String processorClassName(final Pattern pattern, final EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy strategy) {
    String _xifexpression = null;
    if ((strategy == EMFPatternLanguageGeneratorConfig.MatcherGenerationStrategy.NESTED_CLASS)) {
      return EMFPatternLanguageJvmModelInferrerUtil.PROCESSOR_POSTFIX;
    } else {
      String _xblockexpression = null;
      {
        String name = pattern.getName();
        boolean _contains = name.contains(".");
        if (_contains) {
          name = this.realPatternName(pattern);
        }
        String _firstUpper = StringExtensions.toFirstUpper(name);
        _xblockexpression = (_firstUpper + EMFPatternLanguageJvmModelInferrerUtil.PROCESSOR_POSTFIX);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }
  
  /**
   * Returns field name for Variable
   */
  public String fieldName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = StringExtensions.toFirstUpper(_name);
    return ("f" + _firstUpper);
  }
  
  /**
   * Returns parameter name for Variable
   */
  public String parameterName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = null;
    if (_name!=null) {
      _firstUpper=StringExtensions.toFirstUpper(_name);
    }
    return ("p" + _firstUpper);
  }
  
  public String positionConstant(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _upperCase = null;
    if (_name!=null) {
      _upperCase=_name.toUpperCase();
    }
    return ("POSITION_" + _upperCase);
  }
  
  /**
   * Returns correct getter method name for variable.
   * For variable with name 'class' returns getValueOfClass, otherwise returns <code>get#variable.name.toFirstUpper#</code>.
   */
  public String getterMethodName(final Variable variable) {
    String _name = variable.getName();
    boolean _equals = Objects.equal(_name, "class");
    if (_equals) {
      return "getValueOfClass";
    } else {
      String _name_1 = null;
      if (variable!=null) {
        _name_1=variable.getName();
      }
      String _firstUpper = null;
      if (_name_1!=null) {
        _firstUpper=StringExtensions.toFirstUpper(_name_1);
      }
      return ("get" + _firstUpper);
    }
  }
  
  /**
   * Returns correct setter method name for variable.
   * Currently returns <code>set#variable.name.toFirstUpper#</code>.
   */
  public String setterMethodName(final Variable variable) {
    String _name = null;
    if (variable!=null) {
      _name=variable.getName();
    }
    String _firstUpper = null;
    if (_name!=null) {
      _firstUpper=StringExtensions.toFirstUpper(_name);
    }
    return ("set" + _firstUpper);
  }
  
  /**
   * Calls the typeProvider.
   * @return JvmTypeReference pointing the EClass that defines the Variable's type.
   * @see ITypeInferrer
   */
  public JvmTypeReference calculateType(final Variable variable) {
    return this.typeInferrer.getJvmType(variable, variable);
  }
  
  /**
   * Serializes the EObject into Java String variable.
   */
  public CharSequence serializeToJava(final EObject eObject) {
    final String parseString = this.serialize(eObject);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(parseString);
    if (_isNullOrEmpty) {
      return "";
    }
    final String[] splits = parseString.split("[\r\n]+");
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("String patternString = \"\"");
    final StringConcatenation stringRep = ((StringConcatenation) _builder);
    stringRep.newLine();
    for (final String s : splits) {
      {
        stringRep.append((("+\" " + s) + " \""));
        stringRep.newLine();
      }
    }
    stringRep.append(";");
    return stringRep;
  }
  
  /**
   * Serializes the input for Javadoc
   */
  public String serializeToJavadoc(final Pattern pattern) {
    String javadocString = this.serialize(pattern);
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(javadocString);
    if (_isNullOrEmpty) {
      return "Serialization error, check Log";
    }
    javadocString = javadocString.replaceAll(java.util.regex.Pattern.quote("\\\""), Matcher.quoteReplacement("\""));
    javadocString = javadocString.replaceAll("@", "{@literal @}");
    javadocString = javadocString.replaceAll("<", "{@literal <}");
    javadocString = javadocString.replaceAll(">", "{@literal >}");
    return javadocString.trim();
  }
  
  /**
   * Returns the file header comment at the beginning of the text corresponding
   * to the pattern model.
   * The comment text is escaped, so it does not include stars in multi-line comments.
   * 
   * @since 1.3
   */
  public String getFileComment(final PatternModel patternModel) {
    final ICompositeNode patternNode = NodeModelUtils.getNode(patternModel);
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("Generated from ");
    Resource _eResource = patternModel.eResource();
    URI _uRI = null;
    if (_eResource!=null) {
      _uRI=_eResource.getURI();
    }
    _builder.append(_uRI);
    return this.getHeaderComment(patternNode, _builder.toString());
  }
  
  public String getPatternComment(final Pattern pattern) {
    final ICompositeNode patternNode = NodeModelUtils.getNode(pattern);
    return this.getHeaderComment(patternNode, "");
  }
  
  private String getHeaderComment(final ICompositeNode node, final String defaultComment) {
    INode _firstChild = null;
    if (node!=null) {
      _firstChild=node.getFirstChild();
    }
    INode _nextSibling = null;
    if (_firstChild!=null) {
      _nextSibling=_firstChild.getNextSibling();
    }
    return this.extractComment(_nextSibling, defaultComment);
  }
  
  private String extractComment(final INode node, final String defaultComment) {
    if ((node != null)) {
      final EObject grammarElement = node.getGrammarElement();
      if ((grammarElement instanceof TerminalRule)) {
        final TerminalRule rule = ((TerminalRule)grammarElement);
        String _name = rule.getName();
        String _name_1 = this.grammar.getWSRule().getName();
        boolean _equals = Objects.equal(_name, _name_1);
        if (_equals) {
          return this.extractComment(node.getPreviousSibling(), defaultComment);
        } else {
          String _name_2 = rule.getName();
          String _name_3 = this.grammar.getML_COMMENTRule().getName();
          boolean _equals_1 = Objects.equal(_name_2, _name_3);
          if (_equals_1) {
            final String multiLineCommentText = this.escape(node.getText());
            return multiLineCommentText;
          } else {
            String _name_4 = rule.getName();
            String _name_5 = this.grammar.getSL_COMMENTRule().getName();
            boolean _equals_2 = Objects.equal(_name_4, _name_5);
            if (_equals_2) {
              final String singleLineCommentText = this.escape(node.getText());
              return singleLineCommentText;
            }
          }
        }
      }
    }
    return defaultComment;
  }
  
  /**
   * Returns the file header comment at the beginning of the text corresponding
   * to the pattern model containing the given pattern.
   * The comment text is escaped, so it does not include stars in multi-line comments.
   * 
   * @since 1.3
   */
  public String getFileComment(final Pattern pattern) {
    final PatternModel patternModel = EcoreUtil2.<PatternModel>getContainerOfType(pattern, PatternModel.class);
    return this.getFileComment(patternModel);
  }
  
  /**
   * Escapes the input to be usable in literal strings
   */
  public String escapeToQuotedString(final String inputString) {
    String string = inputString;
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(string);
    if (_isNullOrEmpty) {
      return "";
    }
    string = string.replace("\\", "\\\\");
    string = string.replace("\n", "\\n");
    string = string.replace("\t", "\\t");
    string = string.replace("\"", "\\\"");
    return string.trim();
  }
  
  /**
   * Serializes EObject to a String representation. Escapes only the double qoutes.
   */
  private String serialize(final EObject eObject) {
    try {
      final ICompositeNode eObjectNode = NodeModelUtils.getNode(eObject);
      if ((eObjectNode != null)) {
        return this.escape(eObjectNode.getText());
      }
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        if ((this.logger != null)) {
          String _name = eObject.eClass().getName();
          String _plus = ("Error when serializing " + _name);
          this.logger.error(_plus, e);
        }
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return null;
  }
  
  private String escape(final String escapable) {
    if ((escapable == null)) {
      return null;
    }
    String escapedString = escapable.replaceAll("\"", "\\\\\"");
    escapedString = escapedString.replaceAll("\\*+/", "").replaceAll("/*\\*", "");
    return escapedString;
  }
  
  /**
   * Returns the packageName: PatternModel.packageName or "" when nullOrEmpty.
   */
  public String getPackageName(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    }
    return packageName.toLowerCase();
  }
  
  public String getUtilPackageName(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return (_packageName + ".util");
  }
  
  /**
   * @since 1.6
   */
  public String getInternalSpecificationPackage(final Pattern pattern) {
    String _packageName = this.getPackageName(pattern);
    return (_packageName + ".internal");
  }
  
  /**
   * Returns the packageName: PatternModel.packageName + Pattern.name, packageName is ignored, when nullOrEmpty.
   */
  public String getPackageNameOld(final Pattern pattern) {
    EObject _eContainer = pattern.eContainer();
    String packageName = ((PatternModel) _eContainer).getPackageName();
    boolean _isNullOrEmpty = StringExtensions.isNullOrEmpty(packageName);
    if (_isNullOrEmpty) {
      packageName = "";
    } else {
      packageName = (packageName + ".");
    }
    String _name = pattern.getName();
    return (packageName + _name).toLowerCase();
  }
  
  public String getPackagePath(final Pattern pattern) {
    return this.getPackageName(pattern).replace(".", "/");
  }
  
  /**
   * Calculates the correct package path for a selected fqn
   */
  public String getPackagePath(final String fqn) {
    String _xblockexpression = null;
    {
      final Iterable<String> split = Splitter.on(".").split(fqn);
      int _size = IterableExtensions.size(split);
      int _minus = (_size - 1);
      _xblockexpression = IterableExtensions.join(IterableExtensions.<String>take(split, _minus), "/");
    }
    return _xblockexpression;
  }
  
  /**
   * This method returns the pattern name.
   * If the pattern name contains the package (any dot),
   * then removes all segment except the last one.
   */
  public String realPatternName(final String fqn) {
    return IterableExtensions.<String>last(Splitter.on(".").split(fqn));
  }
  
  public JvmType findInferredSpecification(final Pattern pattern) {
    return this.findInferredClass(pattern, BaseGeneratedEMFQuerySpecification.class, BaseGeneratedEMFQuerySpecificationWithGenericMatcher.class);
  }
  
  /**
   * @since 1.7
   */
  public JvmType findMatchClass(final Pattern pattern) {
    final JvmType matchClass = this.findInferredClass(pattern, BasePatternMatch.class);
    if ((matchClass == null)) {
      return this._typeReferences.getTypeForName(GenericPatternMatch.class, pattern).getType();
    } else {
      return matchClass;
    }
  }
  
  /**
   * @since 1.7
   */
  public JvmType findMatcherClass(final Pattern pattern) {
    final JvmType matcherClass = this.findInferredClass(pattern, BaseMatcher.class);
    if ((matcherClass == null)) {
      return this._typeReferences.getTypeForName(GenericPatternMatcher.class, pattern).getType();
    } else {
      return matcherClass;
    }
  }
  
  /**
   * Returns an inferred class with a predefined <em>direct</em> subtype
   */
  public JvmType findInferredClass(final EObject pattern, final Class<?> clazz) {
    return this.findInferredClass(pattern, ((Class<?>[])Conversions.unwrapArray(Collections.<Class<?>>unmodifiableSet(CollectionLiterals.<Class<?>>newHashSet(clazz)), Class.class)));
  }
  
  /**
   * Returns an inferred class with a predefined <em>direct</em> subtype (one of the given values)
   * @param pattern the source pattern
   * @param clazzes a set of classes to check whether the inferred class has any as given values
   * @since 1.6
   */
  public JvmType findInferredClass(final EObject pattern, final Class<?>... clazzes) {
    final Function1<JvmType, Boolean> _function = (JvmType it) -> {
      final Function1<Class<?>, Boolean> _function_1 = (Class<?> clazz) -> {
        return Boolean.valueOf(this.isCompatibleWith(it, clazz));
      };
      return Boolean.valueOf(IterableExtensions.<Class<?>>exists(((Iterable<Class<?>>)Conversions.doWrapArray(clazzes)), _function_1));
    };
    return IterableExtensions.<JvmType>findFirst(Iterables.<JvmType>filter(this.associations.getJvmElements(pattern), JvmType.class), _function);
  }
  
  public boolean isCompatibleWith(final JvmType type, final Class<?> clazz) {
    return (this._typeReferences.is(type, clazz) || 
      ((type instanceof JvmDeclaredType) && IterableExtensions.<JvmTypeReference>exists(((JvmDeclaredType) type).getSuperTypes(), ((Function1<JvmTypeReference, Boolean>) (JvmTypeReference it) -> {
        return Boolean.valueOf(this._typeReferences.is(it, clazz));
      }))));
  }
  
  public boolean isPublic(final Pattern pattern) {
    boolean _isPrivate = PatternLanguageHelper.isPrivate(pattern);
    return (!_isPrivate);
  }
  
  public List<Variable> variables(final XExpression ex) {
    List<Variable> _xblockexpression = null;
    {
      final PatternBody body = EcoreUtil2.<PatternBody>getContainerOfType(ex, PatternBody.class);
      TreeIterator<EObject> _eAllContents = ex.eAllContents();
      Iterator<XExpression> _iterator = CollectionLiterals.<XExpression>newImmutableList(ex).iterator();
      final Function1<XFeatureCall, String> _function = (XFeatureCall it) -> {
        return it.getConcreteSyntaxFeatureName();
      };
      final List<String> valNames = IteratorExtensions.<String>toList(IteratorExtensions.<XFeatureCall, String>map(Iterators.<XFeatureCall>filter(Iterators.<EObject>concat(_eAllContents, _iterator), XFeatureCall.class), _function));
      final Function1<Variable, Boolean> _function_1 = (Variable it) -> {
        return Boolean.valueOf(valNames.contains(it.getName()));
      };
      final Function1<Variable, String> _function_2 = (Variable it) -> {
        return it.getName();
      };
      _xblockexpression = IterableExtensions.<Variable, String>sortBy(IterableExtensions.<Variable>filter(body.getVariables(), _function_1), _function_2);
    }
    return _xblockexpression;
  }
  
  public String expressionMethodName(final XExpression ex) {
    String _expressionPostfix = EMFPatternLanguageJvmModelInferrerUtil.getExpressionPostfix(ex);
    return ("evaluateExpression_" + _expressionPostfix);
  }
  
  private static String getExpressionPostfix(final XExpression xExpression) {
    final Pattern pattern = EcoreUtil2.<Pattern>getContainerOfType(xExpression, Pattern.class);
    Preconditions.checkArgument((pattern != null), "Expression is not inside a pattern");
    int bodyNo = 0;
    EList<PatternBody> _bodies = pattern.getBodies();
    for (final PatternBody patternBody : _bodies) {
      {
        bodyNo = (bodyNo + 1);
        int exNo = 0;
        Collection<XExpression> _allTopLevelXBaseExpressions = PatternLanguageHelper.getAllTopLevelXBaseExpressions(patternBody);
        for (final XExpression xExpression2 : _allTopLevelXBaseExpressions) {
          {
            exNo = (exNo + 1);
            boolean _equals = xExpression.equals(xExpression2);
            if (_equals) {
              String _plus = (Integer.valueOf(bodyNo) + "_");
              return (_plus + Integer.valueOf(exNo));
            }
          }
        }
      }
    }
    throw new QueryProcessingException("Expression not found in pattern", pattern);
  }
  
  /**
   * Output code is intended for generated query specification classes,
   *  since it depends on 'getFeatureLiteral()' / 'getClassifierLiteral()'
   * 
   * <p> the "safe" classifier lookup is used if the result is used for initializing a PParameter
   */
  public StringConcatenationClient serializeInputKey(final IInputKey key, final boolean forParameter) {
    return new StringConcatenationClient() {
      @Override
      protected void appendTo(final StringConcatenationClient.TargetStringConcatenation target) {
        EMFPatternLanguageJvmModelInferrerUtil.this.appendInputKey(target, key, forParameter);
      }
    };
  }
  
  /**
   * Calculates the name of the variable that stores a PParameter for a pattern
   * @since 1.4
   */
  public String getPParameterName(final String parameterName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("parameter_");
    _builder.append(parameterName);
    return _builder.toString();
  }
  
  /**
   * Output code is intended for generated query specification classes,
   *  since it depends on 'getFeatureLiteral()' / 'getClassifierLiteral()'
   * 
   * <p> the "safe" classifier lookup is used if the result is used for initializing a PParameter
   */
  public void appendInputKey(final StringConcatenationClient.TargetStringConcatenation target, final IInputKey key, final boolean forParameter) {
    boolean _matched = false;
    if (key instanceof EStructuralFeatureInstancesKey) {
      _matched=true;
      final EStructuralFeature literal = ((EStructuralFeatureInstancesKey)key).getEmfKey();
      final EClass container = literal.getEContainingClass();
      final String packageNsUri = container.getEPackage().getNsURI();
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("new ");
      target.append(_builder);
      target.append(EStructuralFeatureInstancesKey.class);
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("(getFeatureLiteral(\"");
      _builder_1.append(packageNsUri);
      _builder_1.append("\", \"");
      String _name = container.getName();
      _builder_1.append(_name);
      _builder_1.append("\", \"");
      String _name_1 = literal.getName();
      _builder_1.append(_name_1);
      _builder_1.append("\"))");
      target.append(_builder_1);
    }
    if (!_matched) {
      if (key instanceof EClassTransitiveInstancesKey) {
        _matched=true;
        final EClass literal = ((EClassTransitiveInstancesKey)key).getEmfKey();
        final String packageNsUri = literal.getEPackage().getNsURI();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(EClassTransitiveInstancesKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("((");
        target.append(_builder_1);
        target.append(EClass.class);
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(")");
        CharSequence _classifierGetterName = this.classifierGetterName(forParameter);
        _builder_2.append(_classifierGetterName);
        _builder_2.append("(\"");
        _builder_2.append(packageNsUri);
        _builder_2.append("\", \"");
        String _name = literal.getName();
        _builder_2.append(_name);
        _builder_2.append("\"))");
        target.append(_builder_2);
      }
    }
    if (!_matched) {
      if (key instanceof EClassUnscopedTransitiveInstancesKey) {
        _matched=true;
        final EClass literal = ((EClassUnscopedTransitiveInstancesKey)key).getEmfKey();
        final String packageNsUri = literal.getEPackage().getNsURI();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(EClassUnscopedTransitiveInstancesKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("((");
        target.append(_builder_1);
        target.append(EClass.class);
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(")");
        CharSequence _classifierGetterName = this.classifierGetterName(forParameter);
        _builder_2.append(_classifierGetterName);
        _builder_2.append("(\"");
        _builder_2.append(packageNsUri);
        _builder_2.append("\", \"");
        String _name = literal.getName();
        _builder_2.append(_name);
        _builder_2.append("\"))");
        target.append(_builder_2);
      }
    }
    if (!_matched) {
      if (key instanceof EDataTypeInSlotsKey) {
        _matched=true;
        final EDataType literal = ((EDataTypeInSlotsKey)key).getEmfKey();
        final String packageNsUri = literal.getEPackage().getNsURI();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(EDataTypeInSlotsKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("((");
        target.append(_builder_1);
        target.append(EDataType.class);
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append(")");
        CharSequence _classifierGetterName = this.classifierGetterName(forParameter);
        _builder_2.append(_classifierGetterName);
        _builder_2.append("(\"");
        _builder_2.append(packageNsUri);
        _builder_2.append("\", \"");
        String _name = literal.getName();
        _builder_2.append(_name);
        _builder_2.append("\"))");
        target.append(_builder_2);
      }
    }
    if (!_matched) {
      if (key instanceof JavaTransitiveInstancesKey) {
        _matched=true;
        final String clazz = ((JavaTransitiveInstancesKey)key).getPrettyPrintableName();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("new ");
        target.append(_builder);
        target.append(JavaTransitiveInstancesKey.class);
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("(");
        _builder_1.append(clazz);
        _builder_1.append(".class)");
        target.append(_builder_1);
      }
    }
    if (!_matched) {
      if (Objects.equal(key, BottomTypeKey.class)) {
        _matched=true;
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("unknown type");
        target.append(_builder);
      }
    }
    if (!_matched) {
      if (Objects.equal(key, null)) {
        _matched=true;
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append("null");
        target.append(_builder_1);
      }
    }
  }
  
  private CharSequence classifierGetterName(final boolean forParameter) {
    CharSequence _xifexpression = null;
    if (forParameter) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("getClassifierLiteralSafe");
      _xifexpression = _builder;
    } else {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("getClassifierLiteral");
      _xifexpression = _builder_1;
    }
    return _xifexpression;
  }
}
