package jp.sourceforge.acerola3d.a3;

import java.awt.*;
import java.awt.event.KeyListener;
import javax.vecmath.*;
import java.io.*;

/**
 * A3CanvasとJA3Canvasを統一的に扱うために
 * 導入されたインタフェース。
 */
public interface A3CanvasInterface {

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a);

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a);

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll();

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a);

    /**
     * 背景を削除します。
     */
    public void delBackground();

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a);

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar();

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l);

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l);

    /**
     * KeyListenerを登録します。
     */
    public void addKeyListener(KeyListener l);
    /**
     * 指定されたKeyListenerの登録を抹消します。
     */
    public void removeKeyListener(KeyListener l);

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z);

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z);

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot);

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s);

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera();

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z);

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc);

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z);

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc);

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc();

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w);

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat);

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat();

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z);

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z);

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot);

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s);

    /**
     * カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s);

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale();

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b);
    // マウスナビゲーションのモード設定
    /**
     * ナビゲーションモードです。主に3D仮想空間をマウスで観覧する時の
     * モードを設定するための列挙型ですが、A3オブジェクトの位置などを
     * 編集できるモードも含まれます。ナビゲーションモードを設定すると、
     * a3パッケージがカメラやA3Objectの座標や回転などをコントロール
     * するようになるので、A3Canvas.setCameraLoc()や
     * A3Object.setLoc()などのメソッドと併用する時は注意が必要です。
     * 将来他のモードが追加される可能性があります。
     */
    public enum NaviMode {
        /** a3パッケージがナビゲーションに関与しないモードです。 */
        NONE,
        /** 特定のa3オブジェクトに着目してそれを中心に観覧するモードです。 */
        EXAMINE,
        /** 平面を歩いているようなナビゲーションモードです。 */
        WALK,
        /** 空間を自由に飛ぶようなナビゲーションモードです。 */
        FLY,
        /** 選択されたオブジェクトの位置、回転、大きさを変更できるモードです。 */
        EDIT,
        /** EXAMINEモードと似ていますが、注視点が任意に変更できる観覧モードです。 */
        SIMPLE,
        /** アバタの後方からアバタを追跡するような観覧モードです。A23.setDefaultUpperDirection()などを
         * 用いてZ軸が上になる座標系に設定している場合には正常に機能しないので、ChaseControllerを
         * 用いる方法を検討して下さい。 */
        CHASE,
        /** ユーザ定義のナビゲーションモードです。 */
        USER
    }

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m);

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s);

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed();

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c);
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * アバタの座標を利用して、スクリーン上の点(x,y)を仮想空間上の
     * 点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直な仮想の面を想定して
     * その面上にあると仮定して計算します。その仮想の面は
     * A3CanvasInterfaceに設定されたアバタの座標を含むように
     * カメラからの距離が設定されます。
     */
    public Point3d canvasToVirtualCS(int x,int y);

    /**
     * スクリーン上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直で距離がdisとなる仮想の
     * 面上にあるとして計算するようになっています。この距離disは
     * <em>物理座標系</em>における長さで指定して下さい。
     * 物理座標系における長さとなるので、カメラの拡大縮小率を
     * 1.0以外に設定している場合には注意して下さい。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis);

    /**
     * アバタの座標を利用して、スクリーン上の点(x,y)を物理空間上の
     * 点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直な仮想の面を想定して
     * その面上にあると仮定して計算します。その仮想の面は
     * A3CanvasInterfaceに設定されたアバタの座標を含むように
     * カメラからの距離が設定されます。つまり変換後のZ座標は
     * アバタの物理空間におけるZ座標と等しくなります。
     */
    public Point3d canvasToPhysicalCS(int x,int y);

    /**
     * スクリーン上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * 変換後の座標は、カメラ正面方向に垂直で距離がdisとなる仮想の
     * 面の上にあるとして計算するようになっています。この距離disは
     * 物理座標系における長さで指定して下さい。つまり変換後の
     * z座標は必ず-disになります。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis);

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v);

    /**
     * 物理空間上の点(x,y,z)をスクリーン上の点(x,y)に変換します。
     */
    public Point physicalCSToCanvas(Point3d p);

    /**
     * 仮想空間上の点(x,y,z)をスクリーン上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p);

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v);

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX();

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY();

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ();

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y);
//  ----------J3DGraphics2D(文字描画など)---------
    /**
     * A3Canvasに2Dの描画を行うためのComponent2Dを登録します。
     */
    public void add(Component2D c);
    /**
     * A3Canvasから2Dの描画を行うためのComponent2Dを削除します。
     */
    public void del(Component2D c);
    /**
     * シーンを指定して、A3Canvasに2Dの描画を行うためのComponent2Dを登録します。
     */
    public void add(Component2D c,int scene);
    /**
     * シーンを指定して、A3Canvasから2Dの描画を行うためのComponent2Dを削除します。
     */
    public void del(Component2D c,int scene);
//  ----------おまけ機能---------
    /**
     * A3Canvasに表示されている内容をPNG画像としてファイルに保存します。
     */
    public void saveImage(File file) throws IOException;
//  ----------シーン関係のメソッド---------
    /**
     * シーンを準備します。すでにその番号のシーンが
     * 存在していれば何もしません(既存の内容はそのまま)。
     */
    public void prepareScene(int s);
    /**
     * シーンを切り替えます。
     */
    public void changeActiveScene(int s);
    /**
     * シーンを指定してA3Objectを追加します。
     */
    public void add(A3Object a,int s);
    /**
     * シーンを指定して、A3Objectを削除します。
     */
    public void del(A3Object a,int s);
    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z,int scene);

    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc,int scene);

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w,int scene);

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat,int scene);

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z,int scene);

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot,int scene);

    /**
     * シーンを指定して、カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s,int scene);

    /**
     * シーンを指定して、カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera(int scene);

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z,int scene);

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc,int scene);

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z,int scene);

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc,int scene);

    /**
     * シーンを指定して、カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc(int scene);

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w,int scene);

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat,int scene);

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w,int scene);

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat,int scene);

    /**
     * シーンを指定して、カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat(int scene);

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z,int scene);

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot,int scene);

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z,int scene);

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot,int scene);

    /**
     * シーンを指定して、カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s,int scene);

    /**
     * シーンを指定して、カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s,int scene);

    /**
     * シーンを指定して、カメラの拡大率を返します。
     */
    public double getCameraScale(int scene);
    /**
     * シーンを指定して、ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m,int scene);

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s,int scene);

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed(int scene);

    /**
     * シーンを指定して、A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c,int scene);

    /**
     * シーンを指定して、背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a,int scene);

    /**
     * シーンを指定して、背景を削除します。
     */
    public void delBackground(int scene);

    /**
     * シーンを指定して、アバタをセットします。
     */
    public void setAvatar(A3Object a,int scene);

    /**
     * シーンを指定して、セットされたアバタを取得します。
     */
    public A3Object getAvatar(int scene);
//  ---------- LockedA3の処理 ----------
    /**
     * A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    public void addLockedA3(A3Object a);
    /**
     * 指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    public void delLockedA3(A3Object a);
    /**
     * カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    public void delAllLockedA3();
    /**
     * シーンを指定して、A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    public void addLockedA3(A3Object a,int scene);
    /**
     * シーンを指定して、指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    public void delLockedA3(A3Object a,int scene);
    /**
     * シーンを指定して、カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    public void delAllLockedA3(int scene);
    /**
     * 上方向をY軸とするのかZ軸とするのかの変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d);
    /**
     * シーンを指定して、上方向をY軸とするのかZ軸とするのかの
     * 変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d,int scene);
    /**
     * 上方向をY軸としているのかZ軸としているのかを得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection();
    /**
     * シーンを指定して、上方向をY軸としているのかZ軸としているのか
     * を得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection(int scene);
}
