package jp.sourceforge.acerola3d.a3;

import java.awt.*;

import javax.media.j3d.*;
import javax.vecmath.*;

import java.util.*;

import java.lang.reflect.Constructor;
import jp.sourceforge.acerola3d.a3.A3CanvasInterface.NaviMode;

/**
 * A3CanvasとJA3Canvasの両方で共通して使うことが
 * できるVirtualUniverseのAcerola3Dバージョン。
 */
class A3VirtualUniverse extends VirtualUniverse {
    A3CanvasInterface canvas = null;
    Canvas3D canvas3d = null;
    javax.media.j3d.Locale locale;
    javax.media.j3d.View view;
    BranchGroup rootGroup;
    TransformGroup tGroup;
    Transform3D transform;
    BranchGroup vpGroup;
    ViewPlatform vp;
    HashMap<Integer,Scene> scenes = new HashMap<Integer,Scene>();
    Scene scene;
    TimerBehavior timerBehavior;
    PickingBehavior pickingBehavior;
    CameraBehavior cameraBehavior;
    Light headLight;

    A3VirtualUniverse(A3CanvasInterface c) {
        canvas = c;
        if (canvas instanceof A3Canvas)
            init((Canvas3D)c);
        else if (canvas instanceof JA3Canvas)
            init(((JA3Canvas)canvas).getOffscreenCanvas3D());
        else if (canvas instanceof A3Widget)
            init(((A3Widget)canvas).tCanvas.getOffscreenCanvas3D());
    }
    void init(Canvas3D c3d) {
        canvas3d = c3d;
        locale = new javax.media.j3d.Locale(this);

        PhysicalBody body = new PhysicalBody();
        PhysicalEnvironment environment = new PhysicalEnvironment();

        view = new javax.media.j3d.View();
        view.addCanvas3D(canvas3d);
        view.setPhysicalBody(body);
        view.setPhysicalEnvironment(environment);
        view.setFrontClipDistance(0.1);
        view.setBackClipDistance(100.0);
        view.setUserHeadToVworldEnable(true);

        vpGroup = new BranchGroup();
        tGroup = new TransformGroup();
        tGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        tGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        tGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
        tGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
        vpGroup.addChild(tGroup);
        transform = new Transform3D();
        tGroup.setTransform(transform);
        timerBehavior = new TimerBehavior();
        BoundingSphere bs = new BoundingSphere(new Point3d(0.0,0.0,0.0),
                                               Double.MAX_VALUE);
        timerBehavior.setSchedulingBounds(bs);
        tGroup.addChild(timerBehavior);
        vp = new ViewPlatform();
//System.out.println(vp.getActivationRadius());
        tGroup.addChild(vp);
        headLight = new DirectionalLight();
        headLight.setCapability(Light.ALLOW_STATE_WRITE);
        headLight.setInfluencingBounds(new BoundingSphere(new Point3d(0.0,0.0,0.0),Double.MAX_VALUE));
        headLight.setEnable(true);
        tGroup.addChild(headLight);

        view.attachViewPlatform(vp);

//        vpGroup.setCapability(BranchGroup.ALLOW_DETACH);
//        vpGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);
//        vpGroup.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
//        vpGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);

        pickingBehavior = new PickingBehavior(this,view,tGroup,locale);
        vpGroup.addChild(pickingBehavior);
        BoundingSphere sb = new BoundingSphere(new Point3d(0.0,0.0,0.0),Double.MAX_VALUE);
        pickingBehavior.setSchedulingBounds(sb);
        cameraBehavior = new CameraBehavior(this);
        vpGroup.addChild(cameraBehavior);
        cameraBehavior.setSchedulingBounds(sb);

        rootGroup = new BranchGroup();
        //rootGroup.setCapability(BranchGroup.ALLOW_DETACH);
        //rootGroup.setCapability(BranchGroup.ALLOW_CHILDREN_READ);
        rootGroup.setCapability(BranchGroup.ALLOW_CHILDREN_EXTEND);
        rootGroup.setCapability(BranchGroup.ALLOW_CHILDREN_WRITE);

        rootGroup.addChild(vpGroup);
        
        locale.addBranchGraph(rootGroup);

        AudioDevice mixer = null;
        String mixerClassName = System.getProperty("j3d.audiodevice");
        System.out.println(mixerClassName);
        if (mixerClassName!=null) {
            try {
                Class<?> c = Class.forName(mixerClassName);
                Class<? extends AudioDevice> mixerClass = c.asSubclass(AudioDevice.class);
                Constructor<? extends AudioDevice> constructor = mixerClass.getConstructor(PhysicalEnvironment.class);
                mixer = constructor.newInstance(environment);
            } catch(Exception e) {
                e.printStackTrace();
            }
        }
        if (mixer==null) {
            try {
                mixer = new org.jdesktop.j3d.audioengines.joal.JOALMixer(environment);
            } catch(Exception e) {
                ;
            }
        }
        /*
        if (mixer==null) {
            try {
                //mixer = new com.sun.j3d.audioengines.headspace.HeadspaceMixer(environment);
            } catch(Exception e) {
                ;
            }
        }
        */
        if (mixer==null) {
            try {
                mixer = new com.sun.j3d.audioengines.javasound.JavaSoundMixer(environment);
            } catch(Exception e) {
                ;
            }
        }
        if (mixer == null) {
            System.out.println("null AudioDevice!");
        } else {
            System.out.println(mixer.getClass().getName());
            environment.setAudioDevice(mixer);
            mixer.initialize();
        }

        scene = new Scene(this,0);
        scenes.put(0,scene);
        //rootGroup.addChild(scene.mainGroup);
        scene.activate();
    }

    TimerBehavior getTimerBehavior() {
        return timerBehavior;
    }

    void prepareVirtualUniverse() {
    }

    // A3Objectの追加と削除
    /**
     * A3Objectを追加して表示されるようにします。
     */
    public void add(A3Object a) {
        scene.add(a);
    }

    /**
     * 指定されたA3Objectの登録を削除して表示されないように
     * します。
     */
    public void del(A3Object a) {
        scene.del(a);
    }

    /**
     * 登録されている全てのA3Objectを削除して表示されないようにします。
     *
     */
    public void delAll() {
        scene.delAll();
    }

    /**
     * 背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a) {
        scene.setBackground(a);
    }

    /**
     * 背景を削除します。
     */
    public void delBackground() {
        scene.delBackground();
    }

    /**
     * アバタをセットします。
     */
    public void setAvatar(A3Object a) {
        scene.setAvatar(a);
    }

    /**
     * セットされたアバタを取得します。
     */
    public A3Object getAvatar() {
        return scene.getAvatar();
    }

    // リスナ設定のラッパーメソッド
    /**
     * A3Listenerを登録します。
     */
    public void addA3Listener(A3Listener l) {
        pickingBehavior.addA3Listener(l);
    }

    /**
     * 指定されたA3Listenerの登録を抹消します。
     */
    public void removeA3Listener(A3Listener l) {
        pickingBehavior.removeA3Listener(l);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z) {
        scene.setDefaultCameraLoc(x,y,z);
    }

    /**
     * カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc) {
        scene.setDefaultCameraLoc(loc);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double ay,double z,double w) {
        scene.setDefaultCameraQuat(x,ay,z,w);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat) {
        scene.setDefaultCameraQuat(quat);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z) {
        scene.setDefaultCameraRot(x,y,z);
    }

    /**
     * カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot) {
        scene.setDefaultCameraRot(rot);
    }

    /**
     * カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s) {
        scene.setDefaultCameraScale(s);
    }

    /**
     * カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera() {
        scene.resetCamera();
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z) {
        scene.setCameraLoc(x,y,z);
    }

    /**
     * カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc) {
        scene.setCameraLoc(loc);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z) {
        scene.setCameraLocImmediately(x,y,z);
    }

    /**
     * カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc) {
        scene.setCameraLocImmediately(loc);
    }

    /**
     * カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc() {
        return scene.getCameraLoc();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w) {
        scene.setCameraQuat(x,y,z,w);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat) {
        scene.setCameraQuat(quat);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w) {
        scene.setCameraQuatImmediately(x,y,z,w);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat) {
        scene.setCameraQuatImmediately(quat);
    }

    /**
     * カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat() {
        return scene.getCameraQuat();
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z) {
        scene.setCameraRot(x,y,z);
    }

    /**
     * カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot) {
        scene.setCameraRot(rot);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z) {
        scene.setCameraRotImmediately(x,y,z);
    }

    /**
     * カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot) {
        scene.setCameraRotImmediately(rot);
    }

    /**
     * カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。
     */
    public void setCameraScale(double s) {
        scene.setCameraScale(s);
    }

    /**
     * カメラの拡大率を即時に指定します。
     */
    public void setCameraScaleImmediately(double s) {
        scene.setCameraScaleImmediately(s);
    }

    /**
     * カメラの拡大率を返します。
     */
    public double getCameraScale() {
        return scene.getCameraScale();
    }

    /**
     * ヘッドライトのON,OFFを設定します。
     */
    public void setHeadLightEnable(boolean b) {
        headLight.setEnable(b);
    }

    /**
     * ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m) {
        scene.setNavigationMode(m);
    }

    /**
     * ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s) {
        scene.setNavigationSpeed(s);
    }

    /**
     * ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed() {
        return scene.getNavigationSpeed();
    }

    /**
     * A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c) {
        scene.setA3Controller(c);
    }
//  ----------座標変換とピッキングのためのラッパーメソッド---------
    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算してます。
     */
    public Point3d canvasToVirtualCS(int x,int y) {
        return pickingBehavior.canvasToVirtualCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を仮想空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToVirtualCS(int x,int y,double dis) {
        return pickingBehavior.canvasToVirtualCS(x,y,dis);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * avatarとcameraの距離を利用して計算しています。
     */
    public Point3d canvasToPhysicalCS(int x,int y) {
        return pickingBehavior.canvasToPhysicalCS(x,y);
    }

    /**
     * A3Canvas上の点(x,y)を物理空間上の点(x,y,z)に変換します。
     * カメラとの距離をdis引数で指定された値を用いて計算します。
     */
    public Point3d canvasToPhysicalCS(int x,int y,double dis) {
        return pickingBehavior.canvasToPhysicalCS(x,y,dis);
    }

    /**
     * 物理空間上の点(x,y,z)を仮想空間上の点(x,y,z)に変換します。
     */
    public Vector3d physicalCSToVirtualCS(Vector3d v) {
        Point3d p = pickingBehavior.physicalCSToVirtualCS(new Point3d(v));
        return new Vector3d(p);
    }

    /**
     * 物理空間上の点(x,y,z)をA3Canvas上の点(x,y)に変換します。
     */
    public Point physicalCSToCanvas(Point3d p) {
        return pickingBehavior.physicalCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)をA3Canvas上の点(x,y)に変換します。
     */
    public Point virtualCSToCanvas(Point3d p) {
        return pickingBehavior.virtualCSToCanvas(p);
    }

    /**
     * 仮想空間上の点(x,y,z)を物理空間上の点(x,y,z)に変換します。
     */
    public Vector3d virtualCSToPhysicalCS(Vector3d v) {
        Point3d p = pickingBehavior.virtualCSToPhysicalCS(new Point3d(v));
        return new Vector3d(p);
    }

    /**
     * カメラの座標系(物理空間の座標系)のX軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecX() {
        return physicalCSToVirtualCS(new Vector3d(1.0,0.0,0.0));
    }

    /**
     * カメラの座標系(物理空間の座標系)のY軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecY() {
        return physicalCSToVirtualCS(new Vector3d(0.0,1.0,0.0));
    }

    /**
     * カメラの座標系(物理空間の座標系)のZ軸方向の単位ベクトルを
     * 仮想空間の座標系で表したベクトルを返します。
     */
    public Vector3d getCameraUnitVecZ() {
        return physicalCSToVirtualCS(new Vector3d(0.0,0.0,1.0));
    }

    /**
     * A3Canvas上の点(x,y)にあるA3Objectをピックアップします。
     */
    public A3Object pickA3(int x,int y) {
        return pickingBehavior.pickA3(x,y);
    }
//  ----------シーン関係のメソッド---------
    /**
     * シーンを準備します。すでにその番号のシーンが
     * 存在していれば何もしません(既存の内容はそのまま)。
     */
    public void prepareScene(int scene) {
        Scene s = scenes.get(scene);
        if (s==null) {
            s = new Scene(this,scene);
            scenes.put(scene,s);
        }
    }
    /**
     * シーンを切り替えます。
     */
    public void changeActiveScene(int s) {
        Scene newScene = scenes.get(s);
        if (newScene==null) {
            throw new IllegalArgumentException();
        }
        scene.deactivate();
        scene = newScene;
        //カメラをむりやり変える。本当はCameraBehaviorでやるべき。
        transform.set(scene.cameraNowQ,scene.cameraNowV,scene.cameraNowS);
        tGroup.setTransform(transform);
        scene.activate();
    }
    /**
     * シーンを指定してA3Objectを追加します。
     */
    public void add(A3Object a,int s) {
        Scene tmpScene = scenes.get(s);
        if (tmpScene!=null) {
            tmpScene.add(a);
        } else {
            throw new IllegalArgumentException();
        }
    }
    /**
     * シーンを指定してA3Objectを削除します。
     */
    public void del(A3Object a,int s) {
        Scene tmpScene = scenes.get(s);
        if (tmpScene!=null)
            tmpScene.del(a);
        else
            throw new IllegalArgumentException();
    }
    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(double x,double y,double z,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setDefaultCameraLoc(x,y,z);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラのデフォルトの位置を指定します。
     */
    public void setDefaultCameraLoc(Vector3d loc,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setDefaultCameraLoc(loc);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(double x,double y,double z,double w,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setDefaultCameraQuat(x,y,z,w);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraQuat(Quat4d quat,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setDefaultCameraQuat(quat);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(double x,double y,double z,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setDefaultCameraRot(x,y,z);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラのデフォルトの回転を指定します。
     */
    public void setDefaultCameraRot(Vector3d rot,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setDefaultCameraRot(rot);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラのデフォルトの拡大率を指定します。
     */
    public void setDefaultCameraScale(double s,int scene) {
        Scene sTmp = scenes.get(scene);
        if (sTmp!=null)
            sTmp.setDefaultCameraScale(s);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの位置、回転、拡大率をリセットしてデフォルトに戻します。
     */
    public void resetCamera(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.resetCamera();
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(double x,double y,double z,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraLoc(x,y,z);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの位置を指定します。自動的に補完が働き滑らかにカメラの位置が
     * 変ります。
     */
    public void setCameraLoc(Vector3d loc,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraLoc(loc);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(double x,double y,double z,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraLocImmediately(x,y,z);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの位置を即時に指定します。
     */
    public void setCameraLocImmediately(Vector3d loc,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraLocImmediately(loc);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの現在位置を返します。
     */
    public Vector3d getCameraLoc(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            return s.getCameraLoc();
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(double x,double y,double z,double w,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraQuat(x,y,z,w);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraQuat(Quat4d quat,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraQuat(quat);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(double x,double y,double z,double w,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraQuatImmediately(x,y,z,w);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraQuatImmediately(Quat4d quat,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraQuatImmediately(quat);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの現在の回転を返します。
     */
    public Quat4d getCameraQuat(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            return s.getCameraQuat();
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(double x,double y,double z,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraRot(x,y,z);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を指定します。自動的に補完が働き滑らかにカメラの回転が
     * 変ります。
     */
    public void setCameraRot(Vector3d rot,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraRot(rot);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(double x,double y,double z,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraRotImmediately(x,y,z);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの回転を即時に指定します。
     */
    public void setCameraRotImmediately(Vector3d rot,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setCameraRotImmediately(rot);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの拡大率を指定します。自動的に補完が働き滑らかにカメラの拡大率が
     * 変ります。拡大率がデフォルトの1.0の時は10cmより手前と100mより奥はクリッピングされて
     * 表示されません。拡大率を0.1にすれば1cmから10mの間を表示できるようになり、
     * 10.0にすれば1mから1kmの間を表示できるようになります。
     */
    public void setCameraScale(double s,int scene) {
        Scene sTmp = scenes.get(scene);
        if (sTmp!=null)
            sTmp.setCameraScale(s);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの拡大率を即時に指定します。拡大率がデフォルトの1.0の時は10cmより
     * 手前と100mより奥はクリッピングされて表示されません。拡大率を0.1にすれば
     * 1cmから10mの間を表示できるようになり、10.0にすれば1mから1kmの間を
     * 表示できるようになります。
     */
    public void setCameraScaleImmediately(double s,int scene) {
        Scene sTmp = scenes.get(scene);
        if (sTmp!=null)
            sTmp.setCameraScaleImmediately(s);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、カメラの拡大率を返します。
     */
    public double getCameraScale(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            return s.getCameraScale();
        else
            throw new IllegalArgumentException();
    }
    /**
     * シーンを指定して、ナビゲーションモードを指定します。
     */
    public void setNavigationMode(NaviMode m,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setNavigationMode(m);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを設定します。
     * A3Controllerの作成者はこのスピードを参照して
     * ナビゲーションのスピードを計算することが望まれます．
     */
    public void setNavigationSpeed(double s,int scene) {
        Scene sTmp = scenes.get(scene);
        if (sTmp!=null)
            sTmp.setNavigationSpeed(s);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、ナビゲーションの大まかなスピードを取得します。
     */
    public double getNavigationSpeed(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            return s.getNavigationSpeed();
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、A3Controllerをセットします。これをセットするとナビゲーションモードが
     * USERに自動的にセットされるので、以前設定していたモードは無効になります。
     */
    public void setA3Controller(A3Controller c,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setA3Controller(c);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、背景を表すA3Objectをセットします。
     */
    public void setBackground(A3Object a,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setBackground(a);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、背景を削除します。
     */
    public void delBackground(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.delBackground();
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、アバタをセットします。
     */
    public void setAvatar(A3Object a,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setAvatar(a);
        else
            throw new IllegalArgumentException();
    }

    /**
     * シーンを指定して、セットされたアバタを取得します。
     */
    public A3Object getAvatar(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            return s.getAvatar();
        else
            throw new IllegalArgumentException();
    }
//  ---------- Component2Dの処理 ----------
    public void add(Component2D c) {
        scene.add(c);
    }
    public void del(Component2D c) {
        scene.del(c);
    }
    public void add(Component2D c,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.add(c);
        else
            throw new IllegalArgumentException();
    }
    public void del(Component2D c,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.del(c);
        else
            throw new IllegalArgumentException();
    }
    ArrayList<Component2D> getComponents2D() {
        return scene.getComponents2D();
    }
//  ---------- LockedA3の処理 ----------
    /**
     * A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    void addLockedA3(A3Object a) {
        scene.addLockedA3(a);
    }
    /**
     * 指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    void delLockedA3(A3Object a) {
        scene.delLockedA3(a);
    }
    /**
     * カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    void delAllLockedA3() {
        scene.delAllLockedA3();
    }
    /**
     * シーンを指定して、A3Objectを追加してカメラに対して固定した位置に
     * 表示されるようにします。
     */
    void addLockedA3(A3Object a,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.addLockedA3(a);
        else
            throw new IllegalArgumentException();
    }
    /**
     * シーンを指定して、指定されたA3Objectの登録を削除してカメラに対して固定した
     * 位置に表示されないようにします。
     */
    void delLockedA3(A3Object a,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.delLockedA3(a);
        else
            throw new IllegalArgumentException();
    }
    /**
     * シーンを指定して、カメラに対して固定して表示されるようい登録されている
     * 全てのA3Objectを削除して表示されないようにします。
     */
    void delAllLockedA3(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.delAllLockedA3();
        else
            throw new IllegalArgumentException();
    }
    /**
     * 上方向をY軸とするのかZ軸とするのかの変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d) {
        scene.setUpperDirection(d);
    }
    /**
     * シーンを指定して、上方向をY軸とするのかZ軸とするのかの
     * 変更を行う。
     * デフォルトはY軸で、この場合は特に何もかわらないが、
     * Z軸が設定された場合は表示されるA3Objectが
     * 自動的に回転されて正常な向きで表示されるようになる。
     */
    public void setUpperDirection(A3Object.UpperDirection d,int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            s.setUpperDirection(d);
        else
            throw new IllegalArgumentException();
    }
    /**
     * 上方向をY軸としているのかZ軸としているのかを得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection() {
        return scene.upperDirection;
    }
    /**
     * シーンを指定して、上方向をY軸としているのかZ軸としているのか
     * を得るための関数です。
     */
    public A3Object.UpperDirection getUpperDirection(int scene) {
        Scene s = scenes.get(scene);
        if (s!=null)
            return s.upperDirection;
        else
            throw new IllegalArgumentException();
    }
}
